﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectLockCounter.h"

#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace visrv{ namespace indirect{

    IndirectLockCounter::IndirectLockCounter() NN_NOEXCEPT
    {
        m_Count = 0;
        m_MainThreadId = 0;
        std::memset(&m_Mutex, 0, sizeof(m_Mutex));
        std::memset(&m_ConditionVariable, 0, sizeof(m_ConditionVariable));
    }

    void IndirectLockCounter::Initialize(nn::os::ThreadId mainThreadId) NN_NOEXCEPT
    {
        m_Count = 0;
        m_MainThreadId = mainThreadId;
        nn::os::InitializeMutex(&m_Mutex, false, 0);
        nn::os::InitializeConditionVariable(&m_ConditionVariable);
    }

    void IndirectLockCounter::Finalize() NN_NOEXCEPT
    {
        nn::os::FinalizeConditionVariable(&m_ConditionVariable);
        nn::os::FinalizeMutex(&m_Mutex);
    }

    void IndirectLockCounter::AcquireLock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_MainThreadId, nn::os::GetThreadId(nn::os::GetCurrentThread()));

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            m_Count++;
        }
        nn::os::SignalConditionVariable(&m_ConditionVariable);
    }

    void IndirectLockCounter::ReleaseLock() NN_NOEXCEPT
    {
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            m_Count--;
            NN_SDK_ASSERT_GREATER_EQUAL(m_Count, 0);
        }
        nn::os::SignalConditionVariable(&m_ConditionVariable);
    }

    void IndirectLockCounter::WaitUnlocked() NN_NOEXCEPT
    {
        // TORIAEZU:
        NN_SDK_REQUIRES_EQUAL(m_MainThreadId, nn::os::GetThreadId(nn::os::GetCurrentThread()));

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            while(m_Count > 0)
            {
                nn::os::WaitConditionVariable(&m_ConditionVariable, &m_Mutex);
            }
        }
    }

    bool IndirectLockCounter::IsLocked() NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        return m_Count != 0;
    }

}}}
