﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/applet/applet_FundamentalTypes.h>

#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_CropRegion.h>
#include <nn/vi/vi_ImageTransform.h>
#include <nn/vi/native/vi_NativeType.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerTextureIndexList.h>
#include <nn/vi/fbshare/vi_SharedMemoryPoolLayout.h>
#include <nn/vi/fbshare/vi_SharedTextureOption.h>
#include "../visrv_Config.h"
#include "../visrv_ResourceIdManagement.h"
#include "../util/visrv_TPointerContainer.h"


namespace nn{ namespace visrv{ namespace client{
    class ClientObject;
}}}


namespace nn{ namespace visrv{ namespace fbshare{
    class SharingClientObject;
    class SharedBuffer;
    namespace detail{
        class SharedClientLayer;
        class SharedLowLevelLayer;
    }

    class SharingClientImpl
    {
    public:
        // TORIAEZU: 1 人で最大数作れるようにしておく
        static const int OwningSharedBufferListSize      = SharedBufferCountMax;
        // 最大数作れるようにしておく
        static const int OwningSharedLowLevelLayerListSize =  OwningSharedBufferListSize * SharedLowLevelLayerCountMaxPerSharedBuffer;
        // TORIAEZU: 1 人で最大数作れるようにしておく
        static const int OwningSharedClientLayerListSize = SharedClientLayerCountMax;
        static const int BoundSharedClientLayerListSize  = SharedClientLayerCountMaxPerClient;

    public:
        class OwningSharedBufferList
            : public util::TPointerContainer<SharedBuffer, OwningSharedBufferListSize, util::ContainerThreadSafetyOption_SingleThread>
        {
        public:
            nn::Result Find(SharedBuffer** pOutValue, nn::vi::fbshare::SharedBufferHandle h) NN_NOEXCEPT;
        };

        class OwningSharedLowLevelLayerList
            : public util::TPointerContainer<detail::SharedLowLevelLayer, OwningSharedLowLevelLayerListSize, util::ContainerThreadSafetyOption_SingleThread>
        {
        public:
            nn::Result Find(detail::SharedLowLevelLayer** pOutValue, nn::vi::LayerId layerId) NN_NOEXCEPT;
        };

        class OwningSharedClientLayerList
            : public util::TPointerContainer<detail::SharedClientLayer, OwningSharedClientLayerListSize, util::ContainerThreadSafetyOption_SingleThread>
        {
        public:
            nn::Result Find(detail::SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;
        };

        class BoundSharedClientLayerList
            : public util::TPointerContainer<detail::SharedClientLayer, BoundSharedClientLayerListSize, util::ContainerThreadSafetyOption_SingleThread>
        {
        public:
            nn::Result Find(detail::SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;
        };

    public:
        SharingClientImpl() NN_NOEXCEPT;

        void Initialize(client::ClientObject* pSelf) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        client::ClientObject* GetSelf() NN_NOEXCEPT;
        const client::ClientObject* GetSelf() const NN_NOEXCEPT;

    private:
        friend class SharingClientObject;
        client::ClientObject* m_pSelf;

        // refs to SharedBuffer instances created by this client
        OwningSharedBufferList m_OwningSharedBufferList;

        // refs to SharedLowLevelLayer instatnces created by this client
        OwningSharedLowLevelLayerList m_OwningSharedLowLevelLayerList;

        // refs to SharedClientLayer instances created by this client
        OwningSharedClientLayerList m_OwningSharedClientLayerList;

        // refs to SharedClientLayer instances bound to this client
        BoundSharedClientLayerList m_BoundSharedClientLayerList;
    };


    class SharingClientObject
    {
    public:
        SharingClientObject() NN_NOEXCEPT;

        void InitializeSharing(client::ClientObject* pSelf) NN_NOEXCEPT;
        void FinalizeSharing() NN_NOEXCEPT;

        bool IsAlive() const NN_NOEXCEPT;

    // SharedBuffer --------------------------------------
    public:
        nn::Result CreateSharedBufferStaticStorage(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            uint64_t storageKey,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout
        ) NN_NOEXCEPT;

        nn::Result CreateSharedBufferTransferMemory(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            nn::sf::NativeHandle& transferMemoryHandle,
            size_t transferMemorySize,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout
        ) NN_NOEXCEPT;

        nn::Result CreateSharedBufferProcessHeap(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            ResourceId blockId,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout
        ) NN_NOEXCEPT;

        nn::Result DestroySharedBuffer(nn::vi::fbshare::SharedBufferHandle handle) NN_NOEXCEPT;

        nn::Result RegiterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;
        nn::Result UnregiterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;
        nn::Result GetSharedBufferMemoryHandleId(nn::vi::native::NativeMemoryHandleId* pOutMemHandleId, size_t* pOutSize, nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout, nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;

    private:
        void DestroySharedBufferImpl(SharedBuffer* p) NN_NOEXCEPT;

    // SharedLowLevelLayer ---------------------------------------
    public:
        // SharedLowLevelLayer に ManagedLayer をバインドする。
        // @pre LowLevelLayer は Unbound
        // @post LowLevelLayer は Bound
        nn::Result BindSharedLowLevelLayerToManagedLayer(
            nn::vi::LayerId layerId,
            const char* displayName,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

        // SharedLowLevelLayer に IndirectLayer をバインドする。
        // @pre LowLevelLayer は Unbound
        // @post LowLevelLayer は Bound
        nn::Result BindSharedLowLevelLayerToIndirectLayer(
            nn::vi::IndirectProducerHandleType handle,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

        // SharedLowLevelLayer から ManagedLayer をアンバインドする。
        // @pre LowLevelLayer は Unbound
        // @post LowLevelLayer は Bound
        nn::Result UnbindSharedLowLevelLayer(
            nn::vi::LayerId layerId
        ) NN_NOEXCEPT;

        // LowLevelLayer を SharedBuffer に接続する。
        // @pre LowLevelLayer は Bound
        // @post LowLevelLayer が Connected
        nn::Result ConnectSharedLowLevelLayerToSharedBuffer(
            nn::vi::LayerId layerId,
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer
        ) NN_NOEXCEPT;

        // LowLevelLayer を SharedBuffer から切断する。
        // @pre LowLevelLayer は Connected
        // @post LowLevelLayer は Bound
        nn::Result DisconnectSharedLowLevelLayerFromSharedBuffer(
            nn::vi::LayerId layerId
        ) NN_NOEXCEPT;

        nn::Result GetSharedLowLevelLayerSynchronizedEvent(
            nn::sf::NativeHandle& outNativeHandle,
            nn::vi::LayerId layerId
        ) NN_NOEXCEPT;

        nn::Result CheckSharedLayerSynchronized(
            int* pOutDisplayedFrameBufferIndex,
            nn::vi::LayerId layerId
        ) NN_NOEXCEPT;

    private:
        void UnbindSharedLowLevelLayerImpl(
            detail::SharedLowLevelLayer* pLowLayer
        ) NN_NOEXCEPT;

        void DisconnectSharedLowLevelLayerFromSharedBufferImpl(
            detail::SharedLowLevelLayer* pLowLayer
        ) NN_NOEXCEPT;



    // SharedClientLayer ------------------------------------
    public:
        nn::Result CreateSharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle* pOutHandle,
            nn::applet::AppletResourceUserId userAruid
        ) NN_NOEXCEPT;

        nn::Result DestroySharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle h
        ) NN_NOEXCEPT;

        nn::Result GetSharedClientLayerLayerStacks(
            nn::vi::LayerStackFlagType* pOutValue,
            nn::vi::fbshare::SharedLayerHandle h
        ) NN_NOEXCEPT;

        nn::Result SetSharedClientLayerLayerStacks(
            nn::vi::fbshare::SharedLayerHandle h,
            nn::vi::LayerStackFlagType value
        ) NN_NOEXCEPT;

        nn::Result BindSharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle h,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;
        void DestroySharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT;

        nn::Result UnbindSharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle h
        ) NN_NOEXCEPT;

        nn::Result ConnectSharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle h
        ) NN_NOEXCEPT;

        nn::Result DisconnectSharedClientLayer(
            nn::vi::fbshare::SharedLayerHandle h
        ) NN_NOEXCEPT;

        nn::Result AttachSharedLayerToLowLevelLayer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer,
            ResourceId layerId,
            const nn::vi::fbshare::SharedLayerTextureIndexList& frameBufferIndexList
        ) NN_NOEXCEPT;

        nn::Result StartDetachSharedClientLayerFromLowLevelLayer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result FinishDetachSharedClientLayerFromLowLevelLayer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result GetSharedClientLayerDetachReadyEvent(
            nn::sf::NativeHandle& outSystemEventHandle,
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result ForceDetachSharedLayerFromLowLevelLayer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result AcquireSharedClientLayerFrameBuffer(
            int* pOutFrameBufferIndex,
            nn::vi::native::NativeSync* pOutSync,
            nn::vi::fbshare::SharedLayerTextureIndexList* pOutBufferIndexList,
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result PresentSharedClientLayerFrameBuffer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer,
            int frameBufferIndex,
            const nn::vi::native::NativeSync& sync,
            const nn::vi::CropRegion& crop,
            nn::vi::ImageTransformType transform,
            int presentInterval
        ) NN_NOEXCEPT;

        nn::Result CancelSharedClientLayerFrameBuffer(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer,
            int frameBufferIndex
        ) NN_NOEXCEPT;

        nn::Result GetSharedClientLayerAcquirableEvent(
            nn::sf::NativeHandle& outSystemEventHandle,
            nn::vi::fbshare::SharedLayerHandle hSharedLayer
        ) NN_NOEXCEPT;

        nn::Result FillSharedClientLayerFrameBufferColor(
            nn::vi::fbshare::SharedLayerHandle hSharedLayer,
            int frameBufferIndex,
            int r,
            int g,
            int b,
            int a
        ) NN_NOEXCEPT;

    private:
        void UnbindSharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT;
        void DisconnectSharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT;
        void ForceDetachSharedLayerFromLowLevelLayerImpl(detail::SharedClientLayer* pClientLayer) NN_NOEXCEPT;

    // Detached FrameBuffer Operation ------------------------------------------------------
    public:
        // 指定したインデックスの SharedFrameBuffer を画面に出力する。
        nn::Result PresentDetachedSharedFrameBufferToLowLevelLayer(
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            nn::vi::LayerId layerId,
            int index
        ) NN_NOEXCEPT;

        // 指定したインデックスの SharedFrameBuffer の内容を指定した色で塗りつぶす
        nn::Result FillDetachedSharedFrameBufferColor(
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int frameBufferIndex,
            uint32_t color,
            const nn::vi::fbshare::SharedTextureOption& option
        ) NN_NOEXCEPT;

        // 指定したインデックスの SharedFrameBuffer の画像を読み取る
        nn::Result GetDetachedSharedFrameBufferImage(
            size_t* pOutReadSize,
            void* buffer,
            size_t size,
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int frameBufferIndex
        ) NN_NOEXCEPT;

        // 指定したインデックスの SharedFrameBuffer に画像を書き込む
        nn::Result SetDetachedSharedFrameBufferImage(
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int frameBufferIndex,
            const void* buffer,
            size_t size,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransformType srcTransform
        ) NN_NOEXCEPT;

        // 指定したインデックスの SharedFrameBuffer に画像を書き込む
        nn::Result SetDetachedSharedFrameBufferSubImage(
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int frameBufferIndex,
            int x,
            int y,
            int w,
            int h,
            uint32_t bgColor,
            const void* buffer,
            size_t size,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransformType srcTransform
        ) NN_NOEXCEPT;

        // 起動ロゴを展開する
        nn::Result ExpandStartupLogoOnSharedFrameBuffer(
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int dstIndex,
            const nn::vi::fbshare::SharedTextureOption& option
        );

        // SharedFrameBuffer 間で画像をコピーする
        nn::Result CopyDetachedSharedFrameBufferImage(
            nn::vi::fbshare::SharedBufferHandle hDstSharedBuffer,
            int dstFrameBufferIndex,
            nn::vi::fbshare::SharedBufferHandle hSrcSharedBuffer,
            int srcFrameBufferIndex,
            const nn::vi::fbshare::SharedTextureOption& option,
            nn::vi::LayerStackFlagType stacksFilter,
            nn::vi::LayerStackFlagType nullStacks
        ) NN_NOEXCEPT;

        // SharedFrameBuffer に設定されている ContentParameter の値を取得する。
        // Detached でなくても呼べる。
        nn::Result GetSharedFrameBufferContentParameter(
            nn::vi::LayerStackFlagType* pOutLayerStack,
            nn::vi::CropRegion* pOutCropRegion,
            int32_t* pOutScalingMode,
            uint32_t* pOutTransform,
            int32_t* pOutPresentInterval,
            nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
            int index
        ) NN_NOEXCEPT;


    protected:
        SharingClientImpl m_Sharing;
    };

}}}
