﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharingClientObject.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/service/display/am_DisplayControlConfig.h>
#include <nn/vi/vi_Result.h>

#include "../visrv_Macro.h"
#include "../visrv_Log.h"
#include "../client/visrv_ClientUtility.h"
#include "visrv_SharingServerObject.h"
#include "visrv_SharedBuffer.h"
#include "visrv_SharedBufferPool.h"
#include "detail/visrv_SharedLowLevelLayer.h"
#include "detail/visrv_SharedClientLayer.h"

namespace nn{ namespace visrv{ namespace fbshare{

    nn::Result SharingClientImpl::OwningSharedBufferList::Find(SharedBuffer** pOutValue, nn::vi::fbshare::SharedBufferHandle h) NN_NOEXCEPT
    {
        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(FindImpl(&pBuffer, [&](SharedBuffer* p){ return p->GetHandle() == h; }));
        *pOutValue = pBuffer;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientImpl::OwningSharedLowLevelLayerList::Find(detail::SharedLowLevelLayer** pOutValue, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(FindImpl(&pLayer, [&](detail::SharedLowLevelLayer* p){ return p->GetLayerResourceId() == layerId; }));
        *pOutValue = pLayer;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientImpl::OwningSharedClientLayerList::Find(detail::SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(FindImpl(&pLayer, [&](detail::SharedClientLayer* p){ return p->GetHandle() == h; }));
        *pOutValue = pLayer;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientImpl::BoundSharedClientLayerList::Find(detail::SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(FindImpl(&pLayer, [&](detail::SharedClientLayer* p){ return p->GetHandle() == h; }));
        *pOutValue = pLayer;
        NN_RESULT_SUCCESS;
    }

    //-----------------------------------------------

    SharingClientImpl::SharingClientImpl() NN_NOEXCEPT
        : m_pSelf(nullptr)
    {
    }

    void SharingClientImpl::Initialize(
        client::ClientObject* pSelf
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pSelf);
        m_pSelf = pSelf;
        m_OwningSharedBufferList.Initialize();
        m_OwningSharedLowLevelLayerList.Initialize();
        m_OwningSharedClientLayerList.Initialize();
        m_BoundSharedClientLayerList.Initialize();
    }

    void SharingClientImpl::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_OwningSharedBufferList.GetCount() == 0);
        NN_SDK_REQUIRES(m_OwningSharedLowLevelLayerList.GetCount() == 0);
        NN_SDK_REQUIRES(m_OwningSharedClientLayerList.GetCount() == 0);
        NN_SDK_REQUIRES(m_BoundSharedClientLayerList.GetCount() == 0);
        // NOTE:
        //   解放処理は SharingClientObject が行う
        m_pSelf = nullptr;
        m_BoundSharedClientLayerList.Finalize();
        m_OwningSharedClientLayerList.Finalize();
        m_OwningSharedLowLevelLayerList.Finalize();
        m_OwningSharedBufferList.Finalize();
    }

    client::ClientObject* SharingClientImpl::GetSelf() NN_NOEXCEPT
    {
        return m_pSelf;
    }

    const client::ClientObject* SharingClientImpl::GetSelf() const NN_NOEXCEPT
    {
        return m_pSelf;
    }

    //-------------------------------------------------

    SharingClientObject::SharingClientObject() NN_NOEXCEPT
    {
    }

    void SharingClientObject::InitializeSharing(client::ClientObject* pSelf) NN_NOEXCEPT
    {
        m_Sharing.Initialize(pSelf);
    }

    void SharingClientObject::FinalizeSharing() NN_NOEXCEPT
    {
        // 使用している SharedClientLayer をすべて返却
        if(int n = m_Sharing.m_BoundSharedClientLayerList.GetCount())
        {
            NN_VISRV_LOG_OPENCLOSE("Unbinding all bound SharedClientLayers (%d)\n", n);
            while(auto p = m_Sharing.m_BoundSharedClientLayerList.GetHead())
            {
                UnbindSharedClientLayerImpl(p);
            }
        }

        // 所有している SharedClientLayer をすべて破棄
        if(int n = m_Sharing.m_OwningSharedClientLayerList.GetCount())
        {
            NN_VISRV_LOG_OPENCLOSE("Destroying all owning SharedClientLayers (%d)\n", n);
            while(auto p = m_Sharing.m_OwningSharedClientLayerList.GetHead())
            {
                DestroySharedClientLayerImpl(p);
            }
        }

        // 所有している SharedLowLayer をすべて破棄
        if(int n = m_Sharing.m_OwningSharedLowLevelLayerList.GetCount())
        {
            NN_VISRV_LOG_OPENCLOSE("Unbinding all owning SharedLowLevelLayers (%d)\n", n);
            while(auto p = m_Sharing.m_OwningSharedLowLevelLayerList.GetHead())
            {
                UnbindSharedLowLevelLayerImpl(p);
            }
        }

        // 所有している SharedBuffer をすべて破棄
        if(int n = m_Sharing.m_OwningSharedBufferList.GetCount())
        {
            NN_VISRV_LOG_OPENCLOSE("Destroying all owning SharedBuffers (%d)\n", n);
            while(auto p = m_Sharing.m_OwningSharedBufferList.GetHead())
            {
                DestroySharedBufferImpl(p);
            }
        }

        m_Sharing.Finalize();
    }

    bool SharingClientObject::IsAlive() const NN_NOEXCEPT
    {
        return m_Sharing.GetSelf()->IsAlive();
    }

    // SharedBuffer ----------------------------------------------------------------

    nn::Result SharingClientObject::CreateSharedBufferStaticStorage(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        uint64_t storageKey,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutHandle);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(g_SharedBufferPool.Allocate(&p));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedBufferPool.Free(p));
        NN_SDK_ASSERT(!p->IsInitialized());

        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(p->InitializeOnStaticStorage(
            &h,
            &g_Device,
            &g_GrAlloc,
            &g_SharedBufferStaticStorage,
            storageKey,
            layout,
            m_Sharing.GetSelf(),
            &g_SharedLowLevelLayerManager
        ));
        NN_VISRV_PROCESS_ROLLBACK(p->Finalize());

        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Register(p));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedBufferList.Unregister(p)));

        *pOutHandle = h;
        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedBuffer #%lld created on static storage\n", h._value);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::CreateSharedBufferTransferMemory(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        nn::sf::NativeHandle& transferMemoryHandle,
        size_t transferMemorySize,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutHandle);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(g_SharedBufferPool.Allocate(&p));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedBufferPool.Free(p));
        NN_SDK_ASSERT(!p->IsInitialized());

        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(p->InitializeOnTransferMemory(
            &h,
            &g_Device,
            &g_GrAlloc,
            transferMemoryHandle,
            transferMemorySize,
            layout,
            m_Sharing.GetSelf(),
            &g_SharedLowLevelLayerManager
        ));
        NN_VISRV_PROCESS_ROLLBACK(p->Finalize());

        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Register(p));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedBufferList.Unregister(p)));

        *pOutHandle = h;
        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedBuffer #%lld created on transfer memory\n", h._value);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::CreateSharedBufferProcessHeap(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        ResourceId blockId,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutHandle);
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(blockId != 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(client::ClientObjectAccessor::GetOwnProcessHeapBlockIdList(m_Sharing.GetSelf())->Contains(blockId), nn::vi::ResultNotFound());

        NN_VISRV_PROCESS_START();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(g_SharedBufferPool.Allocate(&p));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedBufferPool.Free(p));
        NN_SDK_ASSERT(!p->IsInitialized());

        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(p->InitializeOnProcessHeap(
            &h,
            &g_Device,
            &g_GrAlloc,
            blockId,
            layout,
            m_Sharing.GetSelf(),
            &g_SharedLowLevelLayerManager
        ));
        NN_VISRV_PROCESS_ROLLBACK(p->Finalize());

        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Register(p));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedBufferList.Unregister(p)));

        *pOutHandle = h;
        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedBuffer #%lld created on process heap %d\n", h._value, blockId);
        NN_RESULT_SUCCESS;
    }



    nn::Result SharingClientObject::DestroySharedBuffer(nn::vi::fbshare::SharedBufferHandle handle) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&p, handle));

        DestroySharedBufferImpl(p);

        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::DestroySharedBufferImpl(SharedBuffer* p) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(p);
        NN_SDK_ASSERT(p->IsInitialized());
        NN_SDK_ASSERT_EQUAL(p->GetOwnerClientObject(), m_Sharing.GetSelf());
        auto h = p->GetHandle();
        NN_UNUSED(h);

        while(auto pLowLayer = p->GetConnectedLowLayerList().GetHead())
        {
            DisconnectSharedLowLevelLayerFromSharedBufferImpl(pLowLayer);
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedBufferList.Unregister(p));
        p->Finalize();
        g_SharedBufferPool.Free(p);
        NN_VISRV_LOG_OPENCLOSE("SharedBuffer #%lld destroyed\n", h._value);
    }

    nn::Result SharingClientObject::RegiterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&p, handle));

        NN_RESULT_DO(p->RegisterImporterAruid(importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::UnregiterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* p = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&p, handle));

        NN_RESULT_DO(p->UnregisterImporterAruid(importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::GetSharedBufferMemoryHandleId(nn::vi::native::NativeMemoryHandleId* pOutMemHandleId, size_t* pOutSize, nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout, nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from global pool
        SharedBuffer* p = nullptr;
        NN_RESULT_DO(g_SharedBufferPool.Find(&p, handle));

        nn::vi::native::NativeMemoryHandleId memId = {};
        size_t size = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_RESULT_DO(p->GetMemoryHandleId(&memId, &size, &layout, importerAruid));
        *pOutMemHandleId = memId;
        *pOutSize        = size;
        *pOutLayout      = layout;
        NN_RESULT_SUCCESS;
    }

    // LowLevelLayer ----------------------------------------------------------------

    nn::Result SharingClientObject::BindSharedLowLevelLayerToManagedLayer(
        nn::vi::LayerId layerId,
        const char* displayName,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_CHECK_NOT_NULL(displayName);

        NN_VISRV_PROCESS_START();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(g_SharedLowLevelLayerManager.BindManagedLayer(
            &pLayer,
            m_Sharing.GetSelf(),
            layerId,
            displayName,
            aruid
        ));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedLowLevelLayerManager.UnbindLayer(pLayer));

        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Register(pLayer));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedLowLevelLayerList.Unregister(pLayer)));

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("ManagedLayer #%lld is bound to SharedLowLevelLayer (aruid=%llu)\n", layerId, aruid);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::BindSharedLowLevelLayerToIndirectLayer(
        nn::vi::IndirectProducerHandleType handle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(g_SharedLowLevelLayerManager.BindIndirectLayer(
            &pLayer,
            m_Sharing.GetSelf(),
            handle,
            aruid
        ));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedLowLevelLayerManager.UnbindLayer(pLayer));

        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Register(pLayer));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedLowLevelLayerList.Unregister(pLayer)));

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("IndirectLayer #%lld is bound to SharedLowLevelLayer (aruid=%llu)\n", handle, aruid);
        NN_RESULT_SUCCESS;
    }
    nn::Result SharingClientObject::UnbindSharedLowLevelLayer(
        nn::vi::LayerId layerId
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLayer, layerId));

        UnbindSharedLowLevelLayerImpl(pLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::UnbindSharedLowLevelLayerImpl(
        detail::SharedLowLevelLayer* pLowLayer
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pLowLayer);
        NN_SDK_REQUIRES(pLowLayer->IsBound());
        auto layerId = pLowLayer->GetLayerResourceId();
        NN_UNUSED(layerId);

        if(pLowLayer->IsConnected())
        {
            DisconnectSharedLowLevelLayerFromSharedBufferImpl(pLowLayer);
        }
        NN_SDK_ASSERT(pLowLayer->GetState() == detail::SharedLowLevelLayerState_Bound);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_OwningSharedLowLevelLayerList.Unregister(pLowLayer));
        g_SharedLowLevelLayerManager.UnbindLayer(pLowLayer);
        NN_VISRV_LOG_OPENCLOSE("LowLevelLayer #%lld is unbound from SharedLowLevelLayer\n", layerId);
    }

    nn::Result SharingClientObject::ConnectSharedLowLevelLayerToSharedBuffer(
        nn::vi::LayerId layerId,
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLayer);
        NN_RESULT_THROW_UNLESS(pLayer->GetState() == detail::SharedLowLevelLayerState_Bound, nn::vi::ResultDenied());

        SharedBuffer* pSharedBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pSharedBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pSharedBuffer);

        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(pLayer->Connect(pSharedBuffer, pSharedBuffer->GetFrameBufferList(), pSharedBuffer->GetFrameBufferCount()));
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Disconnect());

        NN_RESULT_DO(pSharedBuffer->ConnectLowLevelLayer(pLayer));
        NN_VISRV_PROCESS_ROLLBACK(pSharedBuffer->DisconnectLowLevelLayer(pLayer));

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("LowLevelLayer #%lld is connected to SharedBuffer #%lld\n", layerId, hSharedBuffer._value);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::DisconnectSharedLowLevelLayerFromSharedBuffer(
        nn::vi::LayerId layerId
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());

        DisconnectSharedLowLevelLayerFromSharedBufferImpl(pLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::DisconnectSharedLowLevelLayerFromSharedBufferImpl(detail::SharedLowLevelLayer* pLowLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pLowLayer);
        NN_SDK_REQUIRES(pLowLayer->IsConnected());

        if(pLowLayer->IsAttached())
        {
            auto pClientLayer = pLowLayer->GetAttachedClientLayer();
            NN_SDK_ASSERT_NOT_NULL(pClientLayer);
            NN_SDK_ASSERT(pClientLayer->IsAttached());
            NN_SDK_ASSERT_EQUAL(pClientLayer->GetAttachedLowLevelLayer(), pLowLayer);
            ForceDetachSharedLayerFromLowLevelLayerImpl(pClientLayer);
        }
        NN_SDK_ASSERT(!pLowLayer->IsAttached());

        auto layerId = pLowLayer->GetLayerResourceId();
        auto pSharedBuffer = pLowLayer->GetConnectedSharedBuffer();
        NN_SDK_ASSERT_NOT_NULL(pSharedBuffer);
        auto hBuffer = pSharedBuffer->GetHandle();

        pSharedBuffer->DisconnectLowLevelLayer(pLowLayer);
        pLowLayer->Disconnect();
        NN_VISRV_LOG_OPENCLOSE("LowLevelLayer #%lld is disconnected from SharedBuffer #%lld\n", layerId, hBuffer);
        NN_UNUSED(layerId);
        NN_UNUSED(hBuffer);
    }

    nn::Result SharingClientObject::GetSharedLowLevelLayerSynchronizedEvent(
        nn::sf::NativeHandle& outHandle,
        nn::vi::LayerId layerId
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsBound(), nn::vi::ResultDenied());

        nn::sf::NativeHandle h = pLayer->GetSynchronizedEventHandle();

        outHandle = std::move(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::CheckSharedLayerSynchronized(
        int* pOutDisplayedFrameBufferIndex,
        nn::vi::LayerId layerId
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutDisplayedFrameBufferIndex);
        NN_VISRV_CHECK_ALIVE();

        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsBound(), nn::vi::ResultDenied());

        int index = -1;
        NN_RESULT_DO(pLayer->CheckSynchronized(&index));

        *pOutDisplayedFrameBufferIndex = index;
        NN_RESULT_SUCCESS;
    }

    // Create/Destroy SharedClientLayer ----------------------------------------------------------------------
    nn::Result SharingClientObject::CreateSharedClientLayer(
        nn::vi::fbshare::SharedLayerHandle* pOutHandle,
        nn::applet::AppletResourceUserId userAruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutHandle);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();

        // create
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(g_SharedClientLayerManager.Create(&pLayer, m_Sharing.GetSelf(), userAruid));
        NN_VISRV_PROCESS_ROLLBACK(g_SharedClientLayerManager.Destroy(pLayer));

        // register to this client
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Register(pLayer));
        NN_VISRV_PROCESS_ROLLBACK(m_Sharing.m_OwningSharedClientLayerList.Unregister(pLayer));

        auto h = pLayer->GetHandle();

        NN_VISRV_PROCESS_SUCCESS();
        *pOutHandle = h;
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is created\n", h);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::DestroySharedClientLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);
        NN_SDK_ASSERT(pLayer->IsInitialized());

        DestroySharedClientLayerImpl(pLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::DestroySharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);
        NN_SDK_REQUIRES(p->IsInitialized());
        auto h = p->GetHandle();
        NN_UNUSED(h);

        if(p->IsBound())
        {
            UnbindSharedClientLayerImpl(p);
        }
        if(p->IsAttached())
        {
            ForceDetachSharedLayerFromLowLevelLayerImpl(p);
        }
        NN_SDK_ASSERT(!p->IsBound());
        NN_SDK_ASSERT(!p->IsAttached());

        // unregister from this client
        m_Sharing.m_OwningSharedClientLayerList.Unregister(p);
        // destroy
        g_SharedClientLayerManager.Destroy(p);
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is destroyed\n", h);
    }

    // Configure SharedClientLayer ----------------------------------------------------------------------
    nn::Result SharingClientObject::GetSharedClientLayerLayerStacks(
        nn::vi::LayerStackFlagType* pOutValue,
        nn::vi::fbshare::SharedLayerHandle h
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();

        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);
        NN_SDK_ASSERT(pLayer->IsInitialized());

        *pOutValue = pLayer->GetLayerStackFlags();
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::SetSharedClientLayerLayerStacks(
        nn::vi::fbshare::SharedLayerHandle h,
        nn::vi::LayerStackFlagType stacks
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);
        NN_SDK_ASSERT(pLayer->IsInitialized());

        pLayer->SetLayerStackFlags(stacks);
        NN_RESULT_SUCCESS;
    }

    // Bind/Unbind/Connect/Disconnect SharedClientLayer ----------------------------------------------------------------
    nn::Result SharingClientObject::BindSharedClientLayer(
        nn::vi::fbshare::SharedLayerHandle h,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();
        // search from global pool
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(g_SharedClientLayerManager.Find(&pLayer, h));

        NN_RESULT_THROW_UNLESS(pLayer->GetState() == detail::SharedClientLayerState_Initialized, nn::vi::ResultDenied());

        // bind this client to the layer
        NN_RESULT_DO(pLayer->Bind(m_Sharing.GetSelf(), aruid));
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Unbind());

        // bind the layer to this client
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Register(pLayer));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_Sharing.m_BoundSharedClientLayerList.Unregister(pLayer)));

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is bound to aruid=%llu\n", h, aruid.lower);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::UnbindSharedClientLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsBound(), nn::vi::ResultDenied());
        NN_SDK_ASSERT_EQUAL(pLayer->GetUserClient(), m_Sharing.GetSelf());

        UnbindSharedClientLayerImpl(pLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::UnbindSharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);
        NN_SDK_REQUIRES(p->IsBound());
        auto h = p->GetHandle();
        NN_UNUSED(h);

        if(p->IsConnected())
        {
            DisconnectSharedClientLayerImpl(p);
        }
        NN_SDK_ASSERT(!p->IsConnected());

        // unbind the layer from user client
        NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetUserClient()->m_Sharing.m_BoundSharedClientLayerList.Unregister(p));
        // unbind user client from the layer
        p->Unbind();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is unbound\n", h);
    }

    nn::Result SharingClientObject::ConnectSharedClientLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_PROCESS_START();
        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->GetState() == detail::SharedClientLayerState_Bound, nn::vi::ResultDenied());

        // connect
        NN_RESULT_DO(pLayer->Connect());
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Disconnect());

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is connected\n", h);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::DisconnectSharedClientLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, h));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());
        NN_SDK_ASSERT_EQUAL(pLayer->GetUserClient(), m_Sharing.GetSelf());

        // disconnect
        DisconnectSharedClientLayerImpl(pLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::DisconnectSharedClientLayerImpl(detail::SharedClientLayer* p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);
        NN_SDK_REQUIRES(p->IsConnected());
        auto h = p->GetHandle();
        NN_UNUSED(h);
        p->Disconnect();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is disconnected\n", h);
    }

    // Attach/Detach SharedClientLayer ----------------------------------------------------------------

    nn::Result SharingClientObject::AttachSharedLayerToLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        ResourceId layerId,
        const nn::vi::fbshare::SharedLayerTextureIndexList& frameBufferIndexList
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedClientLayer* pClientLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pClientLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pClientLayer);

        // search from own layers
        detail::SharedLowLevelLayer* pLowLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLowLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLowLayer);

        NN_RESULT_THROW_UNLESS(pLowLayer->IsConnected(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pLowLayer->GetAttachState() == detail::SharedLowLevelLayerAttachState_Detached, nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pClientLayer->IsInitialized(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pClientLayer->GetAttachState() == detail::SharedClientLayerAttachState_Detached, nn::vi::ResultDenied());

        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(pLowLayer->Attach(pClientLayer));
        NN_VISRV_PROCESS_ROLLBACK(pLowLayer->Detach());

        NN_RESULT_DO(pClientLayer->Attach(pLowLayer, frameBufferIndexList));
        NN_VISRV_PROCESS_ROLLBACK(pClientLayer->ForceDetach());

        NN_VISRV_PROCESS_SUCCESS();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is attached to LowLevelLayer #%lld\n", hSharedLayer, layerId);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::StartDetachSharedClientLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedClientLayer* pClientLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pClientLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pClientLayer);

        NN_RESULT_THROW_UNLESS(pClientLayer->IsInitialized(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pClientLayer->GetAttachState() == detail::SharedClientLayerAttachState_Attached, nn::vi::ResultDenied());

        NN_RESULT_DO(pClientLayer->StartDetach());
        NN_SDK_ASSERT(pClientLayer->IsDetaching());

        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld: start detaching\n", pClientLayer->GetHandle());
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::FinishDetachSharedClientLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedClientLayer* pClientLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pClientLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pClientLayer);

        NN_RESULT_THROW_UNLESS(pClientLayer->IsInitialized(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pClientLayer->IsDetaching() , nn::vi::ResultDenied());

        auto pLowLayer = pClientLayer->GetAttachedLowLevelLayer();
        NN_SDK_ASSERT_NOT_NULL(pLowLayer);
        NN_SDK_ASSERT_EQUAL(pLowLayer->GetAttachedClientLayer(), pClientLayer);

        NN_RESULT_DO(pClientLayer->FinishDetach());
        pLowLayer->Detach();

        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is safe-detached from LowLevelLayer #%lld\n", pClientLayer->GetHandle(), pLowLayer->GetLayerResourceId());
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::GetSharedClientLayerDetachReadyEvent(
        nn::sf::NativeHandle& outSystemEventHandle,
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedClientLayer* pClientLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pClientLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pClientLayer);

        NN_RESULT_THROW_UNLESS(pClientLayer->IsInitialized(), nn::vi::ResultDenied());

        outSystemEventHandle = pClientLayer->GetDetachReadySystemEventHandle();
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::ForceDetachSharedLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedClientLayer* pClientLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedClientLayerList.Find(&pClientLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pClientLayer);

        NN_RESULT_THROW_UNLESS(pClientLayer->IsInitialized(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pClientLayer->IsAttached(), nn::vi::ResultDenied());

        ForceDetachSharedLayerFromLowLevelLayerImpl(pClientLayer);
        NN_RESULT_SUCCESS;
    }

    void SharingClientObject::ForceDetachSharedLayerFromLowLevelLayerImpl(
        detail::SharedClientLayer* pClientLayer
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pClientLayer);
        NN_SDK_REQUIRES(pClientLayer->IsInitialized());
        NN_SDK_REQUIRES(pClientLayer->IsAttached());

        auto pLowLayer = pClientLayer->GetAttachedLowLevelLayer();
        NN_SDK_ASSERT_NOT_NULL(pLowLayer);
        NN_SDK_ASSERT_EQUAL(pLowLayer->GetAttachedClientLayer(), pClientLayer);

        pClientLayer->ForceDetach();
        pLowLayer->Detach();
        NN_VISRV_LOG_OPENCLOSE("SharedClientLayer #%lld is force-detached from LowLevelLayer #%lld\n", pClientLayer->GetHandle(), pLowLayer->GetLayerResourceId());
    }

    // Acquire/Present SharedClientLayer ----------------------------------------------------------------

    nn::Result SharingClientObject::AcquireSharedClientLayerFrameBuffer(
        int* pOutFrameBufferIndex,
        nn::vi::native::NativeSync* pOutSync,
        nn::vi::fbshare::SharedLayerTextureIndexList* pOutBufferIndexList,
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFrameBufferIndex);
        NN_SDK_REQUIRES_NOT_NULL(pOutSync);
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());

        int index = -1;
        native::NativeRmSync sync = {};
        nn::vi::fbshare::SharedLayerTextureIndexList indexList = {};
        indexList.Reset();
        NN_RESULT_DO(pLayer->Acquire(&index, &sync, &indexList));

        *pOutFrameBufferIndex = index;
        sync.WriteTo(pOutSync);
        *pOutBufferIndexList = indexList;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::PresentSharedClientLayerFrameBuffer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        int frameBufferIndex,
        const nn::vi::native::NativeSync& sync,
        const nn::vi::CropRegion& crop,
        nn::vi::ImageTransformType transform,
        int presentInterval
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());

        native::NativeRmSync rmSync;
        rmSync.ReadFrom(sync);
        NN_RESULT_DO(pLayer->Present(frameBufferIndex, rmSync, crop, transform, presentInterval));

        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::CancelSharedClientLayerFrameBuffer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        int frameBufferIndex
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());

        NN_RESULT_DO(pLayer->Cancel(frameBufferIndex));

        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::GetSharedClientLayerAcquirableEvent(
        nn::sf::NativeHandle& outSystemEventHandle,
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_SDK_ASSERT(pLayer->IsInitialized());

        auto h = pLayer->GetAcquirableSystemEventHandle();

        outSystemEventHandle = std::move(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::FillSharedClientLayerFrameBufferColor(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        int frameBufferIndex,
        int r,
        int g,
        int b,
        int a
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from bound layers
        detail::SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_BoundSharedClientLayerList.Find(&pLayer, hSharedLayer));
        NN_SDK_ASSERT_NOT_NULL(pLayer);

        NN_SDK_ASSERT(pLayer->IsInitialized());
        NN_RESULT_THROW_UNLESS(pLayer->IsConnected(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(pLayer->IsAttached(), nn::vi::ResultDenied());

        NN_RESULT_DO(pLayer->FillBufferColor(frameBufferIndex, r, g, b, a));

        NN_RESULT_SUCCESS;
    }

    // Detached FrameBuffer Operation ----------------------------------------------------------------

    nn::Result SharingClientObject::PresentDetachedSharedFrameBufferToLowLevelLayer(
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        nn::vi::LayerId layerId,
        int index
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        // search from own layers
        detail::SharedLowLevelLayer* pLowLayer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedLowLevelLayerList.Find(&pLowLayer, layerId));
        NN_SDK_ASSERT_NOT_NULL(pLowLayer);

        NN_RESULT_THROW_UNLESS(
            pLowLayer->GetConnectedSharedBuffer()->GetHandle() == hSharedBuffer,
            nn::vi::ResultDenied()
        );
        NN_RESULT_THROW_UNLESS(
            index >= 0 && index < pLowLayer->GetFrameBufferCount(),
            nn::vi::ResultInvalidRange()
        );

        auto pFrameBuffer = pLowLayer->GetFrameBuffer(index);
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        NN_RESULT_DO(pLowLayer->Enqueue(index));
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::FillDetachedSharedFrameBufferColor(
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int frameBufferIndex,
        uint32_t color,
        const nn::vi::fbshare::SharedTextureOption& option
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(frameBufferIndex >= 0 && frameBufferIndex < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[frameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        nn::util::Color4u8 color4;
        std::memcpy(&color4, &color, sizeof(nn::util::Color4u8));

        pBuffer->FillColor(frameBufferIndex, color4, option);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharingClientObject::GetDetachedSharedFrameBufferImage(
        size_t* pOutReadSize,
        void* buffer,
        size_t size,
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int frameBufferIndex
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(frameBufferIndex >= 0 && frameBufferIndex < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[frameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        return pBuffer->GetImage(pOutReadSize, buffer, size, frameBufferIndex);
    }

    nn::Result SharingClientObject::SetDetachedSharedFrameBufferImage(
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int frameBufferIndex,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransformType srcTransform
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(frameBufferIndex >= 0 && frameBufferIndex < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[frameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        return pBuffer->SetImage(frameBufferIndex, buffer, size, dstOption, srcTransform);
    }

    nn::Result SharingClientObject::SetDetachedSharedFrameBufferSubImage(
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int frameBufferIndex,
        int x,
        int y,
        int w,
        int h,
        uint32_t bgColor,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransformType srcTransform
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(frameBufferIndex >= 0 && frameBufferIndex < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[frameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        nn::util::Color4u8 color(
            (bgColor & 0x000000FF) >> 0,
            (bgColor & 0x0000FF00) >> 8,
            (bgColor & 0x00FF0000) >> 16,
            (bgColor & 0xFF000000) >> 24
        );

        return pBuffer->SetSubImage(frameBufferIndex, x, y, w, h, color, buffer, size, dstOption, srcTransform);
    }

    nn::Result SharingClientObject::ExpandStartupLogoOnSharedFrameBuffer(
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int dstIndex,
        const nn::vi::fbshare::SharedTextureOption& option
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(dstIndex >= 0 && dstIndex < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        const void* memory = nullptr;
        size_t memorySize = 0;
        pBuffer->GetStorageInfoForStartupLogo(&memory, &memorySize);
        NN_RESULT_THROW_UNLESS(memory != nullptr, nn::vi::ResultResourceLimit());
        NN_RESULT_THROW_UNLESS(memorySize >= nn::am::service::display::StartupLogoSize, nn::vi::ResultResourceLimit());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[dstIndex];
        NN_RESULT_THROW_UNLESS(!pFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        nn::util::Color4u8 color(
            (nn::am::service::display::StartupLogoBgColor & 0x000000FF) >> 0,
            (nn::am::service::display::StartupLogoBgColor & 0x0000FF00) >> 8,
            (nn::am::service::display::StartupLogoBgColor & 0x00FF0000) >> 16,
            (nn::am::service::display::StartupLogoBgColor & 0xFF000000) >> 24
        );


        auto result = pBuffer->SetSubImage(
            dstIndex,
            nn::am::service::display::StartupLogoPositionX,
            nn::am::service::display::StartupLogoPositionY,
            nn::am::service::display::StartupLogoWidth,
            nn::am::service::display::StartupLogoHeight,
            color,
            memory,
            nn::am::service::display::StartupLogoSize,
            option,
            nn::vi::ImageTransform_None
        );

        nn::os::FlushDataCache(memory, memorySize); // StartupLogoSize だけでよさそうだが一応全範囲やっておく。
        return result;
    }

    nn::Result SharingClientObject::CopyDetachedSharedFrameBufferImage(
        nn::vi::fbshare::SharedBufferHandle hDstSharedBuffer,
        int dstFrameBufferIndex,
        nn::vi::fbshare::SharedBufferHandle hSrcSharedBuffer,
        int srcFrameBufferIndex,
        const nn::vi::fbshare::SharedTextureOption& option,
        nn::vi::LayerStackFlagType stacksFilter,
        nn::vi::LayerStackFlagType nullStacks
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pDstBuffer = nullptr;
        SharedBuffer* pSrcBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pDstBuffer, hDstSharedBuffer));
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pSrcBuffer, hSrcSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pDstBuffer);
        NN_SDK_ASSERT_NOT_NULL(pSrcBuffer);
        NN_SDK_ASSERT(pDstBuffer->IsInitialized());
        NN_SDK_ASSERT(pSrcBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(dstFrameBufferIndex >= 0 && dstFrameBufferIndex < pDstBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(srcFrameBufferIndex >= 0 && srcFrameBufferIndex < pSrcBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pDstFrameBuffer = &pDstBuffer->GetFrameBufferList()[dstFrameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pDstFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());
        auto pSrcFrameBuffer = &pSrcBuffer->GetFrameBufferList()[srcFrameBufferIndex];
        NN_RESULT_THROW_UNLESS(!pSrcFrameBuffer->IsAttachedToClientLayer(), nn::vi::ResultDenied());

        return SharedBuffer::CopyImage(pDstBuffer, dstFrameBufferIndex, pSrcBuffer, srcFrameBufferIndex, option, stacksFilter, nullStacks);
    }

    nn::Result SharingClientObject::GetSharedFrameBufferContentParameter(
        nn::vi::LayerStackFlagType* pOutLayerStack,
        nn::vi::CropRegion* pOutCropRegion,
        int32_t* pOutScalingMode,
        uint32_t* pOutTransform,
        int32_t* pOutPresentInterval,
        nn::vi::fbshare::SharedBufferHandle hSharedBuffer,
        int index
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutLayerStack);
        NN_VISRV_CHECK_NOT_NULL(pOutCropRegion);
        NN_VISRV_CHECK_NOT_NULL(pOutScalingMode);
        NN_VISRV_CHECK_NOT_NULL(pOutTransform);
        NN_VISRV_CHECK_NOT_NULL(pOutPresentInterval);
        NN_VISRV_CHECK_ALIVE();

        SharedBuffer* pBuffer = nullptr;
        NN_RESULT_DO(m_Sharing.m_OwningSharedBufferList.Find(&pBuffer, hSharedBuffer));
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(pBuffer->IsInitialized());

        NN_RESULT_THROW_UNLESS(index >= 0 && index < pBuffer->GetFrameBufferCount(), nn::vi::ResultInvalidRange());

        auto pFrameBuffer = &pBuffer->GetFrameBufferList()[index];
        auto param = pFrameBuffer->GetContentParameter();

        *pOutLayerStack = param.layerStacks;
        *pOutCropRegion = param.cropRegion;
        *pOutScalingMode = param.scalingMode;
        *pOutTransform = param.transform;
        *pOutPresentInterval = param.presentInterval;
        NN_RESULT_SUCCESS;
    }

}}}

