﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedMemoryPoolLayout.h>
#include <nn/vi/fbshare/vi_SharedTextureOption.h>
#include "../visrv_Config.h"
#include "../visrv_ResourceIdManagement.h"
#include "visrv_Device.h"
#include "visrv_GrAlloc.h"
#include "visrv_SharedBufferStaticStorage.h"
#include "../util/visrv_TPointerContainer.h"
#include "../util/visrv_TAruidContainer.h"
#include "../util/visrv_TransferMemoryBinder.h"
#include "detail/visrv_SharedFrameBuffer.h"
#include "detail/visrv_SharedLowLevelLayer.h"
#include "detail/visrv_SharedLowLevelLayerManager.h"

namespace nn{ namespace visrv{ namespace client{
    class ClientObject;
}}}

namespace nn{ namespace visrv{ namespace fbshare{

    class SharedBuffer
    {
    public:
        static const int FrameBufferCountMax   = SharedFrameBufferCountMaxPerSharedBuffer;
        static const int LowLevelLayerCountMax = SharedLowLevelLayerCountMaxPerSharedBuffer;

    public:
        class LowLevelLayerList
            : public util::TPointerContainer<detail::SharedLowLevelLayer, LowLevelLayerCountMax, util::ContainerThreadSafetyOption_SingleThread>
        {
        public:
            nn::Result Find(detail::SharedLowLevelLayer** pOutLayer, ResourceId layerId) NN_NOEXCEPT;
        };

        typedef util::TAruidContainer<SharedBufferImporterCountMaxPerSharedBuffer, util::ContainerThreadSafetyOption_SingleThread> ImporterAruidList;

    public:
        // @brief 無効な SharedBuffer を作る
        // @details
        // Initialize() で初期化する必要がある。
        SharedBuffer() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;
        client::ClientObject* GetOwnerClientObject() NN_NOEXCEPT;
        nn::vi::fbshare::SharedBufferHandle GetHandle() const NN_NOEXCEPT;
        LowLevelLayerList& GetConnectedLowLayerList() NN_NOEXCEPT;
        detail::SharedFrameBuffer* GetFrameBufferList() NN_NOEXCEPT;
        int GetFrameBufferCount() const NN_NOEXCEPT;

        // StartupLogo 用
        void GetStorageInfoForStartupLogo(
            const void** pOutMemory,
            size_t* pOutMemorySize
        ) NN_NOEXCEPT;

        // capsrv 用
        void GetSharedMemoryPoolInfo(
            NvRmMemHandle* pOutMemHandle,
            void** pOutMemory,
            size_t* pOutMemorySize,
            nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout
        ) NN_NOEXCEPT;

        // @brief 初期化する
        // @pre !IsInitialized()
        // @post IsInitialized()
        nn::Result InitializeOnStaticStorage(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            Device* pDevice,
            GrAlloc* pGrAlloc,
            SharedBufferStaticStorage* pStorage,
            uint64_t storageKey,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
            client::ClientObject* pOwner,
            detail::SharedLowLevelLayerManager* pLowLevelLayerManger
        ) NN_NOEXCEPT;

        // @brief 初期化する
        // @pre !IsInitialized()
        // @post IsInitialized()
        nn::Result InitializeOnTransferMemory(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            Device* pDevice,
            GrAlloc* pGrAlloc,
            nn::sf::NativeHandle& transferMemoryHandle,
            size_t transferMemorySize,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
            client::ClientObject* pOwner,
            detail::SharedLowLevelLayerManager* pLowLevelLayerManger
        ) NN_NOEXCEPT;

        // @brief 初期化する
        // @pre !IsInitialized()
        // @post IsInitialized()
        nn::Result InitializeOnProcessHeap(
            nn::vi::fbshare::SharedBufferHandle* pOutHandle,
            Device* pDevice,
            GrAlloc* pGrAlloc,
            ResourceId blockId,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
            client::ClientObject* pOwner,
            detail::SharedLowLevelLayerManager* pLowLevelLayerManger
        ) NN_NOEXCEPT;

        // @brief 破棄する
        void Finalize() NN_NOEXCEPT;

        nn::Result RegisterImporterAruid(nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;
        nn::Result UnregisterImporterAruid(nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;
        nn::Result GetMemoryHandleId(nn::vi::native::NativeMemoryHandleId* pOutMemHandleId, size_t* pOutSize, nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT;

        nn::Result ConnectLowLevelLayer(detail::SharedLowLevelLayer* pLowLayer) NN_NOEXCEPT;
        void DisconnectLowLevelLayer(detail::SharedLowLevelLayer* pLowLayer) NN_NOEXCEPT;

        void FillColor(int index, nn::util::Color4u8 color, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT;
        nn::Result GetImage(size_t* pOutReadSize, void* buffer, size_t size, int index) NN_NOEXCEPT;

        nn::Result SetImage(
            int index,
            const void* buffer,
            size_t size,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransformType srcTransform
        ) NN_NOEXCEPT;

        nn::Result SetSubImage(
            int index,
            int x,
            int y,
            int w,
            int h,
            nn::util::Color4u8 bgColor,
            const void* buffer,
            size_t size,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransformType srcTransform
        ) NN_NOEXCEPT;

        static nn::Result CopyImage(SharedBuffer* pDstBuffer, int dstIndex, SharedBuffer* pSrcBuffer, int srcIndex, const nn::vi::fbshare::SharedTextureOption& option, nn::vi::LayerStackFlagType stacksFilter, nn::vi::LayerStackFlagType nullStacks) NN_NOEXCEPT;

    private:
        // @pre m_RmMemHandle == 0
        // @post m_RmMemHandle != 0
        // @post m_RmSurface が設定済
        nn::Result InitializeMemoryPool(Device* pDevice, void* pMemory, size_t size) NN_NOEXCEPT;

        // @pre m_RmMemHandle != 0
        // @post m_RmMemHandle == 0
        void FinalizeMemoryPool(Device* pDevice) NN_NOEXCEPT;

        nn::Result InitializeFrameBufferList(Device* pDevice, GrAlloc* pGrAlloc, NvRmMemHandle hMem, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT;

        void FinalizeFrameBufferList(Device* pDevice, GrAlloc* pGrAlloc) NN_NOEXCEPT;


    private:
        nn::vi::fbshare::SharedBufferHandle m_Handle;
        client::ClientObject* m_pOwnerClientObject;
        Device* m_pDevice;
        GrAlloc* m_pGrAlloc;
        detail::SharedLowLevelLayerManager* m_pLowLevelLayerManager;

        void* m_pMemory;
        size_t m_MemorySize;
        util::TransferMemoryBinder m_TransferMemory;

        NvRmMemHandle m_RmMemHandle;

        nn::vi::fbshare::SharedMemoryPoolLayout m_FrameBufferLayout;
        detail::SharedFrameBuffer               m_FrameBufferList[FrameBufferCountMax];

        LowLevelLayerList         m_ConnectedLowLevelLayerList;
        ImporterAruidList         m_ImporterAruidList;
    };

}}}
