﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedBuffer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/vi/vi_Result.h>
#include <nvrm_surface.h>

#include "../visrv_Log.h"
#include "../visrv_Macro.h"
#include "../visrv_ResourceIdManagement.h"
#include "../visrv_ProcessHeapBlockManager.h"
#include "detail/visrv_SharedClientLayer.h"
#include "../vic/visrv_VicOperation.h"
#include "../native/visrv_SyncpointWaiter.h"
#include "../native/visrv_ReadWriteToSurface.h"


namespace nn{ namespace visrv{ namespace fbshare{

    nn::Result SharedBuffer::LowLevelLayerList::Find(detail::SharedLowLevelLayer** pOutLayer, ResourceId layerId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(layerId != 0, nn::vi::ResultNotFound());
        detail::SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(FindImpl(&pLayer, [layerId](const detail::SharedLowLevelLayer* e){
            return e->IsInitialized() && e->GetLayerResourceId() == layerId;
        }));
        *pOutLayer = pLayer;
        NN_RESULT_SUCCESS;
    }

    //------------------------------------------

    SharedBuffer::SharedBuffer() NN_NOEXCEPT
        : m_Handle({})
        , m_pOwnerClientObject(nullptr)
        , m_pDevice(nullptr)
        , m_pGrAlloc(nullptr)
        , m_pMemory(nullptr)
        , m_MemorySize(0)
        , m_TransferMemory()
        , m_RmMemHandle()
        , m_FrameBufferLayout()
        , m_FrameBufferList()
        , m_ConnectedLowLevelLayerList()
        , m_ImporterAruidList()
    {
    }

    bool SharedBuffer::IsInitialized() const NN_NOEXCEPT
    {
        return !m_Handle.IsInvalid();
    }

    client::ClientObject* SharedBuffer::GetOwnerClientObject() NN_NOEXCEPT
    {
        return m_pOwnerClientObject;
    }

    nn::vi::fbshare::SharedBufferHandle SharedBuffer::GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

    SharedBuffer::LowLevelLayerList& SharedBuffer::GetConnectedLowLayerList() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_ConnectedLowLevelLayerList;
    }

    detail::SharedFrameBuffer* SharedBuffer::GetFrameBufferList() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_FrameBufferList;
    }

    int SharedBuffer::GetFrameBufferCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_FrameBufferLayout.count;
    }

    void SharedBuffer::GetStorageInfoForStartupLogo(
        const void** pOutMemory,
        size_t* pOutMemorySize
    ) NN_NOEXCEPT
    {
        *pOutMemory     = m_pMemory;
        *pOutMemorySize = m_MemorySize;
    }

    void SharedBuffer::GetSharedMemoryPoolInfo(
        NvRmMemHandle* pOutMemHandle,
        void** pOutMemory,
        size_t* pOutMemorySize,
        nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        *pOutMemHandle  = m_RmMemHandle;
        *pOutMemory     = m_pMemory;
        *pOutMemorySize = m_MemorySize;
        *pOutLayout     = m_FrameBufferLayout;
    }

    //--------------------------------------------------------------------------------------------------

    nn::Result SharedBuffer::InitializeOnStaticStorage(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        Device* pDevice,
        GrAlloc* pGrAlloc,
        SharedBufferStaticStorage* pStorage,
        uint64_t storageKey,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
        client::ClientObject* pOwner,
        detail::SharedLowLevelLayerManager* pLowLevelLayerManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        int bufferCount = layout.count;
        size_t requiredMemorySize = 0;
        for(int32_t i = 0; i < layout.count; i++)
        {
            auto& entry = layout.entries[i];
            uint32_t endPos = entry.offset + entry.size;
            requiredMemorySize = std::max<size_t>(requiredMemorySize, endPos);
        }

        NN_RESULT_THROW_UNLESS(
            bufferCount > 0 && bufferCount <= FrameBufferCountMax,
            nn::vi::ResultInvalidRange()
        );
        NN_RESULT_THROW_UNLESS(requiredMemorySize <= SharedBufferStaticStorage::Size, nn::vi::ResultResourceLimit());

        NN_VISRV_PROCESS_START();

        m_ConnectedLowLevelLayerList.Initialize();
        NN_VISRV_PROCESS_ROLLBACK(m_ConnectedLowLevelLayerList.Finalize());

        void* pMemory = nullptr;
        size_t memorySize =0;
        NN_RESULT_DO(pStorage->AcquireStorage(&pMemory, &memorySize, storageKey));

        //NN_VISRV_LOG_FBSHARE("InitializeMemoryPool...\n");
        NN_RESULT_DO(InitializeMemoryPool(pDevice, pMemory, memorySize));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeMemoryPool(pDevice));

        NN_RESULT_DO(InitializeFrameBufferList(pDevice, pGrAlloc, m_RmMemHandle, layout));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeFrameBufferList(pDevice, pGrAlloc));

        m_pOwnerClientObject = pOwner;
        m_Handle = { AcquireResourceId() };
        m_pDevice = pDevice;
        m_pGrAlloc = pGrAlloc;
        m_pMemory = pMemory;
        m_MemorySize = memorySize;
        m_pLowLevelLayerManager = pLowLevelLayerManager;
        NN_VISRV_PROCESS_SUCCESS();
        *pOutHandle = m_Handle;
        NN_RESULT_SUCCESS;
    }


    nn::Result SharedBuffer::InitializeOnTransferMemory(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        Device* pDevice,
        GrAlloc* pGrAlloc,
        nn::sf::NativeHandle& transferMemoryHandle,
        size_t transferMemorySize,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
        client::ClientObject* pOwner,
        detail::SharedLowLevelLayerManager* pLowLevelLayerManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        int bufferCount = layout.count;
        size_t requiredMemorySize = 0;
        for(int32_t i = 0; i < layout.count; i++)
        {
            auto& entry = layout.entries[i];
            uint32_t endPos = entry.offset + entry.size;
            requiredMemorySize = std::max<size_t>(requiredMemorySize, endPos);
        }

        NN_RESULT_THROW_UNLESS(
            bufferCount > 0 && bufferCount <= FrameBufferCountMax,
            nn::vi::ResultInvalidRange()
        );
        NN_RESULT_THROW_UNLESS(
            transferMemorySize >= requiredMemorySize,
            nn::vi::ResultInvalidRange()
        );

        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(m_TransferMemory.Bind(transferMemoryHandle, transferMemorySize, nn::vi::fbshare::SharedBufferMemoryPermission));
        NN_VISRV_PROCESS_ROLLBACK(m_TransferMemory.Unbind());

        void* pMemory = m_TransferMemory.GetBoundMemory();
        size_t memorySize = m_TransferMemory.GetBoundMemorySize();

        m_ConnectedLowLevelLayerList.Initialize();
        NN_VISRV_PROCESS_ROLLBACK(m_ConnectedLowLevelLayerList.Finalize());

        NN_RESULT_DO(InitializeMemoryPool(pDevice, pMemory, memorySize));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeMemoryPool(pDevice));

        NN_RESULT_DO(InitializeFrameBufferList(pDevice, pGrAlloc, m_RmMemHandle, layout));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeFrameBufferList(pDevice, pGrAlloc));

        m_pOwnerClientObject = pOwner;
        m_Handle = { AcquireResourceId() };
        m_pDevice = pDevice;
        m_pGrAlloc = pGrAlloc;
        m_pMemory = pMemory;
        m_MemorySize = memorySize;
        m_pLowLevelLayerManager = pLowLevelLayerManager;
        NN_VISRV_PROCESS_SUCCESS();
        *pOutHandle = m_Handle;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::InitializeOnProcessHeap(
        nn::vi::fbshare::SharedBufferHandle* pOutHandle,
        Device* pDevice,
        GrAlloc* pGrAlloc,
        ResourceId blockId,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
        client::ClientObject* pOwner,
        detail::SharedLowLevelLayerManager* pLowLevelLayerManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_EQUAL(blockId, 0);

        NN_VISRV_PROCESS_START();

        int bufferCount = layout.count;
        NN_RESULT_THROW_UNLESS(
            bufferCount > 0 && bufferCount <= FrameBufferCountMax,
            nn::vi::ResultInvalidRange()
        );

        void* pMemory = nullptr;
        size_t memorySize =0;
        NN_RESULT_DO(g_ProcessHeapBlockManager.GetMemory(&pMemory, &memorySize, blockId));

        size_t requiredMemorySize = 0;
        for(int32_t i = 0; i < layout.count; i++)
        {
            auto& entry = layout.entries[i];
            uint32_t endPos = entry.offset + entry.size;
            requiredMemorySize = std::max<size_t>(requiredMemorySize, endPos);
        }
        NN_RESULT_THROW_UNLESS(requiredMemorySize <= memorySize, nn::vi::ResultResourceLimit());

        m_ConnectedLowLevelLayerList.Initialize();
        NN_VISRV_PROCESS_ROLLBACK(m_ConnectedLowLevelLayerList.Finalize());

        NN_RESULT_DO(InitializeMemoryPool(pDevice, pMemory, memorySize));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeMemoryPool(pDevice));

        NN_RESULT_DO(InitializeFrameBufferList(pDevice, pGrAlloc, m_RmMemHandle, layout));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeFrameBufferList(pDevice, pGrAlloc));

        m_pOwnerClientObject = pOwner;
        m_Handle = { AcquireResourceId() };
        m_pDevice = pDevice;
        m_pGrAlloc = pGrAlloc;
        m_pMemory = pMemory;
        m_MemorySize = memorySize;
        m_pLowLevelLayerManager = pLowLevelLayerManager;
        NN_VISRV_PROCESS_SUCCESS();
        *pOutHandle = m_Handle;
        NN_RESULT_SUCCESS;
    }


    void SharedBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ConnectedLowLevelLayerList.GetCount(), 0);

        m_ImporterAruidList.Clear();

        FinalizeFrameBufferList(m_pDevice, m_pGrAlloc);
        FinalizeMemoryPool(m_pDevice);
        m_ConnectedLowLevelLayerList.Finalize();

        if(m_TransferMemory.IsBound())
        {
            m_TransferMemory.Unbind();
        }

        m_pLowLevelLayerManager = nullptr;
        m_MemorySize = 0;
        m_pMemory = nullptr;
        m_pGrAlloc = nullptr;
        m_pDevice = nullptr;
        m_Handle = { 0 };
        m_pOwnerClientObject = nullptr;
    }

    nn::Result SharedBuffer::InitializeMemoryPool(Device* pDevice, void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMemory);
        NN_SDK_REQUIRES_GREATER(size, 0u);

        size_t alignment = SharedBufferMemoryUnitSize;
        auto hDevice = pDevice->Get();
        // NOTE: MemKind は GPU マッピングを作るときに必要なのでここでは設定不要
        //NvRmMemKind kind = NvRmMemKind_Generic_16Bx2;

        NVRM_DEFINE_MEM_HANDLE_ATTR(memAttributes);
        NVRM_MEM_HANDLE_SET_ATTR(
            memAttributes,
            static_cast<NvU32>(alignment),
            NvOsMemAttribute_WriteBack, // isCoherent ? NvOsMemAttribute_WriteCombined : NvOsMemAttribute_WriteBack;
            static_cast<NvU32>(size),
            NvRmMemTags_None
        );
        //NVRM_MEM_HANDLE_SET_KIND_ATTR(memAttributes, kind);
        NVRM_MEM_HANDLE_SET_VA_ATTR(memAttributes, reinterpret_cast<NvU64>(pMemory));

        NvError err = NvRmMemHandleAllocAttr(hDevice, &memAttributes, &m_RmMemHandle);
        if(err != NvSuccess)
        {
            NN_VISRV_LOG_ERR("NvRmHandleAllocAttr failed(%d)\n", err);
        }
        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            nn::vi::ResultOperationFailed()
        );

        NN_RESULT_SUCCESS;
    }

    void SharedBuffer::FinalizeMemoryPool(Device* pDevice) NN_NOEXCEPT
    {
        NvRmMemHandleFree(m_RmMemHandle);
        m_RmMemHandle = 0;
    }

    nn::Result SharedBuffer::InitializeFrameBufferList(
        Device* pDevice,
        GrAlloc* pGrAlloc,
        NvRmMemHandle hMem,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        int i = 0;
        NN_VISRV_PROCESS_START();

        NN_VISRV_PROCESS_ROLLBACK(
            for(int j = 0; j < i; j++)
            {
                m_FrameBufferList[j].Finalize(pDevice->Get(), pGrAlloc->Get());
            }
        );
        for(; i < layout.count; i++)
        {
            auto& entry = layout.entries[i];
            NN_RESULT_DO(m_FrameBufferList[i].Initialize(
                pDevice->Get(),
                hMem,
                static_cast<ptrdiff_t>(entry.offset),
                static_cast<size_t>(entry.size),
                static_cast<int>(entry.width),
                static_cast<int>(entry.height),
                pGrAlloc->Get()
            ));
        }

        m_FrameBufferLayout = layout;
        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SharedBuffer::FinalizeFrameBufferList(Device* pDevice, GrAlloc* pGrAlloc) NN_NOEXCEPT
    {
        for(int i = 0; i < m_FrameBufferLayout.count; i++)
        {
            m_FrameBufferList[i].Finalize(pDevice->Get(), pGrAlloc->Get());
        }
        m_FrameBufferLayout = {};
    }

    //--------------------------------------------------------------------------------------------------

    nn::Result SharedBuffer::RegisterImporterAruid(nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(importerAruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultInvalidRange());

        NN_VISRV_LOG_FBSHARE("Registering Importer Aruid\n");
        NN_VISRV_PROCESS_START();

        // 既に Export されていたらエラーにする。
        NN_RESULT_THROW_UNLESS(!m_ImporterAruidList.Contains(importerAruid), nn::vi::ResultAlreadyOpened());

        NN_RESULT_DO(m_ImporterAruidList.Register(importerAruid));
        NN_VISRV_PROCESS_ROLLBACK(m_ImporterAruidList.Unregister(importerAruid));

        NN_RESULT_THROW_UNLESS(NvRmMemHandleExportForAruid(m_RmMemHandle, static_cast<NvU64>(importerAruid.lower)) == NvSuccess, nn::vi::ResultOperationFailed());
        NN_VISRV_PROCESS_ROLLBACK(NvRmMemHandleRemoveExportForAruid(m_RmMemHandle, static_cast<NvU64>(importerAruid.lower)));

        NN_VISRV_LOG_FBSHARE("Exported handle %llu to aruid%llu\n", m_RmMemHandle, importerAruid);
        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::UnregisterImporterAruid(nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(importerAruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultInvalidRange());
        NN_VISRV_LOG_FBSHARE("Unregistering Importer Aruid\n");

        NN_RESULT_DO(m_ImporterAruidList.Unregister(importerAruid));

        auto result = NvRmMemHandleRemoveExportForAruid(m_RmMemHandle, static_cast<NvU64>(importerAruid.lower));
        if(result != NvSuccess)
        {
            NN_VISRV_LOG_FBSHARE_ERR("Removing shared buffer export entry failed: %d\n", result);
        }

        NN_VISRV_LOG_FBSHARE("Unexported handle %llu from aruid%llu\n", m_RmMemHandle, importerAruid);
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::GetMemoryHandleId(
        nn::vi::native::NativeMemoryHandleId* pOutMemHandleId,
        size_t* pOutSize,
        nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout,
        nn::applet::AppletResourceUserId importerAruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutMemHandleId);
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutLayout);
        NN_RESULT_THROW_UNLESS(importerAruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(m_ImporterAruidList.Contains(importerAruid), nn::vi::ResultDenied());

        // The documentation of NvRmMemGetFd says that fd should be closed,
        // but we can see in nvrm_memmgr_hos.cpp, 'Note: On HOS, the "fd" from GetFd must not be closed'.
        *pOutMemHandleId = { static_cast<uint32_t>(NvRmMemGetFd(m_RmMemHandle)) };
        *pOutSize        = m_MemorySize;
        *pOutLayout      = m_FrameBufferLayout;
        NN_RESULT_SUCCESS;
    }

    //--------------------------------------------------------------------------------------------------

    nn::Result SharedBuffer::ConnectLowLevelLayer(detail::SharedLowLevelLayer* pLowLayer) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ConnectedLowLevelLayerList.Register(pLowLayer));
        NN_RESULT_SUCCESS;
    }

    void SharedBuffer::DisconnectLowLevelLayer(detail::SharedLowLevelLayer* pLowLayer) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectedLowLevelLayerList.Unregister(pLowLayer));
    }

    //--------------------------------------------------------------------------------------------------

    namespace {
        native::WriteSurfaceDestinationOption GetWriteSurfaceDestinationOption(const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
        {
            uint32_t out = {};
            if(option.colorOption & nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha)
            {
                out |= native::WriteSurfaceDestinationOption_PreMultipleAlpha;
            }
            if(option.alphaOption & nn::vi::fbshare::SharedTextureAlphaOption_Opaque)
            {
                out |= native::WriteSurfaceDestinationOption_Opaque;
            }
            if(option.transform & nn::vi::ImageTransform_YFlip)
            {
                out |= native::WriteSurfaceDestinationOption_FlipY;
            }
            return static_cast<native::WriteSurfaceDestinationOption>(out);
        }

        native::WriteSurfaceSourceOption GetWriteSurfaceSourceOption(nn::vi::ImageTransformType transform) NN_NOEXCEPT
        {
            uint32_t out = {};
            if(transform == nn::vi::ImageTransform_YFlip)
            {
                out |= native::WriteSurfaceSourceOption_FlipY;
            }
            return static_cast<native::WriteSurfaceSourceOption>(out);
        }
    }

    void SharedBuffer::FillColor(int index, nn::util::Color4u8 color, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0 , m_FrameBufferLayout.count);

        NN_VISRV_LOG_FBSHARE("FillColor %d -> (%d,%d,%d;%d)\n",
            index,
            static_cast<int>(color.GetR()),
            static_cast<int>(color.GetG()),
            static_cast<int>(color.GetB()),
            static_cast<int>(color.GetA())
        );

        auto& e = m_FrameBufferLayout.entries[index];
        NN_UTIL_SCOPE_EXIT
        {
            // Flush
            nn::os::FlushDataCache(reinterpret_cast<uint8_t*>(m_pMemory) + e.offset, e.size);
        };

        native::FillSurfaceRgba8(
            reinterpret_cast<uint8_t*>(m_pMemory) + e.offset,
            e.size,
            reinterpret_cast<uint32_t&>(color),
            GetWriteSurfaceDestinationOption(option)
        );

        // バッファのパラメータを設定。
        detail::SharedFrameBufferContentParameter param = detail::SharedFrameBufferContentParameter::GetDefaultValue();
        param.layerStacks = option.stacks;
        param.transform   = option.transform;
        m_FrameBufferList[index].SetContentParameter(param);
    }

    nn::Result SharedBuffer::GetImage(size_t* pOutReadSize, void* buffer, size_t size, int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_FrameBufferLayout.count);

        auto& e = m_FrameBufferLayout.entries[index];
        NN_UTIL_SCOPE_EXIT
        {
            // Invalidate
            nn::os::FlushDataCache(reinterpret_cast<uint8_t*>(m_pMemory) + e.offset, e.size);
        };

        auto& src = m_FrameBufferList[index];
        auto param = src.GetContentParameter();
        if((param.transform & nn::vi::ImageTransform_YFlip) == 0)
        {
            NN_RESULT_DO(native::ReadFromSurfaceBlockLinearRgba8(
                buffer,
                size,
                reinterpret_cast<char*>(m_pMemory) + e.offset,
                0 /* x */,
                0 /* y */,
                e.width,
                e.height,
                m_FrameBufferList[index].GetSurface().Pitch,
                m_FrameBufferList[index].GetSurface().BlockHeightLog2
            ));
        }
        else
        {
            auto pDst = reinterpret_cast<uint8_t*>(buffer);
            for(int srcY = 0; srcY < src.GetHeight(); srcY++)
            {
                int dstY = src.GetHeight() - 1 - srcY;
                NN_SDK_ASSERT_GREATER_EQUAL(dstY, 0);
                auto stride = 4 * e.width;
                NN_RESULT_DO(native::ReadFromSurfaceBlockLinearRgba8(
                    pDst + stride * dstY,
                    stride,
                    reinterpret_cast<char*>(m_pMemory) + e.offset,
                    0 /* x */,
                    srcY,
                    e.width,
                    1 /* height */,
                    m_FrameBufferList[index].GetSurface().Pitch,
                    m_FrameBufferList[index].GetSurface().BlockHeightLog2
                ));
            }
        }

        *pOutReadSize = 4 * e.width * e.height;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::SetImage(
        int index,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransformType srcTransform
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_FrameBufferLayout.count);

        auto& e = m_FrameBufferLayout.entries[index];
        NN_UTIL_SCOPE_EXIT
        {
            // Flush
            nn::os::FlushDataCache(reinterpret_cast<uint8_t*>(m_pMemory) + e.offset, e.size);
        };

        NN_RESULT_DO(native::WriteToSurfaceBlockLinearRgba8(
            reinterpret_cast<char*>(m_pMemory) + e.offset,
            e.size,
            e.width,
            e.height,
            buffer,
            size,
            0 /* x */,
            0 /* y */,
            e.width,
            e.height,
            m_FrameBufferList[index].GetSurface().Pitch,
            m_FrameBufferList[index].GetSurface().BlockHeightLog2,
            GetWriteSurfaceDestinationOption(dstOption),
            GetWriteSurfaceSourceOption(srcTransform)
        ));

        // バッファのパラメータを設定。
        detail::SharedFrameBufferContentParameter param = detail::SharedFrameBufferContentParameter::GetDefaultValue();
        param.layerStacks = dstOption.stacks;
        param.transform   = dstOption.transform;
        m_FrameBufferList[index].SetContentParameter(param);

        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::SetSubImage(
        int index,
        int x,
        int y,
        int w,
        int h,
        nn::util::Color4u8 bgColor,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransformType srcTransform
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_FrameBufferLayout.count);

        auto& e = m_FrameBufferLayout.entries[index];
        NN_UTIL_SCOPE_EXIT
        {
            // Flush
            nn::os::FlushDataCache(reinterpret_cast<uint8_t*>(m_pMemory) + e.offset, e.size);
        };

        // 背景色で塗りつぶす
        native::FillSurfaceRgba8(
            reinterpret_cast<uint8_t*>(m_pMemory) + e.offset,
            static_cast<size_t>(e.size),
            reinterpret_cast<uint32_t&>(bgColor),
            GetWriteSurfaceDestinationOption(dstOption)
        );

        if(w > 0 && h > 0)
        {
            // 範囲チェックは WriteToSurface の中で行われる
            NN_RESULT_DO(native::WriteToSurfaceBlockLinearRgba8(
                reinterpret_cast<char*>(m_pMemory) + e.offset,
                e.size,
                e.width,
                e.height,
                buffer,
                size,
                x,
                y,
                w,
                h,
                m_FrameBufferList[index].GetSurface().Pitch,
                m_FrameBufferList[index].GetSurface().BlockHeightLog2,
                GetWriteSurfaceDestinationOption(dstOption),
                GetWriteSurfaceSourceOption(srcTransform)
            ));
        }

        // バッファのパラメータを設定。
        detail::SharedFrameBufferContentParameter param = detail::SharedFrameBufferContentParameter::GetDefaultValue();
        param.layerStacks = dstOption.stacks;
        param.transform   = dstOption.transform;
        m_FrameBufferList[index].SetContentParameter(param);

        NN_RESULT_SUCCESS;
    }

    nn::Result SharedBuffer::CopyImage(SharedBuffer* pDstBuffer, int dstIndex, SharedBuffer* pSrcBuffer, int srcIndex, const nn::vi::fbshare::SharedTextureOption& option, nn::vi::LayerStackFlagType stacksFilter, nn::vi::LayerStackFlagType nullStacks) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDstBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pSrcBuffer);
        NN_SDK_REQUIRES(pDstBuffer->IsInitialized());
        NN_SDK_REQUIRES(pSrcBuffer->IsInitialized());
        NN_SDK_REQUIRES_RANGE(dstIndex, 0, pDstBuffer->m_FrameBufferLayout.count);
        NN_SDK_REQUIRES_RANGE(srcIndex, 0, pSrcBuffer->m_FrameBufferLayout.count);

        auto& dst = pDstBuffer->m_FrameBufferList[dstIndex];
        auto& src = pSrcBuffer->m_FrameBufferList[srcIndex];
        auto srcParam = src.GetContentParameter();

        if(pDstBuffer == pSrcBuffer && dstIndex == srcIndex)
        {
            // コピー元とコピー先が同じ場合、レイヤスタックだけ変更する
            srcParam.layerStacks &= stacksFilter;
            if(srcParam.layerStacks == 0)
            {
                srcParam.layerStacks = nullStacks;
            }
            src.SetContentParameter(srcParam);
            NN_RESULT_SUCCESS;
        }

        vic::VicCopySingleDestinationInfo dstInfo = {};
        dstInfo.pSurface = &dst.GetSurface();
        dstInfo.region.x = 0;
        dstInfo.region.y = 0;
        dstInfo.region.width = dst.GetWidth();
        dstInfo.region.height = dst.GetHeight();
        dstInfo.transform = option.transform;

        vic::VicCopySingleSourceInfo srcInfo = {};
        srcInfo.pSurface = &src.GetSurface();
        srcInfo.transform = srcParam.transform;
        if(srcParam.cropRegion.width != 0 && srcParam.cropRegion.height != 0)
        {
            srcInfo.region.x = srcParam.cropRegion.x;
            srcInfo.region.y = srcParam.cropRegion.y;
            srcInfo.region.width = srcParam.cropRegion.width;
            srcInfo.region.height = srcParam.cropRegion.height;
        }
        else
        {
            srcInfo.region.x = 0;
            srcInfo.region.y = 0;
            srcInfo.region.width = src.GetWidth();
            srcInfo.region.height = src.GetHeight();
        }

        vic::VicCopyOption copyOpt = {};
        copyOpt.filter = vic::VicCopyFilter_Bilinear;
        if(option.alphaOption == nn::vi::fbshare::SharedTextureAlphaOption_None)
        {
            copyOpt.alphaMode = vic::VicCopyAlphaMode_Source;
            copyOpt.backgroundColor = nn::util::Color4f(0, 0, 0, 0);
        }
        else
        {
            copyOpt.alphaMode = vic::VicCopyAlphaMode_Opaque;
            copyOpt.backgroundColor = nn::util::Color4f(0, 0, 0, 1);
        }

        NvRmFence copyFence = {};
        NN_RESULT_DO(vic::VicOperation::CopySingle(
            &copyFence,
            dstInfo,
            &vic::g_VicModule,
            srcInfo,
            copyOpt
        ));

        // その場で待つ
        native::SyncpointEntry waitEntry(copyFence);
        native::g_SyncpointWaiter.Enqueue(&waitEntry);
        waitEntry.Wait();

        // Destination 側のバッファのパラメータを設定。
        detail::SharedFrameBufferContentParameter dstParam = detail::SharedFrameBufferContentParameter::GetDefaultValue();
        // レイヤスタックを計算
        {
            auto stacks = srcParam.layerStacks & stacksFilter; // filter で指定されたスタックに制限する
            if(stacks == 0)
            {
                stacks = nullStacks; // どこのスタックにも入らない場合、 nullStacks のスタックに入れる
            }
            dstParam.layerStacks = stacks;
        }
        dstParam.transform = dstInfo.transform;

        pDstBuffer->m_FrameBufferList[dstIndex].SetContentParameter(dstParam);
        NN_RESULT_SUCCESS;
    }

}}}
