﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedLowLevelLayerManager.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include "../../visrv_ServerManager.h"
#include "../../visrv_Macro.h"

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{
    static const int GlobalMultiWaitIndexCount = MultiWaitIndex_SharedLowLevelLayerMax - MultiWaitIndex_SharedLowLevelLayerMin + 1;
    static const int RequiredMultiWaitIndexCount = SharedLowLevelLayerManager::Size * SharedLowLevelLayer::MultiWaitTagCount;

    NN_STATIC_ASSERT(GlobalMultiWaitIndexCount >= RequiredMultiWaitIndexCount);

    const int SharedLowLevelLayerManager::Size;
    const uintptr_t SharedLowLevelLayerManager::MultiWaitTagMin;
    const uintptr_t SharedLowLevelLayerManager::MultiWaitTagMax;

    SharedLowLevelLayerManager::SharedLowLevelLayerManager() NN_NOEXCEPT
        : m_LayerPool()
    {
    }

    void SharedLowLevelLayerManager::Initialize() NN_NOEXCEPT
    {
        m_LayerPool.Initialize();
    }

    void SharedLowLevelLayerManager::Finalize() NN_NOEXCEPT
    {
        m_LayerPool.Finalize();
    }

    nn::Result SharedLowLevelLayerManager::BindManagedLayer(
        SharedLowLevelLayer** pOutLayer,
        client::ClientObject* pClientObject,
        nn::vi::LayerId layerId,
        const char* displayName,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutLayer);
        NN_SDK_REQUIRES_NOT_NULL(pClientObject);
        NN_VISRV_PROCESS_START();

        // allocate
        SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_LayerPool.Allocate(&pLayer));
        NN_VISRV_PROCESS_ROLLBACK(m_LayerPool.Free(pLayer));

        // initialize
        NN_RESULT_DO(pLayer->InitializeForManagedLayer(pClientObject, layerId, displayName, aruid));
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Finalize());
        auto index= m_LayerPool.GetIndex(pLayer);
        pLayer->SetupMultiWaitAdaptor(MultiWaitIndex_SharedLowLevelLayerMin + SharedLowLevelLayer::MultiWaitTagCount * index);

        NN_SDK_ASSERT_EQUAL(pLayer->GetState(), SharedLowLevelLayerState_Bound);
        NN_VISRV_PROCESS_SUCCESS();
        *pOutLayer = pLayer;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedLowLevelLayerManager::BindIndirectLayer(
        SharedLowLevelLayer** pOutLayer,
        client::ClientObject* pClientObject,
        nn::vi::IndirectProducerHandleType handle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutLayer);
        NN_SDK_REQUIRES_NOT_NULL(pClientObject);
        NN_VISRV_PROCESS_START();

        // allocate
        SharedLowLevelLayer* pLayer = nullptr;
        NN_RESULT_DO(m_LayerPool.Allocate(&pLayer));
        NN_VISRV_PROCESS_ROLLBACK(m_LayerPool.Free(pLayer));

        // initialize
        NN_RESULT_DO(pLayer->InitializeForIndirectLayer(pClientObject, handle, aruid));
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Finalize());
        auto index= m_LayerPool.GetIndex(pLayer);
        pLayer->SetupMultiWaitAdaptor(MultiWaitIndex_SharedLowLevelLayerMin + SharedLowLevelLayer::MultiWaitTagCount * index);

        NN_SDK_ASSERT_EQUAL(pLayer->GetState(), SharedLowLevelLayerState_Bound);
        NN_VISRV_PROCESS_SUCCESS();
        *pOutLayer = pLayer;
        NN_RESULT_SUCCESS;
    }

    void SharedLowLevelLayerManager::UnbindLayer(SharedLowLevelLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pLayer);
        pLayer->Finalize();
        m_LayerPool.Free(pLayer);
    }

    void SharedLowLevelLayerManager::HandleMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitTagMin, MultiWaitTagMax);
        int index = (pHolder->userData - MultiWaitTagMin) / SharedLowLevelLayer::MultiWaitTagCount;
        NN_SDK_ASSERT_RANGE(index, 0, Size);

        auto pLayer = m_LayerPool.GetAt(index);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        pLayer->DispatchMultiWait(pHolder);
    }


}}}}
