﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedFrameBuffer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>

#include "../../visrv_Log.h"
#include "../../visrv_Macro.h"
#include "../../native/visrv_SyncpointWaiter.h"

#include <nvrm_surface.h>
#include <nvnflinger_service.h>

namespace nn{ namespace visrv{ namespace fbshare{
    static const NvColorFormat     SharedFrameBufferColorFormat = NvColorFormat_A8B8G8R8;
    static const NvRmSurfaceLayout SharedFrameBufferLayout      = NvRmSurfaceLayout_Blocklinear;
}}}

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{

    SharedFrameBufferContentParameter SharedFrameBufferContentParameter::GetDefaultValue() NN_NOEXCEPT
    {
        SharedFrameBufferContentParameter param = {};
        param.presentInterval = 1;
        return param;
    }

    //------------------------

    SharedFrameBuffer::SharedFrameBuffer() NN_NOEXCEPT
        : m_RmSurface()
        , m_BufferHandle()
        , m_pGraphicBuffer()
        , m_ContentParameter(SharedFrameBufferContentParameter::GetDefaultValue())
        , m_PresentedLowLevelLayerId(0)
        , m_AttachedClientLayerHandle(nn::vi::fbshare::SharedLayerHandle::GetInvalidValue())
    {
    }

    nn::Result SharedFrameBuffer::Initialize(
        NvRmDeviceHandle hDevice,
        NvRmMemHandle hMem,
        ptrdiff_t offset,
        size_t size,
        int width,
        int height,
        alloc_device_t* pGrAlloc
    ) NN_NOEXCEPT
    {
        NN_VISRV_PROCESS_START();

        NN_RESULT_THROW_UNLESS(width > 0, nn::vi::ResultInvalidDimensions());
        NN_RESULT_THROW_UNLESS(height > 0, nn::vi::ResultInvalidDimensions());

        NN_RESULT_DO(InitializeSurface(&m_RmSurface, hDevice, hMem, offset, size, width, height));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeSurface(&m_RmSurface, hDevice));

        NN_RESULT_DO(InitializeGraphicBuffer(&m_BufferHandle, &m_pGraphicBuffer, pGrAlloc, m_RmSurface, width, height));
        NN_VISRV_PROCESS_ROLLBACK(FinalizeGraphicBuffer(&m_BufferHandle, &m_pGraphicBuffer, pGrAlloc));

        m_ContentParameter = SharedFrameBufferContentParameter::GetDefaultValue();

        m_AcquireFence = android::Fence::NO_FENCE;
        m_ReleaseFence = android::Fence::NO_FENCE;

        m_PresentedLowLevelLayerId = 0;
        m_AttachedClientLayerHandle = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();

        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SharedFrameBuffer::Finalize(
        NvRmDeviceHandle hDevice,
        alloc_device_t* pGrAlloc
    ) NN_NOEXCEPT
    {
        m_ReleaseFence.clear();
        m_AcquireFence.clear();
        FinalizeGraphicBuffer(&m_BufferHandle, &m_pGraphicBuffer, pGrAlloc);
        FinalizeSurface(&m_RmSurface, hDevice);

        m_ContentParameter = SharedFrameBufferContentParameter::GetDefaultValue();
        m_PresentedLowLevelLayerId = 0;
        m_AttachedClientLayerHandle = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();
    }

    bool SharedFrameBuffer::IsInitialized() const NN_NOEXCEPT
    {
        return m_pGraphicBuffer != nullptr;
    }

    bool SharedFrameBuffer::IsAttachedToClientLayer() const NN_NOEXCEPT
    {
        return m_AttachedClientLayerHandle != nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();
    }

    nn::Result SharedFrameBuffer::InitializeSurface(
        NvRmSurface* pSurface,
        NvRmDeviceHandle hDevice,
        NvRmMemHandle hMem,
        ptrdiff_t offset,
        size_t size,
        int width,
        int height
        ) NN_NOEXCEPT
    {
        NN_VISRV_PROCESS_START();
        NN_VISRV_PROCESS_ROLLBACK(std::memset(pSurface, 0, sizeof(NvRmSurface)));

        NN_RESULT_THROW_UNLESS(offset >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(size >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS((offset % SharedBufferMemoryUnitSize) == 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS((size % SharedBufferMemoryUnitSize) == 0, nn::vi::ResultInvalidRange());

        const NvU32 surfaceAttributes[] = {
            NvRmSurfaceAttribute_Layout, SharedFrameBufferLayout,
            NvRmSurfaceAttribute_None
        };

        NvRmSurfaceSetup(
            pSurface,
            static_cast<NvU32>(width),
            static_cast<NvU32>(height),
            SharedFrameBufferColorFormat,
            surfaceAttributes
        );

        size_t requiredSize = static_cast<size_t>(NvRmSurfaceComputeSize(pSurface));
        size_t requiredAlignment = static_cast<size_t>(NvRmSurfaceComputeAlignment(hDevice, pSurface));
        NN_VISRV_LOG_FBSHARE("SharedFrameBuffer Memory\n");
        NN_VISRV_LOG_FBSHARE("  Required:   %llu bytes\n", static_cast<uint64_t>(requiredSize));
        NN_VISRV_LOG_FBSHARE("  Offset  :   %llu bytes\n", static_cast<uint64_t>(offset));
        NN_VISRV_LOG_FBSHARE("  Size    :   %llu bytes\n", static_cast<uint64_t>(size));
        NN_SDK_ASSERT_LESS_EQUAL(requiredAlignment, SharedBufferMemoryUnitSize);
        NN_RESULT_THROW_UNLESS(size >= requiredSize, nn::vi::ResultOperationFailed());
        NN_RESULT_THROW_UNLESS((offset % requiredAlignment) == 0, nn::vi::ResultOperationFailed());

        pSurface->hMem = hMem;
        pSurface->Offset = static_cast<NvU32>(offset);

        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SharedFrameBuffer::FinalizeSurface(NvRmSurface* pSurface, NvRmDeviceHandle hDevice) NN_NOEXCEPT
    {
        std::memset(pSurface, 0, sizeof(NvRmSurface));
    }

    nn::Result SharedFrameBuffer::InitializeGraphicBuffer(
        buffer_handle_t* pBufferHandle,
        android::sp<android::GraphicBuffer>* pGraphicBuffer,
        alloc_device_t* pGrAlloc,
        NvRmSurface& surface,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBufferHandle);
        NN_SDK_REQUIRES_NOT_NULL(pGraphicBuffer);
        NN_VISRV_PROCESS_START();

        buffer_handle_t hBuffer = nullptr;
        NN_RESULT_THROW_UNLESS(0 == nvgr_import_nvrmsurface(&surface, 1, &hBuffer), nn::vi::ResultOperationFailed());
        NN_VISRV_PROCESS_ROLLBACK(pGrAlloc->free(pGrAlloc, hBuffer));

        auto pGrBuffer = new android::GraphicBuffer(
            static_cast<uint32_t>(width),
            static_cast<uint32_t>(height),
            android::PIXEL_FORMAT_RGBA_8888,
            GRALLOC_USAGE_HW_RENDER | GRALLOC_USAGE_HW_TEXTURE | GRALLOC_USAGE_HW_FB | GRALLOC_USAGE_HW_COMPOSER,
            //GRALLOC_USAGE_HW_COMPOSER,
            static_cast<uint32_t>(nvgr_get_stride(hBuffer)),
            const_cast<native_handle*>(hBuffer),
            false
        );
        NN_RESULT_THROW_UNLESS(pGrBuffer != nullptr, nn::vi::ResultOperationFailed());

        NN_VISRV_PROCESS_SUCCESS();
        *pBufferHandle = hBuffer;
        *pGraphicBuffer = pGrBuffer;
        NN_RESULT_SUCCESS;
    }

    void SharedFrameBuffer::FinalizeGraphicBuffer(
        buffer_handle_t* pBufferHandle,
        android::sp<android::GraphicBuffer>* pGraphicBuffer,
        alloc_device_t* pGrAlloc
    ) NN_NOEXCEPT
    {
        pGraphicBuffer->clear();
        pGrAlloc->free(pGrAlloc, *pBufferHandle);
        *pBufferHandle = nullptr;
    }

    int SharedFrameBuffer::GetWidth() const NN_NOEXCEPT
    {
        return m_RmSurface.Width;
    }

    int SharedFrameBuffer::GetHeight() const NN_NOEXCEPT
    {
        return m_RmSurface.Height;
    }

    NvRmSurface& SharedFrameBuffer::GetSurface() NN_NOEXCEPT
    {
        return m_RmSurface;
    }

    const android::sp<android::GraphicBuffer>& SharedFrameBuffer::GetGraphicBuffer() NN_NOEXCEPT
    {
        return m_pGraphicBuffer;
    }

    void SharedFrameBuffer::SetContentParameter(const SharedFrameBufferContentParameter& param) NN_NOEXCEPT
    {
        m_ContentParameter = param;
    }

    SharedFrameBufferContentParameter SharedFrameBuffer::GetContentParameter() const NN_NOEXCEPT
    {
        return m_ContentParameter;
    }

    void SharedFrameBuffer::SetAcquireFence(const android::sp<android::Fence>& fence) NN_NOEXCEPT
    {
        if(fence == nullptr)
        {
            m_AcquireFence = android::Fence::NO_FENCE;
        }
        else
        {
            m_AcquireFence = fence;
        }
    }

    android::sp<android::Fence> SharedFrameBuffer::GetAcquireFence() const NN_NOEXCEPT
    {
        return m_AcquireFence;
    }

    void SharedFrameBuffer::SetReleaseFence(const android::sp<android::Fence>& fence) NN_NOEXCEPT
    {
        if(fence == nullptr)
        {
            m_ReleaseFence = android::Fence::NO_FENCE;
        }
        else
        {
            m_ReleaseFence = fence;
        }
    }

    android::sp<android::Fence> SharedFrameBuffer::GetReleaseFence() const NN_NOEXCEPT
    {
        return m_ReleaseFence;
    }

    void SharedFrameBuffer::SetPresentedLowLevelLayerId(ResourceId layerId) NN_NOEXCEPT
    {
        m_PresentedLowLevelLayerId = layerId;
    }

    ResourceId SharedFrameBuffer::GetPresentedLowLevelLayerId() const NN_NOEXCEPT
    {
        return m_PresentedLowLevelLayerId;
    }

    void SharedFrameBuffer::SetAttachedClientLayerHandle(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        m_AttachedClientLayerHandle = h;
    }

    nn::vi::fbshare::SharedLayerHandle SharedFrameBuffer::GetAttachedClientLayerHandle() const NN_NOEXCEPT
    {
        return m_AttachedClientLayerHandle;
    }

}}}}

