﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_PresentationTracerPool.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../visrv_ServerManager.h"

namespace nn{ namespace visrv{ namespace client{

    PresentationTracerPool g_PresentationTracerPool;

    void PresentationTracerPool::Initialize(
        ServerManager* pServerManager,
        uintptr_t multiWaitTagBegin,
        uintptr_t multiWaitTagCount
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(multiWaitTagCount, LayerPresentationTracerCountMax);
        m_Pool.Initialize();

        m_pServerManager = pServerManager;
        m_MultiWaitTagBegin = multiWaitTagBegin;
        m_MultiWaitTagCount = multiWaitTagCount;
    }

    void PresentationTracerPool::Finalize() NN_NOEXCEPT
    {
        m_Pool.Finalize();

        m_pServerManager = nullptr;
        m_MultiWaitTagBegin = 0;
        m_MultiWaitTagCount = 0;
    }

    nn::Result PresentationTracerPool::Acquire(PresentationTracer** pOutTracer) NN_NOEXCEPT
    {
        PresentationTracer* p = nullptr;
        NN_RESULT_DO(m_Pool.Allocate(&p));

        auto index = m_Pool.GetIndex(p);
        p->Initialize(m_MultiWaitTagBegin + index);

        *pOutTracer = p;
        NN_RESULT_SUCCESS;
    }

    void PresentationTracerPool::Release(PresentationTracer* pTracer) NN_NOEXCEPT
    {
        pTracer->Finalize();
        m_Pool.Free(pTracer);
    }

    void PresentationTracerPool::ProcessMultiWaitSignal(nn::os::MultiWaitHolderType* pHolder, bool isUnlinked) NN_NOEXCEPT
    {
        auto index = pHolder->userData - m_MultiWaitTagBegin;
        NN_SDK_ASSERT_RANGE(index, 0u, m_MultiWaitTagCount);

        auto pTracer = m_Pool.GetAt(index);
        NN_SDK_ASSERT_NOT_NULL(pTracer);

        pTracer->ProcessMultiWaitSignalImpl(pHolder, isUnlinked);
    }

    void PresentationTracerPool::LinkMultiWaitImpl(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_REQUIRES_RANGE(pHolder->userData, m_MultiWaitTagBegin, m_MultiWaitTagBegin + m_MultiWaitTagCount);
        m_pServerManager->GetDisplayServerManager()->AddUserWaitHolder(pHolder);
    }

    void PresentationTracerPool::UnlinkMultiWaitImpl(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::UnlinkMultiWaitHolder(pHolder);
    }

}}}
