﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ClientObject.h"

#include <nn/os/os_SdkSystemEventApi.h>

#include "../visrv_Config.h"
#include "../visrv_Log.h"
#include "../service/visrv_DriverConnection.h"
#include "../native/visrv_ParcelIo.h"
#include "../native/visrv_BinderFunctions.h"
#include "visrv_ClientUtility.h"
#include "../native/detail/visrv_TransactionIGraphicBufferProducer.h"

#define NN_VISRV_RELAY_CHECK_ALIVE()  \
    if(!IsAlive()){             \
        return;                 \
    }

#define NN_VISRV_RELAY_FIND_CLIENT_LAYER(VariableName, LayerId)       \
    auto VariableName = ClientUtility::LookupLayerHolderOnClient(&m_LayerHolderList, this, LayerId);


#define NN_VISRV_FIND_BINDER_ENTRY(pVariable, handle)               \
    auto pVariable = m_BinderTable.FindEntryByDriverHandle(handle); \
    if(pVariable == nullptr)                                        \
    {                                                               \
        NN_VISRV_LOG_RELAY_ERR("%s unknown sender handle: %d\n", NN_CURRENT_FUNCTION_NAME, static_cast<int>(handle)); \
        return;                                                     \
    }


namespace nn{ namespace visrv{ namespace client{

    namespace {
        const nn::vi::ProxyName DriverProxyName = { NN_VISRV_RELAY_PORTNAME };

        // WARNING: mutex is required if multithreaded
        static char g_RequestWorkBuffer[4096];

        struct CheckBinderNameInRequestArgument
        {
            bool isOk;
            native::BinderTable* pTable;

            bool isProxyNameExchangeEnabled;
            nn::vi::ProxyName clientProxyName;
        };

        void CheckBinderNameInRequest(flat_binder_object* pObject, void* userPtr) NN_NOEXCEPT
        {
            auto pArg = reinterpret_cast<CheckBinderNameInRequestArgument*>(userPtr);
            if(!pArg->isOk)
            {
                return; // NG
            }

            auto pEntry = pArg->pTable->FindEntryByDriverHandle(static_cast<int32_t>(pObject->handle));
            if(pEntry == nullptr)
            {
                NN_VISRV_LOG_RELAY_ERR("unknown handle in request: %d\n", static_cast<int>(pObject->handle));
                pArg->isOk = false;
                return; // NG
            }

            if(pArg->isProxyNameExchangeEnabled)
            {
                if(std::memcmp(pObject->service_name, &pArg->clientProxyName, sizeof(nn::vi::ProxyName)) == 0)
                {
                    std::memcpy(pObject->service_name, &DriverProxyName, sizeof(nn::vi::ProxyName));
                }
            }
        }

        struct CheckBinderNameInReplyArgument
        {
            nn::vi::DisplayId displayId;
            nn::vi::LayerId layerId;
            android::Parcel* pParcel;
            void* pDataHead;
            native::BinderTable* pTable;
            int32_t requestSenderHandle;
            uint32_t requestCode;

            bool isProxyNameExchangeEnabled;
            nn::vi::ProxyName clientProxyName;
        };

        void CheckBinderNameInReply(flat_binder_object* pObject, void* userPtr) NN_NOEXCEPT
        {
            auto pArg = reinterpret_cast<CheckBinderNameInReplyArgument*>(userPtr);

            auto handle = static_cast<int32_t>(pObject->handle);

            // check if handle is known
            auto pEntry = pArg->pTable->FindEntryByDriverHandle(handle);
#ifdef NN_VISRV_PERMIT_RELAY_ZERO_HANDLE
            if(handle == 0)
            {
                NN_VISRV_LOG_RELAY_WARN(
                    "%s zero reply handle. sender %d: code %d\n",
                    NN_CURRENT_FUNCTION_NAME,
                    static_cast<int>(pArg->requestSenderHandle),
                    static_cast<int>(pArg->requestCode)
                );
            }
            else
#endif
            if(pEntry == nullptr)
            {
                // this is a new binder. try to register
                size_t offset = static_cast<size_t>(reinterpret_cast<char*>(pObject) - reinterpret_cast<char*>(pArg->pDataHead));

                NN_VISRV_LOG_RELAY("Found new binder in reply. handle: %d, offset: %d, type: %d\n",
                    static_cast<int>(handle), static_cast<int>(offset), static_cast<int>(pObject->type)
                );

                // register new binder
                pArg->pParcel->setDataPosition(offset);
                auto pIBinder = pArg->pParcel->readStrongBinder();

                if(!pArg->pTable->AddEntry(pIBinder, native::BinderClass_Unknown, handle, pArg->displayId, pArg->layerId))
                {
                    NN_VISRV_LOG_RELAY_ERR("BinderTable is full!\n");
                }
            }

            // exchange proxy name if necessary.
            if(pArg->isProxyNameExchangeEnabled)
            {
                if(std::memcmp(pObject->service_name, &DriverProxyName, sizeof(nn::vi::ProxyName)) == 0)
                {
                    std::memcpy(pObject->service_name, &pArg->clientProxyName, sizeof(nn::vi::ProxyName));
                }
            }
        }

#ifndef NN_SDK_BUILD_RELEASE
        void DumpTransaction(
            native::BinderClassType binderType,
            uint32_t code,
            const void* requestBuffer,
            size_t requestBufferSize,
            const void* replyBuffer,
            size_t replyBufferSize
        ) NN_NOEXCEPT
        {
            #define NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(Command)               \
                case native::IGraphicBufferProducerFunctionCode_ ## Command :   \
                    {                                                           \
                        native::detail::TransactionIGraphicBufferProducer:: Command ## Request request = {};    \
                        native::detail::TransactionIGraphicBufferProducer:: Command ## Reply reply = {};        \
                        native::detail::TransactionIGraphicBufferProducer::Decode ## Command ## Request(&request, requestBuffer, requestBufferSize);    \
                        native::detail::TransactionIGraphicBufferProducer::Decode ## Command ## Reply(&reply, replyBuffer, replyBufferSize);            \
                        std::memset(requestStr, 0, sizeof(requestStr));         \
                        std::memset(replyStr, 0, sizeof(replyStr));             \
                        request.ToString(requestStr, sizeof(requestStr));       \
                        reply.ToString(replyStr, sizeof(replyStr));             \
                        NN_VISRV_LOG_TRANSACTIONDUMP("%s\n", #Command);             \
                        NN_VISRV_LOG_TRANSACTIONDUMP("  request=%s\n", requestStr); \
                        NN_VISRV_LOG_TRANSACTIONDUMP("  reply  =%s\n", replyStr);   \
                        break;                                                  \
                    }

            #define NN_VISRV_RELAY_TRANSACTION_IGNORE_CASE(Command) \
                case native::IGraphicBufferProducerFunctionCode_ ## Command : break; \

            static int lastCode[2] = {-1, -1};
            //if(code != lastCode[0] && code != lastCode[1])
            //if(m_Constants.GetPermission() == client::ClientPermission_Manager)
            if(binderType == native::BinderClass_IGraphicBufferProducer)
            {
                lastCode[1] = lastCode[0];
                lastCode[0] = code;
                static char requestStr[4096];
                static char replyStr[4096];
                switch(code)
                {
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(RequestBuffer);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(Connect);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(Disconnect);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(DequeueBuffer);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(QueueBuffer);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(SetPreallocatedBuffer);
                    NN_VISRV_RELAY_TRANSACTION_DUMP_CASE(Query);
                default:
                    {
                        NN_VISRV_LOG_TRANSACTIONDUMP("Unknown %d\n", code);
                    }
                }
            }
        }
#endif

    }// anonymous namespace

    void ClientObject::TransactParcel(
        std::int32_t handle,
        std::uint32_t code,
        const void* requestBuffer,
        size_t requestBufferSize,
        void* replyBuffer,
        size_t replyBufferSize,
        std::uint32_t flags
        ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_RELAY("TransactParcel handle %d: code %d\n", static_cast<int>(handle), static_cast<int>(code));
        NN_VISRV_RELAY_CHECK_ALIVE();

        nn::vi::DisplayId displayId = {};
        nn::vi::LayerId layerId = {};
        native::BinderClassType binderType = native::BinderClass_Unknown;

#ifdef NN_VISRV_PERMIT_RELAY_ZERO_HANDLE
        if(handle == 0)
        {
            NN_VISRV_LOG_RELAY_WARN(
                "%s zero sender handle. code %d\n",
                NN_CURRENT_FUNCTION_NAME,
                static_cast<int>(code)
            );
        }
        else
#endif
        {
            NN_VISRV_FIND_BINDER_ENTRY(pBinderEntry, handle);
            displayId = pBinderEntry->GetDisplayId();
            layerId = pBinderEntry->GetLayerId();
            binderType = pBinderEntry->GetBinderClass();
        }
        NN_VISRV_RELAY_FIND_CLIENT_LAYER(pLayerHolder, layerId);
        auto pTracer = (pLayerHolder ? pLayerHolder->pPresentationTracer : nullptr);

        // check binders in request buffer
        {
            if(requestBufferSize > sizeof(g_RequestWorkBuffer))
            {
                NN_VISRV_LOG_RELAY_ERR("Too long request buffer\n");
                return;
            }
            std::memset(g_RequestWorkBuffer, 0, sizeof(g_RequestWorkBuffer));
            std::memcpy(g_RequestWorkBuffer, requestBuffer, requestBufferSize);
            android::Parcel parcel;
            size_t dataPosition;
            native::ParcelIo::OpenParcel(&parcel, &dataPosition, g_RequestWorkBuffer, requestBufferSize);
            CheckBinderNameInRequestArgument arg;
            arg.isOk = true;
            arg.pTable = &m_BinderTable;
            arg.isProxyNameExchangeEnabled = m_Constants.IsProxyNameExchangeEnabled();
            arg.clientProxyName = m_Constants.GetExchangeProxyName();
            parcel.foreachObject(CheckBinderNameInRequest, &arg);
            native::ParcelIo::CloseParcel(&parcel);
            if(!arg.isOk)
            {
                return;
            }
        }

        // relay to nvnflinger process
        service::GetDriverConnection()->TransactParcelAuto(
            handle,
            code,
            nn::sf::InBuffer(reinterpret_cast<const char*>(g_RequestWorkBuffer), requestBufferSize),
            nn::sf::OutBuffer(reinterpret_cast<char*>(replyBuffer), replyBufferSize),
            flags
        );

        // check binders in reply buffer
        {
            android::Parcel parcel;
            size_t dataPosition;
            native::ParcelIo::OpenParcel(&parcel, &dataPosition, replyBuffer, replyBufferSize);
            CheckBinderNameInReplyArgument arg;
            arg.displayId = displayId;
            arg.layerId = layerId;
            arg.pParcel = &parcel;
            arg.pDataHead = reinterpret_cast<char*>(replyBuffer) + dataPosition;;
            arg.pTable = &m_BinderTable;
            arg.requestSenderHandle = handle;
            arg.requestCode = code;
            arg.isProxyNameExchangeEnabled = m_Constants.IsProxyNameExchangeEnabled();
            arg.clientProxyName = m_Constants.GetExchangeProxyName();
            parcel.foreachObject(CheckBinderNameInReply, &arg);
            native::ParcelIo::CloseParcel(&parcel);
        }

        // For debug: dump transaction
#ifndef NN_SDK_BUILD_RELEASE
        DumpTransaction(
            binderType,
            code,
            requestBuffer,
            requestBufferSize,
            replyBuffer,
            replyBufferSize
        );
#endif

        // trace requests which are sent to nvnflinger
        if( pTracer &&
            binderType == native::BinderClass_IGraphicBufferProducer &&
            code == native::IGraphicBufferProducerFunctionCode_QueueBuffer )
        {
            (void)pTracer->RecordQueueBufferRequest(handle, code, requestBuffer, requestBufferSize, flags);
        }

        // signal first presentation event
        if( pLayerHolder &&
            pLayerHolder->layerTexturePresentingEvent.isInitialized &&
            binderType == native::BinderClass_IGraphicBufferProducer &&
            code == native::IGraphicBufferProducerFunctionCode_QueueBuffer )
        {
            nn::os::SignalSystemEvent(&pLayerHolder->layerTexturePresentingEvent.event);
        }

    }// NOLINT(impl/function_size)

    void ClientObject::AdjustRefcount(
        std::int32_t handle,
        std::int32_t diff,
        std::int32_t isStrong
        ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_RELAY("AdjustRefcount handle %d: inc %d: %s\n", static_cast<int>(handle), static_cast<int>(diff), isStrong != 0 ? "strong" : "weak");
        NN_VISRV_RELAY_CHECK_ALIVE();

        if(isStrong)
        {
            NN_VISRV_FIND_BINDER_ENTRY(p, handle);
            p->AddRemoteStrongReferenceCount(diff);
            if(diff < 0 && p->GetRemoteStrongReferenceCount() <= 0)
            {
                NN_VISRV_LOG_RELAY("  handle removed\n");
                p->Finalize();
            }
        }
        else
        {
            if(auto p = m_BinderTable.FindEntryByDriverHandle(handle))
            {
                p->AddRemoteWeakReferenceCount(diff);
            }
        }
    }

    void ClientObject::GetNativeHandle(
        std::int32_t handle,
        std::uint32_t code,
        nn::sf::Out<nn::sf::NativeHandle>& result
        ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_RELAY("GetNativeHandle handle %d: code %d", static_cast<int>(handle), static_cast<int>(code));
        NN_VISRV_RELAY_CHECK_ALIVE();

#ifdef NN_VISRV_PERMIT_RELAY_ZERO_HANDLE
        if(handle != 0)
        {
            NN_VISRV_FIND_BINDER_ENTRY(p, handle);
        }
#else
        NN_VISRV_FIND_BINDER_ENTRY(p, handle);
#endif

        NN_VISRV_LOG_RELAY("  Relay GetNativeHandle\n");
        service::GetDriverConnection()->GetNativeHandle(
            handle,
            code,
            result
            );

    }

}}}
