﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_ScalingMode.h>
#include <nn/vi/vi_RgbRange.h>
#include <nn/vi/vi_ContentType.h>
#include <nn/vi/vi_CmuMode.h>
#include <nn/vi/vi_HotplugState.h>
#include <nn/vi/vi_PowerState.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_DisplayError.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/sf/vi_PolicyLevel.h>
#include "../visrv_Config.h"
#include "../native/visrv_BinderTable.h"

#include "visrv_ClientDisplayHolder.h"
#include "visrv_ClientObjectSmartHolder.h"
#include "visrv_ClientConstants.h"
#include "../indirect/visrv_IndirectLayer.h"
#include "../indirect/visrv_IndirectProducerBinderTable.h"
#include "../indirect/visrv_IndirectCopyImageMapDefer.h"
#include "../fbshare/visrv_SharingClientObject.h"
#include "../util/visrv_TIntegerContainer.h"

namespace nn{ namespace visrv{ namespace client{
    class ClientObjectAccessor;

    class ClientObject
        : public ClientObjectRefcount
        , public fbshare::SharingClientObject
    {
        friend ClientObjectSmartHolder;
    public:
        typedef util::TIntegerContainer<ResourceId, ProcessHeapBlockCountMax, 0, util::ContainerThreadSafetyOption_SingleThread> OwningProcessHeapBlockIdList;

    public:
        NN_IMPLICIT ClientObject(const ClientConstants& constants) NN_NOEXCEPT;
        ~ClientObject() NN_NOEXCEPT;

        bool IsAlive() const NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void CloseAllDisplays() NN_NOEXCEPT;

        const ClientConstants& GetConstants() const NN_NOEXCEPT;

    //---------
    // ProcessHeap
    //---------
        nn::Result AllocateProcessHeapBlock(ResourceId* pOutBlockId, size_t size) NN_NOEXCEPT;
        nn::Result FreeProcessHeapBlock(ResourceId blockId) NN_NOEXCEPT;

    //---------
    // Display
    //---------
        nn::Result ListDisplays(int64_t* pOutCount, nn::vi::DisplayInfo* pOutInfoList, int infoListCapacity) NN_NOEXCEPT;
        nn::Result ListDisplays(int64_t* pOutCount, nn::vi::DisplayInfo* pOutInfoList, int infoListCapacity, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT;
        nn::Result OpenDisplay(nn::vi::DisplayId* pOutDisplayId, const char* name) NN_NOEXCEPT;
        nn::Result OpenDisplay(nn::vi::DisplayId* pOutDisplayId, const char* name, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT;
        nn::Result CloseDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT;

        nn::Result GetDisplayResolution(int64_t* pOutWidth, int64_t* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayLogicalResolution(int* pOutWidth, int* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT;

        nn::Result GetZOrderCountMin(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetZOrderCountMax(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;

        nn::Result SetDisplayMagnification(nn::vi::DisplayId displayId, int x, int y, int width, int height) NN_NOEXCEPT;

    //---------
    // Layer
    //---------
        nn::Result CreateManagedLayer(
            nn::vi::LayerId* pOutLayerId,
            nn::vi::DisplayId displayId,
            nn::vi::LayerSettingsType settings,
            nn::applet::AppletResourceUserId userAruid
            ) NN_NOEXCEPT;

        nn::Result BindManagedLayer(
            size_t* pOutNativeWindowDataSize,
            void* pNativeWindowDataBuffer,
            size_t nativeWindowDataBufferSize,
            nn::vi::LayerId layerId,
            const char* displayName,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;

        nn::Result UnbindManagedLayer(
            nn::vi::LayerId layerId
            ) NN_NOEXCEPT;

        nn::Result CreateStrayLayer(
            nn::vi::LayerId* pOutLayerId,
            size_t* pOutNativeWindowDataSize,
            void* pNativeWindowDataBuffer,
            size_t nativeWindowDataBufferSize,
            nn::vi::DisplayId displayId,
            nn::vi::LayerSettingsType settings
            ) NN_NOEXCEPT;

        nn::Result DestroyManagedLayer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result DestroyStrayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT;

        nn::Result AttachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result DetachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result StartLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result StopLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result StartLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result StopLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result GetLayerPresentationAllFencesExpiredEvent(nn::sf::NativeHandle& outHandle, nn::vi::LayerId layerId) NN_NOEXCEPT;

        nn::Result ConvertScalingMode(std::int64_t* pOutMode, nn::vi::ScalingModeType viMode) NN_NOEXCEPT;
        nn::Result SetLayerVisibility(nn::vi::LayerId layerId, bool isVisible) NN_NOEXCEPT;
        nn::Result SetLayerAlpha(nn::vi::LayerId layerId, float alpha) NN_NOEXCEPT;

        nn::Result SetLayerPosition(nn::vi::LayerId layerId, float x, float y) NN_NOEXCEPT;
        nn::Result SetLayerSize(nn::vi::LayerId layerId, int64_t width, int64_t height) NN_NOEXCEPT;
        nn::Result GetLayerZ(int64_t* pOutZ, nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result SetLayerZ(nn::vi::LayerId layerId, int64_t z) NN_NOEXCEPT;

    //---------
    // LayerConfig
    //---------
        nn::Result AddToLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT;
        nn::Result RemoveFromLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT;
        nn::Result SetLayerStackFlags(nn::vi::LayerId layerId, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT;
        nn::Result GetLayerStackFlags(nn::vi::LayerStackFlagType* pOutStacks, nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result SetLayerConfig(nn::vi::LayerId layerId, nn::vi::LayerConfig config) NN_NOEXCEPT;

        nn::Result SetConductorLayer(nn::vi::LayerId layerId, bool isConductor) NN_NOEXCEPT;

    //---------
    // DisplayMode & Cmu
    //---------
        nn::Result ListDisplayModes(int64_t* pOutCount, nn::vi::DisplayModeInfo* pOutModes, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result ListDisplayRgbRanges(int64_t* pOutCount, nn::vi::RgbRangeType* pOutRanges, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result ListDisplayContentTypes(int64_t* pOutCount, nn::vi::ContentTypeType* pOutTypes, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT;

        nn::Result GetDisplayMode(nn::vi::DisplayModeInfo* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayMode(nn::vi::DisplayId displayId, const nn::vi::DisplayModeInfo& value) NN_NOEXCEPT;
        nn::Result GetDisplayUnderscan(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayUnderscan(nn::vi::DisplayId displayId, int64_t value) NN_NOEXCEPT;
        nn::Result GetDisplayContentType(nn::vi::ContentTypeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayContentType(nn::vi::DisplayId displayId, nn::vi::ContentTypeType value) NN_NOEXCEPT;
        nn::Result GetDisplayRgbRange(nn::vi::RgbRangeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayRgbRange(nn::vi::DisplayId displayId, nn::vi::RgbRangeType value) NN_NOEXCEPT;
        nn::Result GetDisplayCmuMode(nn::vi::CmuModeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayCmuMode(nn::vi::DisplayId displayId, nn::vi::CmuModeType value) NN_NOEXCEPT;
        nn::Result GetDisplayContrastRatio(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayContrastRatio(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT;
        nn::Result GetDisplayHotplugState(nn::vi::HotplugStateType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayGamma(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayGamma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT;
        nn::Result GetDisplayCmuLuma(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result SetDisplayCmuLuma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT;

    //---------
    // DisplayConfig
    //---------
        nn::Result SetDisplayAlpha(nn::vi::DisplayId displayId, float alpha) NN_NOEXCEPT;
        nn::Result SetDisplayLayerStack(nn::vi::DisplayId displayId, nn::vi::LayerStackType id) NN_NOEXCEPT;
        nn::Result SetDisplayPowerState(nn::vi::DisplayId displayId, nn::vi::PowerStateType state) NN_NOEXCEPT;
        nn::Result SetDefaultDisplay(nn::vi::DisplayId id) NN_NOEXCEPT;

    //---------
    // Display Events
    //---------
        nn::Result GetDisplayHotplugEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayModeChangedEvent(nn::os::NativeHandle* pOutHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayVsyncEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayVsyncEventForDebug(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        nn::Result GetDisplayErrorEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT;

    //---------
    // Error Report
    //---------
        nn::Result GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID, nn::vi::DisplayId displayId) NN_NOEXCEPT;

    //---------
    // Layer Events
    //---------
        nn::Result AcquireLayerTexturePresentingEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::LayerId layerId) NN_NOEXCEPT;
        nn::Result ReleaseLayerTexturePresentingEvent(nn::vi::LayerId layerId) NN_NOEXCEPT;

    //---------
    // RelayTransaction
    //---------
        void TransactParcel(
            std::int32_t handle,
            std::uint32_t code,
            const void* requestBuffer,
            size_t requestBufferSize,
            void* replyBuffer,
            size_t replyBufferSize,
            std::uint32_t flags
            ) NN_NOEXCEPT;
        void AdjustRefcount(std::int32_t handle, std::int32_t diff, std::int32_t isStrong) NN_NOEXCEPT;
        void GetNativeHandle(std::int32_t handle, std::uint32_t code, nn::sf::Out<nn::sf::NativeHandle>& result) NN_NOEXCEPT;

    //----------
    // IndirectDisplay
    //----------
        // for manager
        nn::Result CreateIndirectLayer(nn::vi::IndirectLayerHandleType* pOutLayerHandle) NN_NOEXCEPT;
        nn::Result DestroyIndirectLayer(nn::vi::IndirectConsumerHandleType layerHandle) NN_NOEXCEPT;

        nn::Result CreateIndirectProducerEndPoint(
            nn::vi::IndirectProducerHandleType* pOutProducerHandle,
            nn::vi::IndirectLayerHandleType layerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;
        nn::Result DestroyIndirectProducerEndPoint(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT;

        nn::Result CreateIndirectConsumerEndPoint(
            nn::vi::IndirectConsumerHandleType* pOutConsumerHandle,
            nn::vi::IndirectLayerHandleType layerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;
        nn::Result DestroyIndirectConsumerEndPoint(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT;

        nn::Result SetIndirectProducerFlipOffset(
            nn::vi::IndirectLayerHandleType layerHandle,
            nn::vi::IndirectProducerHandleType producerHandle,
            nn::TimeSpan offset
            ) NN_NOEXCEPT;

        // for system
        nn::Result BindIndirectProducerEndPoint(
            size_t* pOutNativeWindowDataSize,
            void* pNativeWindowDataBuffer,
            size_t nativeWindowDataBufferSize,
            nn::vi::IndirectProducerHandleType producerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;

        nn::Result UnbindIndirectProducerEndPoint(
            nn::vi::IndirectProducerHandleType producerHandle
            ) NN_NOEXCEPT;

        nn::Result FlipIndirectProducerBuffer(
            nn::vi::IndirectProducerHandleType producerHandle
            ) NN_NOEXCEPT;

        // for application
        nn::Result GetIndirectLayerImageMap(
            size_t* pOutSize,
            size_t* pOutStride,
            void* pBuffer,
            size_t bufferSize,
            int width,
            int height,
            float sourceRectX,
            float sourceRectY,
            float sourceRectWidth,
            float sourceRectHeight,
            nn::vi::IndirectConsumerHandleType consumerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;

        nn::Result GetIndirectLayerImageMap_I(
            indirect::IndirectMapCopyImageContext* pContext,
            void* pBuffer,
            size_t bufferSize,
            int width,
            int height,
            float sourceRectX,
            float sourceRectY,
            float sourceRectWidth,
            float sourceRectHeight,
            nn::vi::IndirectConsumerHandleType consumerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;

        nn::Result GetIndirectLayerImageMap_II(
            size_t* pOutSize,
            size_t* pOutStride,
            indirect::IndirectMapCopyImageContext* pContext,
            nn::vi::IndirectConsumerHandleType consumerHandle
            ) NN_NOEXCEPT;


        nn::Result GetIndirectLayerImageRequiredMemoryInfo(
            size_t* pOutSize,
            size_t* pOutAlignment,
            int width,
            int height
            ) NN_NOEXCEPT;

        nn::Result BindIndirectConsumerEndPointTransfer(
            nn::os::NativeHandle* pOutIsImageReadyEventHandle,
            bool* pOutIsImageReadyEventHandleManaged,
            nn::os::NativeHandle bufferHandle,
            bool* pIsBufferHandleManaged,
            size_t bufferSize,
            int width,
            int height,
            nn::vi::IndirectConsumerHandleType consumerHandle,
            nn::applet::AppletResourceUserId aruid
            ) NN_NOEXCEPT;

        nn::Result UnbindIndirectConsumerEndPoint(
            nn::vi::IndirectConsumerHandleType consumerHandle
            ) NN_NOEXCEPT;

        nn::Result KickIndirectCopyImageTransfer(
            size_t* pOutSize,
            size_t* pOutStride,
            nn::vi::IndirectConsumerHandleType consumerHandle,
            float sourceRectX,
            float sourceRectY,
            float sourceRectWidth,
            float sourceRectHeight
            ) NN_NOEXCEPT;

        // transaction
        void IndirectDisplayTransactParcel(
            std::int32_t handle,
            std::uint32_t code,
            const void* requestBuffer,
            size_t requestBufferSize,
            void* replyBuffer,
            size_t replyBufferSize,
            std::uint32_t flags
            ) NN_NOEXCEPT;
        void IndirectDisplayAdjustRefcount(std::int32_t handle, std::int32_t diff, std::int32_t isStrong) NN_NOEXCEPT;
        void IndirectDisplayGetNativeHandle(std::int32_t handle, std::uint32_t code, nn::sf::Out<nn::sf::NativeHandle>& result) NN_NOEXCEPT;


    private:
        friend class ClientObjectAccessor;
        ClientConstants                 m_Constants;
        ClientDisplayHolderList         m_DisplayHolderList;
        ClientLayerHolderOnClientList   m_LayerHolderList;
        native::BinderTable             m_BinderTable;
        OwningProcessHeapBlockIdList    m_OwnProcessHeapBlockIdList;

        indirect::IndirectLayerOwnerList    m_OwnIndirectLayerList;
        indirect::IndirectLayerProducerList m_ProducerBoundIndirectLayerList;
        indirect::IndirectLayerConsumerList m_ConsumerBoundIndirectLayerList;
        indirect::IndirectProducerBinderTable m_IndirectProducerBinderTable;
    };

    class ClientObjectAccessor
    {
    public:
        static const ClientConstants* GetConstants(const ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_Constants;
        }

        static ClientDisplayHolderList* GetDisplayHolderList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_DisplayHolderList;
        }

        static ClientLayerHolderOnClientList* GetLayerHolderList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_LayerHolderList;
        }

        static native::BinderTable* GetBinderTable(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_BinderTable;
        }

        static indirect::IndirectLayerOwnerList* GetOwnIndirectLayerList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_OwnIndirectLayerList;
        }

        static indirect::IndirectLayerProducerList* GetProducerBoundIndirectLayerList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_ProducerBoundIndirectLayerList;
        }

        static indirect::IndirectLayerConsumerList* GetConsumerBoundIndirectLayerList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_ConsumerBoundIndirectLayerList;
        }

        static indirect::IndirectProducerBinderTable* GetIndirectProducerBinderTable(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_IndirectProducerBinderTable;
        }

        static ClientObject::OwningProcessHeapBlockIdList* GetOwnProcessHeapBlockIdList(ClientObject* p) NN_NOEXCEPT
        {
            return &p->m_OwnProcessHeapBlockIdList;
        }
    };


}}}
