﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/vi/vi_Display.h>
#include <nn/vi/vi_Result.h>
#include "detail/vi_Common.h"
#include "detail/vi_Globals.h"
#include "detail/vi_DisplayManager.h"
#include "detail/vi_MemoryMap.h"
#include "detail/vi_PlatformDisplayInfo.h"
#include "detail/vi_PlatformDisplayDefines.h"

int nn::vi::ListDisplays(DisplayInfo* pOutDisplays, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(detail::IsInitialized());
    NN_SDK_REQUIRES(count >= 0);

    int written = 0;

    if( pOutDisplays != nullptr )
    {
        for( int i = 0; i < count && i < detail::PlatformDisplayCount; ++i )
        {
            (reinterpret_cast<detail::PlatformDisplayInfo*>(g_DisplayInfoSegment) + i)->WriteInfo(pOutDisplays + i);
            ++written;
        }

        // TODO: enumerate other displays (useful for PC)
    }

    return written;
}

nn::Result nn::vi::OpenDefaultDisplay(Display** pOutDisplay) NN_NOEXCEPT
{
    return OpenDisplay(pOutDisplay, detail::DefaultDisplayName);
}

nn::Result nn::vi::OpenDisplay(Display** pOutDisplay, const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(detail::IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOutDisplay);

    *pOutDisplay = detail::g_DisplayManager->Open(name);

    if( *pOutDisplay == nullptr )
    {
        return nn::vi::ResultNotFound();
    }

    return nn::ResultSuccess();
}

void nn::vi::CloseDisplay(Display* pDisplay) NN_NOEXCEPT
{
    // Necessary since this will destroy layers created from this display
    std::lock_guard<nn::os::Mutex> lock(detail::GetInitializerLock());

    NN_SDK_REQUIRES(detail::IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pDisplay);
    NN_SDK_REQUIRES(pDisplay->IsOpen());

    detail::g_DisplayManager->Close(pDisplay);
}

nn::Result nn::vi::GetDisplayVsyncEvent(nn::os::SystemEventType* pOutEvent, nn::vi::Display* pDisplay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(detail::IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOutEvent);
    NN_SDK_REQUIRES_NOT_NULL(pDisplay);
    NN_SDK_REQUIRES(pDisplay->IsOpen());

    return pDisplay->GetVsyncEvent(pOutEvent);
}
