﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "vi_WindowManager.h"
#include "vi_PlatformDisplayDefines.h"
#include "../vi_Layer.h"

namespace
{
    const UINT CreateWindowMessage  = WM_USER;
    const UINT DestroyWindowMessage = WM_USER + 1;
    const UINT QuitMessage          = WM_USER + 2;

    struct MessageThreadData
    {
        HWND* window;
        nn::os::Event* windowInitialized;
    };

    struct WindowCreationArgs
    {
        HWND* window;
        int width;
        int height;
        bool isTouchscreen;
    };

    LRESULT CALLBACK AppMessageProcedure(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
    {
        switch( msg )
        {
        case WM_CLOSE:
            {
                // Note: This will cause problems if applications are allowed to create more than
                //       one window of type ApplicationClassName.  For NX, this is the current POR.
                //
                //       This will prevent applications from fully testing their shutdown code.
                //       Once there is a sequence defined by the OS on the NX to terminate a process,
                //       we should revisit this and notify applications through this on Windows
                //       as well.
                ::_exit(EXIT_SUCCESS);
            }
            break;
        default:
            break;
        }

        return DefWindowProc(hWnd, msg, wParam, lParam);
    }

    LRESULT CALLBACK MessageProcedure(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
    {
        switch( msg )
        {
        case CreateWindowMessage:
            {
                WindowCreationArgs* args = reinterpret_cast<WindowCreationArgs*>(lParam);

                RECT rect = { 0, 0, args->width, args->height };
                DWORD style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX;
                DWORD exStyle = WS_EX_APPWINDOW;
                AdjustWindowRectEx(&rect, style, FALSE, exStyle);

                *args->window = CreateWindowEx(exStyle,
                                               nn::vi::detail::ApplicationClassName,
                                               nn::vi::detail::WindowTitle,
                                               style,
                                               CW_USEDEFAULT,
                                               CW_USEDEFAULT,
                                               rect.right - rect.left,
                                               rect.bottom - rect.top,
                                               nullptr,
                                               nullptr,
                                               GetModuleHandle(nullptr),
                                               nullptr);

                if( *args->window != nullptr && args->isTouchscreen )
                {
                    RegisterTouchWindow(*args->window, 0);
                }
            }
            break;
        case DestroyWindowMessage:
            {
                DestroyWindow(reinterpret_cast<HWND>(lParam));
            }
            break;
        case QuitMessage:
            {
                PostQuitMessage(0);
            }
            break;
        default:
            break;
        }

        return DefWindowProc(hWnd, msg, wParam, lParam);
    }
}

DWORD WINAPI nn::vi::detail::WindowManager::MessageThread(void* arg) NN_NOEXCEPT
{
    MessageThreadData* data = reinterpret_cast<MessageThreadData*>(arg);

    *data->window = CreateWindowEx(0,
                                   ManagerClassName,
                                   "Nintendo Window Manager",
                                   0,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   nullptr,
                                   nullptr,
                                   GetModuleHandle(nullptr),
                                   nullptr);

    data->windowInitialized->Signal();

    MSG msg;

    while( GetMessage(&msg, NULL, 0, 0) != 0 )
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    return 0;
}

nn::vi::detail::WindowManager::WindowManager() NN_NOEXCEPT
    : m_Dispatcher(nullptr)
    , m_DispatcherInitialized(nn::os::EventClearMode_AutoClear)
    , m_MessageThread(nullptr)
{
}

nn::vi::detail::WindowManager::~WindowManager() NN_NOEXCEPT
{
}

void nn::vi::detail::WindowManager::Initialize() NN_NOEXCEPT
{
    // Hidden window for handling window creation
    WNDCLASSEX windowClass = {
        sizeof(WNDCLASSEX),
        0,
        MessageProcedure,
        0,
        0,
        GetModuleHandle(nullptr),
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        ManagerClassName,
        nullptr,
    };

    NN_ABORT_UNLESS(RegisterClassEx(&windowClass) != 0);

    MessageThreadData data = { &m_Dispatcher, &m_DispatcherInitialized };

    m_MessageThread = CreateThread(nullptr, 4096, MessageThread, &data, 0, nullptr);
    NN_ABORT_UNLESS_NOT_NULL(m_MessageThread);

    m_DispatcherInitialized.Wait();

    // Application window that exposes the native window handle to applications
    windowClass = {
        sizeof( WNDCLASSEX ),
        CS_OWNDC,
        AppMessageProcedure,
        0,
        sizeof(nn::vi::Layer*),
        GetModuleHandle(nullptr),
        nullptr,
        LoadCursor(nullptr, IDC_ARROW),
        static_cast<HBRUSH>(GetStockObject(BLACK_BRUSH)),
        nullptr,
        ApplicationClassName,
        nullptr
    };

    NN_ABORT_UNLESS(RegisterClassEx(&windowClass) != 0);
}

void nn::vi::detail::WindowManager::Finalize() NN_NOEXCEPT
{
    SendMessage(m_Dispatcher, QuitMessage, 0, 0);
    DestroyWindow(m_Dispatcher);

    WaitForSingleObject(m_MessageThread, INFINITE);

    UnregisterClass(ManagerClassName, GetModuleHandle(nullptr));
}

HWND nn::vi::detail::WindowManager::CreateWindow(int width, int height, bool isTouchscreen) NN_NOEXCEPT
{
    HWND window;
    WindowCreationArgs args = { &window, width, height, isTouchscreen };
    SendMessage(m_Dispatcher, CreateWindowMessage, 0, reinterpret_cast<LPARAM>(&args));

    return window;
}

void nn::vi::detail::WindowManager::DestroyWindow(HWND window) NN_NOEXCEPT
{
    SendMessage(m_Dispatcher, DestroyWindowMessage, 0, reinterpret_cast<LPARAM>(window));
}
