﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include "vi_PlatformDisplayInfo.h"

nn::vi::detail::PlatformDisplayInfo::PlatformDisplayInfo(const char* name,
                                                         int layerWidthPixelCountMax,
                                                         int layerHeightPixelCountMax,
                                                         const DisplayPolicy& policy,
                                                         const DisplaySupport& support,
                                                         bool isTouchscreen) NN_NOEXCEPT
    : m_Policy(policy)
    , m_Support(support)
    , m_LayerWidthPixelCountMax(layerWidthPixelCountMax)
    , m_LayerHeightPixelCountMax(layerHeightPixelCountMax)
    , m_isTouchscreen(isTouchscreen)
{
    // TODO: this will truncate names...
    strncpy(m_Name, name, DisplayInfo::NameLengthMax);
}

nn::vi::detail::PlatformDisplayInfo::PlatformDisplayInfo(const PlatformDisplayInfo& rhs)
    : m_Policy(rhs.m_Policy)
    , m_Support(rhs.m_Support)
    , m_LayerWidthPixelCountMax(rhs.m_LayerWidthPixelCountMax)
    , m_LayerHeightPixelCountMax(rhs.m_LayerHeightPixelCountMax)
    , m_isTouchscreen(rhs.m_isTouchscreen)
{
    strncpy(m_Name, rhs.m_Name, DisplayInfo::NameLengthMax);
}

void nn::vi::detail::PlatformDisplayInfo::WriteInfo(DisplayInfo* pOutInfo) const NN_NOEXCEPT
{
    strncpy(pOutInfo->name, m_Name, DisplayInfo::NameLengthMax);

    pOutInfo->hasLayerLimit = m_Policy.HasLayerLimit();
    pOutInfo->layerCountMax = m_Policy.GetLayerCountMax();

    pOutInfo->layerWidthPixelCountMax = m_LayerWidthPixelCountMax;
    pOutInfo->layerHeightPixelCountMax = m_LayerHeightPixelCountMax;
}

bool nn::vi::detail::PlatformDisplayInfo::ValidateLayerDimensions(int width, int height) const NN_NOEXCEPT
{
    return width > 0 &&
           width <= m_LayerWidthPixelCountMax &&
           height > 0 &&
           height <= m_LayerHeightPixelCountMax;
}

bool nn::vi::detail::PlatformDisplayInfo::ValidateZOrder(int z) const NN_NOEXCEPT
{
    return z >= m_Policy.GetZOrderCountMin() &&
           z <= m_Policy.GetZOrderCountMax();
}

bool nn::vi::detail::PlatformDisplayInfo::ValidatePixelFormat(PixelFormat format) const NN_NOEXCEPT
{
    return m_Support.queryPixelFormatSupport(format);
}

bool nn::vi::detail::PlatformDisplayInfo::ValidateScalingMode(ScalingMode mode) const NN_NOEXCEPT
{
    return m_Support.queryScalingModeSupport(mode);
}

const nn::vi::detail::DisplayPolicy& nn::vi::detail::PlatformDisplayInfo::GetPolicy() const NN_NOEXCEPT
{
    return m_Policy;
}

const char* nn::vi::detail::PlatformDisplayInfo::GetName() const NN_NOEXCEPT
{
    return m_Name;
}

const nn::vi::detail::DisplaySupport& nn::vi::detail::PlatformDisplayInfo::GetSupport() const NN_NOEXCEPT
{
    return m_Support;
}

int nn::vi::detail::PlatformDisplayInfo::GetLayerWidthPixelCountMax() const NN_NOEXCEPT
{
    return m_LayerWidthPixelCountMax;
}

int nn::vi::detail::PlatformDisplayInfo::GetLayerHeightPixelCountMax() const NN_NOEXCEPT
{
    return m_LayerHeightPixelCountMax;
}

bool nn::vi::detail::PlatformDisplayInfo::IsTouchscreen() const NN_NOEXCEPT
{
    return m_isTouchscreen;
}
