﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_IntrusiveList.h>
#include "vi_PoolAllocator.h"
#include "vi_Allocator.h"
#include "vi_DisplayFactory.h"
#include "../vi_Display.h"

namespace nn { namespace vi { namespace detail {

    class PlatformDisplayInfo;

    class DisplayManager
    {
    public:
        /**
         * @brief Type for intrusive list of displays.
         */
        typedef nn::util::IntrusiveList<Display, nn::util::IntrusiveListMemberNodeTraits<Display, &Display::m_ListNode>> DisplayList;

        /**
         * @brief Creates a DisplayManager.
         *
         * @param[in] displaySegmentStartAddress Starting memory location for displays.
         * @param[in] displaySegmentSize         Maximum size for display allocations.
         * @param[in] layerSegmentStartAddress   Starting memory location for layers.
         * @param[in] layerSegmentSize           Maximum size for layer allocations.
         * @param[in] platformDisplays           Named platform displays.
         * @param[in] platformDisplayCount       Number of named platform displays.
         *
         * @pre  @a displaySegmentStartAddress != nullptr @n
         *       @a layerSegmentStartAddress != nullptr
         * @post DisplayManager has been initialized.
         */
        DisplayManager(void* displaySegmentStartAddress, size_t displaySegmentSize,
                       void* layerSegmentStartAddress, size_t layerSegmentSize,
                       PlatformDisplayInfo* platformDisplays, int platformDisplayCount) NN_NOEXCEPT;

        /**
         * @brief   Destroys a DisplayManager.
         * @details All displays/layers owned by this object will be destroyed as well.
         */
        ~DisplayManager() NN_NOEXCEPT;

        /**
         * @brief Opens a display by name.
         *
         * @param[in] name Name of the display.
         *
         * @return         A fully-initialized display.
         * @retval nullptr Display by that name could not be found.
         */
        Display* Open(const char* name) NN_NOEXCEPT;

        /**
         * @brief Closes a display.
         *
         * @param[in] pDisplay Display handle.
         *
         * @pre  @a pDisplay != nullptr and @a pDisplay is open.
         * @post Display is closed.
         *
         * @details All layers associated with the display are destroyed as well.
         */
        void Close(Display* pDisplay) NN_NOEXCEPT;

        nn::os::Mutex& GetDisplayListLock() const NN_NOEXCEPT;

        const DisplayList& GetDisplayList() const NN_NOEXCEPT;
    private:
        PoolAllocator<Layer> m_LayerAllocator;   //!< Allocator for layers.

        DisplayList m_OpenDisplays;              //!< List of all displays opened by this process.

        DisplayFactory m_DisplayFactory;

        PlatformDisplayInfo* m_PlatformDisplays; //!< List of named platform displays.
        int m_PlatformDisplayCount;              //!< Number of named platform displays.

        mutable nn::os::Mutex m_Lock;            //!< Lock for protecting display open/close.
    };

}}}
