﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_DisplayModeInfo.h>
#include <hardware/hwcomposer_defs.h>
#include "vi_Android.h"

int nn::vi::detail::ConvertScalingMode(ScalingMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
    case ScalingMode_None:
        return NATIVE_WINDOW_SCALING_MODE_NO_SCALE_CROP;
    case ScalingMode_Exact:
        return NATIVE_WINDOW_SCALING_MODE_FREEZE;
    case ScalingMode_FitToLayer:
        return NATIVE_WINDOW_SCALING_MODE_SCALE_TO_WINDOW;
    case ScalingMode_ScaleAndCrop:
        return NATIVE_WINDOW_SCALING_MODE_SCALE_CROP;
    case ScalingMode_PreserveAspectRatio:
        return NATIVE_WINDOW_SCALING_MODE_SCALE_NO_CROP;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

android::PixelFormat nn::vi::detail::ConvertPixelFormat(PixelFormat format) NN_NOEXCEPT
{
    switch (format)
    {
    case PixelFormat_Rgba8888:
        return android::PIXEL_FORMAT_RGBA_8888;
    case PixelFormat_Rgbx8888:
        return android::PIXEL_FORMAT_RGBX_8888;
    case PixelFormat_Rgb888:
        return android::PIXEL_FORMAT_RGB_888;
    case PixelFormat_Rgb565:
        return android::PIXEL_FORMAT_RGB_565;
    case PixelFormat_Bgra8888:
        return android::PIXEL_FORMAT_BGRA_8888;
    case PixelFormat_Rgba5551:
        return android::PIXEL_FORMAT_RGBA_5551;
    case PixelFormat_Rgba4444:
        return android::PIXEL_FORMAT_RGBA_4444;
    case PixelFormat_StandardRgba8888:
        return android::PIXEL_FORMAT_sRGB_A_8888;
    case PixelFormat_StandardRgbx8888:
        return android::PIXEL_FORMAT_sRGB_X_8888;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void nn::vi::detail::ConvertDisplayMode(DisplayModeInfo* pOutInfo, const android::DisplayInfo& info) NN_NOEXCEPT
{
    pOutInfo->width = info.w;
    pOutInfo->height = info.h;
    pOutInfo->refreshRate = info.fps;
    // TODO: how to query stereo mode?
    pOutInfo->mode = StereoMode_None;
}

nn::vi::RgbRange nn::vi::detail::ConvertRgbRange(int range) NN_NOEXCEPT
{
    switch( range )
    {
    case DISPLAY_RGB_Limited:
        return nn::vi::RgbRange_Limited;
    case DISPLAY_RGB_Full:
        return nn::vi::RgbRange_Full;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int nn::vi::detail::ConvertRgbRange(RgbRange range) NN_NOEXCEPT
{
    switch ( range )
    {
    case nn::vi::RgbRange_Limited:
        return DISPLAY_RGB_Limited;
    case nn::vi::RgbRange_Full:
        return DISPLAY_RGB_Full;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int nn::vi::detail::ConvertPowerState(PowerState state) NN_NOEXCEPT
{
    switch( state )
    {
    case PowerState_Off:
        return HWC_POWER_MODE_OFF;
    case PowerState_On:
        return HWC_POWER_MODE_NORMAL;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

std::uint32_t nn::vi::detail::ConvertLayerSettings(LayerSettings settings) NN_NOEXCEPT
{
    std::uint32_t flags = 0;

    if( settings.Test<nn::vi::LayerFlags::Fullscreen>() )
    {
        flags |= android::ISurfaceComposerClient::eFullscreen;
    }
    if( settings.Test<nn::vi::LayerFlags::Opaque>() )
    {
        flags |= android::ISurfaceComposerClient::eOpaque;
    }

    return flags;
}

bool nn::vi::detail::operator==(const nn::vi::DisplayModeInfo& viInfo, const android::DisplayInfo& androidInfo) NN_NOEXCEPT
{
    // TODO: how to query stereo mode?
    return androidInfo.w == viInfo.width &&
           androidInfo.h == viInfo.height &&
           androidInfo.fps == viInfo.refreshRate;
}

bool nn::vi::detail::operator==(const android::DisplayInfo& androidInfo, const nn::vi::DisplayModeInfo& viInfo) NN_NOEXCEPT
{
    return viInfo == androidInfo;
}
