﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "vi_MemoryManagement.h"

#include <type_traits>
#include <nn/lmem/lmem_UnitHeap.h>
#include "vi_Config.h"
#include "vi_Log.h"

#include "vi_DisplayHolder.h"
#include "vi_LayerHolder.h"


namespace nn{ namespace vi{
    namespace {

        template<typename T, int N>
        class UnitHeap
        {
            NN_DISALLOW_COPY(UnitHeap);
            NN_DISALLOW_MOVE(UnitHeap);
        public:
            UnitHeap() NN_NOEXCEPT
            {
            }

            void Initialize() NN_NOEXCEPT
            {
                m_HeapHandle = nn::lmem::CreateUnitHeap(
                    &m_Storage,
                    sizeof(m_Storage),
                    sizeof(T),
                    nn::lmem::CreationOption_ThreadSafe,
                    NN_ALIGNOF(T),
                    &m_HeapHead
                    );
                NN_SDK_ASSERT_EQUAL(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), N);
            }

            void Finalize() NN_NOEXCEPT
            {
                nn::lmem::DestroyUnitHeap(m_HeapHandle);
            }

            void* Allocate() NN_NOEXCEPT
            {
                return nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
            }

            void Free(void* p) NN_NOEXCEPT
            {
                nn::lmem::FreeToUnitHeap(m_HeapHandle, p);
            }

            void Dump(const char* name) NN_NOEXCEPT
            {
                int allocatable = nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle);
                int allocated = nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle);
                NN_UNUSED(allocatable);
                NN_UNUSED(allocated);
                NN_VI_LOG_MEM("%s % 3d/% 3d\n", name, allocated, allocated + allocatable);
            }

        private:
            nn::lmem::HeapHandle     m_HeapHandle;
            nn::lmem::HeapCommonHead m_HeapHead;
            typename std::aligned_storage<sizeof(T) * N, NN_ALIGNOF(T)>::type m_Storage;
        };

        static const int TotalDisplayCountMax = DisplayCountMax;
        static const int TotalLayerCountMax   = DisplayCountMax * LayerCountPerDisplayMax;

        UnitHeap<DisplayHolder, TotalDisplayCountMax> g_DisplayHolderHeap;
        UnitHeap<LayerHolder  , TotalLayerCountMax  > g_LayerHolderHeap;
    }

    void DumpMemory() NN_NOEXCEPT
    {
#ifdef NN_VI_ENABLE_MEMORY_DUMP
        g_DisplayHolderHeap.Dump("DisplayHolderHeap");
        g_LayerHolderHeap  .Dump("LayerHolderHeap  ");
#endif
    }

    void InitializeMemory() NN_NOEXCEPT
    {
        g_DisplayHolderHeap.Initialize();
        g_LayerHolderHeap.Initialize();
    }

    void FinalizeMemory() NN_NOEXCEPT
    {
        g_LayerHolderHeap.Finalize();
        g_DisplayHolderHeap.Finalize();
    }

    void* AllocateDisplayHolder() NN_NOEXCEPT
    {
        auto p = g_DisplayHolderHeap.Allocate();
        DumpMemory();
        return p;
    }

    void FreeDisplayHolder(void* p) NN_NOEXCEPT
    {
        g_DisplayHolderHeap.Free(p);
        DumpMemory();
    }

    void* AllocateLayerHolder() NN_NOEXCEPT
    {
        auto p = g_LayerHolderHeap.Allocate();
        DumpMemory();
        return p;
    }

    void FreeLayerHolder(void* p) NN_NOEXCEPT
    {
        g_LayerHolderHeap.Free(p);
        DumpMemory();
    }

}}
