﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>

#include <nn/vi/profiler/vi_Profiler.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/sf/vi_DisplayService.sfdl.h>
#include <nn/vi/vi_Result.h>

#include "../vi_CommonUtility.h"
#include "../vi_Config.h"

namespace
{
    std::size_t g_InitializeCount;

    nn::sf::ShimLibraryObjectHolder<nn::visrv::sf::IApplicationRootService> g_ApplicationRootServiceHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
    nn::sf::HipcRef<nn::visrv::sf::IApplicationDisplayService> g_pApplicationService;

    nn::os::Mutex g_InitLock(false);
    nn::vi::DisplayId g_DisplayId;
}

nn::Result nn::vi::profiler::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitLock);

    if( g_InitializeCount == 0 )
    {
        NN_RESULT_DO(GetClientManager().InitializeShimLibraryHolder(&g_ApplicationRootServiceHolder, "vi:u"));
        NN_RESULT_DO(g_ApplicationRootServiceHolder->GetDisplayService(&g_pApplicationService, nn::vi::PolicyLevel_Standard));

        nn::vi::DisplayName name = {};
        nn::util::Strlcpy(name.value, NN_VI_DEFAULT_DISPLAY_NAME, sizeof(name));
        NN_RESULT_DO(g_pApplicationService->OpenDisplay(&g_DisplayId, name));
    }

    NN_SDK_REQUIRES_LESS(g_InitializeCount, std::numeric_limits<decltype(g_InitializeCount)>::max());
    ++g_InitializeCount;

    NN_RESULT_SUCCESS;
}

void nn::vi::profiler::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitLock);

    if( g_InitializeCount == 0 )
    {
        return;
    }

    if( --g_InitializeCount == 0 )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pApplicationService->CloseDisplay(g_DisplayId));

        g_pApplicationService.Reset();
        g_ApplicationRootServiceHolder.FinalizeHolder();
    }
}

nn::Result nn::vi::profiler::GetVsyncEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitLock);

    NN_SDK_REQUIRES(g_InitializeCount > 0);
    NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

    nn::sf::NativeHandle handle;
    NN_RESULT_DO(g_pApplicationService->GetDisplayVsyncEventForDebug(&handle, g_DisplayId));

    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();

    NN_RESULT_SUCCESS;
}
