﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedTextureIndexExchangeTable.h>

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../vi_Log.h"

namespace nn{ namespace vi{ namespace fbshare{

    const int SharedTextureIndexExchangeTable::WindowTextureCountMax;
    const int SharedTextureIndexExchangeTable::BufferTextureCountMax;

    SharedTextureIndexExchangeTable::SharedTextureIndexExchangeTable() NN_NOEXCEPT
    {
        Reset();
    }

    void SharedTextureIndexExchangeTable::Reset() NN_NOEXCEPT
    {
        for(int i = 0; i < WindowTextureCountMax; i++)
        {
            m_ToBufferTextureTable[i] = -1;
        }
        for(int i = 0; i < BufferTextureCountMax; i++)
        {
            m_ToWindowTextureTable[i] = -1;
        }
    }

    void SharedTextureIndexExchangeTable::Reset(int* pBufferTextureIndexList, int count) NN_NOEXCEPT
    {
        Reset();

        if(count > WindowTextureCountMax)
        {
            NN_VI_LOG_WARN("Too many buffer texture indices. given %d, limit %d\n", count, WindowTextureCountMax);
        }
        int nWin = std::min(count, WindowTextureCountMax);
        for(int iWin = 0; iWin < nWin; iWin++)
        {
            int iBuf = pBufferTextureIndexList[iWin];

            if(iBuf < 0 || iBuf >= BufferTextureCountMax)
            {
                NN_VI_LOG_WARN("Invalid buffer texture index. given %d, valid range [%d:%d)\n", iBuf, 0, BufferTextureCountMax);
            }
            else
            {
                m_ToBufferTextureTable[iWin] = iBuf;
                m_ToWindowTextureTable[iBuf] = iWin;
            }
        }
    }

    int SharedTextureIndexExchangeTable::ExchangeToBufferTextureIndex(int windowTextureIndex) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(windowTextureIndex, 0, WindowTextureCountMax);

        if(windowTextureIndex < 0 || windowTextureIndex >= WindowTextureCountMax)
        {
            return -1;
        }

        return m_ToBufferTextureTable[windowTextureIndex];
    }

    int SharedTextureIndexExchangeTable::ExchangeToWindowTextureIndex(int bufferTextureIndex) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(bufferTextureIndex, 0, BufferTextureCountMax);

        if(bufferTextureIndex < 0 || bufferTextureIndex >= BufferTextureCountMax)
        {
            return -1;
        }

        return m_ToWindowTextureTable[bufferTextureIndex];
    }

}}}
