﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedLayerWindowImpl.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/vi/vi_ResultPrivate.h>
#include <nn/vi/fbshare/vi_SharedMemoryPoolLayout.h>
#include "../vi_Macro.h"
#include "../vi_Log.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvrm_sync.h>


namespace nn{ namespace vi{ namespace fbshare{

    const int SharedLayerWindowImpl::BufferCountMin;
    const int SharedLayerWindowImpl::BufferCountMax;

    namespace {
        typedef NVNboolean (*PFNNVNSYNCINITIALIZEFROMNVRMSYNCNVXPROC) (NVNsync *sync, NVNdevice *device, const struct NvRmSyncRec *nvrmSync);

        void SetNvnSyncNoFence(NVNsync* pOutValue, NVNdevice* pDevice, void (*deviceFunction0)()) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(deviceFunction0);
            auto nvnSyncInitializeFromNvRmSyncNVX = reinterpret_cast<PFNNVNSYNCINITIALIZEFROMNVRMSYNCNVXPROC>(deviceFunction0);

            NvRmSyncWithStorage<1> src = {}; // 空に設定される
            nvnSyncInitializeFromNvRmSyncNVX(pOutValue, pDevice, &src);
        }

    }

    SharedLayerWindowImpl::SharedLayerWindowImpl() NN_NOEXCEPT
        : m_pService()
        , m_pDevice()
        , m_pDeviceFunction0(nullptr)
        , m_hBuffer(nn::vi::fbshare::SharedBufferHandle::GetInvalidValue())
        , m_hLayer(nn::vi::fbshare::SharedLayerHandle::GetInvalidValue())
        , m_BufferCount(-1)
        , m_InternalTextureIndex(-1)
        , m_AcquiredTextureIndex(-1)
        , m_FrameCount(0)
        , m_IsPreAcquired(false)
        , m_IsAttachedBufferChanged(false)
        , m_IsInternalWindowReconstructionRequired(false)
        , m_PreAcquiredAttachedTextureIndexList(SharedLayerTextureIndexList::GetInvalidValue())
        , m_AttachedTextureIndexList(SharedLayerTextureIndexList::GetInvalidValue())
        , m_AttachedTextureList()
        , m_MemoryPool()
        , m_TexturePool()
        , m_NativeWindow()
        , m_InternalWindow()
    {
    }

    bool SharedLayerWindowImpl::IsInitialized() const NN_NOEXCEPT
    {
        return m_pDevice != nullptr;
    }

    bool SharedLayerWindowImpl::IsTexturePreAcquired() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_IsPreAcquired;
    }

    bool SharedLayerWindowImpl::IsTextureAcquired() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_AcquiredTextureIndex >= 0;
    }

    nn::Result SharedLayerWindowImpl::Initialize(
        nn::sf::SharedPointer<nn::visrv::sf::ISystemDisplayService> pService,
        NVNdevice* pDevice,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        nn::vi::fbshare::SharedLayerHandle hLayer,
        int bufferCount,
        NVNformat format
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pService);
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        NN_SDK_REQUIRES(!hBuffer.IsInvalid());
        NN_SDK_REQUIRES(!hLayer.IsInvalid());
        NN_SDK_REQUIRES_MINMAX(bufferCount, BufferCountMin, BufferCountMax);

        NN_VI_LOG_DEV("[share]Initialize LayerWindow start\n");
        NN_VI_PROCESS_START();

        nn::vi::native::NativeMemoryHandleId memId = {};
        uint64_t memSize = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};

        NN_VI_LOG_DEV("[share]  SetDeviceFunction\n");
        auto deviceFunction0 = nvnDeviceGetProcAddress(pDevice, "nvnSyncInitializeFromNvRmSyncNVX");
        NN_ABORT_UNLESS_NOT_NULL(deviceFunction0);

        NN_VI_LOG_DEV("[share]  GetSharedMemoryHandle\n");
        NN_RESULT_DO(pService->GetSharedBufferMemoryHandleId(&memId, &memSize, &layout, hBuffer, nn::applet::GetAppletResourceUserId()));

        NN_VI_LOG_DEV("[share]    PoolSize %llu bytes(0x%llX)\n", memSize, memSize);
        NN_VI_LOG_DEV("[share]    Layout %d\n", layout.count);
        for(int i = 0; i < layout.count; i++)
        {
            auto& e = layout.entries[i];
            NN_UNUSED(e);
            NN_VI_LOG_DEV("[share]      [%d] %dx%d 0x%llX-0x%llX (%llu bytes)\n", i, e.width, e.height, e.offset, e.offset + e.size, e.size);
        }

        NN_VI_LOG_DEV("[share]  Initialize MemoryPool\n");
        NN_RESULT_DO(m_MemoryPool.Initialize(pDevice, memId, static_cast<size_t>(memSize), MemoryPoolOption));
        NN_VI_PROCESS_ROLLBACK(m_MemoryPool.Finalize());

        NN_VI_LOG_DEV("[share]  Initialize TexturePool\n");
        NN_RESULT_DO(m_TexturePool.Initialize(pDevice, &m_MemoryPool, layout, format));
        NN_VI_PROCESS_ROLLBACK(m_TexturePool.Finalize());

        NN_VI_LOG_DEV("[share]  Initialize NativeWindow\n");
        NN_RESULT_DO(m_NativeWindow.Initialize(pService, hBuffer, hLayer));
        NN_VI_PROCESS_ROLLBACK(m_NativeWindow.Finalize());

        NN_VI_LOG_DEV("[share]Initialize LayerWindow complete\n");
        NN_VI_PROCESS_SUCCESS();
        m_pService = pService;
        m_pDevice = pDevice;
        m_pDeviceFunction0 = deviceFunction0;
        m_hBuffer = hBuffer;
        m_hLayer = hLayer;
        m_BufferCount = bufferCount;
        m_InternalTextureIndex = -1;
        m_AcquiredTextureIndex = -1;
        m_FrameCount = 0;
        m_IsPreAcquired = false;
        m_IsAttachedBufferChanged = false;
        m_IsInternalWindowReconstructionRequired = false;
        m_PreAcquiredAttachedTextureIndexList.Reset();
        m_AttachedTextureIndexList.Reset();
        std::memset(m_AttachedTextureList, 0, sizeof(m_AttachedTextureList));
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowImpl::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        if(m_AttachedTextureIndexList.GetCount() > 0)
        {
            nvnWindowFinalize(&m_InternalWindow);
            m_AttachedTextureIndexList.Reset();
            std::memset(m_AttachedTextureList, 0, sizeof(m_AttachedTextureList));
        }

        m_NativeWindow.Finalize();
        m_TexturePool.Finalize();
        m_MemoryPool.Finalize();

        m_pService = nullptr;
        m_pDevice = nullptr;
        m_pDeviceFunction0 = nullptr;
        m_hBuffer = nn::vi::fbshare::SharedBufferHandle::GetInvalidValue();
        m_hLayer = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();
        m_BufferCount = -1;
        m_InternalTextureIndex = -1;
        m_AcquiredTextureIndex = -1;
        m_FrameCount = 0;
        m_IsPreAcquired = false;
        m_IsAttachedBufferChanged = false;
        m_IsInternalWindowReconstructionRequired = false;
        m_PreAcquiredAttachedTextureIndexList.Reset();
        m_AttachedTextureIndexList.Reset();
        std::memset(m_AttachedTextureList, 0, sizeof(m_AttachedTextureList));
    }

    nn::Result SharedLayerWindowImpl::PreAcquireTexture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!IsTextureAcquired(), "PresentTexture is required before AcquireTexture");

        if(IsTexturePreAcquired())
        {
            NN_RESULT_SUCCESS;
        }

        bool isAttachedBufferChanged = false;
        SharedLayerTextureIndexList indexList = SharedLayerTextureIndexList::GetInvalidValue();
        NN_RESULT_TRY(m_NativeWindow.PreAcquireTexture(&indexList, timeout))
            NN_RESULT_CATCH(nn::vi::ResultPrivateSharedLayerBufferChanged)
            {
                isAttachedBufferChanged = true;
            }
        NN_RESULT_END_TRY;

        m_IsPreAcquired = true;
        m_IsAttachedBufferChanged = isAttachedBufferChanged;
        m_PreAcquiredAttachedTextureIndexList = indexList;
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowImpl::CancelPreAcquiredTexture() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]canceling pre-acquired texture\n");

        if(!IsTexturePreAcquired())
        {
            NN_VI_LOG_DEV("[share]  no texture is pre-acquired\n");
            return;
        }

        // 一旦 Acquire して即座に Cancel する
        NVNsync sync;
        int textureIndex;
        NVNtexture* pTexture;
        nvnSyncInitialize(&sync, m_pDevice);
        AcquirePreAcquiredTexture(&sync, &textureIndex, &pTexture);
        nvnSyncFinalize(&sync);
        CancelTexture(textureIndex);
    }

    void SharedLayerWindowImpl::AcquirePreAcquiredTexture(NVNsync* pOutTextureAvailableSync, int* pOutTextureIndex, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutTextureAvailableSync);
        NN_SDK_REQUIRES_NOT_NULL(pOutTextureIndex);
        NN_SDK_REQUIRES_NOT_NULL(pOutTexture);
        NN_SDK_REQUIRES(!IsTextureAcquired(), "PresentTexture is required before AcquireTexture");

        if(!IsTexturePreAcquired())
        {
            NN_VI_LOG_DEV("no texture is pre-acquired.\n");
            SetNvnSyncNoFence(pOutTextureAvailableSync, m_pDevice, m_pDeviceFunction0);
            m_InternalTextureIndex = -1;
            m_AcquiredTextureIndex = m_FrameCount % m_BufferCount;
            m_IsPreAcquired = false;
            *pOutTextureIndex = m_AcquiredTextureIndex;
            *pOutTexture = nullptr;
            return;
        }

        // NOTE:
        //   この時点で Acquire 済になっているので（安全な）デタッチはブロックされている。
        //   強制デタッチされても PresentTexture は成功するのでクライアントからの見た目は変わらない。

        // 必要なら InternalWindow を再構築
        {
            if(m_IsAttachedBufferChanged)
            {
                // アタッチ中のバッファが変更された場合 nvnWindow を作り直す。
                NN_VI_LOG_DEV("[share]detected attached buffer changed/ (%d,%d,%d,%d)->(%d,%d,%d,%d)\n",
                    m_AttachedTextureIndexList[0], m_AttachedTextureIndexList[1], m_AttachedTextureIndexList[2], m_AttachedTextureIndexList[3],
                    m_PreAcquiredAttachedTextureIndexList[0], m_PreAcquiredAttachedTextureIndexList[1], m_PreAcquiredAttachedTextureIndexList[2], m_PreAcquiredAttachedTextureIndexList[3]
                );
                ReconstructInternalWindowImpl(m_PreAcquiredAttachedTextureIndexList);
            }
            else if(m_IsInternalWindowReconstructionRequired)
            {
                // Cancel により要求された場合 nvnWindow を作り直す。
                NN_VI_LOG_DEV("[share]internal window reconstruction is required\n");
                ReconstructInternalWindowImpl(m_PreAcquiredAttachedTextureIndexList);
            }
            m_IsAttachedBufferChanged = false;
            m_IsInternalWindowReconstructionRequired = false;
            m_PreAcquiredAttachedTextureIndexList.Reset();
        }

        // InternalWindow から Acquire
        int internalIndex = -1;
        {
            NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&m_InternalWindow, pOutTextureAvailableSync, &internalIndex), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);
            NN_SDK_ASSERT_RANGE(internalIndex, 0, BufferCountMax);
        }

        m_InternalTextureIndex = internalIndex;
        m_AcquiredTextureIndex = m_FrameCount % m_BufferCount;
        m_IsPreAcquired = false;
        *pOutTextureIndex = m_AcquiredTextureIndex;
        *pOutTexture = m_AttachedTextureList[internalIndex];
    }

    void SharedLayerWindowImpl::PresentTexture(NVNqueue* pQueue, int textureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pQueue);
        NN_SDK_REQUIRES(IsTextureAcquired());
        NN_SDK_REQUIRES_EQUAL(textureIndex, m_AcquiredTextureIndex);

        if(m_InternalTextureIndex >= 0)
        {
            nvnQueuePresentTexture(pQueue, &m_InternalWindow, textureIndex);
            // NOTE:
            //   ここから先、（安全な）デタッチされる可能性がある。
        }
        else
        {
            NN_VI_LOG_DEV("no valid buffer is acquired.\n");
        }

        m_FrameCount++;
        m_InternalTextureIndex = -1;
        m_AcquiredTextureIndex = -1;
    }

    void SharedLayerWindowImpl::CancelTexture(int textureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]canceling texture(%d)\n", textureIndex);

        // Acquire されているテクスチャインデックスでなければ無視
        if(textureIndex != m_AcquiredTextureIndex)
        {
            return;
        }

        if(m_InternalTextureIndex >= 0)
        {
            m_NativeWindow.CancelTexture(m_InternalTextureIndex);

            // NOTE:
            //   ここから先、（安全な）デタッチされる可能性がある。

            // nvnWindow の状態が半端になっているため作り直してリセットする必要がある。
            // 次に Acquire するときに作り直す。
            m_IsInternalWindowReconstructionRequired = true;
        }
        else
        {
            NN_VI_LOG_DEV("no valid buffer is acquired.\n");
        }

        // Cancel の場合 FrameCount は増やさない
        m_InternalTextureIndex = -1;
        m_AcquiredTextureIndex = -1;
    }

    void SharedLayerWindowImpl::GetSharedTexture(NVNtexture** pOutTexture, int sharedTextureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_TexturePool.AcquireTexture(pOutTexture, sharedTextureIndex));
    }

    void SharedLayerWindowImpl::ReconstructInternalWindowImpl(const SharedLayerTextureIndexList& indexList) NN_NOEXCEPT
    {
        // 以前の NVNwindow を破棄
        if(m_AttachedTextureIndexList.GetCount() > 0)
        {
            NN_VI_LOG_DEV("[share][lw]  destroy old window\n");
            nvnWindowFinalize(&m_InternalWindow);
            m_AttachedTextureIndexList.Reset();
            std::memset(m_AttachedTextureList, 0, sizeof(m_AttachedTextureList));
        }

        int count = indexList.GetCount();
        NN_ABORT_UNLESS_GREATER_EQUAL(count, BufferCountMin);

        // 新しいインデックスで NVNwindow を作る
        {
            NN_VI_LOG_DEV("[share][lw]  create new window\n");
            for(int64_t iWin = 0; iWin < count; iWin++)
            {
                int iBuf = indexList[iWin];
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_TexturePool.AcquireTexture(&m_AttachedTextureList[iWin], iBuf));
            }

            NVNwindowBuilder builder;
            nvnWindowBuilderSetDevice(&builder, m_pDevice);
            nvnWindowBuilderSetDefaults(&builder);
            nvnWindowBuilderSetNativeWindow(&builder, m_NativeWindow.GetNativeWindowHandle());
            nvnWindowBuilderSetPresentInterval(&builder, 1);
            nvnWindowBuilderSetTextures(&builder, static_cast<int>(count), m_AttachedTextureList);
            NN_ABORT_UNLESS(nvnWindowInitialize(&m_InternalWindow, &builder));
        }

        // インデックスの変換テーブルを再構築
        {
            NN_VI_LOG_DEV("[share][lw]  rebuild exchange table\n");
            int table[BufferCountMax] = {};
            for(int64_t i = 0; i < count; i++)
            {
                table[i] = static_cast<int>(indexList[i]);
            }
            m_NativeWindow.ResetExchangeTable(table, static_cast<int>(count));
        }

        m_AttachedTextureIndexList = indexList;
    }

    void SharedLayerWindowImpl::SetSwapInterval(int value) NN_NOEXCEPT
    {
        m_NativeWindow.SetSwapInterval(value);
    }

    void SharedLayerWindowImpl::SetCrop(int x, int y, int width, int height) NN_NOEXCEPT
    {
        CropRegion crop = {};
        crop.x = x;
        crop.y = y;
        crop.width = width;
        crop.height = height;
        m_NativeWindow.SetCropRegion(crop);
    }

    int SharedLayerWindowImpl::GetTextureWidth(NVNtexture* pTexture) const NN_NOEXCEPT
    {
        return nvnTextureGetWidth(pTexture);
    }

    int SharedLayerWindowImpl::GetTextureHeight(NVNtexture* pTexture) const NN_NOEXCEPT
    {
        return nvnTextureGetHeight(pTexture);
    }

    nn::Result SharedLayerWindowImpl::GetInternalTextureIndex(int* pOutValue, const NVNtexture* pTexture) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_TexturePool.GetTexutureIndex(pOutValue, pTexture);
    }

}}}
