﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/detail/vi_SystemServiceProxy.private.h>

#include <new>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/sf/vi_ServiceName.h>
#include "../vi_Config.h"
#include "../vi_Macro.h"
#include "vi_IndirectLayerTable.h"

#include <binder/HOSServiceManager.h>
#include <utils/String8.h>

namespace nn{ namespace vi{ namespace detail{

    SystemServiceProxy::SystemServiceProxy() NN_NOEXCEPT
        : m_ClientManager(NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER)
        , m_RootServiceHolder(NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER)
    {
        m_RelayProxyName = { NN_VI_DRIVER_SERVICE_NAME };
        m_IndirectDisplayTransactionProxyName = { NN_VI_DEFAULT_INDIRECT_DISPLAY_TRANSACTION_PROXY_NAME };

        NN_STATIC_ASSERT(sizeof(m_IndirectLayerTableStorage) >= sizeof(IndirectLayerTable));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_IndirectLayerTableStorage) >= NN_ALIGNOF(IndirectLayerTable));
        m_pIndirectLayerTable = new(m_IndirectLayerTableStorage) IndirectLayerTable;
    }

    SystemServiceProxy::~SystemServiceProxy() NN_NOEXCEPT
    {
        m_pIndirectLayerTable->~IndirectLayerTable();
        m_pIndirectLayerTable = nullptr;
    }


    nn::Result SystemServiceProxy::Initialize() NN_NOEXCEPT
    {
        NN_VI_PROCESS_START();

        // Root(for System)
        NN_RESULT_DO(InitializeSystemServiceRootImpl(nn::vi::sf::SystemDisplayServiceName));
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemServiceRootImpl());

        // vi:u
        NN_RESULT_DO(InitializeApplicationDisplayServiceImpl(m_RelayProxyName));
        NN_VI_PROCESS_ROLLBACK(FinalizeApplicationDisplayServiceImpl());

        // vi:s
        NN_RESULT_DO(InitializeSystemDisplayServiceImpl());
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemDisplayServiceImpl());

        // relay
        NN_RESULT_DO(InitializeRelayServiceImpl());
        NN_VI_PROCESS_ROLLBACK(FinalizeRelayServiceImpl());

        NN_VI_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemServiceProxy::InitializeIndirectDisplay() NN_NOEXCEPT
    {
        NN_VI_PROCESS_START();

        // Root(for System)
        NN_RESULT_DO(InitializeSystemServiceRootImpl(nn::vi::sf::SystemDisplayServiceName));
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemServiceRootImpl());

        // vi:u
        NN_RESULT_DO(InitializeApplicationDisplayServiceImpl(m_RelayProxyName));
        NN_VI_PROCESS_ROLLBACK(FinalizeApplicationDisplayServiceImpl());

        // vi:s
        NN_RESULT_DO(InitializeSystemDisplayServiceImpl());
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemDisplayServiceImpl());

        // relay(indirect)
        NN_RESULT_DO(InitializeIndirectDisplayTransactionServiceImpl());
        NN_VI_PROCESS_ROLLBACK(FinalizeIndirectDisplayTransactionServiceImpl());

        NN_VI_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemServiceProxy::InitializeForSharedBuffer() NN_NOEXCEPT
    {
        NN_VI_PROCESS_START();

        // Root(for System)
        NN_RESULT_DO(InitializeSystemServiceRootImpl(nn::vi::sf::SystemDisplayServiceName));
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemServiceRootImpl());

        // vi:u
        NN_RESULT_DO(InitializeApplicationDisplayServiceImpl(m_RelayProxyName));
        NN_VI_PROCESS_ROLLBACK(FinalizeApplicationDisplayServiceImpl());

        // vi:s
        NN_RESULT_DO(InitializeSystemDisplayServiceImpl());
        NN_VI_PROCESS_ROLLBACK(FinalizeSystemDisplayServiceImpl());

        NN_VI_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::Finalize() NN_NOEXCEPT
    {
        if(m_pIndirectDisplayTransactionService)
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(m_IndirectDisplayTransactionProxyName.value)), 0);
            m_pIndirectDisplayTransactionService.Reset();
        }
        if(m_pRelayService)
        {
            FinalizeRelayServiceImpl();
        }

        FinalizeSystemDisplayServiceImpl();
        FinalizeApplicationDisplayServiceImpl();
        FinalizeSystemServiceRootImpl();
    }

    //------------------------------------------------------------

    nn::Result SystemServiceProxy::InitializeSystemServiceRootImpl(const char* serviceName) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ClientManager.InitializeShimLibraryHolder(&m_RootServiceHolder, serviceName));
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::FinalizeSystemServiceRootImpl() NN_NOEXCEPT
    {
        m_RootServiceHolder.FinalizeHolder();
    }

    nn::Result SystemServiceProxy::InitializeApplicationDisplayServiceImpl(const nn::vi::ProxyNameSafe& proxyName) NN_NOEXCEPT
    {
        nn::vi::PolicyLevelType policyLevel = nn::vi::PolicyLevel_Standard;

        if(nn::util::Strncmp(proxyName.value, NN_VI_DRIVER_SERVICE_NAME, sizeof(nn::vi::ProxyNameSafe)))
        {
            NN_RESULT_DO(m_RootServiceHolder.GetObject()->GetDisplayService(&m_pApplicationDisplayService, policyLevel));
        }
        else
        {
            NN_RESULT_DO(m_RootServiceHolder.GetObject()->GetDisplayServiceWithProxyNameExchange(&m_pApplicationDisplayService, policyLevel, proxyName.Get()));
        }
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::FinalizeApplicationDisplayServiceImpl() NN_NOEXCEPT
    {
        m_pApplicationDisplayService.Reset();
    }

    nn::Result SystemServiceProxy::InitializeSystemDisplayServiceImpl() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_pApplicationDisplayService->GetSystemDisplayService(&m_pSystemDisplayService));
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::FinalizeSystemDisplayServiceImpl() NN_NOEXCEPT
    {
        m_pSystemDisplayService.Reset();
    }

    nn::Result SystemServiceProxy::InitializeRelayServiceImpl() NN_NOEXCEPT
    {
        NN_VI_PROCESS_START();

        NN_RESULT_DO(m_pApplicationDisplayService->GetRelayService(&m_pRelayService));
        NN_VI_PROCESS_ROLLBACK(m_pRelayService.Reset());

        NN_RESULT_THROW_UNLESS(
            android::defaultHOSServiceManager()->addServiceProxy(android::String8(m_RelayProxyName.value), m_pRelayService) == android::NO_ERROR,
            nn::vi::ResultOperationFailed()
        );
        NN_VI_PROCESS_ROLLBACK(NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(m_RelayProxyName.value)), 0));

        NN_VI_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::FinalizeRelayServiceImpl() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(m_RelayProxyName.value)), 0);
        m_pRelayService.Reset();
    }

    nn::Result SystemServiceProxy::InitializeIndirectDisplayTransactionServiceImpl() NN_NOEXCEPT
    {
        NN_VI_PROCESS_START();

        NN_RESULT_DO(m_pApplicationDisplayService->GetIndirectDisplayTransactionService(&m_pIndirectDisplayTransactionService));
        NN_VI_PROCESS_ROLLBACK(m_pIndirectDisplayTransactionService.Reset());

        NN_RESULT_THROW_UNLESS(
            android::defaultHOSServiceManager()->addServiceProxy(android::String8(m_IndirectDisplayTransactionProxyName.value), m_pIndirectDisplayTransactionService) == android::NO_ERROR,
            nn::vi::ResultOperationFailed()
        );
        NN_VI_PROCESS_ROLLBACK(NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(m_IndirectDisplayTransactionProxyName.value)), 0));

        NN_VI_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SystemServiceProxy::FinalizeIndirectDisplayTransactionServiceImpl() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(m_IndirectDisplayTransactionProxyName.value)), 0);
        m_pIndirectDisplayTransactionService.Reset();
    }

    //------------------------------------------------------------

    nn::sf::SharedPointer<nn::visrv::sf::IApplicationDisplayService> SystemServiceProxy::GetApplicationDisplayService() NN_NOEXCEPT
    {
        return m_pApplicationDisplayService;
    }

    nn::sf::SharedPointer<nn::visrv::sf::ISystemDisplayService> SystemServiceProxy::GetSystemDisplayService() NN_NOEXCEPT
    {
        return m_pSystemDisplayService;
    }

    IndirectLayerTable* SystemServiceProxy::GetIndirectLayerTable() NN_NOEXCEPT
    {
        return m_pIndirectLayerTable;
    }

}}}
