﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "vi_IndirectLayerTable.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>

namespace nn{ namespace vi{ namespace detail{

    void IndirectLayerEntry::Clear() NN_NOEXCEPT
    {
        handle = 0;
        window.Reset();
    }

    //----------------------------------------------------
    const int IndirectLayerTable::Capacity;

    IndirectLayerTable::IndirectLayerTable() NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; i++)
        {
            m_EntryStateList[i] = EntryState_Free;
        }
        for(int i = 0; i < Capacity; i++)
        {
            m_EntryList[i].Clear();
        }
    }

    nn::Result IndirectLayerTable::Allocate(int* pOutIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutIndex);

        for(int i = 0; i < Capacity; i++)
        {
            if(m_EntryStateList[i] == EntryState_Free)
            {
                *pOutIndex = i;
                m_EntryStateList[i] = EntryState_Allocated;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(nn::vi::ResultNotFound());
    }

    void IndirectLayerTable::Free(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, Capacity);
        NN_SDK_REQUIRES_NOT_EQUAL(m_EntryStateList[index], EntryState_Free);

        m_EntryList[index].Clear();
        m_EntryStateList[index] = EntryState_Free;
    }

    int IndirectLayerTable::GetIndex(IndirectLayerEntry* pEntry) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pEntry);
        ptrdiff_t diff = reinterpret_cast<uintptr_t>(pEntry) - reinterpret_cast<uintptr_t>(m_EntryList);
        NN_SDK_REQUIRES(diff % sizeof(IndirectLayerEntry) == 0);
        ptrdiff_t index = diff / sizeof(IndirectLayerEntry);
        if(index < 0 || index >= Capacity)
        {
            return -1;
        }
        return static_cast<int>(index);
    }

    IndirectLayerEntry& IndirectLayerTable::Get(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, Capacity);
        return m_EntryList[index];
    }

}}}
