﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/detail/vi_IndirectLayerImpl.h>

#include "vi_IndirectLayerTable.h"
#include "../native/vi_DeserializeNativeWindow.h"

namespace nn{ namespace vi{ namespace detail{

    nn::Result IndirectLayerImpl::GetLayerHandle(
        IndirectProducerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        IndirectLayerTable* pLayerTable
    ) NN_NOEXCEPT
    {
        IndirectLayerEntry* pEntry = reinterpret_cast<IndirectLayerEntry*>(pLayer);
        int index = pLayerTable->GetIndex(pEntry);
        NN_RESULT_THROW_UNLESS(index >= 0 && index < IndirectLayerTable::Capacity, nn::vi::ResultNotFound());

        *pOutHandle = pEntry->handle;
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayerImpl::RegisterIndirectLayerImpl(
        IndirectLayer** pOutLayer,
        IndirectProducerHandleType handle,
        void* pNativeWindowBuffer,
        size_t nativeWindowBufferSize,
        IndirectLayerTable* pLayerTable
    ) NN_NOEXCEPT
    {
        bool isSuccess = false;

        // allocate entry
        int index = 0;
        NN_RESULT_DO(pLayerTable->Allocate(&index));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pLayerTable->Get(index).Clear();
                pLayerTable->Free(index);
            }
        };

        // create entry
        auto& entry = pLayerTable->Get(index);
        entry.handle = handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            native::DeserializeNativeWindow(&entry.window, pNativeWindowBuffer, static_cast<size_t>(nativeWindowBufferSize))
        );

        // connect
        NN_ABORT_UNLESS_EQUAL(0, native_window_api_connect(entry.window.GetNativeWindow(), NATIVE_WINDOW_API_CPU));

        *pOutLayer = reinterpret_cast<IndirectLayer*>(&entry);
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayerImpl::UnregisterIndirectLayerImpl(
        IndirectLayer* pLayer,
        IndirectLayerTable* pLayerTable
        ) NN_NOEXCEPT
    {
        IndirectLayerEntry* pEntry = reinterpret_cast<IndirectLayerEntry*>(pLayer);
        int index = pLayerTable->GetIndex(pEntry);
        NN_RESULT_THROW_UNLESS(index >= 0 && index < IndirectLayerTable::Capacity, nn::vi::ResultNotFound());

        // disconnect
        NN_ABORT_UNLESS_EQUAL(0, native_window_api_disconnect(pEntry->window.GetNativeWindow(), NATIVE_WINDOW_API_CPU));

        // unregister entry
        pEntry->Clear();
        pLayerTable->Free(index);

        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayerImpl::GetNativeWindow(
        NativeWindowHandle* pOutNativeWindow,
        IndirectLayer* pLayer,
        IndirectLayerTable* pLayerTable
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutNativeWindow);
        NN_SDK_REQUIRES_NOT_NULL(pLayer);
        NN_SDK_REQUIRES_NOT_NULL(pLayerTable);

        IndirectLayerEntry* pEntry = reinterpret_cast<IndirectLayerEntry*>(pLayer);
        int index = pLayerTable->GetIndex(pEntry);
        NN_RESULT_THROW_UNLESS(index >= 0 && index < IndirectLayerTable::Capacity, nn::vi::ResultNotFound());

        *pOutNativeWindow = pEntry->window.GetNativeWindow();
        NN_RESULT_SUCCESS;
    }

}}}
