﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hs/usb_HsPrivateIncludes.h"
#include <nn/usb/usb_Host.h>

#include "hs/usb_HsServiceName.h"

namespace nn {
namespace usb {

enum Quirk
{
    Quirk_NoClearHaltOnEpInit = 1 << 0,
};

static struct {
    uint16_t vid;
    uint16_t pid;
    uint32_t quirks;
} g_Quirks[] = {
    // SIGLO-61518: Joy-Con Charging Grip FW Bug
    { 0x057e, 0x2009, Quirk_NoClearHaltOnEpInit },  // Pro Controller
    { 0x057e, 0x200e, Quirk_NoClearHaltOnEpInit },  // Joy-Con Charging Grip
};

//--------------------------------------------------------------------------
//  Host class implementation
//--------------------------------------------------------------------------
Result Host::Initialize() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::sf::NativeHandle sfEventHandle;
    nn::sf::NativeHandle processHandle(
        static_cast<nn::os::NativeHandle>(nn::dd::GetCurrentProcessHandle()),
        false
    );

    do
    {
        NN_USB_BREAK_UPON_ERROR(
           (m_IsInitialized) ? ResultAlreadyInitialized() : ResultSuccess()
        );

        // We cannot guarantee that the standard allocator is enabled in class
        // driver process (e.g. it's not enalbed in hid process as of now). So
        // we have to use a customized allocator here.
        m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer,
                                               sizeof(m_HeapBuffer),
                                               nn::lmem::CreationOption_NoOption);
        NN_USB_BREAK_UNLESS(m_HeapHandle, ResultMemAllocFailure());
        m_Allocator.Attach(m_HeapHandle);

        NN_USB_BREAK_UPON_ERROR((
            m_Domain.InitializeByName<hs::IClientRootSession, MyAllocator::Policy>(
                &m_RootHandle, &m_Allocator, hs::ServiceName
            )
        ));

        m_Domain.SetSessionCount(1);

        NN_USB_BREAK_UPON_ERROR(
            m_RootHandle->BindClientProcess(processHandle.GetShared())
        );

        NN_USB_BREAK_UPON_ERROR(
            m_RootHandle->GetInterfaceStateChangeEvent(&sfEventHandle)
        );

        nn::os::AttachReadableHandleToSystemEvent(
            &m_IfStateChangeEvent,
            sfEventHandle.GetOsHandle(),
            sfEventHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear
        );
        sfEventHandle.Detach();

        m_IsInitialized = true;
    } while (false);

    return result;
}

Result Host::Finalize() NN_NOEXCEPT
{
    int refCount = 0;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if(!m_IsInitialized) return ResultNotInitialized();

    if((refCount = m_RefCount) > 1)
    {
        NN_SDK_LOG("[usb:hs] Host::Finalize() busy with refCount %d.\n", refCount - 1);
        return ResultResourceBusy();
    }

    m_IsInitialized = false;
    nn::os::DestroySystemEvent(&m_IfStateChangeEvent);
    nn::lmem::DestroyExpHeap(m_HeapHandle);
    m_HeapHandle = nullptr;
    m_RootHandle = nullptr;
    m_Domain.Finalize();
    return ResultSuccess();
}

bool Host::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

Result Host::QueryAllInterfaces(int32_t* pOutIfCount, InterfaceQueryOutput *pOutBuffer, size_t queryBufferSize, DeviceFilter *pDevFilter) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutIfCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pDevFilter);
    nn::sf::OutBuffer data(reinterpret_cast<char *>(pOutBuffer), queryBufferSize);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_RootHandle->QueryAllInterfaces(pOutIfCount, data, *pDevFilter);
    return result;
}

Result Host::QueryAvailableInterfaces(int32_t* pOutIfCount, InterfaceQueryOutput *pOutBuffer, size_t queryBufferSize, DeviceFilter *pDevFilter) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutIfCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pDevFilter);
    nn::sf::OutBuffer data(reinterpret_cast<char *>(pOutBuffer), queryBufferSize);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_RootHandle->QueryAvailableInterfaces(pOutIfCount, data, *pDevFilter);
    return result;
}

Result Host::QueryAcquiredInterfaces(int32_t* pOutIfCount, InterfaceQueryOutput *pOutBuffer, size_t queryBufferSize) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutIfCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    nn::sf::OutBuffer data(reinterpret_cast<char *>(pOutBuffer), queryBufferSize);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_RootHandle->QueryAcquiredInterfaces(pOutIfCount, data);
    return result;
}

Result Host::CreateInterfaceAvailableEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode eventClearMode,
                                         uint8_t filterIndex, DeviceFilter* pDevFilter) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    nn::sf::NativeHandle sfEventHandle;
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);
    NN_ABORT_UNLESS_NOT_NULL(pDevFilter);
    if(!m_IsInitialized) return ResultNotInitialized();
    if((result = m_RootHandle->CreateInterfaceAvailableEvent(&sfEventHandle, filterIndex, *pDevFilter)).IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfEventHandle.GetOsHandle(), sfEventHandle.IsManaged(), eventClearMode);
        sfEventHandle.Detach();
    }
    return result;
}

Result Host::DestroyInterfaceAvailableEvent(nn::os::SystemEventType* pInEvent, uint8_t filterIndex) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pInEvent);
    if(!m_IsInitialized) return ResultNotInitialized();
    if((result = m_RootHandle->DestroyInterfaceAvailableEvent(filterIndex)).IsSuccess())
    {
        nn::os::DestroySystemEvent(pInEvent);
    }
    return result;
}

nn::os::SystemEventType *Host::GetInterfaceStateChangeEvent() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_IfStateChangeEvent : nullptr;
}

Result Host::SetTestMode(uint32_t port, TestMode mode, int32_t offset, uint32_t *pOutStrength) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutStrength);

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    return m_RootHandle->SetTestMode(port, mode, offset, pOutStrength);
}

void Host::PrintInterfaceQuery(InterfaceQueryOutput *pQuery) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pQuery);
    hs::Util::PrintInterfaceQuery(pQuery);
}

void Host::PrintInterfaceProfile(InterfaceProfile *pInterfaceProfile) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pInterfaceProfile);
    hs::Util::PrintInterfaceProfile(pInterfaceProfile);
}

//--------------------------------------------------------------------------
//  HostInterface class implementation
//--------------------------------------------------------------------------
Result HostInterface::Initialize(Host *pHost, InterfaceHandle ifHandle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::sf::NativeHandle stateChangeEventHandle;
    nn::sf::NativeHandle ctrlXferEventHandle;

    NN_ABORT_UNLESS_NOT_NULL(pHost);

    if(m_IsInitialized)
    {
        return ResultAlreadyInitialized();
    }

    nn::sf::OutBuffer deviceProfileBuffer(
        reinterpret_cast<char *>(&m_DeviceProfile), sizeof(m_DeviceProfile)
    );
    nn::sf::OutBuffer interfaceProfileBuffer(
        reinterpret_cast<char *>(&m_InterfaceProfile), sizeof(m_InterfaceProfile)
    );

    result = pHost->m_RootHandle->AcquireUsbIf(
        ifHandle, &m_pHandle, deviceProfileBuffer, interfaceProfileBuffer
    );
    if (result.IsFailure())
    {
        return result;
    }

    result = m_pHandle->GetStateChangeEvent(&stateChangeEventHandle);
    if (result.IsFailure())
    {
        m_pHandle = nullptr;
        return result;
    }

    result = m_pHandle->GetCtrlXferCompletionEvent(&ctrlXferEventHandle);
    if (result.IsFailure())
    {
        m_pHandle = nullptr;
        return result;
    }

    for (auto& entry : g_Quirks)
    {
        if (entry.vid == m_DeviceProfile.deviceDesc.idVendor &&
            entry.pid == m_DeviceProfile.deviceDesc.idProduct )
        {
            m_Quirks = entry.quirks;
            NN_SDK_LOG("Applying USB Quirks %08x to VID %04x PID %04x\n",
                       entry.quirks, entry.vid, entry.pid);
            break;
        }
    }

    m_pHost    = pHost;
    m_pHost->m_RefCount++;

    nn::os::AttachReadableHandleToSystemEvent(
        &m_StateChangeEvent,
        stateChangeEventHandle.GetOsHandle(),
        stateChangeEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    stateChangeEventHandle.Detach();

    nn::os::AttachReadableHandleToSystemEvent(
        &m_CtrlXferCompletionEvent,
        ctrlXferEventHandle.GetOsHandle(),
        ctrlXferEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    ctrlXferEventHandle.Detach();
    m_IsInitialized = true;

    return ResultSuccess();
}

Result HostInterface::Finalize() NN_NOEXCEPT
{
    int refCount = 0;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();

    if((refCount = m_RefCount) > 1)
    {
        NN_SDK_LOG("[usb:hs] HostInterface::Finalize() busy with refCount %d.\n", refCount - 1);
        return ResultResourceBusy();
    }

    m_IsInitialized = false;
    m_pHandle = nullptr;
    m_pHost->m_RefCount--;
    m_pHost   = nullptr;
    memset(&m_InterfaceProfile, 0, sizeof(m_InterfaceProfile));
    memset(&m_DeviceProfile,    0, sizeof(m_DeviceProfile));
    nn::os::DestroySystemEvent(&m_CtrlXferCompletionEvent);
    nn::os::DestroySystemEvent(&m_StateChangeEvent);

    return ResultSuccess();
}

bool HostInterface::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::os::SystemEventType *HostInterface::GetStateChangeEvent() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_StateChangeEvent : nullptr;
}

Result HostInterface::SetInterface(uint8_t bAlternateSetting) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();
    nn::sf::OutBuffer incomingData(reinterpret_cast<char *>(&m_InterfaceProfile), sizeof(InterfaceProfile));
    result = m_pHandle->SetInterface(bAlternateSetting, incomingData);
    return result;
}

Result HostInterface::GetInterface(InterfaceProfile* pOutInterfaceProfile) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutInterfaceProfile);
    if(!m_IsInitialized) return ResultNotInitialized();
    nn::sf::OutBuffer incomingData(reinterpret_cast<char *>(pOutInterfaceProfile), sizeof(InterfaceProfile));
    result = m_pHandle->GetInterface(incomingData);
    return result;
}

Result HostInterface::GetAlternateInterface(InterfaceProfile *pOutInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutInterfaceProfile);
    if(!m_IsInitialized) return ResultNotInitialized();
    nn::sf::OutBuffer incomingData(reinterpret_cast<char *>(pOutInterfaceProfile), sizeof(InterfaceProfile));
    result = m_pHandle->GetAlternateInterface(bAlternateSetting, incomingData);
    return result;
}

Result HostInterface::GetCurrentFrame(FrameNumber *pOutFrame) NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutFrame);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_pHandle->GetCurrentFrame(pOutFrame);
    return result;
}

Result HostInterface::ControlRequest(size_t *pOutTransferredSize, void *pData,
                                     uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue,
                                     uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT
{
    Result   result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutTransferredSize);
    if(!m_IsInitialized) return ResultNotInitialized();

    // Serialize the control transfers
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_USB_RETURN_UPON_ERROR(
        m_pHandle->CtrlXferAsync(
            bmRequestType,
            bRequest,
            wValue,
            wIndex,
            wLength,
            reinterpret_cast<uint64_t>(pData)
        )
    );

    nn::os::WaitSystemEvent(&m_CtrlXferCompletionEvent);
    nn::os::ClearSystemEvent(&m_CtrlXferCompletionEvent);

    XferReport report;
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(&report), sizeof(report));

    NN_USB_RETURN_UPON_ERROR(
        m_pHandle->GetCtrlXferReport(buffer)
    );

    *pOutTransferredSize = report.transferredSize;

    return report.result;
}

Result HostInterface::ResetDevice() NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_pHandle->ResetDevice();
    return result;
}

UsbInterfaceDescriptor& HostInterface::GetDescriptor() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return m_InterfaceProfile.ifDesc;
}

//--------------------------------------------------------------------------
//  HostEndpoint class implementation
//--------------------------------------------------------------------------
Result HostEndpoint::Initialize(HostInterface *pIf, UsbEndpointDescriptor *pEpDescriptor,
                                uint16_t maxUrb, uint32_t maxXferSize) NN_NOEXCEPT
{
    Result result;
    nn::sf::NativeHandle completionEventHandle;
    size_t unused = 0;

    UsbEndpointType      epType      = UsbGetEndpointType(pEpDescriptor);
    EndpointNumber       epNumber    = UsbGetEndpointNumber(pEpDescriptor);
    UsbEndpointDirection epDirection = UsbGetEndpointDirection(pEpDescriptor);

    NN_ABORT_UNLESS_NOT_NULL(pIf);
    if(m_IsInitialized) return ResultAlreadyInitialized();
    if(!pIf->m_IsInitialized) return ResultInterfaceInvalidState();

    result = pIf->m_pHandle->OpenUsbEp(epType, epNumber, epDirection, maxUrb,
                                       maxXferSize, &m_EndpointDescriptor, &m_pHandle);
    if (result.IsFailure())
    {
        return result;
    }

    // Setup the Xfer and Report Ring
    result = m_pHandle->PopulateRing();
    if (result.IsFailure())
    {
        m_pHandle = nullptr;
        return result;
    }

    // Reset device side Data Toggle
    if (epType == UsbEndpointType_Bulk || epType == UsbEndpointType_Int)
    {
        if (!(pIf->m_Quirks & Quirk_NoClearHaltOnEpInit))
        {
            result = pIf->ControlRequest(
                &unused, nullptr,
                BmRequestType(Standard, Endpoint, HostToDevice),
                UsbCtrlXferReq_ClearFeature,
                UsbFeatureSelector_EndpointHalt,
                UsbGetEndpointAddress(epNumber, epDirection),
                0
            );
            if (result.IsFailure())
            {
                m_pHandle = nullptr;
                return result;
            }
        }
    }

    result = m_pHandle->GetCompletionEvent(&completionEventHandle);
    if (result.IsFailure())
    {
        m_pHandle = nullptr;
        return result;
    }

    m_MaxUrbCount    = maxUrb;
    m_pIf            = pIf;
    m_EpType         = epType;
    m_EndpointNumber = epNumber;
    m_EpDirection    = epDirection;
    m_pIf->m_RefCount++;
    m_pIf->m_pHost->m_RefCount++;

    nn::os::AttachReadableHandleToSystemEvent(
        &m_CompletionEvent,
        completionEventHandle.GetOsHandle(),
        completionEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    completionEventHandle.Detach();

    m_IsInitialized = true;

    return ResultSuccess();
}

Result HostEndpoint::Finalize() NN_NOEXCEPT
{
    int refCount = 0;
    int lastRefCount = 0;
    Result result = ResultSuccess();
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if(!m_IsInitialized) return ResultNotInitialized();
    m_IsInitialized = false;

    // This deactivates the endpoint, causing any pending calls to return.
    // Unsuccessful status expected if device has since disconnected.
    (void)m_pHandle->Close();

    // It is necessary to synchronize against any pending calls into this object
    while((refCount = m_RefCount) > 1)
    {
        if(lastRefCount != refCount)
        {
            NN_SDK_LOG("[usb:hs] HostEndpoint::Finalize() endpoint %d waiting on refcount %d...\n",
                       m_EndpointNumber, refCount - 1);
        }
        lastRefCount = refCount;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(25));
    }
    if(lastRefCount > 0)
    {
        NN_SDK_LOG("[usb:hs] HostEndpoint::Finalize() wait done.\n");
    }

    m_ReportRing.Detach();

    // Tear things down
    nn::os::DestroySystemEvent(&m_CompletionEvent);
    m_pIf->m_RefCount--;
    m_pIf->m_pHost->m_RefCount--;
    m_pHandle         = nullptr;
    m_pIf             = nullptr;
    m_EpType          = UsbEndpointType_Invalid;
    m_EndpointNumber  = 0;
    m_EpDirection     = UsbEndpointDirection_Invalid;
    memset(&m_EndpointDescriptor, 0, sizeof(m_EndpointDescriptor));

    return result;
}

bool HostEndpoint::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

Result HostEndpoint::CreateSmmuSpace(void *buffer, uint32_t size) NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    if (!NN_USB_IS_DMA_ALIGNED(buffer) || !NN_USB_IS_DMA_ALIGNED(size))
    {
        return ResultAlignmentError();
    }

    return m_pHandle->CreateSmmuSpace(reinterpret_cast<uint64_t>(buffer), size);
}

Result HostEndpoint::ShareReportRing(void *buffer, uint32_t size) NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    return m_ReportRing.Attach(buffer, size, HsLimitMaxXferPerUrbCount * m_MaxUrbCount);
}

nn::os::SystemEventType* HostEndpoint::GetCompletionEvent() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_CompletionEvent : nullptr;
}

bool HostEndpoint::TimedWaitForCompletion(nn::TimeSpan timeout) NN_NOEXCEPT
{
    bool isCompleted;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return false;

    if (timeout == 0)
    {
        nn::os::WaitSystemEvent(&m_CompletionEvent);
        isCompleted = true;
    }
    else
    {
        isCompleted = nn::os::TimedWaitSystemEvent(&m_CompletionEvent, timeout);

    }

    if (isCompleted)
    {
        nn::os::ClearSystemEvent(&m_CompletionEvent);
    }

    return isCompleted;
}

Result HostEndpoint::PostBuffer(size_t   *pOutTransferredSize,
                                void     *buffer,
                                uint32_t  bytes) NN_NOEXCEPT
{
    Result     result;
    uint32_t   xferId;
    XferReport report;
    uint32_t   count;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    NN_USB_RETURN_UPON_ERROR(
        PostBufferAsync(&xferId, buffer, bytes, 0)
    );

    nn::os::WaitSystemEvent(&m_CompletionEvent);
    nn::os::ClearSystemEvent(&m_CompletionEvent);

    NN_USB_RETURN_UPON_ERROR(GetXferReport(&count, &report, 1));

    // Not supposed to be used with PostBufferAsync()
    NN_SDK_ASSERT_EQUAL(count, 1);
    NN_SDK_ASSERT_EQUAL(xferId, report.id);

    *pOutTransferredSize = report.transferredSize;

    return report.result;
}

Result HostEndpoint::PostBufferAsync(uint32_t  *pOutXferId,
                                     void      *buffer,
                                     uint32_t   bytes,
                                     uint64_t   context) NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    return m_pHandle->PostBufferAsync(
        pOutXferId, reinterpret_cast<uint64_t>(buffer), bytes, context
    );
}

Result HostEndpoint::BatchBufferAsync(uint32_t       *pOutXferId,
                                      void           *buffer,
                                      const uint32_t *pSizeArray,
                                      uint32_t        xferCount,
                                      uint64_t        context,
                                      SchedulePolicy  policy,
                                      uint32_t        frameId) NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    return m_pHandle->BatchBufferAsync(
        pOutXferId, reinterpret_cast<uint64_t>(buffer),
        nn::sf::InBuffer(reinterpret_cast<const char*>(pSizeArray), xferCount * 4), xferCount,
        context, policy, frameId
    );
}

Result HostEndpoint::GetXferReport(uint32_t     *pOutCount,
                                   XferReport   *pOutReport,
                                   uint32_t      count) NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if(!m_IsInitialized) return ResultNotInitialized();

    return m_ReportRing.GetXferReport(pOutCount, pOutReport, count);
}

Result HostEndpoint::ClearEndpointHalt() NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    if((result = m_pHandle->Close()).IsSuccess())
    {
        size_t unused = 0;

        nn::os::ClearSystemEvent(&m_CompletionEvent);

        if((result = m_pIf->ControlRequest(&unused, nullptr,
                                           UsbCtrlXferReqTypeMask_DirHostToDevice |
                                           UsbCtrlXferReqTypeMask_TypeStandard |
                                           UsbCtrlXferReqTypeMask_RecipEndpoint,
                                           UsbCtrlXferReq_ClearFeature,
                                           UsbFeatureSelector_EndpointHalt,
                                           UsbGetEndpointAddress(m_EndpointNumber, m_EpDirection),
                                           0)).IsSuccess())
        {
            result = m_pHandle->ReOpen();
        }
    }

    return result;
}

Result HostEndpoint::Terminate() NN_NOEXCEPT
{
    Result result;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();
    result = m_pHandle->Close();
    return result;
}

UsbEndpointDescriptor& HostEndpoint::GetDescriptor() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return m_EndpointDescriptor;

}

HostEndpoint::ReportRing::ReportRing(nn::sf::SharedPointer<hs::IClientEpSession>& epSession)  NN_NOEXCEPT
    : detail::RingBase<XferReport>()
    , m_Mutex(false)
    , m_Handle(nn::os::InvalidNativeHandle)
    , m_EpSession(epSession)
{
}

HostEndpoint::ReportRing::~ReportRing() NN_NOEXCEPT
{
    Detach();
}

Result HostEndpoint::ReportRing::Attach(void *buffer, uint32_t size, uint32_t itemCount) NN_NOEXCEPT
{
    Result result;
    nn::os::TransferMemoryType xferMem;

    if (!NN_USB_IS_DMA_ALIGNED(buffer) || !NN_USB_IS_DMA_ALIGNED(size))
    {
        return ResultAlignmentError();
    }

    if (size < sizeof(Storage) + sizeof(ItemType) * itemCount)
    {
        return ResultInvalidParameter();
    }

    if (m_Handle != nn::os::InvalidNativeHandle)
    {
        return ResultOperationDenied();
    }

    result = nn::os::CreateTransferMemory(
        &xferMem, buffer, size, nn::os::MemoryPermission_ReadWrite
    );
    if (result.IsFailure())
    {
        goto bail1;
    }

    m_Handle = nn::os::DetachTransferMemory(&xferMem);

    nn::os::DestroyTransferMemory(&xferMem);

    result = m_EpSession->ShareReportRing(nn::sf::NativeHandle(m_Handle, false), size);
    if (result.IsFailure())
    {
        goto bail2;
    }

    m_pStorage  = reinterpret_cast<Storage*>(buffer);
    m_ItemCount = itemCount + 1; // +1 for sentinel
    return ResultSuccess();

bail2:
    nn::os::CloseNativeHandle(m_Handle);
bail1:
    return result;
}

void HostEndpoint::ReportRing::Detach() NN_NOEXCEPT
{
    if (m_Handle != nn::os::InvalidNativeHandle)
    {
        nn::os::CloseNativeHandle(m_Handle);
    }

    m_pStorage = nullptr;
    m_Handle   = nn::os::InvalidNativeHandle;
}

Result HostEndpoint::ReportRing::GetXferReport(uint32_t     *pOutCount,
                                               XferReport   *pOutReport,
                                               uint32_t      count) NN_NOEXCEPT
{
    uint32_t  collectedCount = 0;

    if (m_Handle == nn::os::InvalidNativeHandle)
    {
        return m_EpSession->GetXferReport(
            pOutCount,
            nn::sf::OutBuffer(reinterpret_cast<char*>(pOutReport), sizeof(XferReport) * count),
            count
        );
    }
    else
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        while (collectedCount < count)
        {
            auto *ptr = Shrink();

            if (ptr == nullptr)
            {
                break;
            }

            pOutReport[collectedCount++] = *ptr;
        }

        *pOutCount = collectedCount;

        return ResultSuccess();
    }
}

} // end of namespace usb
} // end of namespace nn
