﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE 1

#include <nn/dd.h>
#include <nn/usb/usb_Device.h>

#include "detail/usb_Util.h"
#include "ds/usb_DsServiceName.h"

using namespace nn::sf;
using namespace nn::dd;

namespace nn {
namespace usb {

//--------------------------------------------------------------------------
//  DsClient class implementation
//--------------------------------------------------------------------------

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsClient"

Result DsClient::Initialize(ComplexId complexId) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Result result = ResultSuccess();
    nn::sf::NativeHandle sfEventHandle;
    nn::sf::NativeHandle processHandle(
        static_cast<nn::os::NativeHandle>(GetCurrentProcessHandle()),
        false
    );

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pInterfaces); i++)
    {
        m_pInterfaces[i] = nullptr;
    }

    do
    {
        m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer,
                                               sizeof(m_HeapBuffer),
                                               nn::lmem::CreationOption_NoOption);
        NN_USB_BREAK_UNLESS(m_HeapHandle, ResultMemAllocFailure());

        m_Allocator.Attach(m_HeapHandle);

        // double parentheses becasue preprocessore take comma as argument delimiter
        NN_USB_BREAK_UPON_ERROR((
            m_Domain.InitializeByName<ds::IDsService, MyAllocator::Policy>(
                &m_Handle, &m_Allocator, ds::ServiceName
            )
        ));
        m_Domain.SetSessionCount(1);

        NN_USB_BREAK_UPON_ERROR(m_Handle->BindComplex(complexId));
        NN_USB_BREAK_UPON_ERROR(m_Handle->BindClientProcess(processHandle.GetShared()));
        NN_USB_BREAK_UPON_ERROR(m_Handle->GetStateChangeEvent(&sfEventHandle));

        nn::os::AttachReadableHandleToSystemEvent(
            &m_StateChangeEvent,
            sfEventHandle.GetOsHandle(),
            sfEventHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear
        );

        sfEventHandle.Detach();

        m_IsInitialized = true;
    } while (false);

    return result;
}


Result DsClient::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;

    int refCount = 0;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if(!m_IsInitialized) return ResultNotInitialized();

    Result result = ResultSuccess();

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pInterfaces); i++)
    {
        if (m_pInterfaces[i] != nullptr)
        {
            result = m_pInterfaces[i]->Disable();
            if (!result.IsSuccess())
            {
                return result;
            }

            result = m_pInterfaces[i]->Finalize();
            if (!result.IsSuccess())
            {
                return result;
            }
        }
    }

    if((refCount = m_RefCount) > 1)
    {
        NN_SDK_LOG("[usb:ds] DsClient::Finalize() busy with refCount %d.\n", refCount - 1);
        return ResultResourceBusy();
    }

    m_IsInitialized = false;
    nn::os::DestroySystemEvent(&m_StateChangeEvent);
    nn::lmem::DestroyExpHeap(m_HeapHandle);
    m_Handle = nullptr;
    m_HeapHandle = NULL;
    m_Domain.Finalize();

    return result;
}

bool DsClient::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::os::SystemEventType *DsClient::GetStateChangeEvent() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_StateChangeEvent : nullptr;
}

Result DsClient::GetState(UsbState *pOutState) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    return m_Handle->GetState(pOutState);
}


Result DsClient::ClearDeviceData() NN_NOEXCEPT
{
    NN_USB_TRACE;

    return m_Handle->ClearDeviceData();
}


Result DsClient::AddUsbStringDescriptor(uint8_t *pIndex, UsbStringDescriptor *pUsbStringDescriptor) NN_NOEXCEPT
{
    NN_USB_TRACE;

    InBuffer usbStringDescriptorBuffer(reinterpret_cast<const char *>(pUsbStringDescriptor), sizeof(UsbStringDescriptor));
    return m_Handle->AddUsbStringDescriptor(pIndex, usbStringDescriptorBuffer);
}


Result DsClient::DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT
{
    NN_USB_TRACE;

    return m_Handle->DeleteUsbStringDescriptor(index);
}


Result DsClient::SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    NN_USB_TRACE;

    InBuffer usbDeviceDescriptorBuffer(reinterpret_cast<const char *>(pUsbDeviceDescriptor), sizeof(UsbDeviceDescriptor));
    return m_Handle->SetUsbDeviceDescriptor(usbDeviceDescriptorBuffer, usbDeviceSpeed);
}


Result DsClient::SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT
{
    NN_USB_TRACE;

    InBuffer binaryObjectStore(reinterpret_cast<const char *>(pData), size);
    return m_Handle->SetBinaryObjectStore(binaryObjectStore);
}


Result DsClient::AddInterface(DsInterface                      *pInterface,
                              SharedPointer<ds::IDsInterface>  *pHandle,
                              uint8_t                          bInterfaceNumber) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    Result result = m_Handle->RegisterInterface(pHandle, bInterfaceNumber);
    if (result.IsFailure())
    {
        return result;
    }

    m_pInterfaces[bInterfaceNumber] = pInterface;

    return ResultSuccess();
}

Result DsClient::DeleteInterface(uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    // the interface will be unregistered in shared object's destructor
    if (m_pInterfaces[ifNum] == nullptr)
    {
        return ResultOperationDenied();
    }

    m_pInterfaces[ifNum] = nullptr;

    return ResultSuccess();
}

Result DsClient::EnableDevice() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    return m_Handle->Enable();
}


Result DsClient::DisableDevice() NN_NOEXCEPT
{
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    return m_Handle->Disable();
}

/*
// see SIGLONTD-3250
Result DsClient::IncreaseSession() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

    // Don't use m_Domain.IncreaseSession() so the implementation is symmetric
    // with DsClient::DecreaseSession()
    return m_Domain.SetSessionCount(m_Domain.GetSessionCount() + 1);
}

// see SIGLONTD-3250
Result DsClient::DecreaseSession() NN_NOEXCEPT
{ NN_USB_LOG_INFO("%s::%s(@%d)\n", kUsbLogClassName, __FUNCTION__,__LINE__);
    std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

    // There is no such API like m_Domain.DecreaseSession().
    return m_Domain.SetSessionCount(m_Domain.GetSessionCount() - 1);
}
*/

//--------------------------------------------------------------------------
//  DsInterface class implementation
//--------------------------------------------------------------------------

#undef NN_USB_CLASS_NAME
#define NN_USB_CLASS_NAME "DsInterface"

Result DsInterface::Initialize(DsClient *pClient, uint8_t bInterfaceNumber) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;
    NativeHandle sfSetupEventHandle;
    NativeHandle sfCtrlInCompletionEventHandle;
    NativeHandle sfCtrlOutCompletionEventHandle;

    if(m_IsInitialized)
    {
        return ResultAlreadyInitialized();
    }

    m_pClient = pClient;
    m_pClient->m_RefCount++;

    m_IsEnabled = false;

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pEndpoints); i++)
    {
        m_pEndpoints[i] = nullptr;
    }

    m_IfNum = bInterfaceNumber;

    result = m_pClient->AddInterface(this, &m_Handle, m_IfNum);
    if (result.IsFailure())
    {
        return result;
    }

    do {
        NN_USB_BREAK_UPON_ERROR(
            m_Handle->GetSetupEvent(&sfSetupEventHandle)
        );
        NN_USB_BREAK_UPON_ERROR(
            m_Handle->GetCtrlInCompletionEvent(&sfCtrlInCompletionEventHandle)
        );
        NN_USB_BREAK_UPON_ERROR(
            m_Handle->GetCtrlOutCompletionEvent(&sfCtrlOutCompletionEventHandle)
        );
    } while (false);

    if (result.IsFailure())
    {
        result = m_pClient->DeleteInterface(m_IfNum);
        m_Handle = nullptr;

        return result;
    }

    nn::os::AttachReadableHandleToSystemEvent(
        &m_SetupEvent,
        sfSetupEventHandle.GetOsHandle(),
        sfSetupEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    sfSetupEventHandle.Detach();

    nn::os::AttachReadableHandleToSystemEvent(
        &m_CtrlInCompletionEvent,
        sfCtrlInCompletionEventHandle.GetOsHandle(),
        sfCtrlInCompletionEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    sfCtrlInCompletionEventHandle.Detach();

    nn::os::AttachReadableHandleToSystemEvent(
        &m_CtrlOutCompletionEvent,
        sfCtrlOutCompletionEventHandle.GetOsHandle(),
        sfCtrlOutCompletionEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    sfCtrlOutCompletionEventHandle.Detach();

    m_IsInitialized = true;

    return result;
}

Result DsInterface::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_USB_ABORT_IF_NULL(m_Handle);

    Result result = ResultSuccess();

    // the interface must be disabled explicitly first
    if (m_IsEnabled)
    {
        return ResultResourceBusy();
    }

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pEndpoints); i++)
    {
        if (m_pEndpoints[i] != nullptr)
        {
            result = m_pEndpoints[i]->Finalize();
            if (!result.IsSuccess())
            {
                return result;
            }
        }
    }

    int refCount = 0;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();

    if((refCount = m_RefCount) > 1)
    {
        NN_SDK_LOG("[usb:ds] DsInterface::Finalize() busy with refCount %d.\n", refCount - 1);
        return ResultResourceBusy();
    }

    m_IsInitialized = false;

    nn::os::DestroySystemEvent(&m_SetupEvent);
    nn::os::DestroySystemEvent(&m_CtrlInCompletionEvent);
    nn::os::DestroySystemEvent(&m_CtrlOutCompletionEvent);

    m_pClient->DeleteInterface(m_IfNum);
    m_Handle  = nullptr;
    m_pClient->m_RefCount--;
    m_pClient = nullptr;

    return result;
}

bool DsInterface::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::os::SystemEventType *DsInterface::GetSetupEvent() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_SetupEvent : nullptr;
}

Result DsInterface::GetSetupPacket(UsbCtrlRequest *pOutSetup) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    OutBuffer pBuffer(reinterpret_cast<char *>(pOutSetup), sizeof(UsbCtrlRequest));

    return m_Handle->GetSetupPacket(pBuffer);
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterface::AppendConfigurationData(UsbDeviceSpeed usbDeviceSpeed, void *pData, uint32_t size) NN_NOEXCEPT
{
    InBuffer data(reinterpret_cast<const char *>(pData), size);
    return m_Handle->AppendConfigurationData(m_IfNum, usbDeviceSpeed, data);
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterface::AddEndpoint(DsEndpoint                      *pEndpoint,
                                uint8_t                          bEndpointAddress,
                                SharedPointer<ds::IDsEndpoint>  *pHandle) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    if (m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    NN_USB_ABORT_IF_NULL(m_Handle);

    Result result = m_Handle->RegisterEndpoint(bEndpointAddress, pHandle);

    if (result.IsFailure())
    {
        return result;
    }

    int epIndex = detail::GetEndpointIndex(bEndpointAddress);
    m_pEndpoints[epIndex] = pEndpoint;

    return ResultSuccess();
}

Result DsInterface::DeleteEndpoint(uint8_t epAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if (!m_IsInitialized) return ResultNotInitialized();

    int epIndex = detail::GetEndpointIndex(epAddress);

    if (m_IsEnabled || !m_pEndpoints[epIndex])
    {
        return ResultOperationDenied();
    }

    m_pEndpoints[epIndex] = nullptr;

    return ResultSuccess();
}

Result DsInterface::Enable() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();
    if (m_IsEnabled) return ResultSuccess();

    NN_USB_ABORT_IF_NULL(m_Handle);
    Result result = m_Handle->Enable();
    if (result.IsSuccess())
    {
        m_IsEnabled = true;
    }

    return result;
}

Result DsInterface::Disable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_USB_ABORT_IF_NULL(m_Handle);

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();
    if (!m_IsEnabled) return ResultSuccess();

    Result result = m_Handle->Disable();
    if (result.IsSuccess())
    {
        m_IsEnabled = false;
    }

    return result;
}

Result DsInterface::CtrlIn(uint32_t *pOutBytesTransferred,
                           void     *buffer,
                           uint32_t  bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();
    if (!m_IsEnabled) return ResultOperationDenied();

    if (!NN_USB_IS_DMA_ALIGNED(buffer))
    {
        return ResultAlignmentError();
    }

    if (bytes != 0)
    {
        FlushDataCache(buffer, bytes);
    }

    NN_USB_ABORT_IF_NULL(m_Handle);
    uint32_t urbId;
    Result result = m_Handle->CtrlInAsync(&urbId,
                                   reinterpret_cast<uint64_t>(buffer),
                                   bytes);
    if (result.IsFailure())
    {
        return result;
    }

    nn::os::WaitSystemEvent(&m_CtrlInCompletionEvent);
    nn::os::ClearSystemEvent(&m_CtrlInCompletionEvent);

    m_Handle->GetCtrlInUrbReport(&m_Report);
    if (m_Report.count == 1 && m_Report.report[0].id == urbId)
    {
        switch (m_Report.report[0].status)
        {
        case UrbStatus_Cancelled:
            result = ResultInterrupted();
            break;

        case UrbStatus_Failed:
            result = ResultTransactionError();
            break;

        case UrbStatus_Finished:
            result = ResultSuccess();
            break;

        default:
            result = ResultInternalStateError();
            break;
        }

        if (pOutBytesTransferred != nullptr)
        {
            *pOutBytesTransferred = m_Report.report[0].transferredSize;
        }
    }
    else
    {
        result = ResultInternalStateError();
    }

    return result;
}


Result DsInterface::CtrlOut(uint32_t *pOutBytesTransferred,
                            void     *pOutBuffer,
                            uint32_t  bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Result result;
    uint32_t urbId;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();
    if (!m_IsEnabled) return ResultOperationDenied();

    if (!NN_USB_IS_DMA_ALIGNED(pOutBuffer))
    {
        return ResultAlignmentError();
    }

    if (bytes != 0)
    {
        InvalidateDataCache(pOutBuffer, bytes);
    }

    NN_USB_ABORT_IF_NULL(m_Handle);
    result = m_Handle->CtrlOutAsync(&urbId,
                                    reinterpret_cast<uint64_t>(pOutBuffer),
                                    bytes);
    if (result.IsFailure())
    {
        return result;
    }

    nn::os::WaitSystemEvent(&m_CtrlOutCompletionEvent);
    nn::os::ClearSystemEvent(&m_CtrlOutCompletionEvent);

    m_Handle->GetCtrlOutUrbReport(&m_Report);
    if (m_Report.count == 1 && m_Report.report[0].id == urbId)
    {
        switch (m_Report.report[0].status)
        {
        case UrbStatus_Cancelled:
            result = ResultInterrupted();
            break;

        case UrbStatus_Failed:
            result = ResultTransactionError();
            break;

        case UrbStatus_Finished:
            result = ResultSuccess();
            break;

        default:
            result = ResultInternalStateError();
            break;
        }

        if (pOutBytesTransferred != nullptr)
        {
            *pOutBytesTransferred = m_Report.report[0].transferredSize;
        }
    }
    else
    {
        result = ResultInternalStateError();
    }

    if (bytes != 0)
    {
        InvalidateDataCache(pOutBuffer, bytes);
    }

    return result;
}

Result DsInterface::CtrlRead(uint32_t *pOutBytesTransferred,
                             void     *pOutBuffer,
                             uint32_t  bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();

    // data stage
    Result result = CtrlOut(pOutBytesTransferred, pOutBuffer, bytes);

    // status stage
    if (result.IsSuccess())
    {
        result = CtrlIn(nullptr, nullptr, 0);
    }

    // stall upon error
    if (result.IsFailure())
    {
        result = CtrlStall();
    }

    return result;
}

Result DsInterface::CtrlWrite(uint32_t *pOutBytesTransferred,
                              void     *buffer,
                              uint32_t  bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();

    // data stage
    Result result = CtrlIn(pOutBytesTransferred, buffer, bytes);

    // status stage
    if (result.IsSuccess())
    {
        result = CtrlOut(nullptr, nullptr, 0);
    }

    // stall upon error
    if (result.IsFailure())
    {
        result = CtrlStall();
    }

    return result;
}

Result DsInterface::CtrlDone() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();

    // status stage
    Result result = CtrlIn(nullptr, nullptr, 0);

    // stall upon error
    if (result.IsFailure())
    {
        result = CtrlStall();
    }

    return result;
}

Result DsInterface::CtrlStall() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if (!m_IsInitialized) return ResultNotInitialized();
    if (!m_IsEnabled) return ResultOperationDenied();

    NN_USB_ABORT_IF_NULL(m_Handle);
    return m_Handle->CtrlStall();
}

//--------------------------------------------------------------------------
//  DsEndpoint class implementation
//--------------------------------------------------------------------------

#undef NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsEndpoint"

Result DsEndpoint::Initialize(DsInterface *pInterface, uint8_t bEndpointAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;
    NativeHandle sfEventHandle;

    NN_ABORT_UNLESS_NOT_NULL(pInterface);
    if(m_IsInitialized) return ResultAlreadyInitialized();

    m_pInterface = pInterface;

    result = m_pInterface->AddEndpoint(this, bEndpointAddress, &m_Handle);
    if (result.IsFailure())
    {
        return result;
    }

    m_Address = bEndpointAddress;

    result = m_Handle->GetCompletionEvent(&sfEventHandle);
    if (result.IsFailure())
    {
        result = m_pInterface->DeleteEndpoint(m_Address);
        m_Handle = nullptr;

        return result;
    }

    pInterface->m_RefCount++;
    pInterface->m_pClient->m_RefCount++;

    nn::os::AttachReadableHandleToSystemEvent(
        &m_CompletionEvent,
        sfEventHandle.GetOsHandle(),
        sfEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );

    sfEventHandle.Detach();
    m_IsInitialized = true;

    return result;
}

Result DsEndpoint::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;

    int refCount = 0;
    int lastRefCount = 0;
    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);

    if(!m_IsInitialized) return ResultNotInitialized();
    m_IsInitialized = false;

    // Cancel all transactions.
    m_Handle->Cancel();

    while((refCount = m_RefCount) > 1)
    {
        if(lastRefCount != refCount)
        {
            NN_SDK_LOG("[usb:ds] DsEndpoint::Finalize() waiting on refcount %d...\n",
                       refCount - 1);
        }
        lastRefCount = refCount;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(25));
    }
    if(lastRefCount > 0)
    {
        NN_SDK_LOG("[usb:ds] DsEndpoint::Finalize() wait done.\n");
    }

    nn::os::DestroySystemEvent(&m_CompletionEvent);
    m_pInterface->m_RefCount--;
    m_pInterface->m_pClient->m_RefCount--;

    Result result = m_pInterface->DeleteEndpoint(m_Address);
    if (result.IsSuccess())
    {
        m_pInterface = nullptr;
        m_Handle     = nullptr;
    }

    return result;
}

bool DsEndpoint::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}


nn::os::SystemEventType* DsEndpoint::GetCompletionEvent() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    return (m_IsInitialized) ? &m_CompletionEvent : nullptr;
}

Result DsEndpoint::PostBufferAsync(uint32_t  *pOutUrbId,
                                   void      *buffer,
                                   uint32_t   bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    if (!NN_USB_IS_DMA_ALIGNED(buffer))
    {
        return ResultAlignmentError();
    }

    NN_USB_ABORT_IF_NULL(m_Handle);

    uint32_t urbId = 0;
    Result result = m_Handle->PostBufferAsync(&urbId,
                                       reinterpret_cast<uint64_t>(buffer),
                                       bytes);
    if (result.IsSuccess())
    {
        *pOutUrbId = urbId;
    }

    return result;
}

Result DsEndpoint::PostBuffer(uint32_t *pOutBytesTransferred,
                              void     *buffer,
                              uint32_t  bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Result result;
    uint32_t urbId;
    UrbReport report;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    // submit async request
    NN_USB_RETURN_UPON_ERROR(PostBufferAsync(&urbId, buffer, bytes));

    // wait for completion
    nn::os::WaitSystemEvent(&m_CompletionEvent);
    nn::os::ClearSystemEvent(&m_CompletionEvent);

    // query for status
    NN_USB_ABORT_IF_NULL(m_Handle);
    m_Handle->GetUrbReport(&report);
    if (report.count == 1 && report.report[0].id == urbId)
    {
        switch (report.report[0].status)
        {
        case UrbStatus_Cancelled:
            result = ResultInterrupted();
            break;

        case UrbStatus_Failed:
            result = ResultTransactionError();
            break;

        case UrbStatus_Finished:
            result = ResultSuccess();
            break;

        default:
            NN_USB_LOG_WARN("urb status = %d\n");
            result = ResultInternalStateError();
            break;
        }

        if (pOutBytesTransferred != nullptr)
        {
            *pOutBytesTransferred = report.report[0].transferredSize;
        }
    }
    else
    {
        NN_USB_LOG_WARN("===== report count %d\n", report.count);
        NN_USB_LOG_WARN("===== expecting id = %08x\n", urbId);

        // Bounds check for really really bad data
        if(report.count > DsLimitRingSize)
        {
            report.count = DsLimitRingSize;
        }

        for (unsigned i = 0; i < report.count; i++)
        {
            NN_USB_LOG_WARN("report[%d].id              = %08x\n", i, report.report[i].id);
            NN_USB_LOG_WARN("report[%d].requestedSize   = %d\n", i, report.report[i].requestedSize);
            NN_USB_LOG_WARN("report[%d].transferredSize = %d\n", i, report.report[i].transferredSize);
            NN_USB_LOG_WARN("report[%d].status          = %d\n", i, report.report[i].status);
        }
        result = ResultInternalStateError();
    }

    return result;
}

Result DsEndpoint::GetUrbReport(UrbReport *pOutReport) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    return m_Handle->GetUrbReport(pOutReport);
}

Result DsEndpoint::Cancel() NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    return m_Handle->Cancel();
}

Result DsEndpoint::SetZeroLengthTransfer(bool isZeroLengthTerminate) NN_NOEXCEPT
{
    NN_USB_TRACE;

    detail::ScopedRefCount<::std::atomic<int>> scopedRefCount(m_RefCount);
    if(!m_IsInitialized) return ResultNotInitialized();

    NN_USB_ABORT_IF_NULL(m_Handle);
    return m_Handle->SetZlt(isZeroLengthTerminate);
}

} // end of namespace usb
} // end of namespace nn
