﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_PmPort.h"
#include "usb_Pm.h"
#include "../detail/usb_Util.h"

namespace nn {
namespace usb {
namespace pm {

Result PmPort::Initialize(Pm *pPm, uint32_t portNumber, detail::UsbComplex *pComplex) NN_NOEXCEPT
{
    m_pPm        = pPm;
    m_pComplex   = pComplex;
    m_PortNumber = portNumber;
    m_IsEnabled  = false;

    auto& port = m_pComplex->GetPlatform()->GetConfig().port[portNumber];

    m_Capability = port.capability;
    m_HsLane     = port.hsLane;
    m_SsLane     = port.ssLane;

    if (IsDualRole())
    {
        // PD will tell us the role later
        m_DataRole = DataRole_Unknown;
    }
    else if (m_Capability & UsbCapability_Host)
    {
        m_DataRole = DataRole_Host;
    }
    else
    {
        m_DataRole = DataRole_Device;
    }

    return ResultSuccess();
}

Result PmPort::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}

Result PmPort::Enable() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    switch (m_DataRole)
    {
    case DataRole_Unknown:
        // ignore
        break;

    case DataRole_Host:
        result = EnableAsHost();
        break;

    case DataRole_Device:
        result = EnableAsDevice();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

Result PmPort::EnableAsHost() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ((m_Capability & UsbCapability_Host) == 0)
    {
        return ResultOperationDenied();
    }

    if (m_IsEnabled)
    {
        NN_USB_LOG_WARN("Bogus request: port %d is already enabled!\n",
                        m_PortNumber);
        return result;
    }

    result = m_pComplex->CreatePort(
        m_HsLane, m_SsLane, m_Capability & ~UsbCapability_Device
    );

    if (result.IsSuccess())
    {
        m_DataRole = DataRole_Host;
        m_IsEnabled = true;
        m_DfpCount++;

        NN_USB_LOG_INFO("Enable port %d as DFP (Host) #%d\n",
                        m_PortNumber, m_DfpCount);
    }

    return result;
}

Result PmPort::EnableAsDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ((m_Capability & UsbCapability_Device) == 0)
    {
        return ResultOperationDenied();
    }

    if (m_IsEnabled)
    {
        NN_USB_LOG_WARN("Bogus request: port %d is already enabled!\n",
                        m_PortNumber);
        return result;
    }

    result = m_pComplex->CreatePort(
        m_HsLane, m_SsLane, m_Capability & ~UsbCapability_Host
    );

    if (result.IsSuccess())
    {
        m_DataRole = DataRole_Device;
        m_IsEnabled = true;
        m_UfpCount++;

        NN_USB_LOG_INFO("Enable port %d as UFP (Device) #%d\n",
                        m_PortNumber, m_UfpCount);
    }

    return result;
}

Result PmPort::Disable() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (!m_IsEnabled)
    {
        NN_USB_LOG_WARN("Bogus request: port %d is already disabled!\n",
                        m_PortNumber);
        return result;
    }

    switch (m_DataRole)
    {
    case DataRole_Host:
        result = m_pComplex->DestroyPort(
            m_HsLane, m_SsLane, m_Capability & ~UsbCapability_Device
        );
        break;

    case DataRole_Device:
        result = m_pComplex->DestroyPort(
            m_HsLane, m_SsLane, m_Capability & ~UsbCapability_Host
        );
        break;

    case DataRole_Unknown:
        // Fallthrough
        // The role cannot be Unknown if the port is already enabled
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    if (result.IsSuccess())
    {
        m_IsEnabled = false;
        m_DisabledCount++;

        NN_USB_LOG_INFO("Disable port %d #%d\n", m_PortNumber, m_DisabledCount);
    }

    return result;
}

void PmPort::ResetRole() NN_NOEXCEPT
{
    if (IsDualRole())
    {
        m_DataRole = DataRole_Unknown;
    }
}

bool PmPort::IsDualRole() const NN_NOEXCEPT
{
    return (m_Capability & UsbCapability_Host) && (m_Capability & UsbCapability_Device);
}

} // end of namespace pm
} // end of namespace usb
} // end of namespace nn
