﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/usb/pd/usb_PdCradle.h>

#include "detail/usb_IPdCradleManager.h"
#include "detail/usb_PdServiceName.h"

#include "usb_PdCreateManager.h"

#define NN_UPD_ACCESS_BY_HIPC

namespace nn { namespace usb { namespace pd {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::usb::pd::detail::IPdCradleManager> g_PdCradleManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeCountMutex(false);

}

void InitializeCradle() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {

        NN_SDK_ASSERT(!g_PdCradleManager);

        // マクロによって、HIPC/DFC を切り替える
        #ifdef NN_UPD_ACCESS_BY_HIPC
            g_PdCradleManager = nn::usb::pd::CreatePdCradleManagerByHipc();
        #else
            g_PdCradleManager = nn::usb::pd::CreatePdCradleManagerByDfc();
        #endif
    }

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void FinalizeCradle() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT(g_PdCradleManager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_PdCradleManager = nullptr;
    }
}


// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenCradleSession( CradleSession* pOutSession ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_PdCradleManager);
    nn::sf::SharedPointer<nn::usb::pd::detail::IPdCradleSession> session;
    auto result = g_PdCradleManager->OpenCradleSession(&session);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseCradleSession( CradleSession* pSession ) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle));
}

Result SetCradleVdo( CradleSession* pSession, Vdo* pVdo, CradleVdmCommand command ) NN_NOEXCEPT
{
    Vdo vdo;
    vdo.Clear();
    if ( pVdo )
    {
        vdo = *pVdo;
    }
    auto result = static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle)->SetCradleVdo(*reinterpret_cast<std::uint32_t*>(pVdo), command);
    return result;
}

Result GetCradleVdo( Vdo* pVdo, CradleSession* pSession, CradleVdmCommand command ) NN_NOEXCEPT
{
    Vdo vdo;
    auto result = static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle)->GetCradleVdo(reinterpret_cast<std::uint32_t*>(&vdo), command);
    if ( pVdo )
    {
        *pVdo = vdo;
        if ( result.IsFailure() )
        {
            pVdo->Clear();
        }
    }
    return result;
}

Result ResetCradleUsbHub( CradleSession* pSession ) NN_NOEXCEPT
{
    auto result = static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle)->ResetCradleUsbHub();
    return result;
}

Result EnableCradleRecovery( bool* pIsSuspended, CradleSession* pSession ) NN_NOEXCEPT
{
    bool isSuspended;
    auto result = static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle)->EnableCradleRecovery(&isSuspended);
    if ( pIsSuspended )
    {
        *pIsSuspended = isSuspended;
        if ( result.IsFailure() )
        {
            *pIsSuspended = false;
        }
    }
    return result;
}

Result DisableCradleRecovery( bool* pIsSuspended, CradleSession* pSession ) NN_NOEXCEPT
{
    bool isSuspended;
    auto result = static_cast<nn::usb::pd::detail::IPdCradleSession*>(pSession->_handle)->DisableCradleRecovery(&isSuspended);
    if ( pIsSuspended )
    {
        *pIsSuspended = isSuspended;
        if ( result.IsFailure() )
        {
            *pIsSuspended = false;
        }
    }
    return result;
}

}}}
