﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/usb/pd/usb_Pd.h>

#include "detail/usb_IPdManager.h"
#include "detail/usb_PdServiceName.h"

#include "usb_PdCreateManager.h"

#define NN_UPD_ACCESS_BY_HIPC

namespace nn { namespace usb { namespace pd {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::usb::pd::detail::IPdManager> g_PdManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeCountMutex(false);

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {

        NN_SDK_ASSERT(!g_PdManager);

        // マクロによって、HIPC/DFC を切り替える
        #ifdef NN_UPD_ACCESS_BY_HIPC
            g_PdManager = nn::usb::pd::CreatePdManagerByHipc();
        #else
            g_PdManager = nn::usb::pd::CreatePdManagerByDfc();
        #endif
    }

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT(g_PdManager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_PdManager = nullptr;
    }
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession( Session* pOutSession ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_PdManager);
    nn::sf::SharedPointer<nn::usb::pd::detail::IPdSession> session;
    auto result = g_PdManager->OpenSession(&session);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseSession( Session* pSession ) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle));
}

// Session がもつ各機能のラップ版
Result BindNoticeEvent( nn::os::SystemEventType* pEvent, Session* pSession ) NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;

    NN_RESULT_DO(static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->BindNoticeEvent(&sfHandle));

    nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                              sfHandle.GetOsHandle(),
                                              sfHandle.IsManaged(),
                                              nn::os::EventClearMode_ManualClear);
    sfHandle.Detach();

    // UnbindSleepButtonEvent で SystemEvent を破棄できるように SessionHandle のメンバ変数にポインタを保存
    pSession->_pEvent = pEvent;

    return nn::ResultSuccess();
}

void UnbindNoticeEvent( Session* pSession ) NN_NOEXCEPT
{
    // SessionHandle が持つpEvent が null の場合は Bind されてないのに呼び出されているので ASSERT
    NN_SDK_ASSERT(pSession->_pEvent != nullptr);

    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->UnbindNoticeEvent();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Bind されていたイベントを破棄する。
    nn::os::DestroySystemEvent(pSession->_pEvent);

    pSession->_pEvent = nullptr;
}

Result GetStatus( Status* pStatus, Session* pSession ) NN_NOEXCEPT
{
    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->GetStatus(pStatus);
    return result;
}

Result GetNotice( Notice* pNotice, Session* pSession ) NN_NOEXCEPT
{
    uint32_t tmp;
    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->GetNotice(&tmp);
    *pNotice = *reinterpret_cast<Notice*>(&tmp);
    return result;
}

Result EnablePowerRequestNotice( Session* pSession ) NN_NOEXCEPT
{
    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->EnablePowerRequestNotice();
    return result;
}

Result DisablePowerRequestNotice( Session* pSession ) NN_NOEXCEPT
{
    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->DisablePowerRequestNotice();
    return result;
}

Result ReplyPowerRequest( Session* pSession, bool isSuccess ) NN_NOEXCEPT
{
    auto result = static_cast<nn::usb::pd::detail::IPdSession*>(pSession->_handle)->ReplyPowerRequest(isSuccess);
    return result;
}

}}}
