﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/usb/pd/driver/usb_Pd.h>

#include "../detail/usb_IPdManager.h"
#include "../detail/usb_IPdSession.h"
#include "../detail/usb_PdServiceName.h"

#include "usb_PdManagerImpl.h"

namespace nn { namespace usb { namespace pd { namespace server {


// IPdSession 実装クラス
class PdManagerImpl::PdSessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、セッションに必要な情報を渡す
    PdSessionImpl(PdManagerImpl* pParent, nn::usb::pd::driver::Session internalSession) NN_NOEXCEPT
        : m_Parent(pParent, true)
        , m_InternalSession(internalSession)
    {}

    // Session のデストラクタで Session のクローズを行う
    ~PdSessionImpl()
    {
        nn::usb::pd::driver::CloseSession(&m_InternalSession);
    }

    // USB::PD ドライバライブラリの API のうち SystemEvent を使わないもの
    nn::Result BindNoticeEvent(nn::sf::Out<nn::sf::NativeHandle> outValue) NN_NOEXCEPT;
    nn::Result UnbindNoticeEvent() NN_NOEXCEPT;
    nn::Result GetStatus(nn::sf::Out<Status> outValue) NN_NOEXCEPT;
    nn::Result GetNotice(nn::sf::Out<uint32_t> outValue) NN_NOEXCEPT;
    Result EnablePowerRequestNotice() NN_NOEXCEPT;
    Result DisablePowerRequestNotice() NN_NOEXCEPT;
    Result ReplyPowerRequest(bool isSuccess) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<PdManagerImpl> m_Parent;

    // USB::PD ライブラリが持つ Session 構造体のメンバ変数
    nn::usb::pd::driver::Session m_InternalSession;

    nn::os::SystemEvent m_SystemEvent;
};

// ここから各 USB::PD API の内部実装
nn::Result PdManagerImpl::PdSessionImpl::BindNoticeEvent(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    Result result;

    result = nn::usb::pd::driver::BindNoticeEvent( m_SystemEvent.GetBase(), m_InternalSession );
    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent( m_SystemEvent.GetBase() );

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *handle = nn::sf::NativeHandle( internalHandle, false );

    return result;
}

nn::Result PdManagerImpl::PdSessionImpl::UnbindNoticeEvent() NN_NOEXCEPT
{
    return nn::usb::pd::driver::UnbindNoticeEvent( m_InternalSession );
}

nn::Result PdManagerImpl::PdSessionImpl::GetStatus(nn::sf::Out<Status> outValue) NN_NOEXCEPT
{
    Result result;
    Status tmp;
    result = nn::usb::pd::driver::GetStatus( &tmp, m_InternalSession );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    *outValue = tmp;

    return result;
}

nn::Result PdManagerImpl::PdSessionImpl::GetNotice(nn::sf::Out<uint32_t> outValue) NN_NOEXCEPT
{
    Result result;
    Notice tmp;
    result = nn::usb::pd::driver::GetNotice( &tmp, m_InternalSession );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    *outValue = tmp.storage;

    return result;
}

nn::Result PdManagerImpl::PdSessionImpl::EnablePowerRequestNotice() NN_NOEXCEPT
{
    Result result;
    result = nn::usb::pd::driver::EnablePowerRequestNotice( m_InternalSession );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

nn::Result PdManagerImpl::PdSessionImpl::DisablePowerRequestNotice() NN_NOEXCEPT
{
    Result result;
    result = nn::usb::pd::driver::DisablePowerRequestNotice( m_InternalSession );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

nn::Result PdManagerImpl::PdSessionImpl::ReplyPowerRequest(bool isSuccess) NN_NOEXCEPT
{
    Result result;
    result = nn::usb::pd::driver::ReplyPowerRequest( m_InternalSession, isSuccess );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

// マネージャーの実装
PdManagerImpl::PdManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

PdManagerImpl::~PdManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result PdManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::usb::pd::detail::IPdSession>> outSession) NN_NOEXCEPT
{
    // USB::PD ドライバライブラリに渡す Session 構造体
    nn::usb::pd::driver::Session internalSession;

    nn::usb::pd::driver::OpenSession(&internalSession);

    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IPdSession, PdManagerImpl::PdSessionImpl>(&m_Allocator, this, internalSession);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}}
