﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   USB::PD ドライバの API インタフェース部分。
 */


#include <nn/usb/pd/driver/usb_Pd.h>
#include <nn/usb/pd/usb_PdCradleLib.h>

#include "../detail/usb_IPdSession.h"
#include "../detail/usb_IPdCradleSession.h"

#include "detail/usb_PdDriver.h"
#include "detail/usb_PdSession.h"

namespace {

nn::usb::pd::driver::detail::Driver s_PdDriver;
nn::usb::pd::driver::detail::Session s_PdSession;

}

namespace nn {
namespace usb {
namespace pd {
namespace driver {

const VdmCommand g_CradleVdmCommandToVdmCommand[CradleVdmCommand_Num] =
{
    VdmCommand_Led,
    VdmCommand_Dp2HdmiFwVersion,
    VdmCommand_PdcHFwVersion,
    VdmCommand_PdcAFwVersion,
    VdmCommand_DeviceState,
    VdmCommand_McuFwVersion,
    VdmCommand_McuFwUpdate,
    VdmCommand_McuSleepEnable,
    VdmCommand_UsbHubReset,
    VdmCommand_UsbHubVBusDetect,
};

// usb プロセス向け

void Initialize() NN_NOEXCEPT
{
    s_PdDriver.Initialize( nn::i2c::SpeedMode(0) );
    s_PdSession.Initialize();
    s_PdDriver.AttachSessionObject( &s_PdSession );
}

void Finalize() NN_NOEXCEPT
{
    s_PdDriver.DetachSessionObject();
    s_PdSession.Finalize();
    s_PdDriver.Finalize();
}

Result Sleep() NN_NOEXCEPT
{
    Result result;
    result = s_PdDriver.SleepRequest();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result MinimumAwake() NN_NOEXCEPT
{
    Result result;
    result = s_PdDriver.MinimumAwakeRequest();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result FullAwake() NN_NOEXCEPT
{
    Result result;
    result = s_PdDriver.FullAwakeRequest();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

// IPC 向け

void OpenSession( Session* pOutSession ) NN_NOEXCEPT
{
    s_PdSession.OpenSession(pOutSession);
}

void CloseSession( Session* pSession ) NN_NOEXCEPT
{
    s_PdSession.CloseSession(*pSession);
}

void OpenCradleSession( CradleSession* pOutSession ) NN_NOEXCEPT
{
    s_PdSession.OpenCradleSession(pOutSession);
}

void CloseCradleSession( CradleSession* pSession ) NN_NOEXCEPT
{
    s_PdSession.CloseCradleSession(*pSession);
}

Result GetHostPdcFirmwareType( uint16_t* pOutValue, const CradleSession& session ) NN_NOEXCEPT
{
    *pOutValue = s_PdDriver.GetHostPdcFirmwareType().storage;
    USBPD_DBG_LOG("GetHostPdcFirmwareType = %04x\n", *pOutValue);
    return ResultSuccess();
}

Result GetHostPdcFirmwareRevision( uint16_t* pOutValue, const CradleSession& session ) NN_NOEXCEPT
{
    *pOutValue = s_PdDriver.GetHostPdcFirmwareRevision();
    USBPD_DBG_LOG("GetHostPdcFirmwareRevision = %04x\n", *pOutValue);
    return ResultSuccess();
}

Result GetHostPdcManufactureId( uint16_t* pOutValue, const CradleSession& session ) NN_NOEXCEPT
{
    *pOutValue = s_PdDriver.GetHostPdcManufactureId();
    USBPD_DBG_LOG("GetHostPdcManufactureId = %04x\n", *pOutValue);
    return ResultSuccess();
}

Result GetHostPdcDeviceId( uint16_t* pOutValue, const CradleSession& session ) NN_NOEXCEPT
{
    *pOutValue = s_PdDriver.GetHostPdcDeviceId();
    USBPD_DBG_LOG("GetHostPdcDeviceId = %04x\n", *pOutValue);
    return ResultSuccess();
}

Result BindNoticeEvent( nn::os::SystemEventType* pEvent, const Session& session ) NN_NOEXCEPT
{
    return s_PdSession.BindNoticeEvent(pEvent, session);
}

Result UnbindNoticeEvent( const Session& session ) NN_NOEXCEPT
{
    return s_PdSession.UnbindNoticeEvent(session);
}

Result GetStatus( Status* pOutValue, const Session& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsOpen(session._sessionId));

    Result result;
    result = s_PdDriver.GetStatus( reinterpret_cast<Status*>(pOutValue), session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result GetNotice( Notice* pOutValue, const Session& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsOpen(session._sessionId));

    Result result;
    result = s_PdDriver.GetNotice( reinterpret_cast<Notice*>(pOutValue), session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result EnablePowerRequestNotice( const Session& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsOpen(session._sessionId));

    Result result;
    result = s_PdDriver.EnablePowerRequestNotice( session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result DisablePowerRequestNotice( const Session& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsOpen(session._sessionId));

    Result result;
    result = s_PdDriver.DisablePowerRequestNotice( session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result ReplyPowerRequest( const Session& session, bool isSuccess ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsOpen(session._sessionId));

    Result result;
    result = s_PdDriver.ReplyPowerRequest( session._sessionId, isSuccess );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

static Result CheckCradle() NN_NOEXCEPT
{
    Result result;

    if ( !s_PdDriver.IsActive() )
    {
        USBPD_WARNING("Cradle is inactive!\n");
        return ResultInactive();
    }
    Status status;
    result = s_PdDriver.GetStatusCore( &status );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( !s_PdDriver.IsCradleFamily( status.GetDeviceType() ) )
    {
        USBPD_WARNING("This device isn't cradle family!\n");
        return ResultInvalidDevice();
    }
    if ( s_PdDriver.IsSleeping() )
    {
        // 上位レイヤは既にスリープ済みで呼ばれることはないはず
        USBPD_WARNING("Driver is sleeping!\n");
        return ResultSleeping();
    }

    return ResultSuccess();
}

static Result AccessCradleVdo( uint32_t* pValue, const CradleSession& session, uint32_t command, VdmSet rw ) NN_NOEXCEPT
{
    Result result;

    NN_ABORT_UNLESS(s_PdSession.IsCradleOpen(session._sessionId));
    result = CheckCradle();
    if ( result.IsFailure() )
    {
        return result;
    }
    if ( command >= CradleVdmCommand_Num )
    {
        return ResultInvalidCommand();
    }

    VdmCommand vdmCommand = g_CradleVdmCommandToVdmCommand[command];
    Vdo vdo;
    vdo.Clear();
    if ( rw == VdmSet_WriteRequest )
    {
        result = s_PdDriver.SendAndReceiveNxVdmWithRequest( &vdo, sizeof(vdo), pValue, VdmSet_WriteRequest, vdmCommand );
//        USBPD_DBG_LOG("SendCradleVdo(%08x, %08x)\n", vdo, vdmCommand);
    }
    else
    {
        result = s_PdDriver.SendAndReceiveNxVdmWithRequest( &vdo, sizeof(vdo), nullptr, VdmSet_ReadRequest, vdmCommand );
        if ( pValue )
        {
            *pValue = vdo.storage;
        }
    }
    if ( result <= ResultVdmError() )
    {
        USBPD_WARNING("Cradle reply is given up!\n");
        return ResultTimeout();
    }
    // UsbHubVBusDetect 成功判定
    if ( vdmCommand == VdmCommand_UsbHubVBusDetect )
    {
        VdmCommonVdo1 cradleUsbHubState;
        cradleUsbHubState.storage = vdo.storage;
        if ( cradleUsbHubState.Get<VdmCommonVdo1::VBusDetectError>() )
        {
            USBPD_WARNING("Cradle USB HUB VBUS undetected reply!\n");
            return ResultErrorReply();
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result SetCradleVdo( const CradleSession& session, uint32_t value, uint32_t command ) NN_NOEXCEPT
{
    Result result;
    result = AccessCradleVdo( &value, session, command, VdmSet_WriteRequest );
    USBPD_DBG_LOG("SetCradleVdo[%d]( %08x, %d )\n", session._sessionId, value, command);

    return result;
}

Result GetCradleVdo( uint32_t* pOutValue, const CradleSession& session, uint32_t command ) NN_NOEXCEPT
{
    Result result;
    result = AccessCradleVdo( pOutValue, session, command, VdmSet_ReadRequest );
    USBPD_DBG_LOG("GetCradleVdo[%d]( %08x, %d )\n", session._sessionId, *pOutValue, command);

    return result;
}

Result ResetCradleUsbHub( const CradleSession& session ) NN_NOEXCEPT
{
    Result result;

    NN_ABORT_UNLESS(s_PdSession.IsCradleOpen(session._sessionId));
    result = CheckCradle();
    if ( result.IsFailure() )
    {
        return result;
    }

    result = s_PdDriver.ResetCradleUsbHub();
    if ( result <= ResultVdmError() )
    {
        USBPD_WARNING("Cradle reply is given up!\n");
        return ResultTimeout();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result EnableCradleRecovery( bool* pIsSuspended, const CradleSession& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsCradleOpen(session._sessionId));

    Result result;
    result = s_PdDriver.EnableCradleRecovery( pIsSuspended, session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result DisableCradleRecovery( bool* pIsSuspended, const CradleSession& session ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(s_PdSession.IsCradleOpen(session._sessionId));

    Result result;
    result = s_PdDriver.DisableCradleRecovery( pIsSuspended, session._sessionId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

} // driver
} // pd
} // usb
} // nn
