﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/dd.h>
#include <nn/util/util_BitPack.h>
#include <nn/usb/pd/usb_PdPdoRdo.h>

namespace nn {
namespace usb {
namespace pd {
namespace driver {
namespace detail {

// I2C
const uint8_t I2cBusIdx = 0;
const uint8_t I2cAddress = 0x18;
const uint8_t I2cBlockSwitchSize = 3;

// GPIO
const uint8_t GpioAlert = 10 * 8 + 4; // GPIO3_PK4
const uint8_t GpioHpd   = 28 * 8 + 1; // GPIO3_PCC1
//  FET1  -> GPIO3_PT1
//  FET2  -> GPIO3_PK5

// Firmware Revision
const uint16_t HostPdcFwRevisionDp1 = 0x0D7C;
const uint16_t HostPdcFwRevisionEp2_2 = 0x0B5B;
const uint16_t HostPdcFwRevisionEp2 = 0x0945;
const uint16_t HostPdcFwRevisionEp1 = 0x0644;
const uint16_t HostPdcFwRevisionFp2 = 0x0529;

// Alert
const uint8_t AlertNum = 15;

/**
 * @brief L1 コマンド
 */
enum L1Command
{
    L1Command_AlertStatus                                       = 0x02, // RO
    L1Command_Status1                                           = 0x03, // RO
    L1Command_Status2                                           = 0x04, // RO
    L1Command_L2Command                                         = 0x05, // WO
    L1Command_ControllerConfiguration1                          = 0x06, // RW
    L1Command_DeviceCapability                                  = 0x07, // RO
    L1Command_PdosSrcConsumerSnkProvider                        = 0x08, // RO
    L1Command_ControllerConfiguration2                          = 0x17, // RW
    L1Command_DisplayPortStatus                                 = 0x18, // RO
    L1Command_DisplayPortAlertEnable                            = 0x19, // RW
    L1Command_VendorConfiguration                               = 0x1A, // RW
    L1Command_AutoNgtSnkInfoNonBattery                          = 0x20, // RW
    L1Command_AutoNgtSnkInfoBattery                             = 0x23, // RW
    L1Command_SystemConfiguration1                              = 0x26, // RW
    L1Command_SystemConfiguration2                              = 0x27, // RW
    L1Command_CurrentPdo                                        = 0x28, // RO
    L1Command_CurrentRdo                                        = 0x2B, // RO
    L1Command_AlertEnable                                       = 0x2E, // RW
    L1Command_SystemConfiguration3                              = 0x2F, // RW
    L1Command_SetRdo                                            = 0x30, // RW
    L1Command_PdosSnkConsumer                                   = 0x33, // RW
    L1Command_PdoSrcProvider                                    = 0x3C, // RW
    L1Command_FirmwareType                                      = 0x4B, // RO
    L1Command_FirmwareRevision                                  = 0x4C, // RO
    L1Command_ManufactureId                                     = 0x4D, // RO
    L1Command_DeviceId                                          = 0x4E, // RO
    L1Command_RevisionId                                        = 0x4F, // RO
    L1Command_IncomingVdm                                       = 0x50, // RO
    L1Command_OutgoingVdm                                       = 0x60  // WO
};

enum L1CommandSize
{
    L1CommandSize_AlertStatus                                   = 2,
    L1CommandSize_Status1                                       = 2,
    L1CommandSize_Status2                                       = 2,
    L1CommandSize_L2Command                                     = 2,
    L1CommandSize_ControllerConfiguration1                      = 2,
    L1CommandSize_DeviceCapability                              = 2,
    L1CommandSize_PdosSrcConsumerSnkProvider                    = 28,
    L1CommandSize_ControllerConfiguration2                      = 2,
    L1CommandSize_DisplayPortStatus                             = 2,
    L1CommandSize_DisplayPortAlertEnable                        = 2,
    L1CommandSize_VendorConfiguration                           = 2,
    L1CommandSize_AutoNgtSnkInfoNonBattery                      = 4,
    L1CommandSize_AutoNgtSnkInfoBattery                         = 4,
    L1CommandSize_SystemConfiguration1                          = 2,
    L1CommandSize_SystemConfiguration2                          = 2,
    L1CommandSize_CurrentPdo                                    = 4,
    L1CommandSize_CurrentRdo                                    = 4,
    L1CommandSize_AlertEnable                                   = 2,
    L1CommandSize_SystemConfiguration3                          = 2,
    L1CommandSize_SetRdo                                        = 4,
    L1CommandSize_PdosSnkConsumer                               = 16,
    L1CommandSize_PdoSrcProvider                                = 28,
    L1CommandSize_FirmwareType                                  = 2,
    L1CommandSize_FirmwareRevision                              = 2,
    L1CommandSize_ManufactureId                                 = 2,
    L1CommandSize_DeviceId                                      = 2,
    L1CommandSize_RevisionId                                    = 2,
    L1CommandSize_IncomingVdm                                   = 28,
    L1CommandSize_OutgoingVdm                                   = 28,

    L1CommandSize_Max                                           = 28
};

/**
 * @brief L2 コマンド
 */
enum L2Command
{
    L2Command_None                                              = 0,

    L2Command_AbortLastSentCommand                              = 0x0101,
    L2Command_SendPowerRoleSwap                                 = 0x0303,
    L2Command_SetSysRdy                                         = 0x0505,
    L2Command_SendGetSourceCap                                  = 0x0606,
    L2Command_SendRdo                                           = 0x0707,
    L2Command_SendHardReset                                     = 0x0808,
    L2Command_StoreSystemConfiguration                          = 0x0909,
    L2Command_UpdatePdoSrcProvider                              = 0x0A0A,
    L2Command_SendGetSinkCap                                    = 0x0B0B,
    L2Command_StoreControllerConfiguration2                     = 0x0C0C,
    L2Command_SystemReset                                       = 0x0D0D,
    L2Command_ResetSysRdy                                       = 0x1010,
    L2Command_SendVdmWithSop                                    = 0x1111,
    L2Command_SendVdmWithSopDash                                = 0x1212,
    L2Command_SendVdmWithSopDoubleDash                          = 0x1313,
    L2Command_AcceptIncomingVdm                                 = 0x1616,
    L2Command_AlternateModeEntered                              = 0x1717,
    L2Command_SendDataRoleSwap                                  = 0x1818,
    L2Command_SendVConnSwap                                     = 0x1919,

    // The following is valid with BM92T3x.
    L2Command_StartDisplayPortEnterMode                         = 0x3131,
    L2Command_StartDisplayPortConfigure                         = 0x3232,
    L2Command_StartDisplayPortConfigureAndStartHandlingHpd      = 0x3636,
    L2Command_StopHandlingHpdAndSendExitDisplayPort             = 0x3B3B,
    L2Command_StartHandlingHpd                                  = 0x3434,
    L2Command_StopHandlingHpd                                   = 0x3939,

    // The following for validation version. Not support in release version.
    L2Command_SendVdmWithSopDashDebug                           = 0x1414,
    L2Command_SendVdmWithSopDoubleDashDebug                     = 0x1515,
    L2Command_SendBistRequestCarrierMode2                       = 0x2424,
    L2Command_SendBistTestData                                  = 0x2B2B,
    L2Command_SendSoftReset                                     = 0x2C2C,
    L2Command_SendBistCarrierMode2ContinuousString              = 0x2F2F
};

/**
 * @brief Status1 コマンドの CommandState
 */
enum Status1CommandState
{
    Status1CommandState_Ready                                   = 0,
    Status1CommandState_Progress                                = 1
};

/**
 * @brief Status1 コマンドの PowerRole
 */
enum Status1PowerRole
{
    Status1PowerRole_Sink                                       = 0,
    Status1PowerRole_Source                                     = 1
};

/**
 * @brief Status1 の PlugOrientation
 */
enum Status1PlugOrientation
{
    Status1PlugOrientation_Cc1                                  = 0,
    Status1PlugOrientation_Cc2                                  = 1
};

/**
 * @brief Status1 コマンドの SpdSnk
 */
enum Status1SpdSnk
{
    Status1SpdSnk_Off                                           = 0,
    Status1SpdSnk_On                                            = 1
};

/**
 * @brief Status1 コマンドの DataRole
 */
enum Status1DataRole
{
    Status1DataRole_NoPlugInsertion                             = 0,
    Status1DataRole_Ufp                                         = 1,    // Device
    Status1DataRole_Dfp                                         = 2,    // Host
    Status1DataRole_AccessoryMode                               = 3
};

/**
 * @brief Status1 コマンドの LastCommand
 */
enum Status1LastCommand
{
    Status1LastCommand_Completed                                = 0,
    Status1LastCommand_Aborted                                  = 2,
    Status1LastCommand_Invalid                                  = 4,
    Status1LastCommand_Rejected                                 = 6,
    Status1LastCommand_Terminated                               = 7
};

/**
 * @brief Status1 コマンドの InternalErrors
 */
enum Status1InternalError
{
    Status1InternalError_None                                   = 0,
    Status1InternalError_Reserved                               = 1,
    Status1InternalError_OvpOvloOcp                             = 2,
    Status1InternalError_UnexpectedError                        = 3
};

/**
 * @brief Status2 の AccessoryMode
 */
enum Status2AccessoryMode
{
    Status2AccessoryMode_None                                   = 0,
    Status2AccessoryMode_Audio                                  = 1,
    Status2AccessoryMode_Debug                                  = 2,
    Status2AccessoryMode_VConnPowered                           = 3
};

/**
 * @brief Status2 コマンドの IncomingVdmType
 */
enum Status2IncomingVdmType
{
    Status2IncomingVdmType_None                                 = 0,
    Status2IncomingVdmType_Sop                                  = 1,
    Status2IncomingVdmType_SopDash                              = 2,
    Status2IncomingVdmType_SopDoubleDash                        = 3,
    Status2IncomingVdmType_SopDashOnDebug                       = 4,
    Status2IncomingVdmType_SopDoubleDashOnDebug                 = 5
};

/**
 * @brief ControllerConfiguration1 コマンドの SpdSrcOff
 */
enum ControllerConfiguration1SpdSrc
{
    ControllerConfiguration1SpdSrc_On                           = 0,
    ControllerConfiguration1SpdSrc_Off                          = 3
};

/**
 * @brief ControllerConfiguration2 コマンドの DataRoleSwapResponse
 */
enum ControllerConfiguration2DataRoleSwapResponse
{
    ControllerConfiguration2DataRoleSwapResponse_AlwaysReject   = 0,
    ControllerConfiguration2DataRoleSwapResponse_UfpAccept      = 1,
    ControllerConfiguration2DataRoleSwapResponse_DfpAccept      = 2,
    ControllerConfiguration2DataRoleSwapResponse_AlwaysAccept   = 3
};

/**
 * @brief ControllerConfiguration2 コマンドの PowerRoleSwapResponse
 */
enum ControllerConfiguration2PowerRoleSwapResponse
{
    ControllerConfiguration2PowerRoleSwapResponse_AlwaysReject  = 0,
    ControllerConfiguration2PowerRoleSwapResponse_SinkAccept    = 1,
    ControllerConfiguration2PowerRoleSwapResponse_SourceAccept  = 2,
    ControllerConfiguration2PowerRoleSwapResponse_AlwaysAccept  = 3
};


/**
 * @brief AlertStatus (0x02) コマンドのビット定義
 */
struct AlertStatus : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > Reserved15;
    typedef nn::util::BitPack16::Field< 14,  1, bool  > NewPdos;                            // Alert when receive Source PDOs or Sink PDOs from far-end device.
    typedef nn::util::BitPack16::Field< 13,  1, bool  > NewContractProvider;                // Alert when accept the request after receiving it from far-end device as Provider.
    typedef nn::util::BitPack16::Field< 12,  1, bool  > NewContractConsumer;                // Alert when negotiation is completed with far-end as Consumer.
    typedef nn::util::BitPack16::Field< 11,  1, bool  > VdmReceived;                        // Alert when received Vendor Defined Messages from far-end device.
    typedef nn::util::BitPack16::Field< 10,  1, bool  > DataRoleSwapComplete;               // Alert when Data Role Swap is completed.
    typedef nn::util::BitPack16::Field<  9,  1, bool  > VBusDrop;                           // Alert when excessive VBUS drop occurs.
    typedef nn::util::BitPack16::Field<  8,  1, bool  > Bist;                               // Alert when PD controller enters BIST mode.
    typedef nn::util::BitPack16::Field<  7,  1, bool  > VConnSwapComplete;                  // Alert when VCONN Source Swap is completed.
    typedef nn::util::BitPack16::Field<  6,  1, bool  > AlternateModeEvent;                 // Alert when Alternate Mode Event occurred.
    typedef nn::util::BitPack16::Field<  5,  1, bool  > CableReset;                         // Alert when Cable Reset send to Cable Plug completed.
    typedef nn::util::BitPack16::Field<  4,  1, bool  > PowerRoleSwapComplete;              // Alert when Power Role Swap is completed.
    typedef nn::util::BitPack16::Field<  3,  1, bool  > PlugInsertRemove;                   // Alert when USB plug insert/Remove is detected.
    typedef nn::util::BitPack16::Field<  2,  1, bool  > CommandCompleteOrAbort;             // Alert when Command is Completed/Accepted/Aborted/Rejected/Expired.
    typedef nn::util::BitPack16::Field<  1,  1, bool  > HardReset;                          // Alert when Hard Reset send to far-end device completed, or when Hard Reset received from far-end device completed.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > Error;                              // Alert when error occurs. Read Status1.Error for details.
};

/**
 * @brief Status1 (0x03) コマンドのビット定義
 */
struct Status1 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > SpdSrc1;                            // 0b: SPDSRC1 OFF, 1b: SPDSRC1 ON
    typedef nn::util::BitPack16::Field< 14,  1, bool  > SpdSnk;                             // 0b: SPDSNK OFF, 1b: SPDSNK ON
    typedef nn::util::BitPack16::Field< 13,  1, bool  > CommandState;                       // 0b: No Command in progress, 1b: Command in progress
    typedef nn::util::BitPack16::Field< 12,  1, bool  > PowerRole;                          // 0b: Port is Sink, 1b: Port is Source
    typedef nn::util::BitPack16::Field< 11,  1, bool  > PlugOrientation;                    // 0b: CC1 side Valid, 1b: CC2 side Valid
    typedef nn::util::BitPack16::Field< 10,  1, bool  > VBus;                               // 0b: Unpowered or vSafe0V, 1b: vSafe5V or negotiated power level
    typedef nn::util::BitPack16::Field<  8,  2, Bit16 > DataRole;                           // 00b: No plug Insertion, 01b: UFP (Device), 10b: DFP (Host), 11b: In Accessory Mode (Not USB Device/Host)
    typedef nn::util::BitPack16::Field<  7,  1, bool  > PlugInsertion;                      // 0b: No plug insert detected, 1b: Plug insert detected
    typedef nn::util::BitPack16::Field<  4,  3, Bit16 > LastCommand;                        // 000b: The last received command is accepted and completed.
                                                                                            // 010b: Last received command aborted by an error.
                                                                                            // 100b: The last received command is invalid.
                                                                                            // 110b: The last received command is rejected by far-end device.
                                                                                            // 111b: The last command is terminated by receiving a wait message (DR_Swap, PR_Swap or VCONN_Swap)
    typedef nn::util::BitPack16::Field<  3,  1, bool  > SpdSrc2;                            // 0b: SPDSRC2 OFF, 1b: SPDSRC2 ON (always 0 in BM92T3x)
    typedef nn::util::BitPack16::Field<  2,  1, bool  > BstEnable;                          // 0b: de-assert BST_EN (Low), 1b: Assert BST_EN (High)
    typedef nn::util::BitPack16::Field<  0,  2, Bit16 > InternalErrors;                     // 00b: No internal Error,
                                                                                            // 01b: Thermal Shut Down (TSD) occurred
                                                                                            // 10b: OVP, OVLO or OCP occurred
                                                                                            // 11b: Unexpected Error occurred
};

/**
 * @brief Status2 (0x04) コマンドのビット定義
 */
struct Status2 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > VBusDrop;                           // 0b: No excessive VBUS drop, 1b: Excessive VBUS drop
    typedef nn::util::BitPack16::Field< 14,  1, bool  > Bist;                               // 0b: No BIST in progress, 1b: BIST in progress
    typedef nn::util::BitPack16::Field< 13,  1, bool  > VConnSource;                        // 0b: Port isn't VCONN Source, 1b: Port is VCONN Source
    typedef nn::util::BitPack16::Field< 12,  1, bool  > ElectronicallyMarkedCable;          // 0b: No Electronically Marked Cable detected, 1b: Electronically Marked Cable detected
                                                                                            // 3-5A supported cable
    typedef nn::util::BitPack16::Field< 10,  2, Bit16 > AccessoryMode;                      // 00b: Not in accessory mode, 01b: Audio Accessory, 10b: Debug Accessory, 11b: VCONN Powered Accessory
    typedef nn::util::BitPack16::Field<  9,  1, bool  > VConnPower;                         // 0b: VCONN off, 1b: VCONN Source
    typedef nn::util::BitPack16::Field<  4,  5, Bit16 > Reserved4;
    typedef nn::util::BitPack16::Field<  3,  1, bool  > LastReceivedPdos;                   // 0b: Source PDOs or no received PDOs, 1b: Sink PDOs
    typedef nn::util::BitPack16::Field<  0,  3, Bit16 > IncomingVdmType;                    // This field indicates a type of SOP* of VDM stored in Incoming VDM (50h)
                                                                                            // 000b: No valid VDM is stored in Incoming VDM
                                                                                            // 001b: SOP
                                                                                            // 010b: SOP'
                                                                                            // 011b: SOP''
                                                                                            // 100b: SOP'_debug
                                                                                            // 101b: SOP''_debug
                                                                                            // This field is set to a value other than 000b if any VDM is received.
                                                                                            // This field is set to 000b if ""Accept Incoming VDM"" command is written to Command (05h)."
};

/**
 * @brief ExecuteL2Command (0x05) コマンドのビット定義
 */
struct ExecuteL2Command : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field<  0, 16, Bit16 > L2Command;
};

/**
 * @brief ControllerConfiguration1 (0x06) コマンドのビット定義
 */
struct ControllerConfiguration1 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 14,  2, Bit16 > SpdSrcOff;                          // 00b: SPDSRC On when operate as source port, 11b: SPDSRC Off
    typedef nn::util::BitPack16::Field< 12,  2, Bit16 > Reserved12;
    typedef nn::util::BitPack16::Field< 10,  2, Bit16 > TurnOnDelay;                        // Turn-on Delay Time (tVINEN_ON/tVOUTEN_ON)
                                                                                            // (invalid with BM92T3x)
                                                                                            // 00b: 1 mili-seconds,
                                                                                            // 01b: 5 mili-seconds
                                                                                            // 10b: 10 mili-seconds,
                                                                                            // 11b: 20 mili-seconds.
    typedef nn::util::BitPack16::Field<  9,  1, bool  > TurnOffDelay;                       // Turn-off Delay Time (tVINEN_OFF/ tVOUTEN_OFF)
                                                                                            // (invalid with BM92T3x)
                                                                                            // 0b: < 250 micro-seconds,
                                                                                            // 1b: 1 mili-seconds
    typedef nn::util::BitPack16::Field<  8,  1, bool  > BatterySupplySupportAtAutoNgtSnk;   // Enable negotiation using AutoNgtSnk Info Battery (23h) at autonegotiation as sink.
                                                                                            // 0b: Disable, doesn’t negotiate with Battery supply PDO.
                                                                                            // 1b: Enable, negotiates with Battery supply PDO.
    typedef nn::util::BitPack16::Field<  7,  1, bool  > UsbCommunicationCapableAtAutoNgtSnk;// 0b: Not required, 1b: Requires USB Communication Capability in Auto Negotiation as Sink
    typedef nn::util::BitPack16::Field<  6,  1, bool  > AutoNgtSnkEnable;                   // 0b: PD controller does not automatically negotiate as Sink
                                                                                            // 1b: PD controller automatically negotiates for a Fixed Source using the AutoNgtSnk Info when operate as Sink.
    typedef nn::util::BitPack16::Field<  5,  1, bool  > VariableSupplySupportAtAutoNgtSnk;  // Enable negotiation using AutoNgtSnk Info Non-Battery (20h) for Variable Supply at auto negotiation as sink.
                                                                                            // 0b: Disable, negotiates with Fixed Supply PDO only,
                                                                                            // 1b: Enable, negotiates with Fixed of Variable Supply PDO.
    typedef nn::util::BitPack16::Field<  4,  1, bool  > EnterSleepMode;                     // 0b: Clear enter sleep mode request, 1b: Whenever possible, PD controller enters lowest possible power consumption level
    typedef nn::util::BitPack16::Field<  2,  2, Bit16 > SleepTimer;                         // Timeout value after expiry of which the PD controller will automatically enter sleep mode.
                                                                                            // 00b: Always active ,
                                                                                            // 01b: 100ms,
                                                                                            // 10-11b: Reserved
    typedef nn::util::BitPack16::Field<  1,  1, bool  > AutoRequestSwapToSnk;               // 0b: PD controller does not automatically initiate Swap request
                                                                                            // 1b: PD controller automatically initiates Swap request.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > Reserved0;
};

/**
 * @brief DeviceCapability (0x07) コマンドのビット定義
 */
struct DeviceCapability : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field<  9,  1, bool  > DisplayPortSourceSupport;           // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  8,  1, bool  > OcpSupport;                         // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  7,  1, bool  > SpdSnkSupport;                      // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  6,  1, bool  > SbrlSupport;                        // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  5,  1, bool  > SpdSrc1Support;                     // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  4,  1, bool  > SpdSrc2Support;                     // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  3,  1, bool  > VOutEnableSupport;                  // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  2,  1, bool  > VInEnableSupport;                   // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  1,  1, bool  > AlertSupport;                       // 0b: Not supported, 1b: Supported
    typedef nn::util::BitPack16::Field<  0,  1, bool  > AlertStatusAck;                     // 0b: Not supported, 1b: Supported
};

/**
 * @brief PdosSrcConsumerSnkProvider (0x08) コマンドの型定義
 */
const uint8_t PdosSrcConsumerSnkProviderNum = 7;
struct PdosSrcConsumerSnkProvider
{
    Pdo Pdos[PdosSrcConsumerSnkProviderNum];
};

/**
 * @brief ControllerConfiguration2 (0x17) コマンドのビット定義
 */
struct ControllerConfiguration2 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field<  8,  2, Bit16 > VBusCurrent;                        // VBUS Current that DFP advertise on CC pin to its port partner (UFP).
                                                                                            // 00: Default values defined by the USB Specification,
                                                                                            // 01: 1.5A,
                                                                                            // 10: 3A,
                                                                                            // 11: Reserved
    typedef nn::util::BitPack16::Field<  7,  1, bool  > ExternallyPowered;                  // The value of this bit is set to Externally Powered bit of Source and Sink PDO.
                                                                                            // If this bit is changed when a port is provider,
                                                                                            // Externally Powered bit of Source PDO is changed and new Source PDO is sent to port partner.
                                                                                            // 0b: Not externally powered,
                                                                                            // 1b: Externally powered. If there is an AC supply, set this bit to '1'.
    typedef nn::util::BitPack16::Field<  5,  2, Bit16 > Reserved;
    typedef nn::util::BitPack16::Field<  4,  1, bool  > VConnSwapResponse;                  // Response to VCONN Source Swap request from far-end device. 0b: Reject, 1b: Accept
    typedef nn::util::BitPack16::Field<  2,  2, Bit16 > DataRoleSwapResponse;               // Response to Data Role Swap request from far-end device.
                                                                                            // 00: Always reject,
                                                                                            // 01: Accept when UFP, reject when DFP,
                                                                                            // 10: Accept when DFP, reject when UFP,
                                                                                            // 11: Always accept
                                                                                            // 10-11b: Reserved
    typedef nn::util::BitPack16::Field<  0,  2, bool  > PowerRoleSwapResponse;              // Response to Power Role Swap request from far-end device.
                                                                                            // 00: Always reject,
                                                                                            // 01: Accept when power sink, reject when power source,
                                                                                            // 10: Accept when power source, reject when power sink,
                                                                                            // 11: Always accept
};

/**
 * @brief DisplayPortStatus (0x18) コマンドのビット定義
 */
struct DisplayPortStatus : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 14,  2, bool  > ExternalMultiplexer;                // Value of MSEL1 and MSEL0
    typedef nn::util::BitPack16::Field<  9,  5, bool  > Reserved;
    typedef nn::util::BitPack16::Field<  8,  1, bool  > IrqHpd;                             // 0 = No IRQ_HPD since last status message. 1 = IRQ_HPD.
    typedef nn::util::BitPack16::Field<  7,  1, bool  > HpdState;                           // 0 = HPD_Low. 1 = HPD_High.
    typedef nn::util::BitPack16::Field<  6,  1, bool  > ExitDisplayPortModeRequest;         // 0 = Maintain current mode.
                                                                                            // 1 = Request exit from DisplayPort Mode (if in DisplayPort Mode).
    typedef nn::util::BitPack16::Field<  5,  1, bool  > UsbConfigurationRequest;            // 0 = Maintain current configuration.
                                                                                            // 1 = Request switch to USB Configuration (if in DisplayPort Configuration).
    typedef nn::util::BitPack16::Field<  4,  1, bool  > MultiFunctionPreferred;             // 0 = No preference for Multi-function.
                                                                                            // 1 = Multi-function preferred.
    typedef nn::util::BitPack16::Field<  3,  1, bool  > Enabled;                            // 0 = Adaptor DisplayPort functionality is disabled.
                                                                                            // 1 = Adaptor DisplayPort functionality is enabled and operational.
    typedef nn::util::BitPack16::Field<  2,  1, bool  > PowerLow;                           // 0 = Adaptor is functioning normally or is disabled.
                                                                                            // 1 = Adaptor has detected low power and disabled DisplayPort support.
    typedef nn::util::BitPack16::Field<  1,  1, bool  > UfpdConnected;                      // 0 = UFP_D is not connected, or adaptor is disabled.
                                                                                            // 1 = UFP_D is connected.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > DfpdConnected;                      // 0 = DFP_D is not connected, or adaptor is disabled.
                                                                                            // 1 = DFP_D is connected.
};

/**
 * @brief DisplayPortAlertEnable (0x19) コマンドのビット定義
 */
struct DisplayPortAlertEnable : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field<  7,  1, bool  > HpdStateOrIrqHpd;                   // If this bit is set to 1, ALERT# is asserted if HPD State status changed or IRQ_HPD is set.
    typedef nn::util::BitPack16::Field<  6,  1, bool  > ExitDisplayPortModeRequest;         // If this bit is set to 1, ALERT# is asserted if Exit DisplayPortMode Request status changed.
    typedef nn::util::BitPack16::Field<  5,  1, bool  > UsbConfigurationRequest;            // If this bit is set to 1, ALERT# is asserted if USB Configuration Request status changed.
    typedef nn::util::BitPack16::Field<  4,  1, bool  > MultiFunctionPreferred;             // If this bit is set to 1, ALERT# is asserted if Multi-function Preferred status changed.
    typedef nn::util::BitPack16::Field<  3,  1, bool  > Enabled;                            // If this bit is set to 1, ALERT# is asserted if Enabled status changed.
    typedef nn::util::BitPack16::Field<  2,  1, bool  > PowerLow;                           // If this bit is set to 1, ALERT# is asserted if Power Low status changed.
    typedef nn::util::BitPack16::Field<  1,  1, bool  > UfpdConnected;                      // If this bit is set to 1, ALERT# is asserted if UFP_D Connected status changed.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > DfpdConnected;                      // If this bit is set to 1, ALERT# is asserted if DFP_D Connected status changed.
};

/**
 * @brief VendorConfiguration (0x1A) コマンドのビット定義
 */
struct VendorConfiguration : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > AllowNoPdsrc2;                      // Specifies behavior when Pdosrc2 doesn’t exist
                                                                                            // if the controller has device capabilities of both Spdsrc1 Support and Spdsrc2 Support
                                                                                            // 0b: The controller acts as power source only when both Pdsrc1 and Pdsrc2 exist.
                                                                                            // 1b: The controller can act as power source if Pdsrc2 doesn’t exist.
                                                                                            //     If Pdsrc2 doesn’t exist, only PDO1 (vSafe5V) is supported
    typedef nn::util::BitPack16::Field<  9,  6, Bit16 > Reserved9;
    typedef nn::util::BitPack16::Field<  8,  1, bool  > ForceNoHost;                        // Force No Host mode
                                                                                            // 0b: Normal operation. If VDDIO is High, it is Host mode. IF VDDIO is low, it is No Host mode.
                                                                                            // 1b: It is always No-Host mode without depending on VDDIO.
    typedef nn::util::BitPack16::Field<  4,  4, Bit16 > Reserved4;
    typedef nn::util::BitPack16::Field<  3,  1, bool  > OcpDetect;                          // Specifyes its behavior after detecting OCP.
                                                                                            // 0b: It recovers automatically
                                                                                            // 1b: It stays halt state until reset is asserted.
                                                                                            // (This is applied to only firmware type -002 and -003.)
    typedef nn::util::BitPack16::Field<  2,  1, bool  > VBusOutputDetectTimeout;            // Enable VBUS output detect timeout.
    typedef nn::util::BitPack16::Field<  1,  1, bool  > OvloOvpDetect;                      // Specifyes its behavior after detecting OVLO/OVP state.
                                                                                            // 0b: It recovers automatically
                                                                                            // 1b: It stays halt state until reset is asserted.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > Reserved0;
};

/**
 * @brief AutoNgtSnkInfoNonBattery (0x20) コマンドのビット定義
 */
struct AutoNgtSnkInfoNonBattery : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 30,  2, Bit32 > PeekPower;                          // Peak Power
                                                                                            // 00b: IMAX equals IOC
                                                                                            // 01b: IMAX equals 130% IOC for 1ms @ 50% duty cycle
                                                                                            // 10b: IMAX equals 150% IOC for 1ms @ 25% duty cycle
                                                                                            // 11b: IMAX equals 200% IOC for 1ms @ 5% duty cycle
    typedef nn::util::BitPack32::Field< 22,  8, Bit32 > MaxVoltate;                         // Maximum voltage in 200mV units
                                                                                            // 00011001b: 5V
                                                                                            // 00111100b: 12V
                                                                                            // 01100100b: 20V
    typedef nn::util::BitPack32::Field< 20,  2, Bit32 > Reserved20;
    typedef nn::util::BitPack32::Field< 12,  8, Bit32 > MinVoltate;                         // Minimum voltage in 200mV units
                                                                                            // 00001101b: 5V
                                                                                            // 00111100b: 12V
                                                                                            // 01100100b: 20V
    typedef nn::util::BitPack32::Field< 10,  2, Bit32 > Reserved10;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaxOpeCurrent;                      // Maximum operating current in 10mA units
};

/**
 * @brief AutoNgtSnkInfoBattery (0x23) コマンドのビット定義
 */
struct AutoNgtSnkInfoBattery : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 30,  2, Bit32 > Reserved30;
    typedef nn::util::BitPack32::Field< 22,  8, Bit32 > MaxVoltate;                         // Maximum voltage in 200mV units
                                                                                            // 00011001b: 5V
                                                                                            // 00111100b: 12V
                                                                                            // 01100100b: 20V
    typedef nn::util::BitPack32::Field< 20,  2, Bit32 > Reserved20;
    typedef nn::util::BitPack32::Field< 12,  8, Bit32 > MinVoltate;                         // Minimum voltage in 200mV units
                                                                                            // 00001101b: 5V
                                                                                            // 00111100b: 12V
                                                                                            // 01100100b: 20V
    typedef nn::util::BitPack32::Field< 10,  2, Bit32 > Reserved10;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaxPower;                           // Maximum Power in 250mW units
};

/**
 * @brief SystemConfiguration1 (0x26) コマンドのビット定義
 */
struct SystemConfiguration1 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > WakeOnPlugInsert;                   // 0b: Not wake on plug insert/remove,
                                                                                            // 1b: Wake on plug insert/remove
    typedef nn::util::BitPack16::Field< 14,  1, bool  > WakeOnSmBus;                        // 0b: Not wake on on any SMBus command
                                                                                            // 1b: Wake on any SMBus command
    typedef nn::util::BitPack16::Field< 13,  1, bool  > Reserved13;
    typedef nn::util::BitPack16::Field< 12,  1, bool  > PdoSrcProvider;                     // 0b: PDO Src Prov (3Ch) not available
                                                                                            // 1b: PDO Src Prov (3Ch) available
    typedef nn::util::BitPack16::Field<  9,  3, Bit16 > PdosSnkConsumer;                    // 000b: PDOs Snk Cons (33h) not supported or no Sink capability,
                                                                                            // 001b: 1 PDO Sink as Consumer supported,
                                                                                            // 010b: 2 PDOs Sink as Consumer supported,
                                                                                            // 011b: 3 PDOs Sink as Consumer supported,
                                                                                            // 100b: 4 PDOs Sink as Consumer supported,
                                                                                            // All other values should be ignored by receiver
    typedef nn::util::BitPack16::Field<  8,  1, bool  > PdosSrcConsumer;                    // 1b: PDOs Src Cons (08h) supported. Should set as 1b always.
    typedef nn::util::BitPack16::Field<  7,  1, bool  > Reserved7;
    typedef nn::util::BitPack16::Field<  6,  1, bool  > AutoNgtSnkWithNoHost;               // 0b: Never use auto negotiation with no host
                                                                                            // 1b: Use auto negotiation when Host is not present
    typedef nn::util::BitPack16::Field<  4,  2, Bit16 > Reserved4;
    typedef nn::util::BitPack16::Field<  0,  4, Bit16 > ReceptacleType;                     // 0000b: Unconfigured
                                                                                            // 0001b-1000b: Reserved
                                                                                            // 1001b: Type-C (Receptacle)
                                                                                            // 1010b: Type C (3A Captive cable)
                                                                                            // 1011b: Type-C (5A Captive cable)
                                                                                            // 1100b-1111b: Reserved
};

/**
 * @brief SystemConfiguration2 (0x27) コマンドのビット定義
 */
struct SystemConfiguration2 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 15,  1, bool  > VInEnableVOutEnableNoHost;          // Should always be set as zero.
    typedef nn::util::BitPack16::Field< 12,  1, bool  > Reserved12;
    typedef nn::util::BitPack16::Field<  9,  3, Bit16 > PdoSrcProviderCount;                // 001b: 1 PDO Source as Provider supported,
                                                                                            // 010b: 2 PDOs Source as Provider supported,
                                                                                            // 011b: 3 PDOs Source as Provider supported,
                                                                                            // 100b: 4 PDOs Source as Provider supported,
                                                                                            // 101b: 5 PDO Source as Provider supported,
                                                                                            // 110b: 6 PDOs Source as Provider supported,
                                                                                            // 111b: 7 PDOs Source as Provider supported.
    typedef nn::util::BitPack16::Field<  8,  1, bool  > BstEnableSupport;                   // 0b: BST_EN is not supported, 1b: BST_EN is supported
    typedef nn::util::BitPack16::Field<  7,  1, bool  > Reserved7;
    typedef nn::util::BitPack16::Field<  6,  1, bool  > AutoTurnOnSpdSnk;                   // 0b: SPDSNK is not turned on automatically.
                                                                                            // 1b: SPDSNK is turned on automatically without set SYS_RDY.
    typedef nn::util::BitPack16::Field<  3,  3, Bit16 > Reserved3;
    typedef nn::util::BitPack16::Field<  2,  1, bool  > ForceNoUsbCommCapableDeadBattery;   // Force No USB Communications Capable mode when Dead Battery
                                                                                            // 0b: Normal operation. USB Communications setting is depended on PDO Src Prov or USB Comm Capable at AutoNgtSnk
                                                                                            // 1b: Always No USB Communications Capable mode without depending on other setting.
    typedef nn::util::BitPack16::Field<  1,  1, bool  > ForceNoUsbCommCapableDfp;           // Force No USB Communications Capable mode when DFP
                                                                                            // 0b: Normal operation. USB Communications setting is depended on PDO Src Prov or USB Comm Capable at AutoNgtSnk
                                                                                            // 1b: Always No USB Communications Capable mode without depending on other setting.
    typedef nn::util::BitPack16::Field<  0,  1, bool  > ForceNoUsbCommCapableUfp;           // Force No USB Communications Capable mode when UFP
                                                                                            // 0b: Normal operation. USB Communications setting is depended on PDO Src Prov or USB Comm Capable at AutoNgtSnk
                                                                                            // 1b: Always No USB Communications Capable mode without depending on other setting.
};

/**
 * @brief CurrentPdo (0x28) コマンドのビット定義
 */
typedef Pdo CurrentPdo;                                                                     // Bitmap is same as PDO described on USB_PD specification.

/**
 * @brief CurrentRdo (0x2B) コマンドのビット定義
 */
typedef Rdo CurrentRdo;                                                                     // Bitmap is same as RDO described on USB_PD specification.

/**
 * @brief AlertEnable (0x2E) コマンドのビット定義
 */
typedef AlertStatus AlertEnable;

/**
 * @brief SystemConfiguration3 (0x2F) コマンドのビット定義
 */
struct SystemConfiguration3 : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 12,  4, Bit16 > DfpAccTime;                         // This field is used when Type-C Role is DRP or when Type-C Role is DFP and at least one Accessory is supported.
                                                                                            // 0000b: reserved
                                                                                            // 0001b: 15ms
                                                                                            // 0010b: 20ms
                                                                                            // 0011b: 25ms
                                                                                            // 0100b: 30ms
                                                                                            // 0101b: 35ms
                                                                                            // 0110b: 40ms
                                                                                            // 0111b: 45ms
                                                                                            // 1000b: 50ms
                                                                                            // 1001b: 55ms
                                                                                            // 1010b: 60ms
                                                                                            // 1011b: 65ms
                                                                                            // 1100b: 70ms
                                                                                            // 1101b-1110b: reserved
                                                                                            // 1111b: infinite. Both DFP time / ACC time and UFP time shall not be set to 1111b simultaneousl
    typedef nn::util::BitPack16::Field<  8,  4, Bit16 > UfpTime;                            // This field is used when Type-C Role is DRP or when Type-C Role is UFP and at least one Accessory is supported.
                                                                                            // 0000b: reserved
                                                                                            // 0001b: 15ms
                                                                                            // 0010b: 20ms
                                                                                            // 0011b: 25ms
                                                                                            // 0100b: 30ms
                                                                                            // 0101b: 35ms
                                                                                            // 0110b: 40ms
                                                                                            // 0111b: 45ms
                                                                                            // 1000b: 50ms
                                                                                            // 1001b: 55ms
                                                                                            // 1010b: 60ms
                                                                                            // 1011b: 65ms
                                                                                            // 1100b: 70ms
                                                                                            // 1101b-1110b: reserved
                                                                                            // 1111b: infinite. Both DFP time / ACC time and UFP time shall not be set to 1111b simultaneously.
    typedef nn::util::BitPack16::Field<  7,  1, bool  > TryDfpSupport;                      // 0b: Type-C Try.DFP is not supported
                                                                                            // 1b: Type-C Try.DFP is supported if Type-C Role is DRP
    typedef nn::util::BitPack16::Field<  6,  1, bool  > PoweredAccessorySupport;            // 0b: Type-C Powered Accessory is not supported
                                                                                            // 1b: Type-C Powered Accessory is supported if Type-C Role is UFP
                                                                                            // (BM92T3x are not supported)
    typedef nn::util::BitPack16::Field<  5,  1, bool  > DebugAccessorySupport;              // 0b: Type-C Debug Accessory is not supported
                                                                                            // 1b: Type-C Debug Accessory is supported
    typedef nn::util::BitPack16::Field<  4,  1, bool  > AudioAccessorySupport;              // 0b: Type-C Audio Accessory is not supported
                                                                                            // 1b: Type-C Audio Accessory is supported
    typedef nn::util::BitPack16::Field<  3,  1, bool  > AutoTurnOffVConn;                   // 0b: VCONN is not turned off automatically
                                                                                            // 1b: VCONN is automatically turned off if cable type detection is completed and the cable isn't active cable or cable type detection failed
    typedef nn::util::BitPack16::Field<  2,  1, bool  > VConnSupport;                       // 0b: VCONN is not supported
                                                                                            // 1b: VCONN is supported
    typedef nn::util::BitPack16::Field<  0,  2, Bit16 > TypeCRole;                          // 00b: UFP
                                                                                            // 01b: DFP
                                                                                            // 10b: DRP
                                                                                            // 11b: DRP
};

/**
 * @brief SetRdo (0x30) コマンドのビット定義
 */
typedef Rdo SetRdo;                                                                         // Bitmap is same as RDO described on USB_PD specification.

/**
 * @brief PdosSnkConsumer (0x33) コマンドのビット定義
 */
const uint8_t PdosSnkConsumerNum = 4;
struct PdosSnkConsumer
{
    Pdo Pdos[PdosSnkConsumerNum];
};

/**
 * @brief PdoSrcProvider (0x3C) コマンドのビット定義
 */
const uint8_t PdoSrcProviderNum = 7;
struct PdoSrcProvider
{
    Pdo Pdos[PdoSrcProviderNum];
};

/**
 * @brief FirmwareType (0x4B) コマンドのビット定義
 */
struct FirmwareType : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field<  8,  8, Bit16 > DefaultId;                          // Indicates the set number of the register default value, when the default is changed by factory option.
    typedef nn::util::BitPack16::Field<  0,  8, Bit16 > Type;                               // Indicates Firmware type, refer to Appendix-A about Firmware type.
};

/**
 * @brief FirmwareRevision (0x4C) コマンドのビット定義
 */
typedef Bit16 FirmwareRevision;                                                             // Indicates Firmware Revision. (There is not this with the consecutive number.)

/**
 * @brief ManufactureId (0x4D) コマンドのビット定義
 */
typedef Bit16 ManufactureId;                                                                // 10DBh

/**
 * @brief DeviceId (0x4E) コマンドのビット定義
 */
typedef Bit16 DeviceId;                                                                     // TBD

/**
 * @brief RevisionId (0x4F) コマンドのビット定義
 */
struct RevisionId : public nn::util::BitPack16
{
    typedef nn::util::BitPack16::Field< 14,  2, Bit16 > UsbPdSpecRev;                       // 00b: Power Delivery 1.0
    typedef nn::util::BitPack16::Field<  8,  6, Bit16 > Reserved8;
    typedef nn::util::BitPack16::Field<  0,  8, Bit16 > DeviceRev;                          // Incrementally higher number represents later revisions
};


} // detail
} // driver
} // pd
} // usb
} // nn
