﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>

#include "usb_PdDriver.h"

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

void Driver::SetSpdSrc( ControllerConfiguration1SpdSrc spdSrc ) NN_NOEXCEPT
{
    Result result;
    util::BitPack16 pre16;
    util::BitPack16 post16;

    result = ReceiveCommand1( &pre16, L1CommandSize_ControllerConfiguration1, L1Command_ControllerConfiguration1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( m_HostPdcFwRevision < HostPdcFwRevisionEp2 ||
         !IsBatterySupported() ||
         pre16.Get<ControllerConfiguration1::SpdSrcOff>() == spdSrc
       )
    {
        // バッテリの無い Copper または EP1 以前では初期値 SpdSrcOff で固定
        USBPD_DBG_LOG("ControllerConfiguration1 = %04x\n", pre16);
    }
    else
    {
        post16 = pre16;
        post16.Set<ControllerConfiguration1::SpdSrcOff>( spdSrc );
        result = SendCommand1( &post16, L1CommandSize_ControllerConfiguration1, L1Command_ControllerConfiguration1 );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        USBPD_DBG_LOG("ControllerConfiguration1 = %04x -> %04x\n", pre16, post16);
    }
}

bool Driver::EnableOnTheGo() NN_NOEXCEPT
{
    Result result;

    if( !IsBatterySupported() )
    {
        // Copper では古い SDK 実行後のみ DFP となっていて過渡的にこのパスに入る。
        // PDC とリレーボックスを初期化するために強制切断して
        // 電源が落ちた後、再度自動で電源が投入される。
        // Copper 再起動後は UFP に戻っているためこのパスを通らなくなる。
        ForceToRemoveDevice();
        USBPD_ABORT("OnTheGo isn't supported on Copper!");
    }

    if ( m_Status1.Get<Status1::PowerRole>() == Status1PowerRole_Source &&
         m_Status2.Get<Status2::VConnSource>() == true
       )
    {
        USBPD_DBG_LOG("OnTheGo is preparing...\n");

        // Charger からの VBUS 5V 出力
        if ( OutputVBusPower() )
        {
            result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            USBPD_DBG_LOG("Status1 = %04x\n", m_Status1);
            result = ReceiveCommand1( &m_Status2, L1CommandSize_Status2, L1Command_Status2 );
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            USBPD_DBG_LOG("Status2 = %04x\n", m_Status2);
            if ( m_Status1.Get<Status1::PowerRole>() == Status1PowerRole_Source &&
                 m_Status2.Get<Status2::VConnSource>() == true &&
                 m_Status1.Get<Status1::VBus>() == true
               )
            {
                USBPD_DBG_LOG("OnTheGo is ready\n");
                return true;
            }
        }

        // Charger への VBUS 入力へ戻す
        if ( m_IsChargerOtgMode )
        {
            InputVBusPower();
        }
        USBPD_WARNING("OnTheGo is failed!\n");
        return false;
    }
    else
    {
        USBPD_DBG_LOG("OnTheGo is disabled\n");
        return false;
    }
}

bool Driver::OutputVBusPower() NN_NOEXCEPT
{
    Result result;

    m_IsVBusOutputCanceled = false;

    // スリープ中は出力禁止
    if ( m_IsSleeping )
    {
        return false;
    }
    // psm 側の PowerRequest 不許可状態
    if ( !m_IsPowerRequestEnabled )
    {
        return false;
    }
    // 挿抜発生後に DFP で無くなった場合は無効
    if ( m_IsVBusOutputCanceled )
    {
        return false;
    }
    // Charger を OTG モードにする依頼をして VBUS から 5V 出力
    NotifyPowerRequest( StatusRequest_PowerOutput );
    result = WaitPowerReply();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    m_IsChargerOtgMode = true;
    // 挿抜発生後に DFP で無くなった場合は無効
    if ( m_IsVBusOutputCanceled )
    {
        return false;
    }

    // Charger からの VBUS 5V 出力待ち
    m_VBusOutputTimerEvent.Stop();
    m_VBusOutputTimerEvent.Clear();
    m_VBusOutputTimerEvent.StartOneShot( VBusOutputCompleteTimeSpan );
    m_VBusOutputTimerEvent.Wait();
    // 挿抜発生後に DFP で無くなった場合は無効
    if ( m_IsVBusOutputCanceled )
    {
        return false;
    }

    return true;
}

bool Driver::InputVBusPower() NN_NOEXCEPT
{
    Result result;

    // スリープ中は psm が自発的に入力にする
    if ( m_IsSleeping )
    {
        m_IsChargerOtgMode = false;
        return true;
    }
    // psm 側の PowerRequest 不許可状態
    if ( !m_IsPowerRequestEnabled )
    {
        m_IsChargerOtgMode = false;
        return true;
    }
    // Charger を ChargeBattery モードへ戻す依頼をする
    NotifyPowerRequest( StatusRequest_PowerInput );
    result = WaitPowerReply();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    m_IsChargerOtgMode = false;

    return true;
}

Result Driver::EnablePowerRequestNoticeCore( int sessionId, bool enable ) NN_NOEXCEPT
{
    if ( !m_PowerRequestNoticeInitialized )
    {
        m_PowerRequestNoticeInitialized = true;
        m_PowerRequestNoticeSessionId = sessionId;
    }
    else if ( sessionId != m_PowerRequestNoticeSessionId )
    {
        return ResultInvalidSession();
    }
    m_IsPowerRequestEnabled = enable;
    m_PowerRequestEnableEvent.Signal();
    return ResultSuccess();
}

Result Driver::EnablePowerRequestNotice( int sessionId ) NN_NOEXCEPT
{
    USBPD_DBG_LOG("EnablePowerRequest\n");
    return EnablePowerRequestNoticeCore( sessionId, true );
}

Result Driver::DisablePowerRequestNotice( int sessionId ) NN_NOEXCEPT
{
    USBPD_DBG_LOG("DisablePowerRequest\n");
    return EnablePowerRequestNoticeCore( sessionId, false );
}

void Driver::NotifyPowerRequest( StatusRequest request ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_IpcStatusMutex);
    m_PowerReplyEvent.Clear();
    // パワー入出力切り替え要求
    m_IpcRequest = request;
    m_IpcNotice[m_PowerRequestNoticeSessionId].Set<Notice::RequestNotice>( true );
    SignalNoticeEvent( m_PowerRequestNoticeSessionId );
}

Result Driver::ReplyPowerRequest( int sessionId, bool isSuccess ) NN_NOEXCEPT
{
//    USBPD_DBG_LOG("ReplyPowerRequest\n");
    Result result = ResultSuccess();
    if ( sessionId != m_PowerRequestNoticeSessionId )
    {
        result =  ResultInvalidSession();
    }
    m_IsPowerReplySuccess = isSuccess;
    m_IpcRequest = StatusRequest_None;
    m_PowerReplyEvent.Signal();
    return result;
}

Result Driver::WaitPowerReply() NN_NOEXCEPT
{
    m_PowerReplyEvent.Wait();
    m_PowerReplyEvent.Clear();
    if ( !m_IsPowerReplySuccess )
    {
        return ResultErrorReply();
    }
    m_IsPowerReplySuccess = false;
    return ResultSuccess();
}

} // detail
} // driver
} // pd
} // usb
} // nn
