﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>
#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>
#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "usb_PdDriver.h"

namespace {

// スレッドに与えるスタックリソース
const size_t ExecuteThreadStackSize = 32 * 1024;
const size_t AlertThreadStackSize = 16 * 1024;
const size_t ReceiveVdmThreadStackSize = 16 * 1024;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_ExecuteThreadStack[ExecuteThreadStackSize];
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_AlertThreadStack[AlertThreadStackSize];
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_ReceiveVdmThreadStack[ReceiveVdmThreadStackSize];

}

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

const char* const g_pDataRoleName[] = {"NoPlug", "UFP", "DFP", "InAccessory"};

static const VdmCommand s_VdmNoticeToCommandTable[VdmNoticeType_Num] =
{
    VdmCommand_DeviceErrorNotice
};

void Driver::Initialize( nn::i2c::SpeedMode i2cSpeed ) NN_NOEXCEPT
{
    if ( m_InitializeCount == 0 )
    {
        Result result;

        // プラットフォーム構成情報読み込み
        GetSettingsItemValue( &m_IsBatterySupported, sizeof(m_IsBatterySupported), "platformconfig", "has_battery" );
        GetSettingsItemValue( &m_IsPdcBootFatalEnabled, sizeof(m_IsPdcBootFatalEnabled), "usb", "show_pdc_boot_fatal" );

        // アラートスレッドイベント初期化
        os::InitializeMultiWait( &m_AlertGpioWaiter );
        // 終了処理用内部イベントの設定（最優先）
        os::InitializeMultiWaitHolder( &m_FinalizeAlertEventHolder, m_FinalizeAlertEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_AlertGpioWaiter, &m_FinalizeAlertEventHolder );
        // スリープ要求イベントの設定
        os::InitializeMultiWaitHolder( &m_SleepAlertRequestEventHolder, m_SleepAlertRequestEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_AlertGpioWaiter, &m_SleepAlertRequestEventHolder );
        // GPIO イベントの設定
        os::InitializeMultiWaitHolder( &m_AlertGpioEventHolder, m_AlertGpioEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_AlertGpioWaiter, &m_AlertGpioEventHolder );

        // 実行スレッドイベント初期化
        os::InitializeMultiWait( &m_ExecuteWaiter );
        // 終了処理用内部イベントの設定（最優先）
        os::InitializeMultiWaitHolder( &m_FinalizeExecuteEventHolder, m_FinalizeExecuteEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_FinalizeExecuteEventHolder );
        // VBUS 出力許可イベントの設定
        os::InitializeMultiWaitHolder( &m_PowerRequestEnableEventHolder, m_PowerRequestEnableEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_PowerRequestEnableEventHolder );
        // スリープ要求イベントの設定
        os::InitializeMultiWaitHolder( &m_SleepRequestEventHolder, m_SleepRequestEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_SleepRequestEventHolder );
        os::InitializeMultiWaitHolder( &m_FullAwakeRequestEventHolder, m_FullAwakeRequestEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_FullAwakeRequestEventHolder );
        // PDC アラートイベントの設定
        for ( int i = 0; i < AlertNum; ++i )
        {
            // Driver コンストラクタで初期化できないため placement new を使う
            m_pAlertEvent[i] = new (&m_AlertEventBuffer[i]) nn::os::Event( os::EventClearMode_ManualClear );
            m_pAlertEvent[i]->Clear();
            os::InitializeMultiWaitHolder( &m_AlertEventHolder[i], m_pAlertEvent[i]->GetBase() );
            if ( i != AlertStatus::CommandCompleteOrAbort::Pos &&
                 i != AlertStatus::VdmReceived::Pos
               )
            {
                os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_AlertEventHolder[i]);
                os::SetMultiWaitHolderUserData( &m_AlertEventHolder[i], static_cast<uintptr_t>(i));
            }
        }
        // Power Contract イベント関連の設定
        os::InitializeMultiWait( &m_NewContractWaiter );
        os::InitializeMultiWaitHolder( &m_NewContractCancelEventHolder, m_NewContractCancelEvent.GetBase() );
        os::InitializeMultiWaitHolder( &m_NewContractEventHolder, m_NewContractEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_NewContractWaiter, &m_NewContractCancelEventHolder );
        os::LinkMultiWaitHolder( &m_NewContractWaiter, &m_NewContractEventHolder );
        // VDM 受信イベントの設定
        for ( int i = 0; i < VdmNoticeType_Num; ++i )
        {
            // Driver コンストラクタで初期化できないため placement new を使う
            m_pNoticeVdmEvent[i] = new (&m_NoticeEventBuffer[i]) nn::os::Event( os::EventClearMode_ManualClear );
            m_pNoticeVdmEvent[i]->Clear();
        }
        // クレードル通知エラーイベントの設定
        os::InitializeMultiWaitHolder( &m_CradleErrorEventHolder, m_pNoticeVdmEvent[VdmNoticeType_DeviceError]->GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_CradleErrorEventHolder );
        // クレードル USB HUB VBUS 検出タイマーイベントの設定
        os::InitializeMultiWaitHolder( &m_CradleUsbHubTimerEventHolder, m_CradleUsbHubTimerEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_CradleUsbHubTimerEventHolder );
        // OVP 回復タイマーイベントの設定
        os::InitializeMultiWaitHolder( &m_OvpRecoverTimerEventHolder, m_OvpRecoverTimerEvent.GetBase() );
        os::LinkMultiWaitHolder( &m_ExecuteWaiter, &m_OvpRecoverTimerEventHolder );

        // BUS プロセス初期化
        InitializeBuses( i2cSpeed );

        // 初期化が完了したら、スレッドを走らせる
        m_IsFinalizing = false;
        os::CreateThread( &m_ExecuteThread, ExecuteThread, this,
                          s_ExecuteThreadStack, ExecuteThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(usb, PdExecute) );
        os::SetThreadNamePointer( &m_ExecuteThread, NN_SYSTEM_THREAD_NAME(usb, PdExecute) ) ;
        os::CreateThread( &m_ReceiveVdmThread, ReceiveVdmThread, this,
                          s_ReceiveVdmThreadStack, ReceiveVdmThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(usb, PdReceiveVdm) );
        os::SetThreadNamePointer( &m_ReceiveVdmThread, NN_SYSTEM_THREAD_NAME(usb, PdReceiveVdm) ) ;
        os::CreateThread( &m_AlertThread, AlertThread, this,
                          s_AlertThreadStack, AlertThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(usb, PdAlert) );
        os::SetThreadNamePointer( &m_AlertThread, NN_SYSTEM_THREAD_NAME(usb, PdAlert) ) ;
        os::StartThread( &m_ExecuteThread );
        os::StartThread( &m_ReceiveVdmThread );
        // Alert スレッドのみ SystemReset の直前に起動し
        // PDC の初期化完了を待つ
        m_InitializePdcEvent.Wait();

        m_InitializeCount++;
    }
}

void Driver::GetSettingsItemValue(void* buffer, size_t bufferSize, const char* name, const char* key) NN_NOEXCEPT
{
    if ( nn::settings::fwdbg::GetSettingsItemValue(buffer, bufferSize, name, key) != bufferSize )
    {
        USBPD_ABORT("Fail to read settings %s.%s.\n", name, key);
    }
}

void Driver::StartAlertThread() NN_NOEXCEPT
{
    gpio::BindInterrupt( m_AlertGpioEvent.GetBase(), &m_AlertSession );
    os::StartThread( &m_AlertThread );
}

void Driver::Finalize() NN_NOEXCEPT
{
    // Initialize された分、Finalize が呼ばれたら、全セッションをクローズし、割り込み管理用スレッドも破棄する。
    NN_SDK_ASSERT(!(m_InitializeCount == 0), "initialize されていません。");

    m_InitializeCount--;
    if( m_InitializeCount == 0 )
    {
        // ExecuteThread 終了処理
        m_FinalizeExecuteEvent.Signal();
        m_MinimumAwakeRequestEvent.Signal();
        m_FullAwakeRequestEvent.Signal();
        os::WaitThread( &m_ExecuteThread );
        os::DestroyThread( &m_ExecuteThread );
        m_FinalizeExecuteEvent.Clear();

        // ReceiveVdmThread 終了処理
        m_IsFinalizing = true;
        m_pAlertEvent[AlertStatus::VdmReceived::Pos]->Signal();
        os::WaitThread( &m_ReceiveVdmThread );
        os::DestroyThread( &m_ReceiveVdmThread );
        m_pAlertEvent[AlertStatus::VdmReceived::Pos]->Clear();

        // AlertThread 終了処理
        m_FinalizeAlertEvent.Signal();  // m_AlertGpioEvent.Signal() が出来ないため
        m_AwakeAlertRequestEvent.Signal();
        os::WaitThread( &m_AlertThread );
        os::DestroyThread( &m_AlertThread );
        m_FinalizeAlertEvent.Clear();

        // 実行スレッドイベント終了処理
        m_OvpRecoverTimerEvent.Stop();
        os::UnlinkMultiWaitHolder( &m_OvpRecoverTimerEventHolder );
        os::FinalizeMultiWaitHolder( &m_OvpRecoverTimerEventHolder );
        m_CradleUsbHubTimerEvent.Stop();
        os::UnlinkMultiWaitHolder( &m_CradleUsbHubTimerEventHolder );
        os::FinalizeMultiWaitHolder( &m_CradleUsbHubTimerEventHolder );
        os::UnlinkMultiWaitHolder( &m_NewContractEventHolder );
        os::FinalizeMultiWaitHolder( &m_NewContractEventHolder );
        os::UnlinkMultiWaitHolder( &m_NewContractCancelEventHolder );
        os::FinalizeMultiWaitHolder( &m_NewContractCancelEventHolder );
        for ( int i = 0; i < VdmNoticeType_Num; ++i )
        {
            m_pNoticeVdmEvent[i]->Clear();
            m_pNoticeVdmEvent[i]->os::Event::~Event();
        }
        os::UnlinkMultiWaitHolder( &m_CradleErrorEventHolder );
        os::FinalizeMultiWaitHolder( &m_CradleErrorEventHolder );
        for ( int i = 0; i < AlertNum; ++i )
        {
            m_pAlertEvent[i]->Clear();
            if ( i != AlertStatus::CommandCompleteOrAbort::Pos &&
                 i != AlertStatus::VdmReceived::Pos
               )
            {
                os::UnlinkMultiWaitHolder( &m_AlertEventHolder[i] );
            }
            os::FinalizeMultiWaitHolder( &m_AlertEventHolder[i] );
            m_pAlertEvent[i]->os::Event::~Event();
        }
        os::UnlinkMultiWaitHolder( &m_FullAwakeRequestEventHolder );
        os::FinalizeMultiWaitHolder( &m_FullAwakeRequestEventHolder );
        os::UnlinkMultiWaitHolder( &m_SleepRequestEventHolder );
        os::FinalizeMultiWaitHolder( &m_SleepRequestEventHolder );
        os::UnlinkMultiWaitHolder( &m_PowerRequestEnableEventHolder );
        os::FinalizeMultiWaitHolder( &m_PowerRequestEnableEventHolder );
        os::UnlinkMultiWaitHolder( &m_FinalizeExecuteEventHolder );
        os::FinalizeMultiWaitHolder( &m_FinalizeExecuteEventHolder );
        os::FinalizeMultiWait( &m_NewContractWaiter );
        os::FinalizeMultiWait( &m_ExecuteWaiter );

        // アラートスレッドイベント終了処理
        os::UnlinkMultiWaitHolder( &m_AlertGpioEventHolder );
        os::FinalizeMultiWaitHolder( &m_AlertGpioEventHolder );
        os::UnlinkMultiWaitHolder( &m_SleepAlertRequestEventHolder );
        os::FinalizeMultiWaitHolder( &m_SleepAlertRequestEventHolder );
        os::UnlinkMultiWaitHolder( &m_FinalizeAlertEventHolder );
        os::FinalizeMultiWaitHolder( &m_FinalizeAlertEventHolder );
        os::FinalizeMultiWait( &m_AlertGpioWaiter );

        // BUS プロセス終了処理
        FinalizeBuses();

        USBPD_DBG_LOG("Finalize\n");
    }
}

void Driver::InitializeBuses( nn::i2c::SpeedMode i2cSpeed ) NN_NOEXCEPT
{
    Result result;

    // I2C 初期化
    i2c::Initialize();
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    if ( i2cSpeed == i2c::SpeedMode(0) )
    {
        i2c::OpenSession( &m_I2cSession, i2c::I2cDevice_Bm92t30mwv );
    }
    else
#endif
    {
        i2c::OpenSessionForDev( &m_I2cSession, I2cBusIdx, I2cAddress, i2c::AddressingMode_BitWidth7, i2cSpeed );
    }

    // GPIO 初期化
    gpio::Initialize();
    gpio::OpenSession( &m_AlertSession, gpio::GpioPadName_CradleIrq );  // == GpioAlert
    gpio::SetDirection( &m_AlertSession, gpio::Direction_Input );
    gpio::SetInterruptMode( &m_AlertSession, gpio::InterruptMode_LowLevel );
    gpio::SetInterruptEnable( &m_AlertSession, true );
    m_AlertGpioEvent.Clear();

    gpio::OpenSession( &m_PdResetSession, gpio::GpioPadName_PdRstN );  // AP_READY -> XRST
    gpio::SetDirection( &m_PdResetSession, gpio::Direction_Output );
    // VCONN_EN(GPIO_PK5) / OTG-OUT-EN(GPIO_DAP4_DIN) / OTG-IN-EN(GPIO_PL0) は psm で初期化
}

void Driver::FinalizeBuses() NN_NOEXCEPT
{
    // GPIO 終了処理
    gpio::CloseSession( &m_PdResetSession );
    gpio::UnbindInterrupt( &m_AlertSession );
    gpio::CloseSession( &m_AlertSession );
    gpio::Finalize();

    // I2C 終了処理
    i2c::CloseSession( m_I2cSession );
    i2c::Finalize();
}

void Driver::InitializePdc() NN_NOEXCEPT
{
    Result result;
    util::BitPack16 tmp16;
    util::BitPack32 tmp32;

#ifdef ENABLE_PDC_I2C_NACK_TEST
    // I2C_NACK テスト時の HW リセット継続（回路上で論理反転されている）
    // ※ 充電不可になり、もし Copper で行ってしまうとシャットダウンすることに注意
    gpio::SetValue( &m_PdResetSession, gpio::GpioValue_High );
    USBPD_WARNING("HwReset forever for NACK of I2C!\n");
#endif
    // PD コン FW ブート待ち
    WaitDeviceId();
    result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Status1 = %04x\n", m_Status1);
    result = ReceiveCommand1( &m_Status2, L1CommandSize_Status2, L1Command_Status2 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Status2 = %04x\n", m_Status2);
    result = ReceiveCommand1( &tmp16, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortStatus = %04x\n", tmp16);

    result = ReceiveCommand1( &tmp32, L1CommandSize_CurrentPdo, L1Command_CurrentPdo );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CurrentPDO = %08x\n", tmp32);
    result = ReceiveCommand1( &tmp32, L1CommandSize_CurrentRdo, L1Command_CurrentRdo );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CurrentRDO = %08x\n", tmp32);

    // PD コンのリセット
    m_HwResetRetryCount = 0;
    if( IsBatterySupported() )
    {
        // HW リセット
        HardwareReset( false );

        // 直前のアラート解除
        result = ReceiveCommand1( &m_AlertStatus, L1CommandSize_AlertStatus, L1Command_AlertStatus );
        USBPD_DBG_LOG("Alert = %04x\n", m_AlertStatus);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // SystemReset のため Alert スレッドをここで起動
        //（ただし HW リセット完了後でなければ I2C から NACK が返って来ることがある）
        StartAlertThread();

        // 対向デバイスとの強制切断
        ForceToRemoveDevice();

        result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
        USBPD_DBG_LOG("Status1 = %04x\n", m_Status1);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        result = ReceiveCommand1( &m_Status2, L1CommandSize_Status2, L1Command_Status2 );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        USBPD_DBG_LOG("Status2 = %04x\n", m_Status2);
    }
    else
    {
        // バッテリがある場合と Alert スレッド起動タイミングを合わせる
        StartAlertThread();
        // バッテリが無い場合に SoC だけリセットされた時のためにダミーの挿入イベント発行
        //   電源 ON 時も SoC 起動までに PDC は Power Contract まで完了している
        m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->Signal();
    }

    // PD コン初期値読み込み
    result = ReceiveCommand1( &m_ManufactureId, L1CommandSize_ManufactureId, L1Command_ManufactureId );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("ManufactureID = %04x\n", m_ManufactureId);
    result = ReceiveCommand1( &m_HostPdcFwType, L1CommandSize_FirmwareType, L1Command_FirmwareType );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("HostPdcFwType = %04x\n", m_HostPdcFwType);
    result = ReceiveCommand1( &m_HostPdcFwRevision, L1CommandSize_FirmwareRevision, L1Command_FirmwareRevision );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("HostPdcFwRevision = %04x\n", m_HostPdcFwRevision);
    // HW リセット後の再設定
    SetupPdc();
#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DBG_LOG)
    // 初期状態取得
    GetInitialStatus();
#endif

    // IPC 用ステータスのアップデート
    UpdateStatus( false );

    // PDC 初期化完了イベント
    m_InitializePdcEvent.Signal();
}

void Driver::FinalizePdc() NN_NOEXCEPT
{
    m_InitializePdcEvent.Clear();
}

void Driver::SetupPdc() NN_NOEXCEPT
{
    Result result;
    util::BitPack16 tmp16;

    // DisplayPort アラート有効化
    m_DisplayPortAlertEnable.Clear();
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::HpdStateOrIrqHpd>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::ExitDisplayPortModeRequest>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::UsbConfigurationRequest>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::MultiFunctionPreferred>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::Enabled>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::PowerLow>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::UfpdConnected>( true );
    m_DisplayPortAlertEnable.Set<DisplayPortAlertEnable::DfpdConnected>( true );
    result = SendCommand1( &m_DisplayPortAlertEnable, L1CommandSize_DisplayPortAlertEnable, L1Command_DisplayPortAlertEnable );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortAlertEnable = %04x\n", m_DisplayPortAlertEnable);

    // VBUS 出力検出タイムアウトを無効に（現在はデフォルトで無効になっている）
    result = ReceiveCommand1( &tmp16, L1CommandSize_VendorConfiguration, L1Command_VendorConfiguration );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("VendorConfiguration = %04x", tmp16);
    if ( tmp16.Get<VendorConfiguration::VBusOutputDetectTimeout>() )
    {
        tmp16.Set<VendorConfiguration::VBusOutputDetectTimeout>( false );
        result = SendCommand1( &tmp16, L1CommandSize_VendorConfiguration, L1Command_VendorConfiguration );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_SDK_LOG(" -> %04x", tmp16);
    }
    NN_SDK_LOG("\n");
}

void Driver::GetInitialStatus() NN_NOEXCEPT
{
    Result result;
    util::BitPack16 tmp16;
    result = ReceiveCommand1( &tmp16, L1CommandSize_ControllerConfiguration1, L1Command_ControllerConfiguration1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("ControllerConfiguration1 = %04x\n", tmp16);
    result = ReceiveCommand1( &tmp16, L1CommandSize_ControllerConfiguration2, L1Command_ControllerConfiguration2 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("ControllerConfiguration2 = %04x\n", tmp16);
    result = ReceiveCommand1( &tmp16, L1CommandSize_SystemConfiguration1, L1Command_SystemConfiguration1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("SystemConfiguration1 = %04x\n", tmp16);
    result = ReceiveCommand1( &tmp16, L1CommandSize_SystemConfiguration2, L1Command_SystemConfiguration2 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("SystemConfiguration2 = %04x\n", tmp16);
    result = ReceiveCommand1( &tmp16, L1CommandSize_SystemConfiguration3, L1Command_SystemConfiguration3 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("SystemConfiguration3 = %04x\n", tmp16);
    result = ReceiveCommand1( &tmp16, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortStatus = %04x\n", tmp16);
}

// CreateThread に渡すメイン関数（要 static）
// クラスメンバに触りたいので中でメンバ関数を呼び出す
void Driver::AlertThread( void* arg ) NN_NOEXCEPT
{
    // メンバ変数を触るために処理本体はメンバ関数で
    Driver* driver = reinterpret_cast<Driver*>(arg);
    driver->AlertThreadBoby();
}

void Driver::AlertThreadBoby() NN_NOEXCEPT
{
    while (1)
    {
        Result result;
        os::MultiWaitHolderType* pEventHolder = os::WaitAny( &m_AlertGpioWaiter );
//        USBPD_DBG_LOG("AlertThread: Wake up!\n");

        // Finalize 中だったら while を抜けて終了
        if ( pEventHolder == &m_FinalizeAlertEventHolder )
        {
            USBPD_DBG_LOG("AlertThread: Finalize\n");
            break;
        }
        // Sleep/Awake 前段処理
        else if ( pEventHolder == &m_SleepAlertRequestEventHolder )
        {
            m_SleepAlertRequestEvent.Clear();
            m_SleepAlertReplyEvent.Signal();
            m_AwakeAlertRequestEvent.Wait();
            m_AwakeAlertRequestEvent.Clear();
        }

        std::lock_guard<nn::os::Mutex> lock(m_AlertMutex);

        AlertStatus alert;
        alert.Clear();
        bool alertGpio = false;
        do
        {
            result = ReceiveCommand1( &m_AlertStatus, L1CommandSize_AlertStatus, L1Command_AlertStatus );
            USBPD_DBG_LOG("AlertThread::Alert = %04x\n", m_AlertStatus);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            alert.storage |= m_AlertStatus.storage;
            alertGpio = gpio::GetValue( &m_AlertSession );
//            USBPD_DBG_LOG("AlertThread::AlertGpio = %d\n", alertGpio);

        } while ( !alertGpio && m_ResetResult.IsSuccess() );

        gpio::ClearInterruptStatus( &m_AlertSession );
        m_AlertGpioEvent.Clear();
        if ( m_ResetResult.IsSuccess() )
        {
            gpio::SetInterruptEnable( &m_AlertSession, true );
        }

        // Alert 関連イベントの発行
        SignalAlertEvents( alert );

        // Sleep/Awake 後段処理
        if ( pEventHolder == &m_SleepAlertRequestEventHolder )
        {
            m_AwakeAlertReplyEvent.Signal();
        }
    }
}

void Driver::SignalAlertEvents( AlertStatus alert ) NN_NOEXCEPT
{
    Result result;

    for ( int i = 0; i < AlertNum; ++i )
    {
        if ( alert.GetBit( i ) )
        {
            switch ( i )
            {
            case AlertStatus::Error::Pos:
                // OVP 時は Power Contract と PMIC の /ACOK 待ちを中断
                m_NewContractCancelEvent.Signal();
                m_PdcResetToAcOkTimerEvent.Signal();
                break;

            case AlertStatus::HardReset::Pos:
                // Hard Reset 発生時に VDM リプライ待ちを中断
                m_IsVdmCanceled = true;
                m_StandardVdmEvent.Signal();
                m_ReplyCommonVdmEvent.Signal();
                break;

            case AlertStatus::PlugInsertRemove::Pos:
                result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                if ( m_pAlertEvent[AlertStatus::NewContractConsumer::Pos]->TryWait() || m_NewContractEvent.TryWait() )
                {
                    // プラグ挿抜によって無効になった Power Contract イベントのクリア
                    USBPD_WARNING("AlertThread: Clear invalid power contract!\n");
                    m_NewContractEvent.Clear();
                    m_pAlertEvent[AlertStatus::NewContractConsumer::Pos]->Clear();
                }
                if ( !m_Status1.Get<Status1::PlugInsertion>() )
                {
                    // プラグ抜去時も Power Contract と PMIC の /ACOK 待ちを中断
                    m_NewContractCancelEvent.Signal();
                    m_PdcResetToAcOkTimerEvent.Signal();
                }
                if ( m_Status1.Get<Status1::DataRole>() != Status1DataRole_Dfp || !m_Status1.Get<Status1::PlugInsertion>() )
                {
                    // DFP でない場合
                    //   VDM リプライ待ちを中断
                    m_IsVdmCanceled = true;
                    m_StandardVdmEvent.Signal();
                    m_ReplyCommonVdmEvent.Signal();
                    //   VBUS 出力待ちをスキップ
//                    USBPD_WARNING("AlertThread: VBUS output wait is canceled!\n");
                    m_IsVBusOutputCanceled = true;
                    m_VBusOutputTimerEvent.Signal();
                }
                break;

            case AlertStatus::NewContractConsumer::Pos:
                m_NewContractEvent.Signal();
                break;

            default:
                break;
            }
            // Status1 読み込みより後に Alert イベント通知
            m_pAlertEvent[i]->Signal();
        }
    }
}

// CreateThread に渡すメイン関数（要 static）
// クラスメンバに触りたいので中でメンバ関数を呼び出す
void Driver::ReceiveVdmThread( void* arg ) NN_NOEXCEPT
{
    // メンバ変数を触るために処理本体はメンバ関数で
    Driver* driver = reinterpret_cast<Driver*>(arg);
    driver->ReceiveVdmThreadBoby();
}

void Driver::ReceiveVdmThreadBoby() NN_NOEXCEPT
{
    while (1)
    {
        Result result;
        m_pAlertEvent[AlertStatus::VdmReceived::Pos]->Wait();
//        USBPD_DBG_LOG("ReceiveVdmThread: Wake up!\n");
        m_pAlertEvent[AlertStatus::VdmReceived::Pos]->Clear();

        // Finalize 中だったら while を抜けて終了
        if ( m_IsFinalizing )
        {
            USBPD_DBG_LOG("ReceiveVdmThread: Finalize\n");
            break;
        }

        // VDM 受信
        VdmUnit vdm[VdmCommandLength_Max];
        result = ReceiveNxVdmCore( &vdm, sizeof(vdm) );

        VdmUnit* v = nullptr;
        nn::os::Event* e = nullptr;
        if ( vdm[0].Get<VdmHeader::VdmType>() == VdmType_StructuredVdm )
        {
            v = m_StandardVdm;
            e = &m_StandardVdmEvent;
        }
        else
        {
            VdmCommand command = VdmCommand( vdm[1].Get<VdmCommonVdo1::Command>() );
            VdmNoticeType n = SearchVdmNoticeType( command );
            if  ( n != VdmNoticeType_None )
            {
                v = m_NoticeVdm[n];
                e = m_pNoticeVdmEvent[n];
            }
            else if ( g_VdmCommandTypeList[command] == VdmCommandType_Ack )
            {
                v = m_ReplyCommonVdm;
                e = &m_ReplyCommonVdmEvent;
            }
        }
        if ( v )
        {
            std::lock_guard<nn::os::Mutex> lock(m_ReceiveVdmMutex);
            memcpy( v, &vdm, sizeof(vdm) );
        }
        if ( e )
        {
            e->Signal();
        }
    }
}

VdmNoticeType Driver::SearchVdmNoticeType( VdmCommand c ) NN_NOEXCEPT
{
    for ( int i = 0; i < VdmNoticeType_Num; ++i )
    {
        if ( s_VdmNoticeToCommandTable[i] == c )
        {
            return VdmNoticeType(i);
        }
    }

    return VdmNoticeType_None;
}

// CreateThread に渡すメイン関数（要 static）
// クラスメンバに触りたいので中でメンバ関数を呼び出す
void Driver::ExecuteThread( void* arg ) NN_NOEXCEPT
{
    // メンバ変数を触るために処理本体はメンバ関数で
    Driver* driver = reinterpret_cast<Driver*>(arg);
    driver->ExecuteThreadBody();
}

void Driver::ExecuteThreadBody() NN_NOEXCEPT
{
    // PD コン初期化
    InitializePdc();

    while (1)
    {
        if ( m_IsPdcBootFatalEnabled )
        {
            // PD コン初期化失敗時の Fatal Error 出力
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_ResetResult);
        }
        else if ( m_ResetResult.IsFailure() )
        {
            USBPD_WARNING("Skip fatal for bad pdc state %08x\n", m_ResetResult.GetInnerValueForDebug());
            // 修理用ファームで強制的に USB デバイスにするための通知
            UpdateStatus( false, Notice::ActiveNotice::Mask );
        }

        os::MultiWaitHolderType* pEventHolder = os::WaitAny( &m_ExecuteWaiter );

        PrintPdcBaseStatuses();

        // 終了要求あり
        if ( pEventHolder == &m_FinalizeExecuteEventHolder )
        {
            USBPD_DBG_LOG("ExecuteThread: Finalize\n");
            break;
        }
        // VBUS 出力許可／禁止通知
        else if ( pEventHolder == &m_PowerRequestEnableEventHolder )
        {
            m_PowerRequestEnableEvent.Clear();
            // SpdSrc 変更後に PlugInsertRemove アラートが発生
            ControllerConfiguration1SpdSrc spdSrc = m_IsPowerRequestEnabled ? ControllerConfiguration1SpdSrc_On : ControllerConfiguration1SpdSrc_Off;
            SetSpdSrc( spdSrc );
            continue;
        }
        // Sleep 要求あり
        else if ( pEventHolder == &m_SleepRequestEventHolder )
        {
            // 起床要求が来るまで ExecuteThread 停止
            SleepAndWaitForMinimumAwake();
            continue;
        }
        // FullAwake 要求あり
        else if ( pEventHolder == &m_FullAwakeRequestEventHolder )
        {
            // クレードルをスリープ解除する
            FullAwake();
            continue;
        }
        // クレードルエラー通知（Alert より低優先度）
        else if ( pEventHolder == &m_CradleErrorEventHolder )
        {
            NotifyCradleError();
            continue;
        }
        // クレードル USB HUB VBUS 検出開始（Alert より低優先度）
        else if ( pEventHolder == &m_CradleUsbHubTimerEventHolder )
        {
            DetectUsbHubInCradle();
            continue;
        }
        // USB 端子の高電圧解除通知（Alert より低優先度）
        else if ( pEventHolder == &m_OvpRecoverTimerEventHolder )
        {
            USBPD_DBG_LOG("ExecuteThread: Stop OvpRecoverTimer\n");
            StopRecoveringOverVoltage();
            continue;
        }

        // Alert イベント処理
        int alertIndex = static_cast<int>( os::GetMultiWaitHolderUserData( pEventHolder ) );
        ExecuteAlertEvents( alertIndex );

        // プラグ挿抜時の FullAwake 準備完了通知
        SignalFullAwakeReadyEvent( alertIndex );
    }

    // PD コン終了処理
    FinalizePdc();
}

void Driver::PrintPdcBaseStatuses() NN_NOEXCEPT
{
    Result result;

    result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Status1 = %04x\n", m_Status1);
    result = ReceiveCommand1( &m_Status2, L1CommandSize_Status2, L1Command_Status2 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Status2 = %04x\n", m_Status2);
}

void Driver::ExecuteAlertEvents( int alertIndex ) NN_NOEXCEPT
{
    Result result;

    USBPD_DBG_LOG("ExecuteThread: Receive Alert = %d\n", alertIndex);

    m_pAlertEvent[alertIndex]->Clear();
    bool isInsertPreCheck = false;
    Bit16 forceNotice = 0;
    switch ( alertIndex )
    {
    case AlertStatus::Error::Pos:
        ClearStatusBase();
        // 実際発生するのは OVP（Over Voltage Protection）のみ
        StartRecoveringOverVoltage();
        break;

    case AlertStatus::HardReset::Pos:
        // PDC の Hard Reset 発生時に Data Role は元に戻り AcOk(VBUS) が発生するが（IAAA-3447）、
        // クレードル MCU は Hard Reset を無視して HPD Attention VDM をリトライするだけ（SIGLO-49238）
        RecoverHardReset();
        break;

    case AlertStatus::PlugInsertRemove::Pos:
        m_CradleAwakeSuspended = false;
        m_IsPdcResetToAcOkTimerEnabled = false;
        ClearStatusBase();
        PrintPlugStatus();
        // OTG モード中に DFP で無かった時は一旦 VBUS を入力に戻す
        DeactivateOnOtgRemove();
        NN_FALL_THROUGH;

    case AlertStatus::NewPdos::Pos:
        // プラグ挿入処理
        ActivateOnPlugInsert( &isInsertPreCheck, &forceNotice );
        break;

    case AlertStatus::NewContractConsumer::Pos:
        if ( m_IsActive && !m_pAlertEvent[AlertStatus::NewPdos::Pos]->TryWait() && m_NewContractEvent.TryWait() )
        {
            // 同期ズレした NewContract イベントをクリア
            m_NewContractEvent.Clear();
            // Google 製 USB Type-C 充電器（非 USB PD）へ対応（SIGLO-67934）
            GetCurrentPower( nullptr, nullptr, nullptr );
            forceNotice = Notice::ActiveNotice::Mask;
        }
        break;

    case AlertStatus::AlternateModeEvent::Pos:
        result = ReceiveCommand1( &m_DisplayPortStatus, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        USBPD_DBG_LOG("DisplayPortStatus = %04x\n", m_DisplayPortStatus);
        break;

    case AlertStatus::Reserved15::Pos:
        USBPD_ABORT("Unexpected alert status!");
        break;

    default:
        break;
    }

    // IPC 用ステータスのアップデート
    UpdateStatus( isInsertPreCheck, forceNotice );
}

void Driver::ClearStatusBase() NN_NOEXCEPT
{
    if ( m_IpcError != StatusError_OverVoltage )
    {
        // StatusError_OverVoltage 状態での PDC の HW リセット後の Insert アラートによって
        // StatusError_OverVoltage が解除されないための対策
        m_IpcError = StatusError_None;
    }
    m_IsActive = false;
    m_IsDisplayPortAlternateMode = false;
    m_IsPowerShortage = false;
    m_CradleDeviceStatus.Clear();
    m_CradlePdcHFwVersion = 0;
    m_CradleMcuFwVersion = 0;
    m_CurrentPdo.Clear();
    m_CurrentRdo.Clear();
    m_SelectedPdo.Clear();
    m_SetRdo.Clear();
    m_IsSendRdoCompleted = false;
    std::memset( &m_IdentityVdm, 0, sizeof(m_IdentityVdm) );
}

void Driver::PrintPlugStatus() NN_NOEXCEPT
{
    const char* pPowerRoleString = m_Status1.Get<Status1::PowerRole>() ? "Source" : "Sink";
    const char* pDataRoleString = g_pDataRoleName[m_Status1.Get<Status1::DataRole>()];
    const char* pPlugString = m_Status1.Get<Status1::PlugInsertion>() ? "Inserted" : "Removed";
    NN_UNUSED(pPowerRoleString);
    NN_UNUSED(pDataRoleString);
    NN_UNUSED(pPlugString);
    USBPD_DBG_LOG("Status1:: %s / %s / %s\n", pPowerRoleString, pDataRoleString, pPlugString );
}

void Driver::DeactivateOnOtgRemove() NN_NOEXCEPT
{
    // OTG モード中に DFP で無かった時は一旦 VBUS を入力に戻す
    if ( m_IsChargerOtgMode && (m_Status1.Get<Status1::DataRole>() != Status1DataRole_Dfp || !m_Status1.Get<Status1::PlugInsertion>()) )
    {
        // プラグが抜けたことをできるだけ早く通知
        UpdateStatus( false );
        // VBUS を入力へ戻す
        InputVBusPower();
        USBPD_DBG_LOG("OnTheGo is finished\n");
    }
}

void Driver::ActivateOnPlugInsert( bool* pIsInsertPreCheck, Bit16* pForceNotice ) NN_NOEXCEPT
{
    *pIsInsertPreCheck = false;
    *pForceNotice = 0;

    if ( !m_Status1.Get<Status1::PlugInsertion>() )
    {
        return;
    }

    if ( m_Status1.Get<Status1::PowerRole>() == Status1PowerRole_Sink )
    {
        // Sink シーケンス（Copper / クレードル / AC アダプタ）
        if ( !EnableOnCopper() && !EnableCradle() )
        {
            return;
        }
        *pIsInsertPreCheck = true;
        *pForceNotice = Notice::ActiveNotice::Mask;
        EnableIsActive();
    }
    else
    {
        // Source シーケンス（OTG デバイス）
        if ( !EnableOnTheGo() )
        {
            return;
        }
        *pForceNotice = Notice::ActiveNotice::Mask;
        EnableIsActive();
    }
}

void Driver::EnableIsActive() NN_NOEXCEPT
{
    // USB 端子の高電圧解除通知
    StopRecoveringOverVoltage();
    m_IsActive = true;
}

bool Driver::IdentifyAcAdaptor( VdmProductId pid ) NN_NOEXCEPT
{
    switch ( pid )
    {
    case VdmAcAdaptorMtmId:
    case VdmAcAdaptorTbtId:
    case VdmAcAdaptorDltId:
    case VdmAcAdaptorFthId:
        return true;
        break;

    default:
        return false;
        break;
    }
}

void Driver::AttachSessionObject( Session* pSession ) NN_NOEXCEPT
{
    m_pSession = pSession;
}

void Driver::DetachSessionObject() NN_NOEXCEPT
{
    m_pSession = nullptr;
}

void Driver::SignalNoticeEvent( int sessionId ) NN_NOEXCEPT
{
    if ( m_pSession )
    {
        m_pSession->NoticeEventCallback( sessionId );
    }
}

void Driver::SignalNoticeEventAll() NN_NOEXCEPT
{
    if ( m_pSession )
    {
        Bit32* p = reinterpret_cast<Bit32*>(&m_IpcStatus);
        NN_UNUSED(p);
        USBPD_DBG_LOG("Status = %08x %08x %08x %08x %08x\n", p[0], p[1], p[2], p[3], p[4]);
        m_pSession->NoticeEventCallback();
    }
}

void Driver::UpdateAlert() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_AlertMutex );
}

void Driver::UpdateStatus( bool isInsertCheck, Bit16 noticeMask ) NN_NOEXCEPT
{
    Status status;
    status.Clear();
    util::BitPack32* pData = &status.m_Data;

//    USBPD_DBG_LOG("UpdateStatus( %d %04x )\n", isInsertCheck, noticeMask);

    // クリア後に再度挿抜イベントが発生していないことを確認
    if ( isInsertCheck &&
         ( m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->TryWait() ||
           !m_Status1.Get<Status1::PlugInsertion>()
         )
       )
    {
//        USBPD_DBG_LOG("m_IsActive final clear!\n");
        m_IsActive = false;
    }
    pData->Set<Status::Active>( m_IsActive );
    pData->Set<Status::Error>( m_IpcError );

    // Active 時の Status を取得
    if ( m_IsPdcBootFatalEnabled || m_ResetResult.IsSuccess() )
    {
        GetStatusOnActive( &status );
    }
    else if ( noticeMask & Notice::ActiveNotice::Mask )
    {
        GetStatusOnFakeActive( &status );
    }

    std::lock_guard<nn::os::Mutex> lock(m_IpcStatusMutex);
    bool isActiveLast = m_IpcStatus.IsActive();
    m_IpcStatus = status;
    // NoticeEvent 通知
    bool isNotice = false;
    if ( status.IsActive() != isActiveLast )
    {
        isNotice = true;
        for ( int i = 0; i < MaxSessions; i++ )
        {
            m_IpcNotice[i].Set<Notice::ActiveNotice>( true );
        }
    }
    if ( noticeMask )
    {
        isNotice = true;
        for ( int i = 0; i < MaxSessions; i++ )
        {
            m_IpcNotice[i].SetAllBitOn( noticeMask );
        }
    }
    if ( isNotice )
    {
        SignalNoticeEventAll();
    }
}

bool Driver::GetStatusOnActive( Status* pStatus ) NN_NOEXCEPT
{
    if ( !m_IsActive )
    {
        return false;
    }

    util::BitPack32* pData = &pStatus->m_Data;

    pData->Set<Status::DisplayPortAlternateMode>( m_IsDisplayPortAlternateMode );

    pData->Set<Status::PlugOrientation>( m_Status1.Get<Status1::PlugOrientation>() );
    // PlugInsertion==false 時は以下不定値
    pData->Set<Status::DataRole>( m_Status1.Get<Status1::DataRole>() );
    pData->Set<Status::PowerRole>( m_Status1.Get<Status1::PowerRole>() );
    pData->Set<Status::AccessoryMode>( m_Status2.Get<Status2::AccessoryMode>() );
    if ( m_Status1.Get<Status1::DataRole>() != Status1DataRole_AccessoryMode )
    {
        // Accessory Mode 時は不定値
        pData->Set<Status::ElectronicallyMarkedCable>( m_Status2.Get<Status2::ElectronicallyMarkedCable>() );
    }
    VdmVendorId vid = m_IdentityVdm.m_IdHeader.Get<VdmIdHeader::VendorId>();
    VdmProductId pid = m_IdentityVdm.m_ProductVdo.Get<ProductVdo::ProductId>();
    Bit8 deviceType = StatusDeviceType_Unknown;
    if ( vid == VdmNintendoId )
    {
        switch ( pid )
        {
        case VdmCradleId:
            // クレードルとリレーボックスの DeviceType 変換
            deviceType = m_CradleDeviceStatus.Get<VdmDeviceState::DeviceType>() + StatusDeviceType_Cradle;
            break;

        case VdmTableDockId:
            deviceType = StatusDeviceType_TableDock;
            break;

        default:
            if ( IdentifyAcAdaptor( pid ) )
            {
                deviceType = StatusDeviceType_AcAdaptor;
            }
            break;
        }
    }
    if ( m_HostPdcFwRevision == HostPdcFwRevisionEp1 && m_CradlePdcHFwVersion == CradlePdcHFwRevisionEp1 )
    {
        // クレードル EP1 以前は Product ID が取れないため FW バージョンで代用
        deviceType = m_CradleDeviceStatus.Get<VdmDeviceState::DeviceType>() + StatusDeviceType_Cradle;
    }
    pData->Set<Status::VdmDeviceType>( deviceType );
    bool isCradleOrRelayBox = deviceType == StatusDeviceType_Cradle || deviceType == StatusDeviceType_RelayBox;
    if ( deviceType == StatusDeviceType_RelayBox )
    {
        // リレーボックスではデバッグログ出力のため UFP と詐称
        // （同様の理由で Copper では UFP から変更せずに PC 直接続と同じ状態にしている）
        pData->Set<Status::DataRole>( StatusDataRole_Ufp );
    }
    if ( IsCradleFamily( static_cast<StatusDeviceType>(deviceType) ) )
    {
        pData->Set<Status::VdmFatalError>( m_CradleDeviceStatus.Get<VdmDeviceState::FatalError>() );
        pData->Set<Status::VdmRelayBoxUsbPower>( m_CradleDeviceStatus.Get<VdmDeviceState::RelayBoxUsbPower>() );
        Bit8 vdmWarning = m_CradleDeviceStatus.Get<VdmDeviceState::PowerDeriveryWarning>();
        if ( vdmWarning == VdmPowerDeriveryWarning_PowerShortage ||
             // DeviceState で PowerShortage が取れない時の対策（クレードル FinalDP から修正）
             (isCradleOrRelayBox && vdmWarning == VdmPowerDeriveryWarning_None && m_IsPowerShortage)
           )
        {
            pData->Set<Status::VdmCradlePowerShortage>( true );
        }
        if ( vdmWarning == VdmPowerDeriveryWarning_AnotherVendor || pData->Get<Status::VdmCradlePowerShortage>() )
        {
            pData->Set<Status::VdmCradleWithUnofficialAcAdaptor>( true );
        }
    }
    else if ( deviceType == StatusDeviceType_Unknown )
    {
        pStatus->m_VendorId = vid;
        pStatus->m_ProductId = pid;
    }
    pStatus->m_CurrentPdo = m_CurrentPdo;
    pStatus->m_CurrentRdo = m_CurrentRdo;

    return true;
}

bool Driver::GetStatusOnFakeActive( Status* pStatus ) NN_NOEXCEPT
{
    pStatus->m_Data.Set<Status::Active>( true );
    pStatus->m_Data.Set<Status::PowerRole>( nn::usb::pd::StatusPowerRole_Sink );
    pStatus->m_Data.Set<Status::DataRole>( nn::usb::pd::StatusDataRole_Ufp );
    pStatus->m_CurrentPdo.Set<pd::Pdo::Voltage>( 5 * 1000 / PdoMilliVoltUnit );
    pStatus->m_CurrentPdo.Set<pd::Pdo::MaximumCurrent>( 0 );
    return true;
}

Result Driver::GetStatus( Status* pStatus, int sessionId ) NN_NOEXCEPT
{
    Result result;
    result = GetStatusCore( pStatus );
//    USBPD_DBG_LOG("GetStatus = %04x\n", m_IsActive);
    if ( sessionId == m_PowerRequestNoticeSessionId )
    {
        pStatus->m_Data.Set<Status::Request>( m_IpcRequest );
    }
    return result;
}

Result Driver::GetStatusCore( Status* pStatus ) NN_NOEXCEPT
{
    if ( !pStatus )
    {
        return ResultInvalidAddress();
    }
    std::lock_guard<nn::os::Mutex> lock(m_IpcStatusMutex);
    *pStatus = m_IpcStatus;
    return ResultSuccess();
}

Result Driver::GetNotice( Notice* pNotice, int sessionId ) NN_NOEXCEPT
{
    if ( !pNotice )
    {
        return ResultInvalidAddress();
    }
    std::lock_guard<nn::os::Mutex> lock(m_IpcStatusMutex);
    *pNotice = m_IpcNotice[sessionId];
    m_IpcNotice[sessionId].Clear();
    return ResultSuccess();
}

} // detail
} // driver
} // pd
} // usb
} // nn
