﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {


const XhciDriver::DmaTrb XhciDriver::ZeroedDmaTrb = {{0}};

void* XhciDriver::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, XhciDriver::ObjectMemAlignmentSize, "XhciDriver");
}

void XhciDriver::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "XhciDriver");
}

Result XhciDriver::Initialize( )
{
    Result result = ResultSuccess();

    do{
        // Announce controller version
        m_ControllerVersion = NN_USB_GET_FIELD32(HcCapBaseVersion, CapRegister.Read32(HcCapBaseOffset));
        NN_USB_LOG_INFO("XhciDriver: Controller version %d.%d.\n",
                        m_ControllerVersion >> 8, m_ControllerVersion & 0xff);

        // Resolve XHCI operational registers
        size_t opRegOffset = NN_USB_GET_FIELD32(HcCapBaseLength, CapRegister.Read32(HcCapBaseOffset));
        OpRegister.Initialize(&CapRegister, opRegOffset, 0x100);

        // Resolve XHCI runtime registers
        size_t rtRegOffset = CapRegister.Read32(HcCapRuntimeRegistersOffset);
        RtRegister.Initialize(&CapRegister, rtRegOffset, 0x100);

        // Resolve XHCI doorbell registers
        size_t dbRegOffset = CapRegister.Read32(HcCapDoorbellRegistersOffset);
        DbRegister.Initialize(&CapRegister, dbRegOffset, sizeof(uint32_t) * 256);

        // Resolve context size
        m_IsContext64b = NN_USB_GET_FIELD32(HccParams64ByteContext, CapRegister.Read32(HccParamsOffset)) ? true : false;

        // Resolve supported slot count potentially limited by driver and/or controller
        m_MaxDeviceSlotCount = NN_USB_GET_FIELD32(HcsParams1MaxSlots, CapRegister.Read32(HcsParams1Offset));
        m_MaxDeviceSlotCount = (m_MaxDeviceSlotCount > LimitMaxDeviceSlotCount) ? LimitMaxDeviceSlotCount : m_MaxDeviceSlotCount;

        // Resolve number of root hub ports
        m_PortCount = NN_USB_GET_FIELD32(HcsParams1MaxPorts, CapRegister.Read32(HcsParams1Offset));

        // Resolve scratchpad buffers software must allocate for HC, ensure we allow it
        m_ScratchPadBufferCount =
            (NN_USB_GET_FIELD32(HcsParams2MaxSpbHi, CapRegister.Read32(HcsParams2Offset)) << 5) |
            NN_USB_GET_FIELD32(HcsParams2MaxSpbLo, CapRegister.Read32(HcsParams2Offset));
        if(m_ScratchPadBufferCount > HsLimitXhciMaxScratchpadBufferCount)
        {
            NN_USB_LOG_ERROR("xHCI ScratchPad Buffer Count = 0x%x, but we support only %d\n",
                             m_ScratchPadBufferCount,
                             HsLimitXhciMaxScratchpadBufferCount);
            result = ResultImplementationLimit();
            break;
        }

        // Resolve event table size
        m_EventRingSegmentTableEntryCount = 1 << NN_USB_GET_FIELD32(HcsParams2ErstMax, CapRegister.Read32(HcsParams2Offset));
        m_EventRingSegmentTableEntryCount = (m_EventRingSegmentTableEntryCount > LimitMaxEventRingSegmentTableEntryCount) ?
            LimitMaxEventRingSegmentTableEntryCount : m_EventRingSegmentTableEntryCount;

        // Reset
        NN_USB_BREAK_UPON_ERROR(ResetController());

        // Resolve controller page size
        m_PageSize = 1 << (12 - 1 +  NN_USB_GET_FIELD32(OpRegPagesize, OpRegister.Read32(OpRegPagesizeOffset)));
        if(m_PageSize != HsLimitXhciPageSize)
        {
            NN_USB_LOG_ERROR("xHCI PAGESIZE = 0x%x, but we support only 0x%d\n", m_PageSize, HsLimitXhciPageSize);
            result = ResultImplementationLimit();
            break;
        }

        // setup device slots
        OpRegister.SetField32(OpRegConfigOffset, NN_USB_MAKE_MASK32(OpRegConfigMaxSlotsEn, m_MaxDeviceSlotCount));

        // DMA structures
        NN_USB_BREAK_UPON_ERROR(InitializeDriverMemory());

        // Start the controller
        NN_USB_BREAK_UPON_ERROR(StartController());
    } while (false);

    // If error, stop the hardware
    if (result.IsFailure())
    {
        OpRegister.Write32(OpRegUsbCmdOffset, 0);
    }

    return result;
}

Result XhciDriver::Finalize()
{
    Result result = ResultSuccess();

    do
    {
        // Stop, ignore the error as falcon could die
        (void)(HaltController());

        // Clear any remaining pending ISR
        OpRegister.Write32(OpRegStatusOffset, OpRegister.Read32(OpRegStatusOffset));

        // Disable interrupts in runtime registers
        RtRegister.ClearField32(RtIntReg0ImanOffset, RtIntReg0ImanIeMask);

        // Cleanup core driver memory
        NN_USB_BREAK_UPON_ERROR(FinalizeDriverMemory());

        // Zero supported device slots
        OpRegister.SetField32(OpRegConfigOffset, NN_USB_MAKE_MASK32(OpRegConfigMaxSlotsEn, 0));

        // Finalize register access objects
        DbRegister.Finalize();
        RtRegister.Finalize();
        OpRegister.Finalize();


    }while(false);

    return result;
}

// Ref [xHCI r1.1] 4.19.6 Port Test Modes
void XhciDriver::SetTestMode(uint32_t port, TestMode mode)
{
    /*
     * For TX1 / Mariko:
     *  - Port 1~4: USB 3.0
     *  - Port 5~8: USB 2.0
     * TestMode applies to only USB 2.0 Ports.
     */
    NN_SDK_REQUIRES_MINMAX(port, 0, 3);

    HubPortNumber portNumber = 5 + port;

    uint32_t setting;

    switch (mode)
    {
    case TestMode_J:
        setting = 1;
        break;

    case TestMode_K:
        setting = 2;
        break;

    case TestMode_Se0Nak:
        setting = 3;
        break;

    case TestMode_Packet:
        setting = 4;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    setting <<= OpRegPortPmscTestModeShift;

    // Stop the xHC first, so we can reset it next
    NN_USB_ABORT_UNLESS_SUCCESS(HaltController());

    // Reset the xHC so all device slots are disabled.
    // Undefined behavior if xHC is not already halted
    NN_USB_ABORT_UNLESS_SUCCESS(ResetController());

    // Start the xHC, so PP=0 can be set.
    NN_USB_ABORT_UNLESS_SUCCESS(StartController());

    // Disable all the USB 2.0 ports (Set PP=0)
    for (HubPortNumber i = 5; i <= 8; i++)
    {
        size_t portScOffset = OpRegPortScOffset + (OpRegPortScSize * (i - 1));
        OpRegister.ClearField32(portScOffset, OpRegPortScPpMask);
    }

    // Stop the xHC
    NN_USB_ABORT_UNLESS_SUCCESS(HaltController());

    // Set Test Mode
    size_t portPmscOffset = OpRegPortPmscOffset + (OpRegPortPmscSize * (portNumber - 1));
    OpRegister.SetField32(portPmscOffset, OpRegPortPmscTestModeMask, setting);
}

Result XhciDriver::CreateDeviceContextAsync(HubPortNumber *portHierarchy,
                                            UsbDeviceSpeed speed,
                                            uint8_t ttDevice, HubPortNumber ttPort,
                                            CreateDeviceContextCallback callback,
                                            void* callbackContext)
{
    Result result = ResultSuccess();

    // Allocate device context
    DeviceContext *pXdc = reinterpret_cast<DeviceContext*>(
        detail::UsbMemoryCalloc(sizeof(DeviceContext), "DeviceContext")
    );
    NN_USB_EXPECT(pXdc != nullptr, ResultMemAllocFailure());

    // Setup device context with passed parameters
    memcpy(pXdc->portHierarchy, portHierarchy, sizeof(pXdc->portHierarchy));

    pXdc->pHc                        = this;
    pXdc->speed                      = speed;
    pXdc->ttParentDevice             = ttDevice;
    pXdc->ttParentPort               = ttPort;
    pXdc->completion.callback.create = callback;
    pXdc->completion.context         = callbackContext;

    // Default state
    pXdc->slotId          = 0;
    pXdc->state           = DeviceContextState_Disabled;

    do
    {
        // Allocate DMA buffers
        NN_USB_BREAK_UPON_ERROR(
            InitializeDmaBuffer(
                &pXdc->outputBuffer, m_PageSize, "DeviceContext.outputBuffer"
            )
        );
        NN_USB_BREAK_UPON_ERROR(
            InitializeDmaBuffer(
                &pXdc->inputBuffer, m_PageSize, "DeviceContext.inputBuffer"
            )
        );
        NN_USB_BREAK_UPON_ERROR(
            DoEnableSlot(pXdc)
        );
    } while(false);

    if (result.IsFailure())
    {
        DestroyDeviceContext(pXdc);
    }

    return result;
}

Result XhciDriver::DestroyDeviceContextAsync(HostControllerDriverDeviceContext* pDeviceContext,
                                             DestroyDeviceContextCallback callback,
                                             void* callbackContext)
{
    Result result = ResultSuccess();
    DeviceContext* pXdc = static_cast<DeviceContext*>(pDeviceContext);

    do
    {
        // Check state - is this allowed?
        NN_USB_BREAK_UNLESS((pXdc->state >= DeviceContextState_Enabled)
                            && (pXdc->state <= DeviceContextState_Configured),
                            ResultInternalStateError());

        pXdc->completion.callback.destroy = callback;
        pXdc->completion.context          = callbackContext;

        NN_USB_BREAK_UPON_ERROR(DoDisableSlot(pXdc->slotId));
    } while (false);

    if (ResultHcCrash::Includes(result))
    {
        // fake the completion
        result = ResultSuccess();
        if (callback)
        {
            callback(m_pHs, result, callbackContext);
        }

        pXdc->state = DeviceContextState_Disabled;
        DestroyDeviceContext(pXdc);
    }

    return result;
}

Result XhciDriver::AddressDeviceAsync(HostControllerDriverDeviceContext* pDeviceContext,
                                      AddressDeviceCallback callback, void* context)
{
    Result result = ResultSuccess();
    DeviceContext* pXdc = static_cast<DeviceContext*>(pDeviceContext);
    XhciEndpoint*  pXep = GetXhciEndpoint(pXdc, 1, true);

    do
    {
        NN_USB_BREAK_UNLESS(pXdc->state == DeviceContextState_Default,
                            ResultInternalStateError());

        pXdc->completion.callback.address  = callback;
        pXdc->completion.context           = context;

        // Reset the ring since TR dequeue pointer will be resetted
        ResetTransferRing(&pXep->transferRing, 1);

        // Update input control context
        DmaInputControlContext* pDmaInCtrlCtx =
            reinterpret_cast<DmaInputControlContext*>(
                pXdc->inputBuffer.u.uintptr + InputCtxControlOffset
            );
        pDmaInCtrlCtx->dw[1] = 3;
        pDmaInCtrlCtx->dw[0] = 0;
        nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);

        /*
         * Ref [xHCI r1.1] 4.6.5
         *
         * With BSR=0, transfer the device from Default to the Addressed state.
         */
        NN_USB_BREAK_UPON_ERROR(DoAddressDevice(pXdc->slotId, pXdc->inputBuffer.ioVa, 0));
    } while (false);

    return result;
}

/*
 * Ref [xHCI r.1. 4.5.2 Slot Context Initialization
 *
 * This is only called by hub. We update the input slot context so next
 * ConfigureEndpoint command will feed the hub characteristics to the xHC.
 */
void XhciDriver::UpdateDeviceContext(HostControllerDriverDeviceContext* pDeviceContext)
{
    DeviceContext* pXdc = static_cast<DeviceContext*>(pDeviceContext);
    DmaSlotCtx* pDmaSlot = reinterpret_cast<DmaSlotCtx*>(
        pXdc->inputBuffer.u.uintptr + InputCtxSlotOffset
    );

    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Addressed);

    NN_USB_SET_FIELD32_2(SlotCtxDw0, pDmaSlot->dw[0],
                         Hub, 1,
                         Mtt, 0);
    NN_USB_SET_FIELD32_1(SlotCtxDw1, pDmaSlot->dw[1],
                         NumberofPorts, pXdc->hub.portCount);
    NN_USB_SET_FIELD32_1(SlotCtxDw2, pDmaSlot->dw[2],
                         TtThinkTime, pXdc->hub.ttThinkTime);

    nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);
}

Result XhciDriver::UpdateEp0Async(HostControllerDriverDeviceContext* pDeviceContext,
                                  uint8_t maxPacketSize0,
                                  UpdateEp0Callback callback,
                                  void *context)
{
    Result result = ResultSuccess();
    DeviceContext* pXdc = static_cast<DeviceContext*>(pDeviceContext);
    XhciEndpoint*  pXep = GetXhciEndpoint(pXdc, 1, true);

    pXdc->completion.callback.update   = callback;
    pXdc->completion.context           = context;

    // Update endpoint context
    pXep->pDmaInputContext->dw[1] =
        (pXep->pDmaInputContext->dw[1] & ~EndpointCtxDw1MaxPacketSizeMask) |
        NN_USB_MAKE_MASK32(EndpointCtxDw1MaxPacketSize, maxPacketSize0);
    nn::dd::FlushDataCache(pXep->pDmaInputContext, EndpointCtxSize);

    // Update input control context
    DmaInputControlContext* pDmaInCtrlCtx =
        reinterpret_cast<DmaInputControlContext*>(pXdc->inputBuffer.u.uintptr + InputCtxControlOffset);
    pDmaInCtrlCtx->dw[1] = 1 << 1;
    pDmaInCtrlCtx->dw[0] = 0;
    nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);

    // Send evaluate context
    NN_USB_RETURN_UPON_ERROR(
        DoEvaluateContext(pXdc->slotId, pXdc->inputBuffer.ioVa)
    );

    pXep->isEnabled = true;

    return result;
}

/*
 * Ref [xHCI r1.1] 4.8.3, Figure 13
 *
 * Transition an Endpoint from Disabled to Running
 */
Result XhciDriver::OpenEpAsync(HostControllerDriverEndpoint *pHcEp)
{
    Result result = ResultSuccess();
    XhciEndpoint* pXep = nullptr;
    DeviceContext* pXdc = nullptr;

    do
    {
        HostEndpoint *pHep = nullptr;

        /*
         * Resolve data structures
         */
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(
            pXdc = static_cast<DeviceContext*>(pHcEp->GetDeviceContext()),
            ResultEndpointStateInvalid()
        );
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(
            pHep = pHcEp->GetHostEndpoint(), ResultEndpointStateInvalid()
        );
        pXep = GetXhciEndpoint(pXdc, pHep->epDir, pHep->epType, pHep->epNumber, false);

        /*
         * Underlying control endpoint is owned by XHCI driver
         */
        if (pHep->epType == UsbEndpointType_Control)
        {
            if(pXep == nullptr)
            {
                return ResultEndpointStateInvalid();
            }

            pXep->pHcEp = pHcEp;
            pHcEp->SetDriverEndpoint(pXep);

            pXep->isEnabled = true;
            pHcEp->ReportOpenComplete(result);
            return result;
        }

        /*
         * User endpoints should never be open at this point
         */
        NN_USB_BREAK_UNLESS(pXep == nullptr, ResultResourceBusy());

        /*
         * Allocate and configure XhciEndpoint
         */
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(pXep = reinterpret_cast<XhciEndpoint*>
                                         (detail::UsbMemoryCalloc(sizeof(XhciEndpoint), "XhciEndpoint")));
        NN_USB_BREAK_UPON_ERROR(SetXhciEndpoint(pXdc, pHep->epDir, pHep->epType, pHep->epNumber, pXep));
        pXep->pDeviceContext = pXdc;
        pXep->pHcEp = pHcEp;
        pHcEp->SetDriverEndpoint(pXep);
        pXep->isEnabled = true;
        pXep->pDmaInputContext  = GetEndpointInputContext (pXdc, pHep->epDir, pHep->epNumber);
        pXep->pDmaOutputContext = GetEndpointOutputContext(pXdc, pHep->epDir, pHep->epNumber);

        pXep->dci = ((pHep->epNumber - 1) << 1) + ((pHep->epDir == UsbEndpointDirection_ToDevice) ? 2 : 3);

        NN_USB_BREAK_UPON_ERROR(
            InitializeTransferRing(&pXep->transferRing,
                                   4096 / 16 - 1,
                                   1)
        );

        /*
         * Resolve endpoint characteristics
         */
        uint32_t maxPacketSize = GetMaxPacketSize(pXep);
        uint32_t maxBurstSize  = GetMaxBurstSize(pXep);

        /*
         * Setup input endpoint context
         */
        memset(pXep->pDmaInputContext, 0, EndpointCtxSize);
        nn::dd::FlushDataCache(pXep->pDmaInputContext, EndpointCtxSize);

        pXep->pDmaInputContext->dw[0] =
            NN_USB_MAKE_MASK32(EndpointCtxDw0State,              EndpointCtxDw0StateDisabled)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Mult,             GetEndpointMult(pHcEp))
            | NN_USB_MAKE_MASK32(EndpointCtxDw0MaxPStreams,      0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Lsa,              0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Interval,         GetEndpointInterval(pHcEp));

        pXep->pDmaInputContext->dw[1] =
            NN_USB_MAKE_MASK32(EndpointCtxDw1Cerr,               (pHep->epType == UsbEndpointType_Isoc) ? 0 : 3)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1EpType,           GetEndpointCtxDw1EpType(pHep))
            | NN_USB_MAKE_MASK32(EndpointCtxDw1Hid,              0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1MaxBurstSize,     maxBurstSize)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1MaxPacketSize,    maxPacketSize);

        pXep->pDmaInputContext->qw[1] =
            pXep->transferRing.dmaTrbList.ioVa
            | NN_USB_MAKE_MASK32(EndpointCtxQw1Dcs,              pXep->transferRing.ccs);

        pXep->pDmaInputContext->dw[4] =
            NN_USB_MAKE_MASK32(EndpointCtxDw4MaxEsitPayloadLo, GetMaxEsitPayload(pXep))  |
            NN_USB_MAKE_MASK32(EndpointCtxDw4AverageTrbLength, GetAverageTrbLength(pXep));
        nn::dd::FlushDataCache(pXep->pDmaInputContext, EndpointCtxSize);

        /*
         * Update input slot context entry count
         */
        DmaSlotCtx* pDmaSlot = reinterpret_cast<DmaSlotCtx*>(
            pXdc->inputBuffer.u.uintptr + InputCtxSlotOffset
        );
        NN_USB_SET_FIELD32_1(SlotCtxDw0, pDmaSlot->dw[0],
                             ContextEntries, GetSlotContextEntryCount(pXdc));

        /*
         * Ref [xHCI r1.1] 4.6.6
         *
         *   "The Add Context flag A1 and Drop Context flags D0 and D1 of the
         *    Input Control Context (in the Input Context) shall be cleared to
         *    ‘0’... A0 shall be set to ‘1’..."
         *
         * Update input control context to add this endpoint
         */
        DmaInputControlContext* pDmaInCtrlCtx =
            reinterpret_cast<DmaInputControlContext*>(pXdc->inputBuffer.u.uintptr + InputCtxControlOffset);
        pDmaInCtrlCtx->dw[0] = 0;
        pDmaInCtrlCtx->dw[1] = MakeInputContextControlMask(pHep) | InputControlCtxDw1AddSlotMask;

        nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);

        /*
         * Send configure endpoint command
         */
        NN_USB_BREAK_UPON_ERROR(
            DoConfigureEndpoint(pXdc->slotId, pXdc->inputBuffer.ioVa, 0, pXep)
        );
    }while(false);

    if(result.IsFailure() && (pXep != nullptr) && (pXdc != nullptr))
    {
        FinalizeEndpoint(pXdc, pXep);
    }

    return result;
} // NOLINT(impl/function_size)'

/*
 * Ref [xHCI r1.1] 4.8.3, Figure 13
 *
 * Transition an Endpoint from Error / Halted to Stopped
 */
Result XhciDriver::ResetEpAsync(HostControllerDriverEndpoint *pHcEp)
{
    Result result = ResultSuccess();

    HostEndpoint  *pHep;
    DeviceContext *pXdc;
    XhciEndpoint  *pXep;
    TransferRing  *pTr;

    // Resolve data structures
    pXdc = static_cast<DeviceContext*>(pHcEp->GetDeviceContext());
    NN_USB_EXPECT(pXdc != nullptr, ResultEndpointStateInvalid());

    pHep = pHcEp->GetHostEndpoint();
    NN_USB_EXPECT(pHep != nullptr, ResultEndpointStateInvalid());

    pXep = GetXhciEndpoint(pXdc, pHep->epDir, pHep->epType, pHep->epNumber, true);
    NN_USB_EXPECT(pXep != nullptr, ResultEndpointStateInvalid());

    pTr  = &pXep->transferRing;

    // Check current Ep State
    nn::dd::InvalidateDataCache(pXep->pDmaOutputContext, sizeof(DmaEndpointContext));
    char epState = NN_USB_GET_FIELD32(EndpointCtxDw0State,
                                      pXep->pDmaOutputContext->dw[0]);

    switch (epState)
    {
    case EndpointCtxDw0StateDisabled:
        NN_USB_LOG_ERROR("Trying to reset an Endpoint while it's in Disabled state!\n");
        result = ResultEndpointStateInvalid();
        break;

    case EndpointCtxDw0StateRunning:
        // Running: StopEndpoint -> SetTrDequeuePointer
        result = DoStopEndpoint(pXdc->slotId, pXep->dci, 0);
        break;

    case EndpointCtxDw0StateHalted:
        // Halted: ResetEndpoint -> SetTrDequeuePointer
        result = DoResetEndpoint(pXdc->slotId, pXep->dci, 0);
        break;

    case EndpointCtxDw0StateStopped:
    case EndpointCtxDw0StateError:
        // Stopped / Error: SetTrDequeuePointer
        ResetTransferRing(pTr, 1);
        result = DoSetTrDequeuePointer(pXdc->slotId, pXep->dci, pTr->dmaTrbList.ioVa, 1, 0, 0);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

/*
 * Ref [xHCI r1.1] 4.8.3, Figure 13
 *
 * Transition an Endpoint from non-Disabled to Disabled
 */
Result XhciDriver::CloseEpAsync(HostControllerDriverEndpoint *pHcEp)
{
    //HostEndpoint *pHep = pHcEp->GetHostEndpoint();
    Result result = ResultSuccess();

    do
    {
        HostEndpoint *pHep;
        DeviceContext* pXdc;
        XhciEndpoint* pXep;

        /*
         * Resolve data structures
         */
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(
            pXdc = static_cast<DeviceContext*>(pHcEp->GetDeviceContext()),
            ResultEndpointStateInvalid()
        );
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(
            pHep = pHcEp->GetHostEndpoint(),
            ResultEndpointStateInvalid()
        );
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(
            pXep = GetXhciEndpoint(pXdc, pHep->epDir, pHep->epType, pHep->epNumber, true),
            ResultEndpointStateInvalid()
        );
        pXep->isEnabled = false;

        /*
         * Underlying control endpoint is owned by XHCI driver
         */
        if (pHep->epType == UsbEndpointType_Control)
        {
            pHcEp->SetDriverEndpoint(nullptr);
            pXep->pHcEp = nullptr;
            pHcEp->ReportCloseComplete(ResultSuccess());
            return result;
        }

        /*
         * Update input slot context entry count
         */
        DmaSlotCtx* pDmaSlot = reinterpret_cast<DmaSlotCtx*>(pXdc->inputBuffer.u.uintptr + InputCtxSlotOffset);
        pDmaSlot->dw[0] = (pDmaSlot->dw[0] & ~SlotCtxDw0ContextEntriesMask) |
            NN_USB_MAKE_MASK32(SlotCtxDw0ContextEntries, GetSlotContextEntryCount(pXdc));

        /*
         * Update input control context to drop this endpoint
         */
        DmaInputControlContext* pDmaInCtrlCtx =
            reinterpret_cast<DmaInputControlContext*>(pXdc->inputBuffer.u.uintptr + InputCtxControlOffset);
        pDmaInCtrlCtx->dw[0] = MakeInputContextControlMask(pHep);
        pDmaInCtrlCtx->dw[1] = InputControlCtxDw1AddSlotMask;

        nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);

        /*
         * Send configure endpoint command
         */
        result = DoConfigureEndpoint(pXdc->slotId, pXdc->inputBuffer.ioVa, 0, pXep);
        if (ResultHcCrash::Includes(result))
        {
            /*
             * Falcon is dead, fake the completion, otherwise device tearing
             * down would wait indefinitely for the completion.
             */
            result = ResultSuccess();
            pXep->pHcEp->ReportCloseComplete(result);
            FinalizeEndpoint(pXdc, pXep);
        }

        NN_USB_BREAK_UPON_ERROR(result);
    }while(false);


    return result;
}

Result XhciDriver::SubmitUrbAsync(UsbRequestBlock *pUrb)
{
    HostControllerDriverEndpoint *pHcEp = pUrb->m_pHcEp;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();
    Result result = ResultSuccess();

    switch (pHep->epType)
    {
    case UsbEndpointType_Control:
        result = HandleControlUrbSubmit(pUrb);
        break;

    case UsbEndpointType_Bulk:
    case UsbEndpointType_Int:
        result = HandleNormalUrbSubmit(pUrb);
        break;

    case UsbEndpointType_Isoc:
        result = HandleIsochUrbSubmit(pUrb);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

Result XhciDriver::GetRootHubPortStatus(HubPortNumber portNumber, UsbHubPortStatus *pPs)
{
    Result result = ResultSuccess();
    pPs->wPortStatus = UsbPortStatus_Zero;
    pPs->wPortChange = UsbPortStatusChange_Zero;

    NN_ABORT_UNLESS_MINMAX(portNumber, 1, m_PortCount);

    size_t portScOffset = OpRegPortScOffset + (OpRegPortScSize * (portNumber - 1));
    uint32_t portScVal = OpRegister.Read32(portScOffset);

    /*
     * Get static port status
     */
    // Get port link state, which has same bit position
    pPs->wPortStatus = (portScVal & OpRegPortScPlsMask);

    switch(NN_USB_GET_FIELD32(OpRegPortScSpeed, portScVal))
    {
    case 3:
        pPs->wPortStatus |= UsbPortStatus_HighSpeed;
        break;
    case 2:
        pPs->wPortStatus |= UsbPortStatus_LowSpeed;
        break;
    case 1:
        // Full
        break;
    default:
        break;
    }
    if(portScVal & OpRegPortScCcsMask) pPs->wPortStatus |= UsbPortStatus_Connection;
    if(portScVal & OpRegPortScPedMask) pPs->wPortStatus |= UsbPortStatus_Enable;
    if(portScVal & OpRegPortScOcaMask) pPs->wPortStatus |= UsbPortStatus_Overcurrent;
    if(portScVal & OpRegPortScPrMask)  pPs->wPortStatus |= UsbPortStatus_Reset;
    if(portScVal & OpRegPortScPpMask)  pPs->wPortStatus |= UsbPortStatus_Power;

    /*
     * Get status change flags
     */
    if(portScVal & OpRegPortScCscMask) pPs->wPortChange |= UsbPortStatusChange_Connection;
    if(portScVal & OpRegPortScPecMask) pPs->wPortChange |= UsbPortStatusChange_Enable;
    if(portScVal & OpRegPortScOccMask) pPs->wPortChange |= UsbPortStatusChange_Overcurrent;
    if(portScVal & OpRegPortScWrcMask) pPs->wPortChange |= UsbPortStatusChange_BhReset;
    if(portScVal & OpRegPortScPrcMask) pPs->wPortChange |= UsbPortStatusChange_Reset;
    if(portScVal & OpRegPortScPlcMask) pPs->wPortChange |= UsbPortStatusChange_LinkState;
    if(portScVal & OpRegPortScCecMask) pPs->wPortChange |= UsbPortStatusChange_ConfigError;

    NN_USB_LOG_TRACE("XhciDriver::GetRootHubPortStatus(%d)\n",portNumber);
    NN_USB_LOG_TRACE("  wPortStatus = 0x%04x\n",pPs->wPortStatus);
    NN_USB_LOG_TRACE("  wPortChange = 0x%04x\n",pPs->wPortChange);

    return result;
}

Result XhciDriver::ClearRootHubPortStatus(HubPortNumber portNumber, uint16_t wPortChange)
{
    Result result = ResultSuccess();
    uint32_t portScWrite1ToClear = 0;

    NN_ABORT_UNLESS_MINMAX(portNumber, 1, m_PortCount);

    if (wPortChange & UsbPortStatusChange_Connection)  portScWrite1ToClear |= OpRegPortScCscMask;
    if (wPortChange & UsbPortStatusChange_Enable)      portScWrite1ToClear |= OpRegPortScPecMask;
    if (wPortChange & UsbPortStatusChange_Overcurrent) portScWrite1ToClear |= OpRegPortScOccMask;
    if (wPortChange & UsbPortStatusChange_BhReset)     portScWrite1ToClear |= OpRegPortScWrcMask;
    if (wPortChange & UsbPortStatusChange_Reset)       portScWrite1ToClear |= OpRegPortScPrcMask;
    if (wPortChange & UsbPortStatusChange_LinkState)   portScWrite1ToClear |= OpRegPortScPlcMask;
    if (wPortChange & UsbPortStatusChange_ConfigError) portScWrite1ToClear |= OpRegPortScCecMask;

    OpRegister.SetField32(OpRegPortScOffset + (OpRegPortScSize * (portNumber - 1)),
                          OpRegPortScCommandMask, portScWrite1ToClear);

    return result;
}

Result XhciDriver::SetRootHubPortAttribute(HubPortNumber portNumber, PortAttribute attribute, bool isTrue)
{
    Result result = ResultSuccess();
    size_t portScOffset = OpRegPortScOffset + (OpRegPortScSize * (portNumber - 1));

    NN_ABORT_UNLESS_MINMAX(portNumber, 1, m_PortCount);

    switch (attribute)
    {
    case PortAttribute_Reset:
        if (isTrue)
        {
            // trigger the reset
            OpRegister.SetField32(portScOffset, OpRegPortScCommandMask, OpRegPortScPrMask);
        }
        break;
    case PortAttribute_Suspend:
        break;
    case PortAttribute_Power:
        if (isTrue)
        {
            OpRegister.SetField32(portScOffset, OpRegPortScCommandMask, OpRegPortScPpMask);
        }
        else
        {
            OpRegister.ClearField32(portScOffset, OpRegPortScCommandMask | OpRegPortScPpMask);
        }

        /*
         * Ref [xHCI r1.1], 5.4.8
         *
         *   "If PPC = ‘1’ software is responsible for waiting 20 ms after asserting
         *   PP, before attempting to change the state of the port."
         *
         *   "After modifying PP, software shall read PP and confirm that it is reached
         *   its target state before modifying it again, undefined behavior may occur
         *   if this procedure is not followed."
         *
         * N.B. this doesn't affect TX1 since its PPC=0.
         */
        if (CapRegister.TestField32(HccParamsOffset, HccParamsPpcMask))
        {
            if (isTrue)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
            }

            if (OpRegister.TestField32(portScOffset, OpRegPortScPpMask) != isTrue)
            {
                NN_USB_LOG_WARN("Time out while %s root port %d power\n",
                                isTrue ? "setting" : "clearing",
                                portNumber);
                return ResultHardwareTimeout();
            }
        }

        break;

    case PortAttribute_Owner:
        break;
    case PortAttribute_Enable:
        if (!isTrue)
        {
            //  OpRegister.SetField32(portScOffset, OpRegPortScPrMask | OpRegPortScCommandMask, OpRegPortScPedMask);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

Result XhciDriver::ResetRootHub()
{
    Result result = ResultSuccess();


    return result;
}

UsbDeviceSpeed XhciDriver::GetPortSpeed(HubPortNumber portNumber)
{
    UsbDeviceSpeed speed = UsbDeviceSpeed_Invalid;

    if ((portNumber > 0) && (portNumber <= m_PortCount))
    {
        size_t portScOffset = OpRegPortScOffset + (OpRegPortScSize * (portNumber - 1));
        uint32_t portScVal = OpRegister.Read32(portScOffset);
        if(portScVal & OpRegPortScCcsMask)
        {
            /*
             * Ref [xHCI r1.1] 7.2.2.1.1, Table 157
             *
             * TODO: TX1 uses the default mapping (i.e. PSIC=0) so this is
             *       correct. We probably should actually parse the xHCI
             *       Extended Capabilities.
             */
            switch(NN_USB_GET_FIELD32(OpRegPortScSpeed, portScVal))
            {
            case 1:
                speed = UsbDeviceSpeed_Full;
                break;
            case 2:
                speed = UsbDeviceSpeed_Low;
                break;
            case 3:
                speed = UsbDeviceSpeed_High;
                break;
            case 4:
                speed = UsbDeviceSpeed_Super;
                break;
            default:
                break;
            }
        }
    }

    return speed;
}

void XhciDriver::Isr()
{
    uint32_t rtIman;
    uint32_t opStatus = OpRegister.Read32(OpRegStatusOffset);

    if(!(opStatus & OpRegStatusEintMask))
    {
        //NN_USB_LOG_ERROR("XhciDriver::Isr() Spurious.\n");
        //return;
    }

    if(opStatus & OpRegStatusFatalMask)
    {
        NN_USB_LOG_ERROR("XhciDriver::Isr() Fatal.\n");
        return;
    }

    // Clear op int status
    OpRegister.Write32(OpRegStatusOffset, opStatus);

    // Acknowledge PCI interrupt
    if((rtIman = RtRegister.Read32(RtIntReg0ImanOffset)) & RtIntReg0ImanIpMask)
    {
        RtRegister.Write32(RtIntReg0ImanOffset, rtIman);
    }

    if(opStatus & OpRegStatusPortMask)
    {
        NN_USB_LOG_TRACE("XhciDriver::Isr() - Port\n");
        HostControllerDriver::ReportPortEvent();
    }
    if(opStatus & OpRegStatusHaltMask)
    {
        NN_USB_LOG_ERROR("XhciDriver::Isr() - Host System Error\n");
    }
    if(opStatus & OpRegStatusFatalMask)
    {
        NN_USB_LOG_ERROR("XhciDriver::Isr() - Host System Error\n");
    }
    if(opStatus & OpRegStatusHceMask)
    {
        NN_USB_LOG_ERROR("XhciDriver::Isr() - Host Controller Error\n");
    }
    PollEventRing();
}

void XhciDriver::PollEventRing()
{
    int32_t trbCount;
    uint8_t ccs = m_EventCcs;
    int32_t index = m_EventIndex;

    nn::dd::InvalidateDataCache(m_pDmaEvents, sizeof(DmaTrb) * MaxEventTrbs);

    // For potentially all event TRBs
    for(trbCount=0; trbCount < MaxEventTrbs; trbCount++)
    {
        index = (m_EventIndex + trbCount) % MaxEventTrbs;
        DmaTrb* pTrb = m_pDmaEvents + index;
        uint32_t dw3 = pTrb->dw[3];
        uint8_t pcs = NN_USB_GET_FIELD32(EventTrbDw3C, dw3);

        if(ccs != pcs)
        {
            break;
        }

        switch(NN_USB_GET_FIELD32(EventTrbDw3TrbType, dw3))
        {
        case EventTrbTypeTransfer:
            HandleTransferCompletionEvent(pTrb);
            break;

        case EventTrbTypeCommandCompletion:
            HandleCommandCompletionEvent(pTrb);
            break;

        case EventTrbTypePortStatusChange:
        {
            int portId = NN_USB_GET_FIELD32(EventTrbDw0PortId, pTrb->dw[0]);

            size_t portScOffset = OpRegPortScOffset + (OpRegPortScSize * (portId - 1));
            uint32_t portScVal = OpRegister.Read32(portScOffset);

            NN_USB_LOG_INFO("### WPR  DR WOE WDE WCE CAS CEC PLC PRC OCC WRC "
                            "PEC CSC LWS PIC SPD  PP PLS  PR OCA PED CCS\n");
            NN_USB_LOG_INFO("%3d %3d %3d %3d %3d %3d %3d %3d %3d %3d %3d %3d "
                            "%3d %3d %3d %3d %3d %3d %3d %3d %3d %3d %3d\n",
                            portId,
                            (portScVal >> 31) & 0x01,
                            (portScVal >> 30) & 0x01,
                            (portScVal >> 27) & 0x01,
                            (portScVal >> 26) & 0x01,
                            (portScVal >> 25) & 0x01,
                            (portScVal >> 24) & 0x01,
                            (portScVal >> 23) & 0x01,
                            (portScVal >> 22) & 0x01,
                            (portScVal >> 21) & 0x01,
                            (portScVal >> 20) & 0x01,
                            (portScVal >> 19) & 0x01,
                            (portScVal >> 18) & 0x01,
                            (portScVal >> 17) & 0x01,
                            (portScVal >> 16) & 0x01,
                            (portScVal >> 14) & 0x03,
                            (portScVal >> 10) & 0x0f,
                            (portScVal >>  9) & 0x01,
                            (portScVal >>  5) & 0x0f,
                            (portScVal >>  4) & 0x01,
                            (portScVal >>  3) & 0x01,
                            (portScVal >>  1) & 0x01,
                            (portScVal >>  0) & 0x01);
            break;
        }

        case EventTrbTypeBandwidthRequest:
            NN_USB_LOG_INFO("EventTrbTypeBandwidthRequest\n");
            break;

        case EventTrbTypeDoorbell:
            NN_USB_LOG_INFO("EventTrbTypeDoorbell\n");
            break;

        case EventTrbTypeHostController:
            NN_USB_LOG_INFO("EventTrbTypeHostController\n");
            break;

        case EventTrbTypeDeviceNotification:
            NN_USB_LOG_INFO("EventTrbTypeDeviceNotification\n");
            break;

        case EventTrbTypeMfindexWrap:
            NN_USB_LOG_INFO("EventTrbTypeMfindexWrap\n");
            break;

        default:
            NN_USB_LOG_INFO("XhciDriver::PollEventRing() unknown TRB type %d\n",
                            NN_USB_GET_FIELD32(EventTrbDw3TrbType, dw3));
            break;
        }

        // toggle ccs when we wrap back
        if (index == (MaxEventTrbs - 1))
        {
            ccs ^= 1;
        }
    }

    m_EventCcs   = ccs;
    m_EventIndex = (m_EventIndex + trbCount) % MaxEventTrbs;

    if(trbCount != 0)
    {
        nn::dd::DeviceVirtualAddress ioVa = m_DmaEventsIoVa + sizeof(DmaTrb) * m_EventIndex;
        RtRegister.Write32(RtIntReg0ErdpLoOffset, static_cast<uint32_t>(ioVa) | RtIntReg0ErdpLoEhbMask);
        RtRegister.Write32(RtIntReg0ErdpHiOffset, static_cast<uint32_t>(ioVa >> 32));
    }
} // NOLINT(impl/function_size)

void XhciDriver::HandleTransferCompletionEvent(DmaTrb* pEventTrb)
{
    DmaTrb eventTrb = *pEventTrb; // make a safe coherent copy

    // Parse completion event
    size_t  residual       = NN_USB_GET_FIELD32(EventTrbDw2TransferLength,
                                                eventTrb.dw[2]);
    uint8_t completionCode = NN_USB_GET_FIELD32(EventTrbDw2CompletionCode,
                                                eventTrb.dw[2]);
    int     slotId         = NN_USB_GET_FIELD32(EventTrbDw3SlotId,
                                                eventTrb.dw[3]);
    int     dci            = NN_USB_GET_FIELD32(EventTrbDw3EndpointId,
                                                eventTrb.dw[3]);

    Result result = CheckTransferTrbCompletionCode(completionCode);

    if (ResultXhcIsocRingStarve::Includes(result))
    {
        // This event doesn't point to an active transfer TRB
        NN_USB_LOG_WARN("Ep<%d:%d>: Transfer Ring Underrun / Overrun\n", slotId, dci);
        return;
    }

    /*
     * Ref [xHCI r1.1] 4.10.2.6
     *
     * TODO: Perhaps check Host System Errors if the event TRB doesn't
     * contain valid data (invalid slot ID / DCI / TRB pointer / etc).
     */

    // Get the device this transfer belongs to
    DeviceContext* pXdc = GetXhciDeviceContext(slotId);
    if (pXdc == nullptr)
    {
        // This can be observed during MCCI connection exercise
        NN_USB_LOG_ERROR(
            "Ep<%d:%d>: Transfer Event TRB refers to invalid device slot!\n",
            slotId, dci
        );
        return;
    }

    // Get endpoint and transfer ring
    XhciEndpoint* pXep = GetXhciEndpoint(pXdc, dci, true);
    if (pXep == nullptr)
    {
        NN_USB_LOG_ERROR(
            "Ep<%d:%d>: Transfer Event TRB refers to invalid DCI!\n",
            slotId, dci
        );
        return;
    }

    TransferRing    *pTr      = &pXep->transferRing;
    UsbRequestBlock *pUrb     = nullptr;
    MetaTrb         *pMetaTrb = nullptr;

    // Find the TRB which generates this event
    for (int32_t count = 0; count < pTr->pendCount; count++)
    {
        int trbIndex;
        nn::dd::DeviceVirtualAddress trbIoVa;

        trbIndex = (pTr->pendIndex + count) % pTr->totalCount;
        pMetaTrb = pTr->pMetaTrbList + trbIndex;
        trbIoVa  = pTr->dmaTrbList.ioVa + (trbIndex * sizeof(DmaTrb));

        // Yeah, get you!
        if (trbIoVa == eventTrb.qw[0])
        {
            pUrb = pMetaTrb->pUrb;

            // Consume all TDs thus far
            int32_t consumedTds = count + 1;
            pTr->pendIndex  = (pTr->pendIndex + consumedTds) % pTr->totalCount;
            pTr->pendCount -= consumedTds;

            break;
        }
    }

    if (pUrb == nullptr)
    {
        NN_USB_LOG_ERROR(
            "Ep<%d:%d>: Transfer Event TRB refers to invalid Transfer TRB!"
            " Completion code %d (%s)\n",
            slotId, dci, completionCode, GetCompletionString(completionCode)
        );
        return;
    }

    auto& xfer = pUrb->m_Xfer[pUrb->m_CompletionCount];

    /*
     * Trust abnormal completion code from the last in a chain
     *
     * Note: For a stalled ctrl-in transfer, the sequence usually looks like
     *       SETUP -> DATA_IN (short) -> STATUS (stall). We shall update the
     *       transferred length upon short, but overall result of the transfer
     *       shall be stall.
     */
    if (result.IsFailure())
    {
        xfer.result = result;
    }

    // Update transferred size
    if (xfer.xferredSize == 0)
    {
        xfer.xferredSize = xfer.size - (pMetaTrb->remainingSize + residual);
    }

    if (completionCode != CompletionCodeSuccess   &&
        completionCode != CompletionCodeShortPacket)
    {
        NN_USB_LOG_WARN(
            "Ep<%d:%d>: Transfer of size %d(%d) completion code %d (%s)\n",
            slotId, dci,
            xfer.size, xfer.xferredSize,
            completionCode, GetCompletionString(completionCode)
        );
    }

    switch (completionCode)
    {
    case CompletionCodeSuccess:
        if (pMetaTrb->sequence == 0)
        {
            pUrb->Complete();
        }
        else
        {
            // BUG: This shouldn't happen
            NN_USB_LOG_ERROR("Premature TRB Completion Success!\n");
        }
        break;

    case CompletionCodeStopped:
    case CompletionCodeStoppedLengthInvalid:
    case CompletionCodeStoppedShortPacket:
        /*
         * Ref [xHCI r1.1] 4.6.9
         *
         * If the transaction is interrupted by Stop Endpoint Command, that
         * means the transaction has been timed out. The URB will be completed
         * when we finish resetting the endpoint.
         */
        break;

    case CompletionCodeShortPacket:
        /*
         * Ref [xHCI r1.1] 4.10.1.1.2
         *
         * Short Packet Event will propagate to the last TRB with IOC=1.
         */
        if (pMetaTrb->sequence != 0)
        {
            // Do nothing now
        }
        else
        {
            // Now we complete the URB
            pUrb->Complete();
        }
        break;

    case CompletionCodeTrbError:
        // BUG: This is definitely our bug
        NN_USB_ABORT("BUG: TRB Error Detected!\n");
        break;

    case CompletionCodeStallError:
    case CompletionCodeTransactError:
    case CompletionCodeBabbleError:
    case CompletionCodeDataBufferError:
    case CompletionCodeSplitTransactionError:
        /*
         * Ref [xHCI r1.1] 4.8.3, 4.10.2
         *
         * Error conditions will halt the endpoint, in which case the pipe
         * schedule will be stopped and no more Transfer Event TRB will be
         * triggerred. We should complete the URB and report the error
         * immediately.
         */
        pUrb->Complete();
        break;

    case CompletionCodeMissedServiceError:
        /*
         * Ref HandleIsochUrbSubmit(). We use 1 TRB per Isoch TD, so this is
         * the right time to complete the TD.
         */
        pUrb->Complete();
        break;

    case CompletionCodeIsochBufferOverrun:
        /*
         * Ref SIGLO-61378. We see this error when the WiiU USB Mic is
         * directly connected to root hub Type-C port.
         */
        pUrb->Complete();
        break;

    default:
        // FIXME: What should we do for other errors?
        NN_USB_LOG_ERROR(
            "Ep<%d:%d>: Transfer Event TRB with completion code %d (%s) "
            "unhandled!\n",
            slotId, dci,
            completionCode, GetCompletionString(completionCode)
        );
        break;
    }

    /*
     * Ref [xHCI r1.1] 4.6.8.1
     *
     * TODO: Maybe implement Soft Retry for USB Transaction Error?
     */
} // NOLINT(impl/function_size)

void XhciDriver::HandleCommandCompletionEvent(DmaTrb* pEventTrb)
{
    Command *pCmd;
    nn::dd::DeviceVirtualAddress pendCmdIoVa;
    int completionCode = NN_USB_GET_FIELD32(EventTrbDw2CompletionCode,
                                            pEventTrb->dw[2]);
    Result result = CheckCommandTrbCompletionCode(completionCode);

    if (ResultXhcCommandRingStopped::Includes(result))
    {
        // This event doesn't point to an active command TRB
        NN_USB_LOG_INFO("Command Ring Stopped!\n");
        return;
    }

    pCmd        = m_CommandRing + m_CommandPendIndex;
    pendCmdIoVa = m_DmaCommandRingIoVa + sizeof(DmaTrb) * m_CommandPendIndex;

    int commandCode = NN_USB_GET_FIELD32(CommandTrbDw3TrbType,
                                         pCmd->pCommandTrb->dw[3]);

    // Sanity checkes
    if (m_CommandPendCount == 0)
    {
        // BUG: It must be a software or hardware bug if we reach here
        NN_USB_LOG_ERROR("Receive Command Completion Event TRB while there is "
                         "no pending command!\n");
        return;
    }

    if (pendCmdIoVa != pEventTrb->qw[0])
    {
        // BUG: It must be a software or hardware bug if we reach here
        NN_USB_LOG_ERROR("Command Completion Event TRB refers to invalid "
                         "pending command!n");
        return;
    }

    // Now handle the Event TRB
    pCmd->status = result;
    if(pCmd->status.IsFailure())
    {
        NN_USB_LOG_WARN("TRB command %d (%s) completed with code %d (%s).\n",
                        commandCode,
                        GetTrbName(commandCode),
                        completionCode,
                        GetCompletionString(completionCode));
    }

    if(pCmd->callback != nullptr)
    {
        pCmd->pEventTrb = pEventTrb;
        (this->*(pCmd->callback))(pCmd);
        pCmd->callback = nullptr;
    }

    m_CommandPendIndex  = (m_CommandPendIndex + 1) % MaxCommandTrbs;
    m_CommandPendCount -= 1;
}

void XhciDriver::FlushCommandRing()
{
    while (m_CommandPendCount)
    {
        Command *pCmd   = m_CommandRing + m_CommandPendIndex;
        int commandCode = NN_USB_GET_FIELD32(CommandTrbDw3TrbType,
                                             pCmd->pCommandTrb->dw[3]);

        NN_USB_LOG_WARN("Flush TRB command %d (%s)\n",
                        commandCode, GetTrbName(commandCode));

        pCmd->status = ResultHcCrash();

        if (pCmd->callback != nullptr)
        {
            pCmd->pEventTrb = nullptr;
            (this->*(pCmd->callback))(pCmd);
            pCmd->callback = nullptr;
        }

        m_CommandPendIndex  = (m_CommandPendIndex + 1) % MaxCommandTrbs;
        m_CommandPendCount -= 1;
    }
}

void XhciDriver::OnControllerStall()
{
    m_pHs->LockStackMutex();

    // Make sure no more command will be submitted
    m_IsControllerStall = true;

    // Now complete all pending commands
    FlushCommandRing();

    m_pHs->UnlockStackMutex();
}

Result XhciDriver::InitializeDmaBuffer(DmaBuffer* pBuf, size_t size, const char* pDebugName)
{
    Result result = ResultSuccess();

    pBuf->u.pVoid = m_pPlatform->DoSmmu().CallocMemory(&pBuf->ioVa, size, pDebugName);

    if (pBuf->u.pVoid != nullptr)
    {
        pBuf->size       = size;
        pBuf->pDebugName = pDebugName;
    }
    else
    {
        result = ResultMemAllocFailure();
    }

    return result;
}

Result XhciDriver::FinalizeDmaBuffer(DmaBuffer* pBuf)
{
    Result result = ResultSuccess();

    result = m_pPlatform->DoSmmu().FreeMemory(pBuf->u.pVoid, pBuf->pDebugName);
    if (result.IsSuccess())
    {
        pBuf->u.pVoid    = nullptr;
        pBuf->ioVa       = 0;
        pBuf->size       = 0;
        pBuf->pDebugName = nullptr;
    }

    return result;
}

Result XhciDriver::InitializeDriverMemory()
{
    Result result = ResultSuccess();

    do
    {
        nn::dd::DeviceVirtualAddress scratchPadBufferArrayIoVa = 0;
        nn::dd::DeviceVirtualAddress deviceContextArrayIoVa = 0;

        /*
         * Setup Scratchpads
         */
        if(m_ScratchPadBufferCount > 0)
        {
            // 1st page for the pointer array, the remains are the actual buffers
            NN_STATIC_ASSERT(
                sizeof(nn::dd::DeviceVirtualAddress) * HsLimitXhciMaxScratchpadBufferCount <= HsLimitXhciPageSize
            );
            m_pDmaScratchPadBufferArray = reinterpret_cast<nn::dd::DeviceVirtualAddress*>(
                m_pPlatform->DoSmmu().CallocMemory(
                    &scratchPadBufferArrayIoVa,
                    m_PageSize * (1 + m_ScratchPadBufferCount),
                    "DmaScratchPadBufferArray"
                )
            );
            NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_pDmaScratchPadBufferArray);

            for(uint32_t i = 0; i < m_ScratchPadBufferCount; i++)
            {
                m_pDmaScratchPadBufferArray[i] = scratchPadBufferArrayIoVa + m_PageSize * (1 + i);
            }

            nn::dd::FlushDataCache(m_pDmaScratchPadBufferArray, m_PageSize * (1 + m_ScratchPadBufferCount));
        }

        /*
         * Setup Device Context Base Address Array
         */
        m_pDmaDeviceContextArray = reinterpret_cast<nn::dd::DeviceVirtualAddress*>(
            m_pPlatform->DoSmmu().CallocMemory(
                &deviceContextArrayIoVa,
                sizeof(nn::dd::DeviceVirtualAddress) * LimitMaxDeviceSlotCount,
                "DmaDeviceContextArray"
            )
        );
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_pDmaDeviceContextArray);

        // Slot zero of Device Context Base Address Array points to scratchpad arrray
        m_pDmaDeviceContextArray[0] = scratchPadBufferArrayIoVa;

        nn::dd::FlushDataCache(m_pDmaDeviceContextArray, sizeof(nn::dd::DeviceVirtualAddress) * LimitMaxDeviceSlotCount);

        OpRegister.Write32(OpRegDcbaapLoOffset, static_cast<uint32_t>(deviceContextArrayIoVa));
        OpRegister.Write32(OpRegDcbaapHiOffset, static_cast<uint32_t>(deviceContextArrayIoVa >> 32));

        /*
         * Setup Event TRB Ring
         */
        m_pDmaEvents = reinterpret_cast<DmaTrb*>(
            m_pPlatform->DoSmmu().CallocMemory(
                &m_DmaEventsIoVa,
                sizeof(DmaTrb) * MaxEventTrbs,
                "DmaEventTrbs"
            )
        );
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_pDmaEvents);

        nn::dd::FlushDataCache(m_pDmaEvents, sizeof(DmaTrb) * MaxEventTrbs);

        RtRegister.Write32(RtIntReg0ErdpLoOffset, static_cast<uint32_t>(m_DmaEventsIoVa));
        RtRegister.Write32(RtIntReg0ErdpHiOffset, static_cast<uint32_t>(m_DmaEventsIoVa >> 32));

        /*
         * Setup Event Ring Segment Table
         */
        nn::dd::DeviceVirtualAddress erstIoVa = 0;
        m_pDmaEventRingSegmentTable = reinterpret_cast<DmaErst*>(
            m_pPlatform->DoSmmu().CallocMemory(
                &erstIoVa,
                sizeof(DmaErst) * m_EventRingSegmentTableEntryCount,
                "DmaEventRingSegmentTable"
            )
        );
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_pDmaEventRingSegmentTable);

        // assign Event TRB Ring to ERST
        // N.B. we assume that there is only 1 segment
        NN_SDK_ASSERT(m_EventRingSegmentTableEntryCount == 1);
        m_pDmaEventRingSegmentTable->ringSegmentBaseAddress = m_DmaEventsIoVa;
        m_pDmaEventRingSegmentTable->ringSegmentSize        = MaxEventTrbs;
        nn::dd::FlushDataCache(m_pDmaEventRingSegmentTable, sizeof(DmaErst) * m_EventRingSegmentTableEntryCount);

        RtRegister.Write32(RtIntReg0ErstszOffset,
                           NN_USB_MAKE_MASK32(RtIntReg0ErstszField, m_EventRingSegmentTableEntryCount));
        RtRegister.Write32(RtIntReg0ErstbaLoOffset, static_cast<uint32_t>(erstIoVa));
        RtRegister.Write32(RtIntReg0ErstbaHiOffset, static_cast<uint32_t>(erstIoVa >> 32));

        /*
         * Setup Command Ring, +1 for link TRB.
         */
        m_pDmaCommandRing = reinterpret_cast<DmaTrb*>(
            m_pPlatform->DoSmmu().CallocMemory(
                &m_DmaCommandRingIoVa,
                sizeof(DmaTrb) * (MaxCommandTrbs + 1),
                "DmaCommandTrbs"
            )
        );
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_pDmaCommandRing);

        // make it a ring, last points to first
        m_pDmaCommandRing[MaxCommandTrbs].qw[0] = m_DmaCommandRingIoVa;

        nn::dd::FlushDataCache(m_pDmaCommandRing, sizeof(DmaTrb) * (MaxCommandTrbs + 1));

        OpRegister.Write32(OpRegCrcrLoOffset,
                           static_cast<uint32_t>(m_DmaCommandRingIoVa) | OpRegCrcrLoRcsMask);
        OpRegister.Write32(OpRegCrcrHiOffset, static_cast<uint32_t>(m_DmaCommandRingIoVa >> 32));

    }while(false);

    if(result.IsFailure())
    {
        FinalizeDriverMemory();
    }

    return result;
}

Result XhciDriver::FinalizeDriverMemory()
{
    Result result = ResultSuccess();

    /*
     * Teardown Event Ring Segment Table
     */
    if(m_pDmaCommandRing != nullptr)
    {
        m_pPlatform->DoSmmu().FreeMemory(m_pDmaCommandRing, "DmaCommandTrbs");
        m_pDmaCommandRing = nullptr;
    }

    /*
     * Teardown Event Ring Segment Table
     */
    if(m_pDmaEventRingSegmentTable != nullptr)
    {
        m_pPlatform->DoSmmu().FreeMemory(m_pDmaEventRingSegmentTable, "DmaEventRingSegmentTable");
        m_pDmaEventRingSegmentTable = nullptr;
    }

    /*
     * Teardown Event TRB Ring
     */
    if(m_pDmaEvents != nullptr)
    {
        m_pPlatform->DoSmmu().FreeMemory(m_pDmaEvents, "DmaEventTrbs");
        m_pDmaEvents    = nullptr;
        m_DmaEventsIoVa = 0;
    }

    /*
     * Teardown Device Context Array
     */
    if(m_pDmaDeviceContextArray != nullptr)
    {
        m_pPlatform->DoSmmu().FreeMemory(m_pDmaDeviceContextArray, "DmaDeviceContextArray");
        m_pDmaDeviceContextArray = nullptr;
    }

    /*
     * Teardown Scratchpads
     */
    if(m_pDmaScratchPadBufferArray != nullptr)
    {
        m_pPlatform->DoSmmu().FreeMemory(m_pDmaScratchPadBufferArray, "DmaScratchPadBufferArray");
        m_pDmaScratchPadBufferArray = nullptr;
    }

    return result;
}

Result XhciDriver::InitializeEndpoint0(DeviceContext* pXdc)
{
    Result result = ResultSuccess();

    do
    {
        XhciEndpoint* pXep;

        /*
         * Allocate and configure XhciEndpoint
         */
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(pXep = reinterpret_cast<XhciEndpoint*>
                                         (detail::UsbMemoryCalloc(sizeof(XhciEndpoint), "XhciEndpoint0")));
        NN_USB_BREAK_UPON_ERROR(SetXhciEndpoint(pXdc, UsbEndpointDirection_Control, UsbEndpointType_Control, 0, pXep));

        pXep->pDeviceContext = pXdc;
        pXep->pHcEp = nullptr;
        pXep->dci = 1;

        pXep->pDmaInputContext  = GetEndpointInputContext (pXdc, UsbEndpointDirection_Control, 0);
        pXep->pDmaOutputContext = GetEndpointOutputContext(pXdc, UsbEndpointDirection_Control, 0);
        NN_USB_BREAK_UPON_ERROR(
            InitializeTransferRing(&pXep->transferRing, 8, 1)
        );

        /*
         * Setup input endpoint context, see [xHCI r1.1] 6.2.3.1
         */
        memset(pXep->pDmaInputContext, 0, EndpointCtxSize);
        nn::dd::FlushDataCache(pXep->pDmaInputContext, EndpointCtxSize);
        pXep->pDmaInputContext->dw[0] =
            NN_USB_MAKE_MASK32(EndpointCtxDw0State,              EndpointCtxDw0StateDisabled)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Mult,             0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0MaxPStreams,      0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Lsa,              0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw0Interval,         0);
        pXep->pDmaInputContext->dw[1] =
            NN_USB_MAKE_MASK32(EndpointCtxDw1Cerr,               3)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1EpType,           EndpointCtxDw1EpTypeControl)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1Hid,              0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1MaxBurstSize,     0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw1MaxPacketSize,    Util::GetControlTransferMaxPacketSize(pXdc->speed));
        pXep->pDmaInputContext->qw[1] =
            pXep->transferRing.dmaTrbList.ioVa
            | NN_USB_MAKE_MASK32(EndpointCtxQw1Dcs,              pXep->transferRing.ccs);
        pXep->pDmaInputContext->dw[4] =
            NN_USB_MAKE_MASK32(EndpointCtxDw4MaxEsitPayloadLo,   0)
            | NN_USB_MAKE_MASK32(EndpointCtxDw4AverageTrbLength, 8);
        nn::dd::FlushDataCache(pXep->pDmaInputContext, EndpointCtxSize);
        pXdc->pControlEndpoint = pXep;
    }while(false);

    return result;
}

// This can be called only after slot has been disabled, since default endpoint operation is intrinsic with slot
Result XhciDriver::FinalizeEndpoint0(DeviceContext* pXdc)
{
    Result result = ResultSuccess();

    do
    {
        XhciEndpoint* pXep = GetXhciEndpoint(pXdc, UsbEndpointDirection_Control, UsbEndpointType_Control, 0, false);

        // Anything to do?
        if(pXep == nullptr) break;

        // Unlink in upper
        if(pXep->pHcEp != nullptr)
        {
            pXep->pHcEp->SetDriverEndpoint(nullptr);
        }

        // Tear down transfer ring
        NN_USB_BREAK_UPON_ERROR(FinalizeTransferRing(&pXep->transferRing));

        // Clear host controller driver reference to EP0
        NN_USB_BREAK_UPON_ERROR(SetXhciEndpoint(pXdc, UsbEndpointDirection_Control, UsbEndpointType_Control, 0, nullptr));

        // Gone
        detail::UsbMemoryFree(pXep, "XhciEndpoint0");

    }while(false);

    return result;
}

Result XhciDriver::FinalizeEndpoint(DeviceContext* pXdc, XhciEndpoint* pXep)
{
    Result result = ResultSuccess();

    do
    {
        HostEndpoint *pHep;
        HostControllerDriverEndpoint *pHcEp;

        NN_USB_BREAK_UPON_NULL_WITH_ERROR(pHcEp = pXep->pHcEp, ResultEndpointStateInvalid());
        NN_USB_BREAK_UPON_NULL_WITH_ERROR(pHep = pHcEp->GetHostEndpoint(), ResultEndpointStateInvalid());

        // control endpoint shall not use this method
        if(UsbEndpointIsControl(pHep->pDescriptor)) return result;

        // DMA rings
        NN_USB_LOG_UPON_ERROR(FinalizeTransferRing(&pXep->transferRing));

        // Disassociate with device
        NN_USB_LOG_UPON_ERROR(SetXhciEndpoint(pXdc, pHep->epDir, pHep->epType, pHep->epNumber, nullptr));

        // Disassociate with higher level endpoint
        pHcEp->SetDriverEndpoint(nullptr);

        // any stale reference shall crash
        memset(pXep, 0, sizeof(XhciEndpoint));
        detail::UsbMemoryFree(pXep, "XhciEndpoint");

    }while(false);

    return result;
}

Result XhciDriver::StartController()
{
    Result result = ResultSuccess();

    // Setup interrupt registers
    RtRegister.Write32(RtIntReg0ImodOffset, RtIntReg0ImodDefault);
    RtRegister.SetField32(RtIntReg0ImanOffset, RtIntReg0ImanIeMask);
    OpRegister.Write32(OpRegStatusOffset, OpRegister.Read32(OpRegStatusOffset));

    // Enable
    OpRegister.Write32(OpRegUsbCmdOffset,
                       OpRegUsbCmdRunMask | OpRegUsbCmdEieMask | OpRegUsbCmdHseieMask);

    // Wait for host controller to stop saying it is halted
    if(!OpRegister.PollField32(OpRegStatusOffset, OpRegStatusHaltMask, 0,
                               nn::TimeSpan::FromMilliSeconds(100), 100))
    {
        NN_USB_LOG_ERROR("XhciDriver::StartController() - Controller won't start.\n");
        result = ResultHardwareTimeout();
    }

    return result;
}

/*
 * Stop the controller
 */
Result XhciDriver::HaltController()
{
    Result result = ResultSuccess();

    // Since this method may be used at unusual times
    if(!OpRegister.IsInitialized()) return ResultNotInitialized();

    // Disable interrupts first
    OpRegister.ClearField32(OpRegUsbCmdOffset,
                            OpRegUsbCmdEieMask | OpRegUsbCmdHseieMask | OpRegUsbCmdEweMask);

    // Stop it
    NN_USB_LOG_TRACE("XhciDriver::HaltController() stopping controller.\n");
    OpRegister.Write32(OpRegUsbCmdOffset, 0);

    // Wait for controller to stop it
    if(!OpRegister.PollField32(OpRegStatusOffset, OpRegStatusHaltMask, OpRegStatusHaltMask,
                               nn::TimeSpan::FromSeconds(1), 100))
    {
        NN_USB_LOG_ERROR("XhciDriver::HaltController() - Controller not responding.\n");
        result = ResultHardwareTimeout();
    }

    return result;
}

/*
 * Reset a halted HC.
 *
 * This resets pipelines, timers, counters, state machines, etc.
 * Transactions will be terminated immediately, and operational registers
 * will be set to their defaults.
 */
Result XhciDriver::ResetController()
{
    Result result = ResultSuccess();

    NN_USB_LOG_TRACE("XhciDriver::Reset() - Resetting controller...\n");

    // Controller should be halted before continuing
    if(!OpRegister.TestField32(OpRegStatusOffset, OpRegStatusHaltMask))
    {
        NN_USB_LOG_ERROR("XhciDriver::Reset() - Cannot proceed because controller not halted.\n");
        return ResultResourceBusy();
    }

    // Issue reset
    OpRegister.Write32(OpRegUsbCmdOffset, OpRegUsbCmdResetMask);

    // Wait for reset to finish
    if(!OpRegister.PollField32(OpRegUsbCmdOffset, OpRegUsbCmdResetMask, 0,
                               nn::TimeSpan::FromSeconds(10), 100))
    {
        NN_USB_LOG_ERROR("XhciDriver::Reset() - Command timeout.\n");
        return ResultHardwareTimeout();
    }

    // Wait for controller to become ready
    if(!OpRegister.PollField32(OpRegStatusOffset, OpRegStatusCnrMask, 0,
                               nn::TimeSpan::FromSeconds(10), 100))
    {
        NN_USB_LOG_ERROR("XhciDriver::Reset() - Controller not ready.\n");
        return ResultHardwareTimeout();
    }

    NN_USB_LOG_TRACE("done.\n");

    return result;
}

//----------------------------------------------------------------------------
// xHC Command Interface
//----------------------------------------------------------------------------

Result XhciDriver::DoEnableSlot(DeviceContext *pXdc)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.dw[3] = NN_USB_MAKE_VALUE32_1(CommandTrbDw3,
                                      TrbType, CommandTrbTypeEnableSlot);

    NN_USB_XHCI_LOG("Issue xHCI Command: Enable Slot\n");
    return DoCommandAsync(trb, &XhciDriver::OnEnableSlotCompletion, pXdc);
}

Result XhciDriver::DoDisableSlot(int slotId)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.dw[3] = NN_USB_MAKE_VALUE32_2(CommandTrbDw3,
                                      TrbType, CommandTrbTypeDisableSlot,
                                      SlotId,  slotId);

    NN_USB_XHCI_LOG("Issue xHCI Command: Disable Slot %d\n", slotId);
    return DoCommandAsync(trb, &XhciDriver::OnDisableSlotCompletion, nullptr);
}

Result XhciDriver::DoAddressDevice(int slotId,
                                   nn::dd::DeviceVirtualAddress inputContext,
                                   int bsr)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.qw[0] = inputContext;

    trb.dw[3] = NN_USB_MAKE_VALUE32_3(CommandTrbDw3,
                                      TrbType, CommandTrbTypeAddressDevice,
                                      SlotId,  slotId,
                                      Bsr,     bsr);

    NN_USB_XHCI_LOG("Issue xHCI Command: Address Device %d, BSR=%d\n", slotId, bsr);
    return DoCommandAsync(trb, &XhciDriver::OnAddressDeviceCompletion, nullptr);
}

Result XhciDriver::DoConfigureEndpoint(int slotId,
                                       nn::dd::DeviceVirtualAddress inputContext,
                                       int dc,
                                       XhciEndpoint *pXep)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.qw[0] = inputContext;

    trb.dw[3] = NN_USB_MAKE_VALUE32_3(CommandTrbDw3,
                                      TrbType, CommandTrbTypeConfigureEndpoint,
                                      SlotId,  slotId,
                                      Dc,      dc);

    NN_USB_XHCI_LOG("Issue xHCI Command: Configure Endpoint <%d:%d>\n", slotId, pXep->dci);
    return DoCommandAsync(trb, &XhciDriver::OnConfigureEndpointCompletion, pXep);
}

Result XhciDriver::DoEvaluateContext(int slotId,
                                     nn::dd::DeviceVirtualAddress inputContext)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.qw[0] = inputContext;

    trb.dw[3] = NN_USB_MAKE_VALUE32_2(CommandTrbDw3,
                                      TrbType, CommandTrbTypeEvaluateContext,
                                      SlotId,  slotId);

    NN_USB_XHCI_LOG("Issue xHCI Command: Evaluate Context %d\n", slotId);
    return DoCommandAsync(trb, &XhciDriver::OnEvaluateContextCompletion, nullptr);
}

Result XhciDriver::DoResetEndpoint(int slotId, int dci, int tsp)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.dw[3] = NN_USB_MAKE_VALUE32_4(CommandTrbDw3,
                                      TrbType,    CommandTrbTypeResetEndpoint,
                                      SlotId,     slotId,
                                      EndpointId, dci,
                                      Tsp,        tsp);

    // TODO: Evaluate the significance of [xHCI r1.1] P.106, Note #3

    NN_USB_XHCI_LOG("Issue xHCI Command: Reset Endpoint <%d:%d>\n", slotId, dci);
    return DoCommandAsync(trb, &XhciDriver::OnResetEndpointCompletion, nullptr);
}

Result XhciDriver::DoStopEndpoint(int slotId, int dci, int sp)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.dw[3] = NN_USB_MAKE_VALUE32_4(CommandTrbDw3,
                                      TrbType,    CommandTrbTypeStopEndpoint,
                                      SlotId,     slotId,
                                      EndpointId, dci,
                                      Sp,         sp);

    NN_USB_XHCI_LOG("Issue xHCI Command: Stop Endpoint <%d:%d>\n", slotId, dci);
    return DoCommandAsync(trb, &XhciDriver::OnStopEndpointCompletion, nullptr);
}

Result XhciDriver::DoSetTrDequeuePointer(int slotId, int dci,
                                         nn::dd::DeviceVirtualAddress dequeuePointer,
                                         int dcs, int streamId, int sct)
{
    DmaTrb trb = ZeroedDmaTrb;

    trb.dw[0] =
        (dequeuePointer & CommandTrbDw0NewTrDeqPtrLoMask) |
        NN_USB_MAKE_VALUE32_2(CommandTrbDw0,
                              Sct,   sct,
                              Dcs,   dcs);

    trb.dw[1] = dequeuePointer >> 32;

    trb.dw[2] = NN_USB_MAKE_VALUE32_1(CommandTrbDw2,
                                      StreamId, streamId);

    trb.dw[3] = NN_USB_MAKE_VALUE32_3(CommandTrbDw3,
                                      TrbType,    CommandTrbTypeSetTrDequeuePointer,
                                      SlotId,     slotId,
                                      EndpointId, dci);

    NN_USB_XHCI_LOG("Issue xHCI Command: Set TR Dequeue Pointer <%d:%d>\n", slotId, dci);
    return DoCommandAsync(trb, &XhciDriver::OnSetTrDequeuePointerCompletion, nullptr);
}

Result XhciDriver::DoCommandAsync(DmaTrb& commandTrb,
                                  DoCommandCallback callback, void *data)
{
    if (m_IsControllerStall)
    {
        NN_USB_LOG_ERROR("xHC: Command failed (Dead Bird)\n");
        return ResultHcCrash();
    }

    // Make sure the command ring is not full
    if(m_CommandPendCount >= MaxCommandTrbs)
    {
        NN_USB_LOG_ERROR("xHC: Command Ring Overrun!\n");
        return ResultResourceBusy();
    }

    DmaTrb  *pTrb = m_pDmaCommandRing + m_CommandFreeIndex;
    Command *pCmd = m_CommandRing     + m_CommandFreeIndex;

    *pTrb = commandTrb;
    pTrb->dw[3] |= m_CommandPcs;
    nn::dd::FlushDataCache(pTrb, sizeof(DmaTrb));

    pCmd->callback    = callback;
    pCmd->data        = data;
    pCmd->pCommandTrb = pTrb;
    pCmd->pEventTrb   = nullptr;
    pCmd->status      = ResultSuccess();

    // Command is officially queued
    m_CommandFreeIndex++;
    m_CommandPendCount++;

    // Now handle the Link TRB
    if (m_CommandFreeIndex == MaxCommandTrbs)
    {
        pTrb = m_pDmaCommandRing + m_CommandFreeIndex;
        pTrb->dw[3] = NN_USB_MAKE_VALUE32_3(LinkTrbDw3,
                                            C,       m_CommandPcs,
                                            Tc,      1,
                                            TrbType, XferTrbTypeLink);
        nn::dd::FlushDataCache(pTrb, sizeof(DmaTrb));

        m_CommandFreeIndex = 0;
        m_CommandPcs ^= 1;
    }

    // Ring the bell
    DbRegister.Write32(0, 0);

    return ResultSuccess();
}

/*
 * Ref [xHCI r1.1] 4.3.3
 */
void XhciDriver::OnEnableSlotCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pEventTrb   = pCmd->pEventTrb;
    DeviceContext *pXdc = reinterpret_cast<DeviceContext*>(pCmd->data);

    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Disabled);

    if (result.IsFailure())
    {
        (*pXdc->completion.callback.create)(m_pHs, result, pXdc, pXdc->completion.context);
        DestroyDeviceContext(pXdc);
        return;
    }

    pXdc->state = DeviceContextState_Enabled;
    pXdc->slotId = NN_USB_GET_FIELD32(EventTrbDw3SlotId, pEventTrb->dw[3]);

    NN_USB_ABORT_UNLESS(m_pDevices[pXdc->slotId] == nullptr);

    m_pDevices[pXdc->slotId] = pXdc;

    // Encode route string
    uint32_t ctxRouteString = 0;
    for (int32_t i = 0; i < (HsLimitMaxHubHierarchyCount - 1); i++)
    {
        if(pXdc->portHierarchy[i + 1] >= 1)
        {
            ctxRouteString |= ((0xF & pXdc->portHierarchy[i + 1]) << (i * 4));
        }
    }

    /*
     * Ref [xHCI r1.1] 7.2.2.1.1, Table 157
     *
     * TODO: TX1 uses the default mapping (i.e. PSIC=0) so this is
     *       correct. We probably should actually parse the xHCI
     *       Extended Capabilities.
     */
    uint32_t ctxSpeed = 0;
    if (pXdc->speed == UsbDeviceSpeed_Low)
    {
        ctxSpeed = 2;
    }
    else if (pXdc->speed == UsbDeviceSpeed_Full)
    {
        ctxSpeed = 1;
    }
    else if (pXdc->speed == UsbDeviceSpeed_High)
    {
        ctxSpeed = 3;
    }
    else // UsbDeviceSpeed_Super
    {
        ctxSpeed = 4;
    }

    // Prepare input context before Address Device
    DmaInputControlContext *pDmaInCtrlCtx;
    DmaSlotCtx             *pDmaSlot;
    /*
     * Ref [xHCI r1.1] 4.6.5
     *
     *   "The Add Context flags A0 and A1 of the Input Control Context
     *    data structure (in the Input Context) shall be set to ‘1’, and
     *    all remaining Add Context and Drop Context flags shall all be
     *    cleared to ‘0’."
     *
     * Initialize the Input Control Context
     */
    pDmaInCtrlCtx = reinterpret_cast<DmaInputControlContext*>(
        pXdc->inputBuffer.u.uintptr + InputCtxControlOffset
    );
    pDmaInCtrlCtx->dw[0] = 0;
    pDmaInCtrlCtx->dw[1] = InputControlCtxDw1AddSlotMask | InputControlCtxDw1AddEndpoint0Mask;
    pDmaInCtrlCtx->dw[7] = 0;

    // Initialize the Input Slot Context
    pDmaSlot = reinterpret_cast<DmaSlotCtx*>(
        pXdc->inputBuffer.u.uintptr + InputCtxSlotOffset
    );
    pDmaSlot->dw[0] = NN_USB_MAKE_VALUE32_5(SlotCtxDw0,
                                            RouteString,    ctxRouteString,
                                            Speed,          ctxSpeed,
                                            ContextEntries, 1,
                                            Hub,            0,
                                            Mtt,            0);
    pDmaSlot->dw[1] = NN_USB_MAKE_VALUE32_1(SlotCtxDw1,
                                            RootHubPortNumber, pXdc->portHierarchy[0]);
    pDmaSlot->dw[2] = NN_USB_MAKE_VALUE32_2(SlotCtxDw2,
                                            TtHubSlotId,  pXdc->ttParentDevice,
                                            TtPortNumber, pXdc->ttParentPort);
    pDmaSlot->dw[3] = NN_USB_MAKE_VALUE32_1(SlotCtxDw3,
                                            SlotState, SlotCtxDw3SlotStateEnable);

    nn::dd::FlushDataCache(pXdc->inputBuffer.u.pVoid, pXdc->inputBuffer.size);

    // Initialize the input default control Endpoint 0 Context
    result = InitializeEndpoint0(pXdc);
    if (result.IsFailure())
    {
        (*pXdc->completion.callback.create)(m_pHs, result, pXdc, pXdc->completion.context);
        DestroyDeviceContext(pXdc);
        return;
    }

    // Patch device context on to main array
    m_pDmaDeviceContextArray[pXdc->slotId] = pXdc->outputBuffer.ioVa;
    nn::dd::FlushDataCache(m_pDmaDeviceContextArray,
                           sizeof(nn::dd::DeviceVirtualAddress) * LimitMaxDeviceSlotCount);

    /*
     * Ref [xHCI r1.1] 4.6.5
     *
     * Now issue the Address Device Command with BSR=1, which transfers the
     * device from Enabled to the Default state. SET_ADDRESS request is not
     * sent until AddressDeviceAsync()
     */
    result = DoAddressDevice(pXdc->slotId, pXdc->inputBuffer.ioVa, 1);
    if (result.IsFailure())
    {
        (*pXdc->completion.callback.create)(m_pHs, result, pXdc, pXdc->completion.context);
        DestroyDeviceContext(pXdc);
    }
}

void XhciDriver::OnDisableSlotCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];

    NN_USB_ABORT_UNLESS(pXdc != nullptr);

    if(pXdc->completion.callback.destroy != nullptr)
    {
        (*pXdc->completion.callback.destroy)(m_pHs, result, pXdc->completion.context);
    }

    if (result.IsSuccess())
    {
        pXdc->state = DeviceContextState_Disabled;
        DestroyDeviceContext(pXdc);
    }
}

void XhciDriver::OnAddressDeviceCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];

    NN_USB_ABORT_UNLESS(pXdc != nullptr);

    int bsr = NN_USB_GET_FIELD32(CommandTrbDw3Bsr, pCommandTrb->dw[3]);
    DmaSlotCtx* pDmaSlot = reinterpret_cast<DmaSlotCtx*>(
        pXdc->outputBuffer.u.uintptr +  DeviceCtxSlotOffset
    );

    switch (pXdc->state)
    {
    case DeviceContextState_Enabled:
        NN_USB_ABORT_UNLESS(bsr == 1);

        (*pXdc->completion.callback.create)(m_pHs, result, pXdc, pXdc->completion.context);

        if (result.IsSuccess())
        {
            pXdc->state = DeviceContextState_Default;
        }
        else
        {
            DestroyDeviceContext(pXdc);
        }

        break;

    case DeviceContextState_Default:
        NN_USB_ABORT_UNLESS(bsr == 0);

        if (result.IsSuccess())
        {
            // Now we are addressed, check Output Context to get the allocated address
            nn::dd::InvalidateDataCache(pDmaSlot, sizeof(DmaSlotCtx));
            pXdc->address = NN_USB_GET_FIELD32(SlotCtxDw3UsbDeviceAddress, pDmaSlot->dw[3]);

            pXdc->state = DeviceContextState_Addressed;
        }

        (*pXdc->completion.callback.address)(m_pHs, result, pXdc->completion.context);

        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void XhciDriver::OnConfigureEndpointCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];
    XhciEndpoint  *pXep = reinterpret_cast<XhciEndpoint*>(pCmd->data);

    NN_USB_ABORT_UNLESS(pXdc != nullptr);
    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Addressed ||
                        pXdc->state == DeviceContextState_Configured);

    if (pXep->isEnabled)   // We are trying to add the endpoint
    {
        pXep->pHcEp->ReportOpenComplete(result);
        if (result.IsFailure())
        {
            FinalizeEndpoint(pXdc, pXep);
        }
    }
    else                   // We are trying to drop the endpoint
    {
        pXep->pHcEp->ReportCloseComplete(result);
        FinalizeEndpoint(pXdc, pXep);
    }
}

void XhciDriver::OnEvaluateContextCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];

    NN_USB_ABORT_UNLESS(pXdc != nullptr);
    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Default   ||
                        pXdc->state == DeviceContextState_Addressed ||
                        pXdc->state == DeviceContextState_Configured);

    (*pXdc->completion.callback.update)(m_pHs, result, pXdc->completion.context);
}

void XhciDriver::OnResetEndpointCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId,     pCommandTrb->dw[3]);
    int     dci         = NN_USB_GET_FIELD32(CommandTrbDw3EndpointId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];
    XhciEndpoint  *pXep = GetXhciEndpoint(pXdc, dci, true);
    TransferRing  *pTr  = &pXep->transferRing;

    NN_USB_ABORT_UNLESS(pXdc != nullptr);
    NN_USB_ABORT_UNLESS(pXep != nullptr);
    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Default   ||
                        pXdc->state == DeviceContextState_Addressed ||
                        pXdc->state == DeviceContextState_Configured);

    if (result.IsFailure())
    {
        pXep->pHcEp->ReportResetComplete(result);
        return;
    }

    ResetTransferRing(pTr, 1);

    result = DoSetTrDequeuePointer(slotId, dci, pTr->dmaTrbList.ioVa, 1, 0, 0);
    if (result.IsFailure())
    {
        pXep->pHcEp->ReportResetComplete(result);
        return;
    }
}

void XhciDriver::OnStopEndpointCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId,     pCommandTrb->dw[3]);
    int     dci         = NN_USB_GET_FIELD32(CommandTrbDw3EndpointId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];
    XhciEndpoint  *pXep = GetXhciEndpoint(pXdc, dci, true);
    TransferRing  *pTr  = &pXep->transferRing;

    NN_USB_ABORT_UNLESS(pXdc != nullptr);
    NN_USB_ABORT_UNLESS(pXep != nullptr);
    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Default   ||
                        pXdc->state == DeviceContextState_Addressed ||
                        pXdc->state == DeviceContextState_Configured);

    if (result.IsFailure())
    {
        pXep->pHcEp->ReportResetComplete(result);
        return;
    }

    ResetTransferRing(pTr, 1);

    result = DoSetTrDequeuePointer(slotId, dci, pTr->dmaTrbList.ioVa, 1, 0, 0);
    if (result.IsFailure())
    {
        pXep->pHcEp->ReportResetComplete(result);
        return;
    }
}

void XhciDriver::OnSetTrDequeuePointerCompletion(Command *pCmd)
{
    Result  result      = pCmd->status;
    DmaTrb *pCommandTrb = pCmd->pCommandTrb;
    int     slotId      = NN_USB_GET_FIELD32(CommandTrbDw3SlotId,     pCommandTrb->dw[3]);
    int     dci         = NN_USB_GET_FIELD32(CommandTrbDw3EndpointId, pCommandTrb->dw[3]);
    DeviceContext *pXdc = m_pDevices[slotId];
    XhciEndpoint  *pXep = GetXhciEndpoint(pXdc, dci, true);

    NN_USB_ABORT_UNLESS(pXdc != nullptr);
    NN_USB_ABORT_UNLESS(pXep != nullptr);
    NN_USB_ABORT_UNLESS(pXdc->state == DeviceContextState_Default   ||
                        pXdc->state == DeviceContextState_Addressed ||
                        pXdc->state == DeviceContextState_Configured);

    pXep->pHcEp->ReportResetComplete(result);
}

Result XhciDriver::DestroyDeviceContext(HostControllerDriverDeviceContext* pDeviceContext)
{
    Result result = ResultSuccess();

    DeviceContext* pXdc = static_cast<DeviceContext*>(pDeviceContext);

    // nothing to do if nothing provided
    if(pXdc == nullptr) return result;

    if((pXdc->slotId > 0) && (pXdc->slotId < LimitMaxDeviceSlotCount))
    {
        m_pDmaDeviceContextArray[pXdc->slotId] = 0;
        nn::dd::FlushDataCache(m_pDmaDeviceContextArray,
                               sizeof(nn::dd::DeviceVirtualAddress) * LimitMaxDeviceSlotCount);
        FinalizeEndpoint0(pXdc);
        m_pDevices[pXdc->slotId] = nullptr;
    }

    // finalize DMA buffers
    FinalizeDmaBuffer(&pXdc->outputBuffer);
    FinalizeDmaBuffer(&pXdc->inputBuffer);

    // Free the context structure itself
    detail::UsbMemoryFree(pXdc, "DeviceContext");

    return result;
}

const char* XhciDriver::GetTrbName(int trbType)
{
    const char *trbName[] = {
        "Reserved",
        "Normal",
        "Setup Stage",
        "Data Stage",
        "Status Stage",
        "Isoch",
        "Link",
        "Event Data",
        "No-Op",
        "Enable Slot Command",

        "Disable Slot Command",
        "Address Device Command",
        "Configure Endpoint Command",
        "Evaluate Context Command",
        "Reset Endpoint Command",
        "Stop Endpoint Command",
        "Set TR Dequeue Pointer Command",
        "Reset Device Command",
        "Force Event Command",
        "Negotiate Bandwidth Command",

        "Set Latency Tolerance Value Command",
        "Get Port Bandwidth Command",
        "Force Header Command",
        "No Op Command",
        "Reserved",
        "Reserved",
        "Reserved",
        "Reserved",
        "Reserved",
        "Reserved",

        "Reserved",
        "Reserved",
        "Transfer Event",
        "Command Completion Event",
        "Port Status Change Event",
        "Bandwidth Request Event",
        "Doorbell Event",
        "Host Controller Event",
        "Device Notification Event",
        "MFINDEX Wrap Event",
    };

    if (trbType < NN_USB_ARRAY_COUNT32(trbName))
    {
        return trbName[trbType];
    }

    return "Unknown TRB Type";
}

const char* XhciDriver::GetCompletionString(int completionCode)
{
    const char *completionString[] = {
        "Invalid",
        "Success",
        "Data Buffer Error",
        "Babble Detected Error",
        "USB Transaction Error",
        "TRB Error",
        "Stall Error",
        "Resource Error",
        "Bandwidth Error",
        "No Slots Available Error",

        "Invalid Stream Type Error",
        "Slot Not Enabled Error",
        "Endpoint Not Enabled Error",
        "Short Packet",
        "Ring Underrun",
        "Ring Overrun",
        "VF Event Ring Full Error",
        "Parameter Error",
        "Bandwidth Overrun Error",
        "Context State Error",

        "No Ping Response Error",
        "Event Ring Rull Error",
        "Incompatible Device Error",
        "Missed Service Error",
        "Command Ring Stopped",
        "Command Aborted",
        "Stopped",
        "Stopped - Length Invalid",
        "Stopped - Short Packet",
        "Max Exit Latency Too Large Error",

        "Reserved",
        "Isoch Buffer Overrun",
        "Event Lost Error",
        "Undefined Error",
        "Invalid Stream ID Error",
        "Secondary Bandwidth Error",
        "Split Transaction Error",
    };

    if (completionCode < NN_USB_ARRAY_COUNT32(completionString))
    {
        return completionString[completionCode];
    }

    return "Unkonwn Completion Code";
}

Result XhciDriver::CheckTransferTrbCompletionCode(uint8_t completionCode)
{
    Result result;

    switch(completionCode)
    {
    case CompletionCodeSuccess:
        result = ResultSuccess();
        break;

    case CompletionCodeDataBufferError:
        result = ResultXhcDataBufferError();
        break;

    case CompletionCodeBabbleError:
    case CompletionCodeTransactError:
    case CompletionCodeStreamTypeError:
    case CompletionCodeEventLostError:
    case CompletionCodeInvalidStreamIdError:
    case CompletionCodeSplitTransactionError:
        result = ResultTransactionError();
        break;

    case CompletionCodeTrbError:
        NN_USB_ABORT("Transfer TRB parameter invalid or out of range!\n");
        break;

    case CompletionCodeStallError:
        result = ResultStalled();
        break;

    case CompletionCodeEndpointNotEnabledError:
        NN_USB_ABORT("Doorbell is rung while the EP is in Disabled state!\n");
        break;

    case CompletionCodeShortPacket:
        result = ResultShortPacket();
        break;

    case CompletionCodeRingUnderrun:
    case CompletionCodeRingOverrun:
        result = ResultXhcIsocRingStarve();
        break;

    case CompletionCodeBandwidthOverrunError:
        // Max ESIT Payload, should have been enforced in HandleIsochUrbSubmit()
        NN_USB_ABORT("Isoc bandwidth overrun!\n");
        break;

    case CompletionCodeNoPingResponseError:
        NN_USB_ABORT("MaxExitLatency=0, should never get No Ping Response Error!\n");
        break;

    case CompletionCodeIncompatibleDeviceError:
        result = ResultIncompatibleDevice();
        break;

    case CompletionCodeMissedServiceError:
        result = ResultXhcIsocMissedService();
        break;

    case CompletionCodeStopped:
    case CompletionCodeStoppedLengthInvalid:
    case CompletionCodeStoppedShortPacket:
        result = ResultXferStopped();
        break;

    case CompletionCodeIsochBufferOverrun:
        result = ResultIsochBufferOverrun();
        break;

    case CompletionCodeUndefinedError:
        result = ResultXhcUndefinedError();
        break;

    default:
        /*
         * TODO: I am a coward. Let's change this to abort after 3.0NUP.
         */
        NN_USB_LOG_ERROR("Unhandled Transfer TRB Completion code %d (%s)!\n",
                         completionCode, GetCompletionString(completionCode));
        result = ResultInternalStateError();
        break;
    }

    return result;
}

Result XhciDriver::CheckCommandTrbCompletionCode(uint8_t completionCode)
{
    Result result;

    switch(completionCode)
    {
    case CompletionCodeSuccess:
        result = ResultSuccess();
        break;

    case CompletionCodeTransactError:
        // This could be returned by Address Device Command
        result = ResultTransactionError();
        break;

    case CompletionCodeTrbError:
        NN_USB_ABORT("Command TRB parameter invalid or out of range!\n");
        break;

    case CompletionCodeResourceError:
    case CompletionCodeBandwidthError:
    case CompletionCodeNoSlotAvailableError:
    case CompletionCodeSecondaryBandwidthError:
    case CompletionCodeEventLostError:
        result = ResultXhcCommandFailure();
        break;

    case CompletionCodeSlotNotEnabledError:
        NN_USB_ABORT("Command is isseed to a disabled Device Slot!\n");
        break;

    case CompletionCodeVfEventRingFullError:
        NN_USB_ABORT("Should never receive a VF event!\n");
        break;

    case CompletionCodeParameterError:
        /*
         * This error is observed when we remove the cradle while AddressDevice
         * command is running: Attach the cradle, then remove in about 2400ms.
         * It could be easily reproduced with MCCI.
         */
        NN_USB_LOG_ERROR("Context parameter is invalid!\n");
        result = ResultXhcParameterError();
        break;

    case CompletionCodeContextStateError:
        NN_USB_ABORT("Command is issued to transition from an illegal context state!");
        break;

    case CompletionCodeIncompatibleDeviceError:
        result = ResultIncompatibleDevice();
        break;

    case CompletionCodeCommandRingStopped:
        result = ResultXhcCommandRingStopped();
        break;

    case CompletionCodeCommandAborted:
        result = ResultXhcCommandAborted();
        break;

    case CompletionCodeMaxExitLatencyTooLargeError:
        NN_USB_ABORT("MaxExitLatency=0, should never get No Ping Response Error!\n");
        break;

    case CompletionCodeUndefinedError:
        result = ResultXhcUndefinedError();
        break;

    default:
        /*
         * TODO: I am a coward. Let's change this to abort after 3.0NUP.
         */
        NN_USB_LOG_ERROR("Unhandled Command TRB Completion code %d (%s)!\n",
                         completionCode, GetCompletionString(completionCode));
        result = ResultInternalStateError();
        break;
    }

    return result;
}

XhciDriver::DmaEndpointContext*
XhciDriver::GetEndpointInputContext(DeviceContext* pDeviceContext,
                                    UsbEndpointDirection direction,
                                    uint8_t epNumber)
{
    size_t offset;

    // Software BUG if epNumber out of range
    NN_ABORT_UNLESS(epNumber >= 0);
    NN_ABORT_UNLESS(epNumber < 16);

    switch (direction)
    {
    case UsbEndpointDirection_Control:
        NN_ABORT_UNLESS(epNumber == 0);
        // input control + slot
        offset = EndpointCtxSize + EndpointCtxSize;
        break;

    case UsbEndpointDirection_ToDevice:
        // input control + ...
        offset = EndpointCtxSize + EndpointCtxSize * 2 * epNumber;
        break;

    case UsbEndpointDirection_ToHost:
        // input control + slot + ...
        offset = EndpointCtxSize + EndpointCtxSize + EndpointCtxSize * 2 * epNumber;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return reinterpret_cast<DmaEndpointContext*>(
        pDeviceContext->inputBuffer.u.uintptr + offset
    );
}

XhciDriver::DmaEndpointContext*
XhciDriver::GetEndpointOutputContext(DeviceContext* pDeviceContext,
                                     UsbEndpointDirection direction,
                                     uint8_t epNumber)
{
    size_t offset;

    // Software BUG if epNumber out of range
    NN_ABORT_UNLESS(epNumber >= 0);
    NN_ABORT_UNLESS(epNumber < 16);

    switch (direction)
    {
    case UsbEndpointDirection_Control:
        NN_ABORT_UNLESS(epNumber == 0);
        // slot
        offset = EndpointCtxSize;
        break;

    case UsbEndpointDirection_ToDevice:
        offset = 0x00 +  EndpointCtxSize * 2 * epNumber;
        break;

    case UsbEndpointDirection_ToHost:
        // slot + ...
        offset = EndpointCtxSize + EndpointCtxSize * 2 * epNumber;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return reinterpret_cast<DmaEndpointContext*>(
        pDeviceContext->outputBuffer.u.uintptr + offset
    );
}

XhciDriver::DeviceContext* XhciDriver::GetXhciDeviceContext(int32_t slotId)
{
    DeviceContext* pXdc = nullptr;
    if((slotId >= 0) && (slotId < m_MaxDeviceSlotCount))
    {
        pXdc = m_pDevices[slotId];
    }
    return pXdc;
}

XhciDriver::XhciEndpoint* XhciDriver::GetXhciEndpoint(DeviceContext* pXdc, UsbEndpointDirection epDir,
                                                      UsbEndpointType epType, EndpointNumber epNumber,
                                                      bool isOperationalRequired)
{
    XhciEndpoint* pXep = nullptr;
    if((epNumber == 0) || (epType == UsbEndpointType_Control))
    {
        pXep = pXdc->pControlEndpoint;
    }
    else if((epDir == UsbEndpointDirection_ToDevice) && ((epNumber>=1) && (epNumber<=15)))
    {
        pXep = pXdc->pOutEndpoints[epNumber - 1];
    }
    else if((epDir == UsbEndpointDirection_ToHost) && ((epNumber>=1) && (epNumber<=15)))
    {
        pXep = pXdc->pInEndpoints[epNumber - 1];
    }

    if((pXep != nullptr) && (pXep->pHcEp == nullptr) && isOperationalRequired)
    {
        // endpoint is not operational
        pXep = nullptr;
    }

    return pXep;
}

XhciDriver::XhciEndpoint* XhciDriver::GetXhciEndpoint(DeviceContext* pXdc, uint8_t deviceContextIndex, bool isOperationalRequired)
{
    // See figure 73 in XHCI spec to know about device context index (DCI)
    XhciEndpoint* pXep = nullptr;
    if(deviceContextIndex == 1)
    {
        pXep = pXdc->pControlEndpoint;
    }
    else if(deviceContextIndex & 1)
    {
        pXep = pXdc->pInEndpoints[(deviceContextIndex - 3) >> 1];
    }
    else if(deviceContextIndex != 0)
    {
        pXep = pXdc->pOutEndpoints[(deviceContextIndex - 2) >> 1];
    }

    if((pXep != nullptr) && (pXep->pHcEp == nullptr) && isOperationalRequired)
    {
        // endpoint is not operational
        pXep = nullptr;
    }

    return pXep;
}

Result XhciDriver::SetXhciEndpoint(DeviceContext* pXdc, UsbEndpointDirection epDir,
                                   UsbEndpointType epType, EndpointNumber epNumber,
                                   XhciEndpoint* pXep)
{
    Result result = ResultSuccess();

    NN_ABORT_UNLESS_MINMAX(epNumber, 0, 15);

    if((epNumber == 0) || (epType == UsbEndpointType_Control))
    {
        pXdc->pControlEndpoint = pXep;
    }
    else if(epDir == UsbEndpointDirection_ToDevice)
    {
        pXdc->pOutEndpoints[epNumber - 1] = pXep;
    }
    else // epDir == UsbEndpointDirection_ToHost
    {
        pXdc->pInEndpoints[epNumber - 1] = pXep;
    }

    return result;
}

/*
 * Ref [xHCI r1.1] 6.2.3.6
 */
uint32_t XhciDriver::GetEndpointInterval(HostControllerDriverEndpoint *pHcEp)
{
    uint32_t interval = 0;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();
    UsbEndpointDescriptor* pDescriptor = pHep->pDescriptor;

    switch (pHcEp->GetDeviceSpeed())
    {
    case UsbDeviceSpeed_Low:
        if (UsbEndpointIsInt(pDescriptor))
        {
            uint32_t microFrame = pDescriptor->bInterval * 8;

            // SIGLO-70764: Explicitly check for "!=0", only to make Coverity happy.
            if (microFrame != 0)
            {
                interval = Util::FindLastSetBit(microFrame) - 1;
            }
            else
            {
                NN_USB_ABORT("Invalid endpoint bInterval\n");
            }
        }
        else if (UsbEndpointIsControl(pDescriptor))
        {
            // FIXME: what about low speed ctrl?
            interval = pDescriptor->bInterval;
        }
        else // bulk or isoch
        {
            NN_USB_LOG_WARN("Low speed device cannot use bulk or isoch transfers!\n");
            interval = 0;
        }
        break;

    case UsbDeviceSpeed_Full:
        if (UsbEndpointIsInt(pDescriptor))
        {
            uint32_t microFrame = pDescriptor->bInterval * 8;

            // SIGLO-70764: Explicitly check for "!=0", only to make Coverity happy.
            if (microFrame != 0)
            {
                interval = Util::FindLastSetBit(microFrame) - 1;
            }
            else
            {
                NN_USB_ABORT("Invalid endpoint bInterval\n");
            }
        }
        else if (UsbEndpointIsIsoc(pDescriptor))
        {
            interval = (pDescriptor->bInterval - 1) + 3;
        }
        else // ctrl or bulk
        {
            // FIXME: what about full speed ctrl and bulk?
            interval = pDescriptor->bInterval;
        }
        break;

    case UsbDeviceSpeed_High:
        if (UsbEndpointIsInt(pDescriptor) || UsbEndpointIsIsoc(pDescriptor))
        {
            interval = pDescriptor->bInterval - 1;
        }
        else // ctrl or bulk
        {
            // Max NAK rate
            interval = pDescriptor->bInterval;
        }
        break;

    case UsbDeviceSpeed_Super:
    case UsbDeviceSpeed_SuperPlus:
        if (UsbEndpointIsInt(pDescriptor) || UsbEndpointIsIsoc(pDescriptor))
        {
            interval = pDescriptor->bInterval - 1;
        }
        else // ctrl or bulk
        {
            /*
             * "For SuperSpeed bulk and control endpoints, the Interval field
             *  shall not be used by the xHC."
             */
            interval = 0;
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return interval;
}

uint32_t XhciDriver::GetEndpointMult(HostControllerDriverEndpoint *pHcEp)
{
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();
    if ((pHcEp->GetDeviceSpeed() != UsbDeviceSpeed_Super) || !UsbEndpointIsIsoc(pHep->pDescriptor))
    {
        return 0;
    }
    return pHep->pCompanionDescriptor->bmAttributes & 0x03;
}

uint8_t XhciDriver::GetEndpointCtxDw1EpType(HostEndpoint *pHep)
{
    uint8_t epType = EndpointCtxDw1EpTypeNotValid;
    switch (pHep->epType)
    {
    case UsbEndpointType_Control:
        epType = EndpointCtxDw1EpTypeControl;
        break;
    case UsbEndpointType_Bulk:
        epType = (UsbEndpointIsDeviceToHost(pHep->pDescriptor)) ? EndpointCtxDw1EpTypeBulkIn : EndpointCtxDw1EpTypeBulkOut;
        break;
    case UsbEndpointType_Isoc:
        epType = (UsbEndpointIsDeviceToHost(pHep->pDescriptor)) ? EndpointCtxDw1EpTypeIsochIn : EndpointCtxDw1EpTypeIsochOut;
        break;
    case UsbEndpointType_Int:
        epType = (UsbEndpointIsDeviceToHost(pHep->pDescriptor)) ? EndpointCtxDw1EpTypeIntIn : EndpointCtxDw1EpTypeIntOut;
        break;
    default:
        break;
    }
    return epType;
}

int32_t XhciDriver::GetSlotContextEntryCount(DeviceContext* pXdc)
{
    uint8_t entries = 1;  // Always need one for the default control endpoint0
    for(int32_t i = 31; i >= 2; i--)
    {
        int32_t index = (i - 2) >> 1;
        if(i & 1)
        {
            if((pXdc->pInEndpoints[index] != nullptr) && (pXdc->pInEndpoints[index]->isEnabled))
            {
                entries = i;
                break;
            }
        }
        else
        {
            if((pXdc->pOutEndpoints[index] != nullptr) && (pXdc->pOutEndpoints[index]->isEnabled))
            {
                entries = i;
                break;
            }
        }
    }

    return entries;
}

uint32_t XhciDriver::MakeInputContextControlMask(HostEndpoint *pHep)
{
    uint32_t mask = 0;
    if(UsbGetEndpointNumber(pHep->pDescriptor) == 0)
    {
        mask = InputControlCtxDw1AddEndpoint0Mask;
    }
    else if(UsbEndpointIsHostToDevice(pHep->pDescriptor))
    {
        mask = InputControlCtxDw1AddEndpoint1OutMask << ((UsbGetEndpointNumber(pHep->pDescriptor) - 1) << 1);
    }
    else if(UsbEndpointIsDeviceToHost(pHep->pDescriptor))
    {
        mask = InputControlCtxDw1AddEndpoint1InMask << ((UsbGetEndpointNumber(pHep->pDescriptor) - 1) << 1);
    }
    return mask;
}

Result XhciDriver::InitializeTransferRing(TransferRing* pTr, int32_t trbCount, uint8_t cycleState)
{
    Result result = ResultSuccess();

    do
    {
        int32_t trbCountIncludingLink = trbCount + 1;

        /*
         * Ref [xHCI r1.1] Table 54
         *
         * Transfer Ring shouldn't span 64K boundary. Here we make sure it is
         * page aligned and doesn't span page boundary.
         *
         * It should be checked by upper level. It's a BUG if the invalid
         * parameter reaches here.
         */
        NN_USB_ABORT_UNLESS(trbCountIncludingLink * sizeof(DmaTrb) <= 0x1000);

        memset(pTr, 0, sizeof(TransferRing));
        pTr->totalCount    = trbCount;

        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
            pTr->pMetaTrbList = reinterpret_cast<MetaTrb*>(
                detail::UsbMemoryCalloc(sizeof(MetaTrb) * trbCountIncludingLink, "MetaTrb")
            )
        );

        NN_USB_BREAK_UPON_ERROR(
            InitializeDmaBuffer(&pTr->dmaTrbList,
                                sizeof(DmaTrb) * trbCountIncludingLink,
                                "TransferRing")
        );

        ResetTransferRing(pTr, cycleState);
    } while (false);

    if (result.IsFailure())
    {
        FinalizeTransferRing(pTr);
    }

    return result;
}

void XhciDriver::ResetTransferRing(TransferRing* pTr, uint8_t cycleState)
{
    // transfer ring cannot be running when this is called!
    pTr->freeIndex = 0;
    pTr->pendIndex = 0;
    pTr->pendCount = 0;
    pTr->ccs       = (cycleState) ? 1 : 0;
    memset(pTr->dmaTrbList.u.pVoid, 0, pTr->dmaTrbList.size);

    // If the cycle state is 0, set the cycle bit to 1 for all the TRBs
    if(!cycleState)
    {
        for(int32_t index=0; index < pTr->totalCount; index++)
        {
            pTr->dmaTrbList.u.pTrb[index].dw[3] |= XferTrbDw3CMask;
        }
    }

    // make the last link TRB in the segment point to the first transfer TRB
    pTr->dmaTrbList.u.pTrb[pTr->totalCount].qw[0] = pTr->dmaTrbList.ioVa;
    pTr->dmaTrbList.u.pTrb[pTr->totalCount].dw[3] |=
        (NN_USB_MAKE_MASK32(XferTrbDw3TrbType, XferTrbTypeLink) | LinkTrbDw3TcMask);

    // make coherent
    nn::dd::FlushDataCache(pTr->dmaTrbList.u.pVoid, pTr->dmaTrbList.size);
}

Result XhciDriver::FinalizeTransferRing(TransferRing* pTr)
{
    Result result = ResultSuccess();
    if(pTr->pMetaTrbList != nullptr)
    {
        detail::UsbMemoryFree(pTr->pMetaTrbList, "MetaTrb");
        pTr->pMetaTrbList = nullptr;
    }
    pTr->freeIndex     = 0;
    pTr->pendIndex     = 0;
    pTr->pendCount     = 0;
    pTr->totalCount    = 0;
    result = FinalizeDmaBuffer(&pTr->dmaTrbList);
    return result;
}

uint32_t XhciDriver::GetCurrentFrameId()
{
    return NN_USB_GET_FIELD32(
        RtMfindexIndex, RtRegister.Read32(RtMfindexOffset)
    ) >> 3;
}

uint32_t XhciDriver::CalculateTrbCount(uint32_t dataSize)
{
    return NN_USB_ROUNDUP_SIZE(dataSize, 0x10000) / 0x10000;
}

/*
 * Ref [xHCI r1.1] 4.11.2.4. `n' starts from 0
 */
uint32_t XhciDriver::CalculateTdSize(UsbRequestBlock *pUrb, uint32_t xferNum, uint32_t n)
{
    HostEndpoint *pHep = pUrb->m_pHcEp->GetHostEndpoint();
    uint32_t tdXferSize    = pUrb->m_Xfer[xferNum].size;
    uint32_t trbCount      = CalculateTrbCount(tdXferSize);
    uint32_t maxPacketSize = NN_USB_GET_FIELD32(UsbEndpointMaxPacketSize_Payload,
                                                pHep->pDescriptor->wMaxPacketSize);
    uint32_t tdPacketCount = NN_USB_ROUNDUP_SIZE(tdXferSize, maxPacketSize) / maxPacketSize;

    if (n == trbCount - 1)
    {
        // last TRB
        return 0;
    }
    else
    {
        uint32_t xferredSize = (n + 1) * (1024 * 64);
        uint32_t xferredPacketCount = NN_USB_ROUNDUP_SIZE(xferredSize, maxPacketSize) / maxPacketSize;
        uint32_t tdSize = tdPacketCount - xferredPacketCount;

        return tdSize > 31 ? 31 : tdSize;
    }
}

uint32_t XhciDriver::GetMaxPacketSize(XhciEndpoint *pXep)
{
    HostControllerDriverEndpoint *pHcEp = pXep->pHcEp;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();

    return NN_USB_GET_FIELD32(UsbEndpointMaxPacketSize_Payload,
                              pHep->pDescriptor->wMaxPacketSize);
}

uint32_t XhciDriver::GetMaxBurstSize(XhciEndpoint *pXep)
{
    uint32_t maxBurstSize = 0;
    HostControllerDriverEndpoint *pHcEp = pXep->pHcEp;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();

    switch (pHcEp->GetDeviceSpeed())
    {
    case UsbDeviceSpeed_Super:
        maxBurstSize = pHep->pCompanionDescriptor->bMaxBurst;
        break;

    case UsbDeviceSpeed_High:
        if (UsbEndpointIsIsoc(pHep->pDescriptor) ||
            UsbEndpointIsInt(pHep->pDescriptor))
        {
            maxBurstSize = NN_USB_GET_FIELD32(UsbEndpointMaxPacketSize_TxnPerUframe,
                                              pHep->pDescriptor->wMaxPacketSize);
        }
        break;

    default:
        maxBurstSize = 0;
        break;
    }

    return maxBurstSize;
}

/*
 * Ref [xHCI r1.1] 4.14.1
 *
 * TDPC: Transfer Descriptor Packet Count
 */
uint32_t XhciDriver::CalculateTdpc(XhciEndpoint *pXep, uint32_t tdXferSize)
{
    uint32_t maxPacketSize = GetMaxPacketSize(pXep);
    uint32_t tdpc = NN_USB_ROUNDUP_SIZE(tdXferSize, maxPacketSize) / maxPacketSize;

    // ZLP increments TDPC by 1
    if (tdpc == 0)
    {
        tdpc = 1;
    }

    return tdpc;
}

/*
 * Ref [xHCI r1.1] 4.14.1.1. We use suggested "reasonable initial value".
 */
uint32_t XhciDriver::GetAverageTrbLength(XhciEndpoint *pXep)
{
    HostControllerDriverEndpoint *pHcEp = pXep->pHcEp;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();

    switch (pHep->epType)
    {
    case UsbEndpointType_Control:
        return 8;

    case UsbEndpointType_Int:
        return 1024 * 1;

    case UsbEndpointType_Bulk:
    case UsbEndpointType_Isoc:
        return 1024 * 3;

    default:
        NN_UNEXPECTED_DEFAULT;
        return 0;
    }
}

/*
 * Ref [xHCI r1.1] 4.14.2
 */
uint32_t XhciDriver::GetMaxEsitPayload(XhciEndpoint *pXep)
{
    uint32_t maxEsitPayload = 0;
    uint32_t maxPacketSize = GetMaxPacketSize(pXep);
    uint32_t maxBurstSize  = GetMaxBurstSize(pXep);
    HostControllerDriverEndpoint *pHcEp = pXep->pHcEp;
    HostEndpoint *pHep = pHcEp->GetHostEndpoint();

    // Only applies to interrupt or isochronous endpoints
    if (UsbEndpointIsControl(pHep->pDescriptor) || UsbEndpointIsBulk(pHep->pDescriptor))
    {
        return 0;
    }

    switch (pHcEp->GetDeviceSpeed())
    {
    case UsbDeviceSpeed_Super:
        maxEsitPayload = pHep->pCompanionDescriptor->wBytesPerInterval;
        break;

    case UsbDeviceSpeed_SuperPlus:
        NN_USB_LOG_ERROR("TODO: GetMaxEsitPayload for SSP\n");
        break;

    default:
        maxEsitPayload = maxPacketSize * (maxBurstSize + 1);
        break;
    }

    return maxEsitPayload;
}

/*
 * Ref [xHCI r1.1] 4.11.2.3
 *
 * TBC: TRB Transfer Burst Count
 *
 * TBC = ROUNDUP ( TDPC / ( Max Burst Size + 1 ) ) - 1
 */
uint32_t XhciDriver::CalculateTbc(XhciEndpoint *pXep, uint32_t tdXferSize)
{
    uint32_t maxBurstSize  = GetMaxBurstSize(pXep);
    uint32_t tdpc = CalculateTdpc(pXep, tdXferSize);

    return NN_USB_ROUNDUP_SIZE(tdpc, maxBurstSize + 1) / (maxBurstSize + 1) - 1;
}

/*
 * Ref [xHCI r1.1] 4.11.2.3
 *
 * TLBPC: Transfer Last Burst Packet Count
 *
 * IsochBurstResiduePackets = TDPC MODULUS ( Max Burst Size + 1 )
 * TLBPC = IF ( IsochBurstResiduePackets == 0 )
 *         THEN Max Burst Size
 *         ELSE IsochBurstResiduePackets - 1
 */
uint32_t XhciDriver::CalculateTlbpc(XhciEndpoint *pXep, uint32_t tdXferSize)
{
    uint32_t maxBurstSize = GetMaxBurstSize(pXep);
    uint32_t residue = CalculateTdpc(pXep, tdXferSize) % (maxBurstSize + 1);

    if (residue == 0)
    {
        return maxBurstSize;
    }
    else
    {
        return residue - 1;
    }
}

Result XhciDriver::OpenTd(TransferRing *pTr)
{
    // The first acquired TRB is never the Link TRB
    pTr->freeIndex %= pTr->totalCount;

    return ResultSuccess();
}

int XhciDriver::AcquireTrb(TransferRing *pTr)
{
    if (pTr->freeIndex == pTr->totalCount)
    {
        // This Link TRB must be in the middle of the TD, set CH=1
        DmaTrb& linkTrb = pTr->dmaTrbList.u.pTrb[pTr->totalCount];
        linkTrb.dw[3] = NN_USB_MAKE_VALUE32_5(LinkTrbDw3,
                                              C,       pTr->ccs,
                                              Tc,      1,
                                              Ch,      1,
                                              Ioc,     0,
                                              TrbType, XferTrbTypeLink);
        nn::dd::FlushDataCache(&linkTrb, sizeof(linkTrb));

        pTr->ccs      ^= 1;
        pTr->freeIndex = 0;
    }

    pTr->pendCount++;

    return pTr->freeIndex++;
}

Result XhciDriver::CloseTd(TransferRing *pTr)
{
    if (pTr->freeIndex == pTr->totalCount)
    {
        // It's caller's responsibility to ensure CH of last TRB is 0
        DmaTrb& linkTrb = pTr->dmaTrbList.u.pTrb[pTr->totalCount];
        linkTrb.dw[3] = NN_USB_MAKE_VALUE32_5(LinkTrbDw3,
                                              C,       pTr->ccs,
                                              Tc,      1,
                                              Ch,      0,
                                              Ioc,     0,
                                              TrbType, XferTrbTypeLink);
        nn::dd::FlushDataCache(&linkTrb, sizeof(linkTrb));

        pTr->ccs      ^= 1;
        pTr->freeIndex = 0;
    }

    return ResultSuccess();
}

/*
 * Ref [xHCI r1.1] 3.2.9, 4.11.2.2
 */
Result XhciDriver::HandleControlUrbSubmit(UsbRequestBlock *pUrb)
{
    Result result = ResultSuccess();
    HostControllerDriverEndpoint *pHcEp = pUrb->m_pHcEp;
    //HostEndpoint *pHep = pHcEp->GetHostEndpoint();
    XhciEndpoint *pXep = reinterpret_cast<XhciEndpoint *>(pHcEp->GetDriverEndpoint());
    if(pXep==nullptr) return ResultEndpointStateInvalid();
    TransferRing *pTr = &pXep->transferRing;

    UsbCtrlRequest& ctrlRequest = pUrb->m_CtrlRequest;

    // make sure we have enough free TRBs
    int32_t stages = 2; // setup + status
    if (ctrlRequest.wLength > 0)
    {
        stages += 1; // data
    }
    if (pTr->totalCount - pTr->pendCount < stages)
    {
        return ResultTrbAllocFailure();
    }

    // Prepare the TDs

    int index = pTr->freeIndex;
    DmaTrb  *pDmaTrb      = nullptr;
    MetaTrb *pMetaTrb     = nullptr;
    DmaTrb  *pSetupDmaTrb = nullptr;

    int xferType;
    int dataDir;
    int statusDir;

    switch (pUrb->m_Token)
    {
    case UsbBusToken_Invalid:
        xferType  = XferTrbCtrlSetupStageNoData;
        dataDir   = 0; // actually doesn't care
        statusDir = 1;
        break;

    case UsbBusToken_In:
        xferType  = XferTrbCtrlSetupStageInData;
        dataDir   = 1;
        statusDir = 0;
        break;

    case UsbBusToken_Out:
        xferType  = XferTrbCtrlSetupStageOutData;
        dataDir   = 0;
        statusDir = 1;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    // Setup Stage TD
    OpenTd(pTr);

    index    = AcquireTrb(pTr);
    pDmaTrb  = pTr->dmaTrbList.u.pTrb + index;
    pMetaTrb = pTr->pMetaTrbList      + index;

    pSetupDmaTrb = pDmaTrb;  // save for later use

    pMetaTrb->pUrb          = pUrb;
    pMetaTrb->sequence      = --stages;
    pMetaTrb->remainingSize = 0;
    pMetaTrb->bufSize       = 0;

    std::memcpy(const_cast<uint64_t*>(&pDmaTrb->qw[0]), &ctrlRequest, sizeof(UsbCtrlRequest));

    pDmaTrb->dw[2] = NN_USB_MAKE_VALUE32_2(SetupStageTrbDw2,
                                           TransferLength, 8,
                                           IntTarget,      0);

    // Cycle bit is toggled later when all stages are ready
    pDmaTrb->dw[3] = NN_USB_MAKE_VALUE32_5(SetupStageTrbDw3,
                                           C,       ~pTr->ccs,
                                           Ioc,     0,
                                           Idt,     1,
                                           Trt,     xferType,
                                           TrbType, XferTrbTypeSetupStage);

    // No need to flush data cache here since we will toggle cycle bit later

    CloseTd(pTr);

    // Data Stage TD, if any
    if (xferType != XferTrbCtrlSetupStageNoData)
    {
        OpenTd(pTr);

        index    = AcquireTrb(pTr);
        pDmaTrb  = pTr->dmaTrbList.u.pTrb + index;
        pMetaTrb = pTr->pMetaTrbList      + index;

        pMetaTrb->pUrb          = pUrb;
        pMetaTrb->sequence      = --stages;
        pMetaTrb->remainingSize = 0;
        pMetaTrb->bufSize       = pUrb->m_BufferSize;

        pDmaTrb->qw[0] = pUrb->m_IoVa;

        pDmaTrb->dw[2] = NN_USB_MAKE_VALUE32_3(DataStageTrbDw2,
                                               TransferLength, pUrb->m_BufferSize,
                                               TdSize,         0,
                                               IntTarget,      0);

        pDmaTrb->dw[3] = NN_USB_MAKE_VALUE32_5(DataStageTrbDw3,
                                               C,       pTr->ccs,
                                               Isp,     dataDir ? 1 : 0,
                                               Ch,      0,
                                               TrbType, XferTrbTypeDataStage,
                                               Dir,     dataDir);

        nn::dd::FlushDataCache(pDmaTrb, sizeof(DmaTrb));

        CloseTd(pTr);
    }

    // Status Stage TD
    OpenTd(pTr);

    index    = AcquireTrb(pTr);
    pDmaTrb  = pTr->dmaTrbList.u.pTrb + index;
    pMetaTrb = pTr->pMetaTrbList      + index;

    pMetaTrb->pUrb          = pUrb;
    pMetaTrb->sequence      = --stages;
    pMetaTrb->remainingSize = 0;
    pMetaTrb->bufSize       = 0;

    pDmaTrb->qw[0] = 0;

    pDmaTrb->dw[2] = NN_USB_MAKE_VALUE32_1(StatusStageTrbDw2,
                                           IntTarget,      0);

    /*
     * Ref [xHCI r1.1] 6.4.1.2 Control TRBs
     *   "The IOC flag should only be set in the Status Stage TRB of a Control
     *    transfer"
     */
    pDmaTrb->dw[3] = NN_USB_MAKE_VALUE32_5(StatusStageTrbDw3,
                                           C,       pTr->ccs,
                                           Ch,      0,
                                           Ioc,     1,
                                           TrbType, XferTrbTypeStatusStage,
                                           Dir,     statusDir);

    nn::dd::FlushDataCache(pDmaTrb, sizeof(DmaTrb));

    CloseTd(pTr);

    /*
     * Ref [xHCI r1.1] 3.2.9 Control Transfers
     *   "Software “constructs” a control transfer by placing either two (Setup
     *    Stage and Status Stage), or three (Setup Stage, Data Stage, and Status
     *    Stage) TDs on the Transfer Ring BEFORE ringing the doorbell."
     *
     * Toggle cycle bit of Setup Stage TRB then ring the bell
     */
    pSetupDmaTrb->dw[3] ^= 1;
    nn::dd::FlushDataCache(pSetupDmaTrb, sizeof(DmaTrb));
    DbRegister.Write32(pXep->pDeviceContext->slotId << 2,
                       NN_USB_MAKE_MASK32(DbRegTarget, DbRegTargetEndpoint0));

    return result;
} // NOLINT(impl/function_size)

Result XhciDriver::HandleNormalUrbSubmit(UsbRequestBlock *pUrb)
{
    HostControllerDriverEndpoint *pHcEp = pUrb->m_pHcEp;
    XhciEndpoint *pXep = reinterpret_cast<XhciEndpoint *>(pHcEp->GetDriverEndpoint());
    TransferRing *pTr = &pXep->transferRing;
    uintptr_t dataPtr = pUrb->m_IoVa;
    DmaTrb *firstTrb  = nullptr;
    int totalTrbCount = 0;

    // do we have enough free TRBs?
    for (uint32_t i = 0; i < pUrb->m_XferCount; i++)
    {
        totalTrbCount += CalculateTrbCount(pUrb->m_Xfer[i].size);
    }
    if (pTr->totalCount - pTr->pendCount < totalTrbCount)
    {
        return ResultTrbAllocFailure();
    }

    // submit all the xfer
    for (uint32_t i = 0; i < pUrb->m_XferCount; i++)
    {
#define LAST_XFER (i + 1 == pUrb->m_XferCount)
        uint32_t  dataSize = pUrb->m_Xfer[i].size;
        uint32_t  trbCount = CalculateTrbCount(dataSize);

        OpenTd(pTr);

        // create and chain all necessary TRBs, but leave the cycle bit
        // of the first TRB to be handled in the end
        for (uint32_t j = 0; j < trbCount; j++)
        {
#define LAST_TRB  (j + 1 == trbCount)

            int      index   = AcquireTrb(pTr);
            DmaTrb&  dmaTrb  = pTr->dmaTrbList.u.pTrb[index];
            MetaTrb& metaTrb = pTr->pMetaTrbList[index];

            metaTrb.pUrb          = pUrb;
            metaTrb.sequence      = trbCount - j - 1;
            metaTrb.bufSize       = LAST_TRB ? dataSize : 0x10000;
            metaTrb.remainingSize = dataSize - metaTrb.bufSize;

            dmaTrb.qw[0] = dataPtr;
            dmaTrb.dw[2] = NN_USB_MAKE_VALUE32_2(
                XferTrbDw2,
                TransferLength, metaTrb.bufSize,
                TdSize,         CalculateTdSize(pUrb, i, j)
            );
            dmaTrb.dw[3] = NN_USB_MAKE_VALUE32_6(
                XferTrbDw3,
                C,       firstTrb == nullptr ? ~pTr->ccs : pTr->ccs,
                Isp,     pUrb->m_Token == UsbBusToken_In ? 1 : 0,
                Ch,      LAST_TRB ? 0 : 1,
                Ioc,     LAST_TRB ? 1 : 0,
                Bei,     LAST_XFER ? 0 : 1,
                TrbType, XferTrbTypeNormal
            );

            nn::dd::FlushDataCache(&dmaTrb, sizeof(DmaTrb));

            dataSize = metaTrb.remainingSize;
            dataPtr += metaTrb.bufSize;

            if (firstTrb == nullptr)
            {
                firstTrb = &dmaTrb;
            }
#undef LAST_TRB
        }

        CloseTd(pTr);
#undef LAST_XFER
    }

    // now toggle cycle bit of the first TRB
    firstTrb->dw[3] ^= 1;
    nn::dd::FlushDataCache(firstTrb, sizeof(DmaTrb));

    // ring the bell
    DbRegister.Write32(pXep->pDeviceContext->slotId << 2,
                       NN_USB_MAKE_MASK32(DbRegTarget, pXep->dci));

    return ResultSuccess();
}

Result XhciDriver::HandleIsochUrbSubmit(UsbRequestBlock *pUrb)
{
    Result result = ResultSuccess();
    HostControllerDriverEndpoint *pHcEp = pUrb->m_pHcEp;
    XhciEndpoint *pXep = reinterpret_cast<XhciEndpoint *>(pHcEp->GetDriverEndpoint());
    TransferRing *pTr  = &pXep->transferRing;

    int      totalTrbCount = 0;
    uint32_t xferCount     = 0;

    uintptr_t dataPtr = pUrb->m_IoVa;
    DmaTrb *firstTrb  = nullptr;

    uint32_t esit     = 1 << GetEndpointInterval(pHcEp);
    uint32_t maxEsitPayload = GetMaxEsitPayload(pXep);

    uint32_t frameId;
    uint32_t startFrameId;
    uint32_t endFrameId;
    uint32_t currentFrameId = GetCurrentFrameId();

    uint32_t ist = NN_USB_GET_FIELD32(
        HcsParams2Ist, CapRegister.Read32(HcsParams2Offset)
    );

    for (uint32_t i = 0; i < pUrb->m_XferCount; i++)
    {
        /*
         * Ref [xHCI r1.1] 4.14.2.1
         *   "Software shall not define a TD Transfer Size for a TD of an Isoch
         *    endpoint that exceeds the Max ESIT Payload."
         */
        if (pUrb->m_Xfer[i].size > maxEsitPayload)
        {
            return ResultMaxEsitPayload();
        }

        /*
         * Ref [xHCI r1.1] 4.14.2
         *   "Per the USB specifications, the Maximum Allowed ESIT Payload of a
         *    FS Interrupt, FS Isoch, HS Interrupt, HS Isoch, SS Interrupt, or
         *    SS Isoch periodic pipe are defined as 64B, 1KB, 3KB, 3KB, 3KB, and
         *    48KB, respectively."
         *
         * So a single TRB (64KB) should be sufficient for a non-SS+ Isoch Xfer.
         *
         * TODO: handle SS+
         */
        totalTrbCount += 1;
    }

    // do we have enough free TRBs?
    if (pTr->totalCount - pTr->pendCount < totalTrbCount)
    {
        return ResultTrbAllocFailure();
    }

    if (pUrb->m_Policy == SchedulePolicy_Absolute)
    {
        frameId = pUrb->m_FrameId % 2048;
    }
    else  // SchedulePolicy_Relative
    {
        frameId = (currentFrameId + pUrb->m_FrameId) % 2048;
    }

    // FIXME: double check the calculation below
    if (ist & 0x08) // IST[2...0] is frame count
    {
        startFrameId = (currentFrameId + (ist & 0x07) + 1) % 2048;
    }
    else // IST[2...0] is microframe count
    {
        startFrameId = (currentFrameId + 1 + 1) % 2048;
    }
    endFrameId = (currentFrameId + 895) % 2048;

    /*
     * Ref [xHCI r1.1] 4.11.2.5
     *
     * FrameId must be in the Valid Frame Window.
     */
    if (endFrameId > startFrameId)
    {
        if (frameId < startFrameId || frameId > endFrameId)
        {
            result = ResultNotInFrameWindow();
        }
    }
    else
    {
        if (frameId < startFrameId && frameId > endFrameId)
        {
            result = ResultNotInFrameWindow();
        }
    }

    if (result.IsFailure())
    {
        // NN_USB_LOG_WARN("FrameId not in the Valid Frame Window\n"
        //                 "  - FrameId       : %d\n"
        //                 "  - Window        : [%d, %d]\n"
        //                 "  - CurrentFrameId: %d\n"
        //                 "  - IST           : %d\n",
        //                 frameId,
        //                 startFrameId, endFrameId,
        //                 currentFrameId,
        //                 ist);
        return result;
    }

    // submit all the xfers
    for (uint32_t i = 0; i < pUrb->m_XferCount; i++)
    {
#define LAST_XFER (i + 1 == pUrb->m_XferCount)
        uint32_t  dataSize = pUrb->m_Xfer[i].size;

        OpenTd(pTr);

        int      index   = AcquireTrb(pTr);
        DmaTrb&  dmaTrb  = pTr->dmaTrbList.u.pTrb[index];
        MetaTrb& metaTrb = pTr->pMetaTrbList[index];

        metaTrb.pUrb          = pUrb;
        metaTrb.sequence      = 0;
        metaTrb.bufSize       = dataSize;
        metaTrb.remainingSize = 0;

        dmaTrb.qw[0] = dataPtr;
        dmaTrb.dw[2] = NN_USB_MAKE_VALUE32_1(
            XferTrbDw2,
            TransferLength, metaTrb.bufSize
        );
        dmaTrb.dw[3] = NN_USB_MAKE_VALUE32_8(
            IsochTrbDw3,
            C,       firstTrb == nullptr ? ~pTr->ccs : pTr->ccs,
            Isp,     pUrb->m_Token == UsbBusToken_In ? 1 : 0,
            Ioc,     1,
            Tbc,     CalculateTbc(pXep, metaTrb.bufSize),
            Bei,     LAST_XFER ? 0 : 1,
            TrbType, XferTrbTypeIsoch,
            Tlbpc,   CalculateTlbpc(pXep, metaTrb.bufSize),
            FrameId, (frameId + ((esit * xferCount) >> 3)) % 2048
        );

        nn::dd::FlushDataCache(&dmaTrb, sizeof(DmaTrb));

        if (firstTrb == nullptr)
        {
            firstTrb = &dmaTrb;
        }

        dataPtr += dataSize;

        CloseTd(pTr);

        xferCount++;
#undef LAST_XFER
    }

    // now toggle cycle bit of the first TRB
    firstTrb->dw[3] ^= 1;
    nn::dd::FlushDataCache(firstTrb, sizeof(DmaTrb));

    // ring the bell
    DbRegister.Write32(pXep->pDeviceContext->slotId << 2,
                       NN_USB_MAKE_MASK32(DbRegTarget, pXep->dci));

    return ResultSuccess();
} // NOLINT(impl/function_size)

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
