﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

uint32_t Util::RoundupDmaBufferSize(uint32_t nominalSize)
{
    uint32_t roundedUpSize = NN_USB_ROUNDUP_SIZE(nominalSize, HwLimitDmaBufferAlignmentSize);
    return roundedUpSize;
}

const char* Util::GetClassCodeDescription(uint8_t classCode)
{
    const char *description = "UNKNOWN";
    struct ClassCodeEntry
    {
        uint8_t    code;
        const char *description;
    };
    static const ClassCodeEntry classCodes[] = {
        { 0x00,  "PER INTERFACE"        },
        { 0x01,  "AUDIO"                },
        { 0x02,  "COMM"                 },
        { 0x03,  "HID"                  },
        { 0x05,  "PHYSICAL"             },
        { 0x06,  "STILL IMAGE"          },
        { 0x07,  "PRINTER"              },
        { 0x08,  "MASS STORAGE"         },
        { 0x09,  "HUB"                  },
        { 0x0a,  "CDC DATA"             },
        { 0x0b,  "CSCID"                },
        { 0x0d,  "CONTENT SECURITY"     },
        { 0x0e,  "VIDEO"                },
        { 0x11,  "BILLBOARD"            },
        { 0xe0,  "WIRELESS CONTROLLER"  },
        { 0xef,  "MISC"                 },
        { 0xfe,  "APPLICATION"          },
        { 0xff,  "VENDOR"               },
    };
    for (uint32_t index = 0; index < (sizeof(classCodes) / sizeof(classCodes[0])); index++)
    {
        const ClassCodeEntry *pE = &classCodes[index];
        if (classCode == pE->code)
        {
            description = pE->description;
            break;
        }
    }
    return description;
}

Result Util::GetEpIntervalInUs(UsbDeviceSpeed devSpeed,  UsbEndpointDescriptor *pEpDesc, uint32_t *pRetInterval)
{
    Result result = ResultSuccess();
    uint32_t intervalInUs = 0;

    switch (UsbGetEndpointType(pEpDesc))
    {
    case UsbEndpointType_Control:
        if (devSpeed == UsbDeviceSpeed_High)
        {
            // uframes per NAK
            intervalInUs = pEpDesc->bInterval * 125;
        }
        break;
    case UsbEndpointType_Isoc:
        if (devSpeed == UsbDeviceSpeed_High ||
            devSpeed == UsbDeviceSpeed_Super)
        {
            intervalInUs = (1 << (pEpDesc->bInterval - 1)) * 125;
        }
        else if(devSpeed == UsbDeviceSpeed_Full)
        {
            intervalInUs = (1 << (pEpDesc->bInterval - 1)) * 1000;
        }
        else
        {
            intervalInUs = 1000;
        }
        break;
    case UsbEndpointType_Bulk:
        if (devSpeed == UsbDeviceSpeed_High)
        {
            if (UsbEndpointIsHostToDevice(pEpDesc))
            {
                // uframes per NAK
                intervalInUs = pEpDesc->bInterval * 125;
            }
        }
        break;
    case UsbEndpointType_Int:
        if (devSpeed == UsbDeviceSpeed_High ||
            devSpeed == UsbDeviceSpeed_Super)
        {
            intervalInUs = (1 << (pEpDesc->bInterval - 1)) * 125;
        }
        else
        {
            intervalInUs = pEpDesc->bInterval * 1000;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    *pRetInterval = intervalInUs;

    return result;
}

bool Util::EvaluateDeviceId(DeviceFilter *pFilter, UsbInterfaceDescriptor *pIfDesc,
                            UsbDeviceDescriptor *pDevDesc)
{
    /* Check validity of descriptors, cannot match if device and interface
       is not valid */
    if ((pFilter == nullptr) || (pIfDesc == nullptr) || (pDevDesc == nullptr))
    {
        return false;
    }
    if (pIfDesc->bLength  < UsbDescriptorSize_Interface ||
        pDevDesc->bLength < UsbDescriptorSize_Device     )
    {
        return false;
    }

    /* Compare this registration against the offered device. These checks
       attempt to disqualify based upon client specified filter parameters. */

    if (pFilter->matchFlags & DeviceFilterMatchFlags_Vendor)
    {
        if (pDevDesc->idVendor != pFilter->idVendor)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_Product)
    {
        if (pDevDesc->idProduct != pFilter->idProduct)
        {
            return false;
        }
    }
    // "Lo > Hi" naturally yields no match
    if (pFilter->matchFlags & DeviceFilterMatchFlags_DeviceLo)
    {
        if (pDevDesc->bcdDevice < pFilter->bcdDeviceLo)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_DeviceHi)
    {
        if (pDevDesc->bcdDevice > pFilter->bcdDeviceHi)
        {
            return false;
        }
    }

    if (pFilter->matchFlags & DeviceFilterMatchFlags_DeviceClass)
    {
        if (pDevDesc->bDeviceClass != pFilter->bDeviceClass)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_DeviceSubClass)
    {
        if (pDevDesc->bDeviceSubClass != pFilter->bDeviceSubClass)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_DeviceProtocol)
    {
        if (pDevDesc->bDeviceProtocol != pFilter->bDeviceProtocol)
        {
            return false;
        }
    }

    if (pFilter->matchFlags & DeviceFilterMatchFlags_InterfaceClass)
    {
        if (pIfDesc->bInterfaceClass != pFilter->bInterfaceClass)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_InterfaceSubClass)
    {
        if (pIfDesc->bInterfaceSubClass != pFilter->bInterfaceSubClass)
        {
            return false;
        }
    }
    if (pFilter->matchFlags & DeviceFilterMatchFlags_InterfaceProtocol)
    {
        if (pIfDesc->bInterfaceProtocol != pFilter->bInterfaceProtocol)
        {
            return false;
        }
    }

    return true;
}

UsbEndpointMask Util::GetEndpointMask(UsbEndpointDirection epDir, EndpointNumber epNumber)
{
    UsbEndpointMask mask = 0;
    switch (epDir)
    {
    case UsbEndpointDirection_ToDevice:
        mask = (1 << epNumber);
        break;
    case UsbEndpointDirection_ToHost:
        mask = (1 << (epNumber + UsbLimitMaxEndpointPairCount));
        break;
    case UsbEndpointDirection_Control:
        mask = 1;
        break;
    default:
        mask = 0;
        break;
    }
    return mask;
}

UsbEndpointMask Util::GetEndpointMask(HostEndpoint *pHep)
{
    return Util::GetEndpointMask(pHep->epDir, pHep->epNumber);
}

UsbEndpointMask Util::GetEndpointFromMask(UsbEndpointMask mask, UsbEndpointDirection *pEpDir, EndpointNumber *pEndpointNumber)
{
    UsbEndpointMask foundMask = 0;
    for (int i = 0; i < (UsbLimitMaxEndpointPairCount * 2); i++)
    {
        if ((foundMask = ((1 << i) & mask)))
        {
            if (i < UsbLimitMaxEndpointPairCount)
            {
                *pEpDir    = UsbEndpointDirection_ToDevice;
                *pEndpointNumber = i;
            }
            else
            {
                *pEpDir    = UsbEndpointDirection_ToHost;
                *pEndpointNumber = i - UsbLimitMaxEndpointPairCount;
            }
            break;
        }
    }
    return foundMask;
}

uint32_t Util::GetControlTransferMaxPacketSize(UsbDeviceSpeed devSpeed)
{
    uint32_t size = 0;

    // See [xHCI r1.1] 6.2.3.1, [USB 2.0] 5.5.3, [USB 3.1] 4.4.5.1
    switch(devSpeed)
    {
    case UsbDeviceSpeed_Low:
    case UsbDeviceSpeed_Full:
        size = 8;
        break;
    case UsbDeviceSpeed_High:
        size = 64;
        break;
    case UsbDeviceSpeed_Super:
        size = 512;
        break;
    default:
        break;
    }

    return size;
}

const char* Util::GetEndpointDescription(UsbEndpointDescriptor *epd)
{
    const char *description = "?";
    switch (UsbGetEndpointType(epd))
    {
    case UsbEndpointType_Control:
        description = "CONTROL";
        break;
    case UsbEndpointType_Isoc:
        description = (UsbEndpointIsDeviceToHost(epd)) ? "ISOC-IN" : "ISOC-OUT";
        break;
    case UsbEndpointType_Bulk:
        description = (UsbEndpointIsDeviceToHost(epd)) ? "BULK-IN" : "BULK-OUT";
        break;
    case UsbEndpointType_Int:
        description = (UsbEndpointIsDeviceToHost(epd)) ? "INTERRUPT-IN" : "INTERRUPT-OUT";
        break;
    default:
        break;
    }
    return description;
}

const char* Util::GetEndpointDescription(UsbEndpointType epType, UsbEndpointDirection epDirection)
{
    const char *description = "?";
    switch (epType)
    {
    case UsbEndpointType_Control:
        description = "CONTROL";
        break;
    case UsbEndpointType_Isoc:
        description = (UsbEndpointDirection_ToHost==epDirection) ? "ISOC-IN" : "ISOC-OUT";
        break;
    case UsbEndpointType_Bulk:
        description = (UsbEndpointDirection_ToHost==epDirection) ? "BULK-IN" : "BULK-OUT";
        break;
    case UsbEndpointType_Int:
        description = (UsbEndpointDirection_ToHost==epDirection) ? "INTERRUPT-IN" : "INTERRUPT-OUT";
        break;
    default:
        break;
    }
    return description;
}

const char* Util::GetSpeedDescription(UsbDeviceSpeed speed)
{
    const char *description = "?";
    switch (speed)
    {
    case UsbDeviceSpeed_Low:
        description = "Ls";
        break;
    case UsbDeviceSpeed_Full:
        description = "Fs";
        break;
    case UsbDeviceSpeed_High:
        description = "Hs";
        break;
    case UsbDeviceSpeed_Super:
        description = "Ss";
        break;
    default:
        break;
    }
    return description;
}

void Util::PrintDescriptors(UsbConfigDescriptor *pCfgDesc, UsbDeviceDescriptor *pDevDesc, UsbDeviceSpeed speed,
                            DeviceInterface *ifArray, HostEndpoint *epIn, HostEndpoint *epOut)
{
    NN_SDK_LOG("  Device:\n");
    NN_SDK_LOG("    bLength             = 0x%x\n", pDevDesc->bLength);
    NN_SDK_LOG("    bDescriptorType     = 0x%x\n", pDevDesc->bDescriptorType);
    NN_SDK_LOG("    bcdUSB              = 0x%x\n", pDevDesc->bcdUSB);
    NN_SDK_LOG("    bDeviceClass        = 0x%x (%s)\n", pDevDesc->bDeviceClass,
               Util::GetClassCodeDescription(pDevDesc->bDeviceClass));
    NN_SDK_LOG("    bDeviceSubClass     = 0x%x\n", pDevDesc->bDeviceSubClass);
    NN_SDK_LOG("    bDeviceProtocol     = 0x%x\n", pDevDesc->bDeviceProtocol);
    NN_SDK_LOG("    bMaxPacketSize0     = 0x%x\n", pDevDesc->bMaxPacketSize0);
    NN_SDK_LOG("    idVendor            = 0x%x\n", pDevDesc->idVendor);
    NN_SDK_LOG("    idProduct           = 0x%x\n", pDevDesc->idProduct);
    NN_SDK_LOG("    bcdDevice           = 0x%x\n", pDevDesc->bcdDevice);
    NN_SDK_LOG("    iManufacturer       = 0x%x\n", pDevDesc->iManufacturer);
    NN_SDK_LOG("    iProduct            = 0x%x\n", pDevDesc->iProduct);
    NN_SDK_LOG("    iSerialNumber       = 0x%x\n", pDevDesc->iSerialNumber);
    NN_SDK_LOG("    bNumConfigurations  = 0x%x\n", pDevDesc->bNumConfigurations);
    if (pCfgDesc != nullptr)
    {
        NN_SDK_LOG("  Configuration:\n");
        NN_SDK_LOG("    bLength             = 0x%x\n", pCfgDesc->bLength);
        NN_SDK_LOG("    bDescriptorType     = 0x%x\n", pCfgDesc->bDescriptorType);
        NN_SDK_LOG("    wTotalLength        = 0x%x\n", pCfgDesc->wTotalLength);
        NN_SDK_LOG("    bNumInterfaces      = 0x%x\n", pCfgDesc->bNumInterfaces);
        NN_SDK_LOG("    bConfigurationValue = 0x%x\n", pCfgDesc->bConfigurationValue);
        NN_SDK_LOG("    iConfiguration      = 0x%x\n", pCfgDesc->iConfiguration);
        NN_SDK_LOG("    bmAttributes        = 0x%x\n", pCfgDesc->bmAttributes);
        NN_SDK_LOG("    bMaxPower           = 0x%x\n", pCfgDesc->bMaxPower);
    }
    for (int32_t ifIndex = 0; ifIndex < HsLimitMaxInterfacesPerConfigurationCount; ifIndex++)
    {
        UsbInterfaceDescriptor *pIfDesc;
        DeviceInterface *pDif = &ifArray[ifIndex];
        if ((pIfDesc = pDif->pActAltSetting) != nullptr)
        {
            NN_SDK_LOG("  Interface %d:\n", ifIndex);
            NN_SDK_LOG("    bLength             = 0x%x\n", pIfDesc->bLength);
            NN_SDK_LOG("    bDescriptorType     = 0x%x\n", pIfDesc->bDescriptorType);
            NN_SDK_LOG("    bInterfaceNumber    = 0x%x\n", pIfDesc->bInterfaceNumber);
            NN_SDK_LOG("    bAlternateSetting   = 0x%x\n", pIfDesc->bAlternateSetting);
            NN_SDK_LOG("    bNumEndpoints       = 0x%x\n", pIfDesc->bNumEndpoints);
            NN_SDK_LOG("    bInterfaceClass     = 0x%x (%s)\n", pIfDesc->bInterfaceClass,
                       Util::GetClassCodeDescription(pIfDesc->bInterfaceClass));
            NN_SDK_LOG("    bInterfaceSubClass  = 0x%x\n", pIfDesc->bInterfaceSubClass);
            NN_SDK_LOG("    bInterfaceProtocol  = 0x%x\n", pIfDesc->bInterfaceProtocol);
            NN_SDK_LOG("    iInterface          = 0x%x\n", pIfDesc->iInterface);
            for (EndpointNumber epNumber = 1; epNumber <= Device::MaxUserEndpointPairCount; epNumber++)
            {
                int32_t epIndex = epNumber - 1;
                for (int32_t dir = 0; dir < 2; dir++)
                {
                    HostEndpoint *pHeP = (dir == 0) ? &epIn[epIndex] : &epOut[epIndex];
                    if ((1 << ifIndex) & pHeP->ifMask)
                    {
                        uint32_t intervalInUs = 0;
                        NN_SDK_LOG("    Endpoint %d (%s):\n", epNumber,
                                   Util::GetEndpointDescription(pHeP->pDescriptor));
                        NN_SDK_LOG("      bLength           = 0x%x\n", pHeP->pDescriptor->bLength);
                        NN_SDK_LOG("      bDescriptorType   = 0x%x\n", pHeP->pDescriptor->bDescriptorType);
                        NN_SDK_LOG("      bEndpointAddress  = 0x%x\n", pHeP->pDescriptor->bEndpointAddress);
                        NN_SDK_LOG("      bmAttributes      = 0x%x\n", pHeP->pDescriptor->bmAttributes);
                        NN_SDK_LOG("      wMaxPacketSize    = 0x%x\n", pHeP->pDescriptor->wMaxPacketSize);
                        Util::GetEpIntervalInUs(speed, pHeP->pDescriptor, &intervalInUs);
                        NN_SDK_LOG("      bInterval         = 0x%x (%d us)\n",
                                   pHeP->pDescriptor->bInterval, intervalInUs);
                        if(pHeP->pCompanionDescriptor->bLength != 0)
                        {
                            NN_SDK_LOG("      Companion Descriptor\n");
                            NN_SDK_LOG("        bLength           = 0x%x\n", pHeP->pCompanionDescriptor->bLength);
                            NN_SDK_LOG("        bDescriptorType   = 0x%x\n", pHeP->pCompanionDescriptor->bDescriptorType);
                            NN_SDK_LOG("        bMaxBurst         = 0x%x\n", pHeP->pCompanionDescriptor->bMaxBurst);
                            NN_SDK_LOG("        bmAttributes      = 0x%x\n", pHeP->pCompanionDescriptor->bmAttributes);
                            NN_SDK_LOG("        wBytesPerInterval = 0x%x\n", pHeP->pCompanionDescriptor->wBytesPerInterval);
                        }
                    }
                }
            }
        }
    }
}

void Util::PrintInterfaceQuery(InterfaceQueryOutput *pQuery)
{
    NN_SDK_LOG("Interface with handle=0x%x:\n", pQuery->ifProfile.handle);
    NN_SDK_LOG("  Device debug name = %s\n", pQuery->deviceProfile.deviceDebugName);
    NN_SDK_LOG("  Device speed      = %s\n", Util::GetSpeedDescription(pQuery->deviceProfile.deviceSpeed));
    NN_SDK_LOG("  Device UID        = 0x%x\n", pQuery->deviceProfile.deviceUid);
    NN_SDK_LOG("  Config descriptor:\n");
    NN_SDK_LOG("    bConfigurationValue = 0x%x\n", pQuery->deviceProfile.cfgDesc.bConfigurationValue);
    NN_SDK_LOG("    bmAttributes        = 0x%x\n", pQuery->deviceProfile.cfgDesc.bmAttributes);
    NN_SDK_LOG("    bMaxPower           = 0x%x\n", pQuery->deviceProfile.cfgDesc.bMaxPower);
    NN_SDK_LOG("    bNumInterfaces      = %d\n",   pQuery->deviceProfile.cfgDesc.bNumInterfaces);
    NN_SDK_LOG("  Device descriptor:\n");
    NN_SDK_LOG("    idVendor            = 0x%x\n", pQuery->deviceProfile.deviceDesc.idVendor);
    NN_SDK_LOG("    idProduct           = 0x%x\n", pQuery->deviceProfile.deviceDesc.idProduct);
    NN_SDK_LOG("    bDeviceClass        = 0x%x (%s)\n", pQuery->deviceProfile.deviceDesc.bDeviceClass,
               Util::GetClassCodeDescription(pQuery->deviceProfile.deviceDesc.bDeviceClass));
    NN_SDK_LOG("    bDeviceSubClass     = 0x%x (%s)\n",
               pQuery->deviceProfile.deviceDesc.bDeviceSubClass,
               Util::GetClassCodeDescription(pQuery->deviceProfile.deviceDesc.bDeviceSubClass));
    NN_SDK_LOG("    bDeviceProtocol     = 0x%x\n", pQuery->deviceProfile.deviceDesc.bDeviceProtocol);
    NN_SDK_LOG("    bcdDevice           = 0x%x\n", pQuery->deviceProfile.deviceDesc.bcdDevice);
    NN_SDK_LOG("    bcdbcdUSB           = 0x%x\n", pQuery->deviceProfile.deviceDesc.bcdUSB);
    NN_SDK_LOG("    bNumConfigurations  = 0x%x\n", pQuery->deviceProfile.deviceDesc.bNumConfigurations);
    NN_SDK_LOG("  Interface:\n");
    NN_SDK_LOG("    altSettingCount     = %d\n", pQuery->ifProfile.altSettingCount);
    NN_SDK_LOG("    bInterfaceNumber    = %d\n", pQuery->ifProfile.ifDesc.bInterfaceNumber);
    NN_SDK_LOG("    bInterfaceClass     = 0x%x (%s)\n",  pQuery->ifProfile.ifDesc.bInterfaceClass,
               Util::GetClassCodeDescription(pQuery->ifProfile.ifDesc.bInterfaceClass));
    NN_SDK_LOG("    bInterfaceSubClass  = %d\n", pQuery->ifProfile.ifDesc.bInterfaceSubClass);
    NN_SDK_LOG("    bInterfaceProtocol  = %d\n", pQuery->ifProfile.ifDesc.bInterfaceProtocol);
    NN_SDK_LOG("    bNumEndpoints       = %d\n", pQuery->ifProfile.ifDesc.bNumEndpoints);
    NN_SDK_LOG("    Endpoints:\n");
    for (int32_t i = 0; i < (Device::MaxUserEndpointPairCount * 2); i++)
    {
        int32_t epIndex = i % Device::MaxUserEndpointPairCount;
        EndpointNumber epNumber = epIndex + 1;
        UsbEndpointDescriptor *pEpd = (i < Device::MaxUserEndpointPairCount) ?
            (pQuery->ifProfile.epInDesc + epIndex) : (pQuery->ifProfile.epOutDesc + epIndex);
        NN_UNUSED(epNumber);
        if (UsbEndpointIsValid(pEpd))
        {
            uint32_t intervalInUs = 0;
            NN_SDK_LOG("      Endpoint %d (%s):\n", epNumber, Util::GetEndpointDescription(pEpd));
            NN_SDK_LOG("        bLength           = 0x%x\n", pEpd->bLength);
            NN_SDK_LOG("        bDescriptorType   = 0x%x\n", pEpd->bDescriptorType);
            NN_SDK_LOG("        bEndpointAddress  = 0x%x\n", pEpd->bEndpointAddress);
            NN_SDK_LOG("        bmAttributes      = 0x%x\n", pEpd->bmAttributes);
            NN_SDK_LOG("        wMaxPacketSize    = 0x%x\n", pEpd->wMaxPacketSize);
            Util::GetEpIntervalInUs(pQuery->deviceProfile.deviceSpeed, pEpd, &intervalInUs);
            NN_SDK_LOG("        bInterval         = 0x%x (%d us)\n",
                       pEpd->bInterval, intervalInUs);
        }
    }
}

void Util::PrintInterfaceProfile(InterfaceProfile *pInterfaceProfile)
{
    NN_SDK_LOG("    handle              = 0x%x:\n", pInterfaceProfile->handle);
    NN_SDK_LOG("    altSettingCount     = %d\n", pInterfaceProfile->altSettingCount);
    NN_SDK_LOG("    bInterfaceNumber    = %d\n", pInterfaceProfile->ifDesc.bInterfaceNumber);
    NN_SDK_LOG("    bInterfaceClass     = 0x%x (%s)\n", pInterfaceProfile->ifDesc.bInterfaceClass,
               Util::GetClassCodeDescription(pInterfaceProfile->ifDesc.bInterfaceClass));
    NN_SDK_LOG("    bInterfaceSubClass  = %d\n", pInterfaceProfile->ifDesc.bInterfaceSubClass);
    NN_SDK_LOG("    bInterfaceProtocol  = %d\n", pInterfaceProfile->ifDesc.bInterfaceProtocol);
    NN_SDK_LOG("    bNumEndpoints       = %d\n", pInterfaceProfile->ifDesc.bNumEndpoints);
    NN_SDK_LOG("    Endpoints:\n");
    for (int32_t i = 0; i < (Device::MaxUserEndpointPairCount * 2); i++)
    {
        int32_t epIndex = i % Device::MaxUserEndpointPairCount;
        EndpointNumber epNumber = epIndex + 1;
        UsbEndpointDescriptor *pEpd = (i < Device::MaxUserEndpointPairCount) ?
            (pInterfaceProfile->epInDesc + epIndex) : (pInterfaceProfile->epOutDesc + epIndex);
        NN_UNUSED(epNumber);
        if (UsbEndpointIsValid(pEpd))
        {
            NN_SDK_LOG("      Endpoint %d (%s):\n", epNumber, Util::GetEndpointDescription(pEpd));
            NN_SDK_LOG("        bLength           = 0x%x\n", pEpd->bLength);
            NN_SDK_LOG("        bDescriptorType   = 0x%x\n", pEpd->bDescriptorType);
            NN_SDK_LOG("        bEndpointAddress  = 0x%x\n", pEpd->bEndpointAddress);
            NN_SDK_LOG("        bmAttributes      = 0x%x\n", pEpd->bmAttributes);
            NN_SDK_LOG("        wMaxPacketSize    = 0x%x\n", pEpd->wMaxPacketSize);
            NN_SDK_LOG("        bInterval         = 0x%x\n", pEpd->bInterval);
        }
    }
}

bool Util::IsValidEndpointDescriptor(UsbEndpointDescriptor *pEpDesc, UsbDeviceSpeed speed)
{
    if (pEpDesc->bLength < UsbDescriptorSize_Endpoint)
    {
        NN_USB_LOG_WARN("Endpoint bLength invalid\n");
        return false;
    }

    if (UsbGetEndpointNumber(pEpDesc) == 0)
    {
        NN_USB_LOG_WARN("Endpoint number cannot be zero\n");
        return false;
    }

    auto epType = UsbGetEndpointType(pEpDesc);

    switch (speed)
    {
    case UsbDeviceSpeed_Low:
        if (epType == UsbEndpointType_Int)
        {
            if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 255))
            {
                NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                return false;
            }
        }
        break;

    case UsbDeviceSpeed_Full:
        if (epType == UsbEndpointType_Int)
        {
            if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 255))
            {
                NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                return false;
            }
        }

        if (epType == UsbEndpointType_Isoc)
        {
            if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 16))
            {
                NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                return false;
            }
        }
        break;

    case UsbDeviceSpeed_High:
        if (epType == UsbEndpointType_Isoc || epType == UsbEndpointType_Int)
        {
            if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 16))
            {
                NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                return false;
            }
        }
        break;

    case UsbDeviceSpeed_Super:
    case UsbDeviceSpeed_SuperPlus:
        if (epType == UsbEndpointType_Isoc)
        {
            if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 16))
            {
                NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                return false;
            }
        }
        else if (epType == UsbEndpointType_Int)
        {
            if ((pEpDesc->bmAttributes & 0x30) == 0x10) // Notification Int Ep
            {
                if (!detail::IsInRange<int>(pEpDesc->bInterval, 8, 16))
                {
                    NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                    return false;
                }
            }
            else
            {
                if (!detail::IsInRange<int>(pEpDesc->bInterval, 1, 16))
                {
                    NN_USB_LOG_WARN("Endpoint bInterval out of range\n");
                    return false;
                }
            }
        }

        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return true;
}

// For example: Passing 0x80000001 returns 32
uint32_t Util::FindLastSetBit(uint32_t mask32)
{
    uint32_t bit;
    if (mask32 == 0)
    {
        return 0;
    }
    for (bit = 1; mask32 != 1; bit++)
    {
        mask32 = (uint32_t)mask32 >> 1;
    }
    return (bit);
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
