﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once


namespace nn { namespace usb { namespace hs {

template<class ManagedObjectType, typename HandleType, typename getFailResultType, uint32_t maxHandles, uint32_t sequenceInit>
class SafeHandleManager
{
public:
    SafeHandleManager()
        : m_RollingHandleSequence(sequenceInit)
    {
        memset(m_HandleTable, 0, sizeof(m_HandleTable));
    }
    ~SafeHandleManager()
    {
    }

    Result Assign(ManagedObjectType *pObject, HandleType *pRetHandle)
    {
        Result result = ResultHandleAllocError();
        // find free handle
        for (uint32_t index = 0; index < m_HandleTableEntries; index++)
        {
            if (m_HandleTable[index].pObject == nullptr)
            {
                // increment rolling handle sequence
                if (++m_RollingHandleSequence == 0)
                {
                    m_RollingHandleSequence = 1;
                }

                // put handle in global lookup table
                *pRetHandle = MakeSafeHandle(index);
                m_HandleTable[index].pObject = pObject;
                m_HandleTable[index].handle = *pRetHandle;
                result = ResultSuccess();
                break;
            }
        }

        return result;
    }

    Result Release(HandleType handle)
    {
        Result result = getFailResultType();
        uint32_t index = ParseSafeHandle(handle);
        if (index < m_HandleTableEntries)
        {
            if (m_HandleTable[index].handle == handle)
            {
                m_HandleTable[index].handle  = -1;
                m_HandleTable[index].pObject = nullptr;
                result = ResultSuccess();
            }
        }
        return result;
    }

    ManagedObjectType* Get(HandleType handle)
    {
        ManagedObjectType *pObject = nullptr;
        uint32_t index = ParseSafeHandle(handle);
        if (index < m_HandleTableEntries)
        {
            if (m_HandleTable[index].handle == handle)
            {
                pObject = m_HandleTable[index].pObject;
            }
        }
        return pObject;
    }

private:
    struct HandleTableEntry
    {
        HandleType         handle;
        ManagedObjectType *pObject;
    };
    const uint8_t    m_SafeHandleBits = 16;
    const uint32_t   m_SequenceMask = 0xffffffffUL << m_SafeHandleBits;
    const uint32_t   m_IndexMask    = 0xffffffffUL >> (32 - m_SafeHandleBits);
    const uint32_t   m_HandleTableEntries = maxHandles;
    uint32_t         m_RollingHandleSequence;
    HandleTableEntry m_HandleTable[maxHandles];

    HandleType MakeSafeHandle(uint32_t index)
    {
        return (HandleType)(
            (m_RollingHandleSequence << m_SafeHandleBits) | (index & m_IndexMask)
        );
    }
    uint32_t ParseSafeHandle(HandleType handle)
    {
        return (uint32_t)(
            ((uint32_t)handle) & m_IndexMask
        );
    }
};

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

