﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

// Limit inclusion scope of this big file
#include "usb_HsXusbFirmware-soc-tegra21x.h"
#include "usb_HsXusbFirmware-soc-tegra210b01.h"

namespace nn { namespace usb { namespace hs {

Result PlatformControllerTegra30::Initialize(detail::UsbComplex* pComplex,
                                             detail::UsbController::Config* pConfig) NN_NOEXCEPT
{
    // Initialize base class
    NN_USB_ABORT_UNLESS_SUCCESS(
        PlatformController::Initialize(pComplex, pConfig)
    );

    m_pComplex = static_cast<detail::UsbComplexTegra21x*>(pComplex);

    const auto& platformConfig = m_pPlatform->GetConfig();

    m_IsRootPortPowerControlled = platformConfig.isRootPortPowerControlled;

    // Mariko (T214) uses different firmware than TX1 (T210)
    switch (platformConfig.socName)
    {
    case detail::UsbPlatform::SocName_Tx1:
        m_pFirmwareImage = FirmwareImage21x;
        m_FirmwareSize   = sizeof(FirmwareImage21x);
        break;

    case detail::UsbPlatform::SocName_Mariko:
        m_pFirmwareImage = FirmwareImage210b01;
        m_FirmwareSize   = sizeof(FirmwareImage210b01);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    NN_USB_ABORT_UNLESS_SUCCESS(
        DoSmmu().Map(&m_FalconFirmwareIoVa,
                     m_pFirmwareImage,
                     m_FirmwareSize,
                     nn::dd::MemoryPermission_ReadWrite,
                     nn::dd::MemoryPermission_ReadOnly)
    );

    return ResultSuccess();
}

Result PlatformControllerTegra30::Finalize() NN_NOEXCEPT
{
    NN_USB_ABORT_UNLESS_SUCCESS(DoSmmu().Unmap(m_pFirmwareImage));
    NN_USB_ABORT_UNLESS_SUCCESS(PlatformController::Finalize());

    return ResultSuccess();
}

Result PlatformControllerTegra30::Enable() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    const UsbSsHubDescriptor superspeedHubDescriptor = {
            NN_USB_SS_HUB_DESCRIPTOR_SIZE(1),  // bDescLength
            UsbSsHubDescriptorType,            // bDescriptorType
            m_SuperSpeedPortCount,             // bNbrPorts
            0,                                 // wHubCharacteristics
            5,                                 // bPwrOn2PwrGood
            0,                                 // bHubContrCurrent
            0,                                 // bHubHdrDecLat
            0,                                 // bHubDelay
            {0}                                // deviceRemovable
    };
    const UsbHubDescriptor standardHubDescriptor = {
            NN_USB_HUB_DESCRIPTOR_SIZE(1),     // bDescLength
            UsbHubDescriptorType,              // bDescriptorType
            m_HighSpeedPortCount,              // bNbrPorts
            0,                                 // wHubCharacteristics
            5,                                 // bPwrOn2PwrGood
            0,                                 // bHubContrCurrent
            {0},                               // deviceRemovable
            0,                                 // portPwrCtrlMask
    };

    // Am I the first one to enable the controller?
    if (m_ControllerRefCount > 0)
    {
        m_ControllerRefCount++;

        NN_USB_LOG_INFO("Host Controller Enabled (ref++)\n");
        return ResultSuccess();
    }

    // Enable base class platform controller
    NN_USB_RETURN_UPON_ERROR(PlatformController::Enable());

    // Port power is at known state of all off
    m_RootPortPowerRefCount = 0;

#if defined (NN_BUILD_CONFIG_SPEC_NX)
    // Begin with all root port VBUS OFF (these are active low, so "true" is OFF)
    SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchB1En, true);
    SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchB2En, true);
    SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchB3En, true);

    // Begin with master root port VBUS OFF (active high, so "false" is OFF)
    SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchAEn, false);
#endif

    // Controller capability registers
    CapRegister.Initialize(&Register, 0, 0x10);

    // Enable XUSB subsystem
    NN_USB_RETURN_UPON_ERROR(EnableXusb());

    // Hold the big lock before we proceed
    NN_USB_ABORT_IF_NULL(LockStack());
    NN_UTIL_SCOPE_EXIT {
        UnlockStack();
    };

    // Create XhciDriver
    m_pHostControllerDriver = new XhciDriver(
        m_pHs, StartupMode_Default, this, CapRegister
    );
    if (m_pHostControllerDriver == nullptr)
    {
        result = ResultMemAllocFailure();
        return result;
    }

    NN_UTIL_SCOPE_EXIT {
        if (result.IsFailure())
        {
            delete m_pHostControllerDriver;
            m_pHostControllerDriver = nullptr;
        }
    };

    // Create SuperSpeed Root Hub
    m_pSuperSpeedRootHub = new HubDevice(
        m_pHs, nullptr, nullptr, -1, UsbDeviceSpeed_Super, nullptr, StartupMode_Default
    );
    if (m_pSuperSpeedRootHub == nullptr)
    {
        result = ResultMemAllocFailure();
        return result;
    }

    NN_UTIL_SCOPE_EXIT {
        if (result.IsFailure())
        {
            delete m_pSuperSpeedRootHub;
            m_pSuperSpeedRootHub = nullptr;
        }
    };

    // Create HighSpeed Root Hub
    m_pHighSpeedRootHub = new HubDevice(
        m_pHs, nullptr, nullptr, -1, UsbDeviceSpeed_High, nullptr, StartupMode_Default
    );
    if (m_pHighSpeedRootHub == nullptr)
    {
        result = ResultMemAllocFailure();
        return result;
    }

    NN_UTIL_SCOPE_EXIT {
        if (result.IsFailure())
        {
            delete m_pHighSpeedRootHub;
            m_pHighSpeedRootHub = nullptr;
        }
    };

    // Initialize XhciDriver
    NN_USB_RETURN_UPON_ERROR(m_pHostControllerDriver->Initialize());
    NN_UTIL_SCOPE_EXIT {
        if (result.IsFailure())
        {
            m_pHostControllerDriver->Finalize();
        }
    };

    // Initialize SuperSpped and HighSpeed Root Hub
    NN_USB_ABORT_UPON_ERROR(
        m_pSuperSpeedRootHub->InitializeRootHub(&superspeedHubDescriptor, m_pHostControllerDriver, 1)
    );
    NN_USB_ABORT_UPON_ERROR(
        m_pHighSpeedRootHub->InitializeRootHub (&standardHubDescriptor,   m_pHostControllerDriver, 5)
    );

    detail::UsbController::Enable();

    m_ControllerRefCount++;
    NN_USB_LOG_INFO("Host Controller Enabled\n");
    return ResultSuccess();
} // NOLINT(impl/function_size)

Result PlatformControllerTegra30::Disable() NN_NOEXCEPT
{
    Hs* pHs = nullptr;
    Result result = ResultSuccess();

    // Am I the last one to disable the controller?
    if (m_ControllerRefCount > 1)
    {
        m_ControllerRefCount--;
        NN_USB_LOG_INFO("Host Controller Disabled (ref--)\n");
        return ResultSuccess();
    }

    do
    {
        RootHubTerminationCallbackContext rootHubDestroyContext;

        /*
         * Teardown superspeed hub
         */
        if((pHs = PlatformController::LockStack()) != nullptr)
        {
            if(m_pSuperSpeedRootHub == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultInternalStateError());
            }

            if(m_pHostControllerDriver == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultInternalStateError());
            }

            NN_USB_BREAK_UPON_ERROR(
                m_pSuperSpeedRootHub->TerminateAsync(
                    RootHubTerminationOptions,
                    PlatformControllerTegra30::RootHubTerminationCallback,
                    &rootHubDestroyContext
                )
            );

            PlatformController::UnlockStack();
            pHs = nullptr;
        }

        // Wait for device termination
        rootHubDestroyContext.m_Semaphore.Acquire();
        NN_USB_RETURN_UPON_ERROR(rootHubDestroyContext.m_Status);

        /*
         * Teardown high speed hub
         */
        if((pHs = PlatformController::LockStack()) != nullptr)
        {
            if(m_pHighSpeedRootHub == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultInternalStateError());
            }

            if(m_pHostControllerDriver == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultInternalStateError());
            }

            NN_USB_BREAK_UPON_ERROR(
                m_pHighSpeedRootHub->TerminateAsync(
                    RootHubTerminationOptions,
                    PlatformControllerTegra30::RootHubTerminationCallback,
                    &rootHubDestroyContext
                )
            );
            PlatformController::UnlockStack();
            pHs = nullptr;
        }

        // Wait for device termination
        rootHubDestroyContext.m_Semaphore.Acquire();
        NN_USB_RETURN_UPON_ERROR(rootHubDestroyContext.m_Status);

        /*
         * Cleanup
         */
        if((pHs = PlatformController::LockStack()) != nullptr)
        {
            NN_USB_BREAK_UPON_ERROR(m_pSuperSpeedRootHub->Finalize());
            NN_USB_BREAK_UPON_ERROR(m_pHighSpeedRootHub->Finalize());
            NN_USB_BREAK_UPON_ERROR(m_pHostControllerDriver->Finalize());

            delete m_pSuperSpeedRootHub;
            m_pSuperSpeedRootHub = nullptr;
            delete m_pHighSpeedRootHub;
            m_pHighSpeedRootHub = nullptr;
            delete m_pHostControllerDriver;
            m_pHostControllerDriver = nullptr;

            // No point for Falcon error recovery
            m_pComplex->GetResetEvent()->Clear();

            PlatformController::UnlockStack();
            pHs = nullptr;

            detail::UsbController::Disable();
        }

        /*
         * Disable subsystem.
         * DO NOT LockStack() because the following deadlock could happen:
         *   1. We LockStack() here in PM thread (due to hotplug or power event)
         *   2. ISR thread runs SmiIrqHandler() which blocks on LockStack()
         *   3. DisableXusb() calls FreeIrq(), which waits for current IRQ Handler to finish
         *   4. Deadlock!
         */
        NN_USB_BREAK_UPON_ERROR(DisableXusb());

        // Don't touch any more registers
        CapRegister.Finalize();

    }while (false);

    if(pHs != nullptr)
    {
        PlatformController::UnlockStack();
    }

    // Disable base class platform controller
    if (result.IsSuccess())
    {
        PlatformController::Disable();
        m_ControllerRefCount--;
        NN_USB_LOG_INFO("Host Controller Disabled\n");
    }

    return result;
}

uintptr_t PlatformControllerTegra30::GetBaseAddress() NN_NOEXCEPT
{
    return 0; // not used
}

detail::RegisterBlock* PlatformControllerTegra30::GetBaseRegister() NN_NOEXCEPT
{
    return &Register;
}

Result PlatformControllerTegra30::SetTestMode(uint32_t port, TestMode mode,
                                              int32_t offset, uint32_t *pOutStrength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHostControllerDriver);

    if (port >= m_pPlatform->GetConfig().portCount)
    {
        NN_USB_LOG_WARN("Physical port %d doesn't exist!\n", port);
        return ResultInvalidParameter();
    }

    m_pComplex->OverrideDriveStrength(port, offset, pOutStrength);
    m_pHostControllerDriver->SetTestMode(port, mode);

    return ResultSuccess();
}

Result PlatformControllerTegra30::Bind(Hs* pHs) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_USB_RETURN_UPON_ERROR(PlatformController::Bind(pHs));
    if(IsEnabled() &&  (m_pSuperSpeedRootHub == nullptr))
    {
        result = Enable();
    }

    NN_USB_LOG_INFO("PlatformControllerTegra30::Bind() %d\n", result);
    return result;
}

Result PlatformControllerTegra30::Unbind() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    return result;
}

Result PlatformControllerTegra30::EnableXusb()
{
    Result result = ResultSuccess();

    do
    {
        // 1.05V rail
        // This is managed by Eris PCIe

        // Enable PLLe
        // This is managed by Eris PCIe

        // Release and power XusbSs & XusbHost
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbSs, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbHost, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbSs, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbHost, true));

        // Enable clocks for XusbSs & XusbHost
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbHost, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbSs, true));

        // Program Bar0 Space
        Register.SetField32(XusbHostConfigurationOffset, XusbHostConfigurationEnFpciMask);
        Register.SetField32(XusbCfg4Offset, XusbCfg4BaseAddressMask, 0x70090000);
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(150));
        Register.SetField32(XusbCfg1Offset,
                            XusbCfg1BusMasterMask | XusbCfg1MemorySpaceMask |XusbCfg1IoSpaceMask);
        Register.SetField32(XusbHostIntrMaskOffset, XusbHostIntrMaskIpIntMask);
        Register.SetField32(XusbHostClkgateHysteresisOffset, 0x80);

        // Check the chip ID for sanity
        m_DeviceId = (Register.Read32(XusbCfg0Offset) >> 16) & 0xffff;
        NN_USB_ABORT_UNLESS(XusbCfg0Tegra21xDevId == m_DeviceId);

        // De-assert PEx USB PAD PLL
        // This is managed by Eris PCIe

        // Boot Falcon
        NN_USB_BREAK_UPON_ERROR(FalconLoadFirmware(4, 0xf, 0, false));

        // Tell Falcon to become enabled
        NN_USB_BREAK_UPON_ERROR(FalconSendMessageAsync(FalconMailboxCommand_Enabled, 0));

        // Register Falcon's Management ISR
        nn::os::InitializeInterruptEvent(&m_SmiIrqEvent, Usb3HostSmiInterruptName,
                                         nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_SmiIrqHolder, &m_SmiIrqEvent);
        NN_USB_BREAK_UPON_ERROR(m_pPlatform->RequestIrq(&m_SmiIrqHolder, &m_SmiIrqHandler));

        // Register Host ISR
        nn::os::InitializeInterruptEvent(&m_HostIrqEvent, Usb3HostInterruptName,
                                         nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_HostIrqHolder, &m_HostIrqEvent);
        NN_USB_BREAK_UPON_ERROR(m_pPlatform->RequestIrq(&m_HostIrqHolder, &m_HostIrqHandler));
    } while (false);

    return result;
}

Result PlatformControllerTegra30::DisableXusb()
{
    Result result = ResultSuccess();

    do
    {
        // Unregister Host ISR
        m_pPlatform->FreeIrq(&m_HostIrqHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HostIrqHolder);
        nn::os::FinalizeInterruptEvent(&m_HostIrqEvent);

        // Unregister Falcon's Management ISR
        m_pPlatform->FreeIrq(&m_SmiIrqHolder);
        nn::os::FinalizeMultiWaitHolder(&m_SmiIrqHolder);
        nn::os::FinalizeInterruptEvent(&m_SmiIrqEvent);

        // Reset Falcon to make it just stop
        CsbWrite(XusbFalcCpuCtlOffset, XusbFalcCpuCtlHresetMask);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        // Deinit firmware
        CsbWrite(XusbCsbMpIloadBaseLoOffset, 0);
        CsbWrite(XusbCsbMpIloadBaseHiOffset, 0);

        // Clear any pending ISRs
        Register.Write32(XusbCfgAruSmiIntrOffset, Register.Read32(XusbCfgAruSmiIntrOffset));

        // Disable PCI related
        Register.SetField32(XusbHostClkgateHysteresisOffset, 0x14);
        Register.Write32(XusbHostIntrMaskOffset, 0);
        Register.ClearField32(XusbCfg1Offset,
                              XusbCfg1BusMasterMask | XusbCfg1MemorySpaceMask |XusbCfg1IoSpaceMask);
        Register.SetField32(XusbCfg4Offset, XusbCfg4BaseAddressMask, 0);
        Register.ClearField32(XusbHostConfigurationOffset, XusbHostConfigurationEnFpciMask);

        // Assert PEx USB PAD PLL reset
        // This is managed by Eris PCIe

        // Put IP into reset
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbHost, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbSs, true));

        // Stop IP clocks
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbHost, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbSs, false));

        // Disable IP power
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbHost, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbSs, false));

        // Disable PLLe
        // This is managed by Eris PCIe

        // Poweroff power rails
        // PowerDomain_Max77620_Ldo1 is managed by Eris PCIe

    } while (false);

    return result;
}

uint32_t PlatformControllerTegra30::CsbRead(uint32_t address)
{
    uint32_t val32;
    Register.Write32(XusbCfgAruC11CsbrangeOffset,
                     address >> XusbCfgAruC11CsbrangePageSelectShift);

    val32 = Register.Read32(XusbCfgCsbBaseAddrOffset +
                            (XusbCfgCsbBaseAddrPageOffsetMask & address));
    return val32;
}

void PlatformControllerTegra30::CsbWrite(uint32_t address, uint32_t data)
{
    Register.Write32(XusbCfgAruC11CsbrangeOffset,
                     address >> XusbCfgAruC11CsbrangePageSelectShift);
    Register.Write32(XusbCfgCsbBaseAddrOffset +
                     (XusbCfgCsbBaseAddrPageOffsetMask & address),
                     data);
}

Result PlatformControllerTegra30::FalconLoadFirmware(int32_t ssPadCount, uint8_t ssPortMap,
                                                     uint8_t hsicPortCount, bool isResetAru)
{
    Result result = ResultSuccess();
    FirmwareConfigTable *pFirmwareConfig;

    do
    {
        pFirmwareConfig = reinterpret_cast<FirmwareConfigTable*>(m_pFirmwareImage);

        // Program SS port map config
        pFirmwareConfig->ss_portmap = static_cast<uint8_t>(((1 << ssPadCount) - 1) & ssPortMap);
        pFirmwareConfig->num_hsic_port = hsicPortCount;

        // First thing, reset the ARU. By the time we get to
        // loading boot code below, reset would be complete.
        // alternatively we can busy wait on rst pending bit.

        // Don't reset during ELPG/LP0 exit path
        if (isResetAru)
        {
            Register.Write32(XusbCfgAruRstOffset, 1);
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1500));
        }

        if (CsbRead(XusbCsbMpIloadBaseLoOffset) != 0)
        {
            NN_USB_LOG_ERROR("Firmware already loaded, Falcon state 0x%x\n",
                             CsbRead(XusbFalcCpuCtlOffset));
            result = ResultInternalStateError();
            break;
        }

        NN_USB_LOG_INFO("Booting Falcon firmare version %02x.%02x %s from 0x%x (+0x%x).\n",
                        (pFirmwareConfig->version_id >> 24) & 0xff,
                        (pFirmwareConfig->version_id >> 16) & 0xff,
                        (pFirmwareConfig->build_log == 0x1) ? "debug" : "release",
                        m_FalconFirmwareIoVa, m_FirmwareSize);

        // Program the size of DFI into ILOAD_ATTR
        CsbWrite(XusbCsbMpIloadAttrOffset, m_FirmwareSize);

        // Make firmware image in memory coherenty
        nn::dd::FlushDataCache(pFirmwareConfig, m_FirmwareSize);

        // Boot code of the firmware reads the ILOAD_BASE_LO register
        // to get to the start of the dfi in system memory.
        CsbWrite(XusbCsbMpIloadBaseLoOffset, m_FalconFirmwareIoVa + sizeof(FirmwareConfigTable));
        CsbWrite(XusbCsbMpIloadBaseHiOffset, 0);

        // Set BOOTPATH to 1 in APMAP Register. Bit 31 is APMAP_BOOTMAP
        CsbWrite(XusbCsbMpApmapOffset, XusbCsbMpApmapBootpathMask);

        // Invalidate L2IMEM
        CsbWrite(XusbCsbMpL2imemopTrigOffset, XusbCsbMpL2imemopTrigInvalidateAllMask);

        // Initiate fetch of Bootcode from system memory into L2IMEM.
        // Program BootCode location and size in system memory.
        CsbWrite(XusbCsbMpL2imemopSizeOffset,
                 NN_USB_MAKE_MASK32(XusbCsbMpL2imemopSizeSrcOffset, pFirmwareConfig->boot_codetag / ImemBlockSize) |
                 NN_USB_MAKE_MASK32(XusbCsbMpL2imemopSizeSrcCount, pFirmwareConfig->boot_codesize / ImemBlockSize));

        // Trigger L2IMEM Load operation
        CsbWrite(XusbCsbMpL2imemopTrigOffset, XusbCsbMpL2imemopTrigLoadLockedResultMask);

        // Setup Falcon Auto-fill
        CsbWrite(XusbFalcImfillCtlOffset,
                 NN_USB_ROUNDUP_SIZE(pFirmwareConfig->boot_codesize,ImemBlockSize) / ImemBlockSize);
        CsbWrite(XusbFalcImfillRng1Offset,
                 NN_USB_MAKE_MASK32(XusbFalcImfillRng1TagLo, pFirmwareConfig->boot_codetag / ImemBlockSize) |
                 NN_USB_MAKE_MASK32(XusbFalcImfillRng1TagHi,
                                    ((pFirmwareConfig->boot_codetag + pFirmwareConfig->boot_codesize) / ImemBlockSize) - 1));
        CsbWrite(XusbFalcDmactlOffset, 0);

        // Wait for RESULT_VLD to get set
        bool isL2imemopresultValid = false;
        os::Tick timeoutTick = nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(10)) + nn::os::GetSystemTick();
        do
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(55));
            isL2imemopresultValid = (CsbRead(XusbCsbMempoolL2imemopResultOffset) &
                                     XusbCsbMempoolL2imemopResultValidMask) ? true : false;
        }while(!isL2imemopresultValid && (nn::os::GetSystemTick() <= timeoutTick));

        if(!isL2imemopresultValid)
        {
            NN_USB_LOG_ERROR("DMA controller not ready 0x08%x\n",CsbRead(XusbCsbMempoolL2imemopResultOffset));
            result = ResultHardwareTimeout();
            break;
        }
        CsbWrite(XusbFalcBootvecOffset, pFirmwareConfig->boot_codetag);

        // Start Falcon CPU
        CsbWrite(XusbFalcCpuCtlOffset, XusbFalcCpuCtlStartCpuMask);
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1500));
        uint32_t cpuCtlVal = CsbRead(XusbFalcCpuCtlOffset);
        NN_USB_LOG_INFO("Falcon state 0x%x.\n", cpuCtlVal);
        if(XusbFalcCpuCtlHaltedMask & cpuCtlVal)
        {
            result = ResultHardwareTimeout();
            break;
        }

    }while(false);

    return result;
}

Result PlatformControllerTegra30::FalconSendMessageAsync(FalconMailboxCommand command, uint32_t data)
{
    Result result = ResultSuccess();

    do
    {
        // Wait for Falcon to become idle
        if(!Register.PollField32(XusbCfgAruMboxOwnerOffset, XusbCfgAruMboxOwnerIdMask, 0,
                                 nn::TimeSpan::FromMilliSeconds(20), 20))
        {
            NN_USB_LOG_WARN("PlatformControllerTegra30::FalconSendMessageAsync() - timeout waiting for idle.\n");
            result = ResultResourceBusy();
            break;
        }

        // Acquire
        Register.Write32(XusbCfgAruMboxOwnerOffset, XusbCfgAruMboxOwnerIdSoftware);
        if(!Register.PollField32(XusbCfgAruMboxOwnerOffset, XusbCfgAruMboxOwnerIdMask, XusbCfgAruMboxOwnerIdSoftware,
                                 nn::TimeSpan::FromMilliSeconds(20), 20))
        {
            NN_USB_LOG_WARN("PlatformControllerTegra30::FalconSendMessageAsync() - timeout waiting for acquire.\n");
            result = ResultResourceBusy();
            break;
        }

        // Write the data + the command
        Register.Write32(XusbCfgAruMboxDataInOffset,
                         NN_USB_MAKE_MASK32(XusbCfgAruMboxDataInData, data) |
                         NN_USB_MAKE_MASK32(XusbCfgAruMboxDataInType, command));

        // Set interrupt enables
        Register.SetField32(XusbCfgAruMboxCmdOffset,
                            XusbCfgAruMboxCmdIntEnMask | XusbCfgAruMboxCmdFalconIntEnMask);

    }while(false);

    return result;
}

void PlatformControllerTegra30::FalconHandleIrq()
{
    // Sample status
    uint32_t status = Register.Read32(XusbCfgAruSmiIntrOffset);

    // Clear pending
    Register.Write32(XusbCfgAruSmiIntrOffset, status);

    // Falcon
    if(XusbCfgAruSmiIntrFwReinitMask & status)
    {
        /*
         * Falcon could hang when we tearing down host stack, so we have to
         * check this due to execution race.
         */
        if(m_pHostControllerDriver != nullptr)
        {
            m_pHostControllerDriver->OnControllerStall();

            // Let the PM to do the serialization for us
            m_pComplex->RequestReset();

            NN_USB_LOG_ERROR("Falcon died: Try to recover by resetting the controller!\n");
        }
        else
        {
            NN_USB_LOG_ERROR("Falcon died: Ignored becasue host stack is being torn down!\n");
        }

        return;
    }

    // Mailbox
    if(XusbCfgAruSmiIntrMboxMask & status)
    {
        // What Falcon sent to us
        uint32_t mboxOutValue = Register.Read32(XusbCfgAruMboxDataOutOffset);
        FalconMailboxCommand command = static_cast<FalconMailboxCommand>
            (NN_USB_GET_FIELD32(XusbCfgAruMboxDataOutType, mboxOutValue));
        uint32_t data = NN_USB_GET_FIELD32(XusbCfgAruMboxDataOutData, mboxOutValue);

        // What we may send back to Falcon
        uint32_t replyData = data;
        FalconMailboxCommand replyCommand = FalconMailboxCommand_Invalid;

        NN_USB_LOG_TRACE("Mailbox: command=%d, data=0x%x.\n", command, data);

        switch(command)
        {
        case FalconMailboxCommand_Enabled:
            break;
        case FalconMailboxCommand_IncFalcClock:
        case FalconMailboxCommand_DecFalcClock:
        case FalconMailboxCommand_IncSspiClock:
        case FalconMailboxCommand_DecSspiClock:
            // T210 ignores this for now
            replyCommand = FalconMailboxCommand_Ack;
            break;
        case FalconMailboxCommand_SetBw:
        {
            uint32_t bytesPerSecond = data << 10;
            NN_USB_LOG_TRACE("Mailbox: Falcon bandwidth request of %d kB/s.\n",
                            (bytesPerSecond + 1023) / 1024);
            break;
        }
        case FalconMailboxCommand_SetSsPwrGating:
        case FalconMailboxCommand_SetSsPwrUngating:
            break;
        case FalconMailboxCommand_SaveDfeCtleCtx:
            break;
        case FalconMailboxCommand_StartHsicIdle:
            break;
        case FalconMailboxCommand_StopHsicIdle:
            break;
        case FalconMailboxCommand_DbcWakeStack:
        case FalconMailboxCommand_HsicPretendConnect:
        case FalconMailboxCommand_ResetSspi:
        case FalconMailboxCommand_DisableSsLpfsDetection:
            break;
        case FalconMailboxCommand_EnableSsLpfsDetection:
            break;
        default:
            NN_USB_LOG_WARN("Mailbox: Unhandled response type %d.\n",
                            command);
            break;
        }

        // Is a response required?
        if(replyCommand == FalconMailboxCommand_Invalid)
        {
            // No, just reset interrupt and clear ownership of mailbox
            Register.ClearField32(XusbCfgAruMboxCmdOffset,
                                  XusbCfgAruMboxCmdSmiIntEnMask);
            Register.Write32(XusbCfgAruMboxOwnerOffset, 0);
        }
        else
        {
            // Reply with command and data determined in cases above
            Register.Write32(XusbCfgAruMboxDataInOffset,
                             NN_USB_MAKE_MASK32(XusbCfgAruMboxDataInType, replyCommand) |
                             NN_USB_MAKE_MASK32(XusbCfgAruMboxDataInData, replyData));
            Register.SetField32(XusbCfgAruMboxCmdOffset,
                                XusbCfgAruMboxCmdIntEnMask |
                                XusbCfgAruMboxCmdFalconIntEnMask);
        }
    }
}

void PlatformControllerTegra30::SmiIrqHandler()
{
    Hs* pHs = nullptr;
    if((pHs = PlatformController::LockStack()) != nullptr)
    {
        FalconHandleIrq();
        PlatformController::UnlockStack();
    }
}

void PlatformControllerTegra30::HostIrqHandler()
{
    Hs* pHs = nullptr;
    if((pHs = PlatformController::LockStack()) != nullptr)
    {
        if(m_pHostControllerDriver != nullptr)
        {
            m_pHostControllerDriver->Isr();
        }
        PlatformController::UnlockStack();
    }
}

void PlatformControllerTegra30::RootHubTerminationCallback(Device *pDevice, Result status, void *context)
{
    RootHubTerminationCallbackContext* pContext = reinterpret_cast<RootHubTerminationCallbackContext*>(context);
    pContext->m_Status = status;
    pContext->m_Semaphore.Release();
}

int PlatformControllerTegra30::SetRootHubPortPower(HubPortNumber hubPortNumber, bool isPowerOn) NN_NOEXCEPT
{
    int refCount = PlatformController::SetRootHubPortPower(hubPortNumber, isPowerOn);

#if defined (NN_BUILD_CONFIG_SPEC_NX)
    bool isActionNeeded = true;
    nn::gpio::GpioPadName gpioPadName;

    switch(hubPortNumber)
    {
        case 2:
            gpioPadName = nn::gpio::GpioPadName_UsbSwitchB1En;
            break;
        case 3:
            gpioPadName = nn::gpio::GpioPadName_UsbSwitchB2En;
            break;
        case 4:
            gpioPadName = nn::gpio::GpioPadName_UsbSwitchB3En;
            break;
        default:
            isActionNeeded = false;
            break;
    }

    if(isActionNeeded)
    {
        if((refCount == 0) && !isPowerOn)
        {
            // Turn off port control (active low)
            NN_USB_LOG_INFO("Root port %d power OFF.\n", hubPortNumber);
            SetGpioOutputValue(gpioPadName, true);

            // If all ports are off, turn off the master (active high)
            m_RootPortPowerRefCount--;
            if(m_RootPortPowerRefCount == 0)
            {
                NN_USB_LOG_INFO("Root port common power OFF.\n");
                SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchAEn, false);
            }
        }
        else if((refCount == 1) && isPowerOn)
        {
            // If this is the first port to turn on, turn on the master (active high)
            if(m_RootPortPowerRefCount == 0)
            {
                NN_USB_LOG_INFO("Root port common power ON.\n");
                SetGpioOutputValue(nn::gpio::GpioPadName_UsbSwitchAEn, true);
            }
            m_RootPortPowerRefCount++;

            // Turn on port control (active low)
            NN_USB_LOG_INFO("Root port %d power ON.\n", hubPortNumber);
            SetGpioOutputValue(gpioPadName, false);
        }
    }
#endif

    return refCount;
}

void PlatformControllerTegra30::SetGpioOutputValue(nn::gpio::GpioPadName gpioPadName, bool isHigh) NN_NOEXCEPT
{
    if(m_IsRootPortPowerControlled)
    {
        nn::gpio::GpioPadSession session;
        nn::gpio::OpenSession(&session, gpioPadName);
        nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
        nn::gpio::SetValue(&session, isHigh ? nn::gpio::GpioValue_High : nn::gpio::GpioValue_Low);
        nn::gpio::CloseSession(&session);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10)); // delay requested by NCL hardware
    }
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
