﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

//--------------------------------------------------------------------------
//  HubDevice implementation
//--------------------------------------------------------------------------

void* HubDevice::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, Device::ObjectMemAlignmentSize, "HubDevice");
}

void HubDevice::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "HubDevice");
}

Result HubDevice::InitializeRootHub(const UsbHubDescriptor* pDescriptor, HostControllerDriver *pHostControllerDriver,
                                    HubPortNumber basePortNumber)
{
    Result result = ResultSuccess();
    m_IsRootHub               = true;
    m_IsSuperSpeedHub         = false;
    m_RootBasePortNumber      = basePortNumber;
    m_HubDescriptor.standard  = *pDescriptor;
    m_pHc                     = pHostControllerDriver;
    m_PortCount               = pDescriptor->bNbrPorts;

    result = Device::InitializeAsync(nullptr, nullptr);

    return result;
}

Result HubDevice::InitializeRootHub(const UsbSsHubDescriptor* pDescriptor, HostControllerDriver *pHostControllerDriver,
                                    HubPortNumber basePortNumber)
{
    Result result = ResultSuccess();
    m_IsRootHub               = true;
    m_IsSuperSpeedHub         = true;
    m_RootBasePortNumber      = basePortNumber;
    m_HubDescriptor.super     = *pDescriptor;
    m_pHc                     = pHostControllerDriver;
    m_PortCount               = pDescriptor->bNbrPorts;

    result = Device::InitializeAsync(nullptr, nullptr);

    return result;
}

Result HubDevice::Finalize()
{
    Result result = ResultSuccess();
    NN_USB_LOG_INFO("HubDevice::Finalize()\n");

    if (m_IsRootHub)
    {
        m_pHc->ClearPortEventHandler(this);
    }

    for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_Ports); i++)
    {
        if (m_Ports[i] != nullptr)
        {
            m_Ports[i]->Finalize();
            delete m_Ports[i];
            m_Ports[i] = nullptr;
        }
    }

    if (m_pDmaBuffers != nullptr)
    {
        m_pPlatform->DoSmmu().Unmap(m_pDmaBuffers, Util::RoundupDmaBufferSize(sizeof(DmaBuffers)));
        detail::UsbMemoryFree(m_pDmaBuffers, "DmaBuffers");
        m_pDmaBuffers = nullptr;
    }

    if(result.IsSuccess())
    {
        result = Device::Finalize();
    }

    return result;
}

int HubDevice::GetObjectName(char *pObjName, uint32_t maxSize)
{
    char hcName[HsLimitMaxDebugNameSize] = { 0 };
    m_pHc->GetObjectName(hcName, sizeof(hcName));
    if(m_IsRootHub)
    {
        return nn::util::SNPrintf(pObjName, maxSize,
                                  "%sRootHubDevice%d-L%d/P%d/AXX",
                                  Util::GetSpeedDescription(GetRootDevice()->GetSpeed()),
                                  Device::GetUid(), m_Level, m_ParentPortNumber);
    }
    else
    {
        return nn::util::SNPrintf(pObjName, maxSize,
                                  "%sHubDevice%d-L%d/P%d/A%02x",
                                  Util::GetSpeedDescription(GetRootDevice()->GetSpeed()),
                                  Device::GetUid(), m_Level, m_ParentPortNumber, m_Address);
    }

}

HubDevice::Port* HubDevice::GetParentPort(Device *pChildDevice)
{
    HubDevice::Port *pParentPort = nullptr;
    HubDevice *pParentHub;
    if ((pParentHub = static_cast<HubDevice *>(pChildDevice->GetParentDevice())) != nullptr)
    {
        HubPortNumber parentPortNum = pChildDevice->GetParentPort();
        if ((parentPortNum > 0) && ((unsigned)parentPortNum <= NN_USB_ARRAY_COUNT32(pParentHub->m_Ports)))
        {
            pParentPort = pParentHub->m_Ports[parentPortNum - 1];
        }
    }
    return pParentPort;
}

Result HubDevice::StartHub()
{
    Result result = ResultSuccess();

    do
    {
        // Create & initialize ports
        for (int32_t i = 0; i < m_PortCount; i++)
        {
            if ((m_Ports[i] = new Port(m_pHs, this, i + 1)) == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultMemAllocFailure());
            }
            NN_USB_BREAK_UPON_ERROR(m_Ports[i]->Initialize());
        }
        if (result.IsSuccess())
        {
            for (int32_t i = 0; i < m_PortCount; i++)
            {
                m_Ports[i]->QueueFsmEvent(Port::Event_Try);
            }
        }
    } while (false);

    return result;
}

Result HubDevice::ArmPortIndicationInterrupt()
{
    Result result = ResultSuccess();
    if ( !m_IsPortIndicationArmed)
    {
        if (m_IsRootHub)
        {
            if(!m_IsRootHubPortEventConfigured)
            {
                uint32_t portMask = 0;
                for(int32_t i=0; i < m_PortCount; i++)
                {
                    portMask |= (1 << (m_RootBasePortNumber + i));
                }
                m_pHc->SetPortEventHandler(this, StaticRootHubPortEventCallback, portMask);
                m_IsRootHubPortEventConfigured = true;
            }
            m_IsPortIndicationArmed = true;
        }
        else if (m_pIntHep != nullptr)
        {
            HostControllerDriverEndpoint *pHcEp = m_pIntHep->pHcEp;
            UsbRequestBlock *pUrb = nullptr;
            if ((result = pHcEp->CreatePrivateUrb(nn::dd::GetCurrentProcessHandle(),
                                                  (uint64_t)&m_pDmaBuffers->hubEventBuffer,
                                                  NN_USB_HUB_EVENT_XFER_SIZE(m_PortCount),
                                                  false,
                                                  UsbBusToken_In,
                                                  StaticExternalHubPortEventCallback,
                                                  this,
                                                  0,
                                                  &pUrb)).IsSuccess())
            {
                if ((result = pUrb->Submit()).IsSuccess())
                {
                    m_IsPortIndicationArmed = true;
                }
                else
                {
                    pHcEp->DestroyUrb(pUrb);
                }
            }
        }
    }
    return result;
}

void HubDevice::HandlePortEvent(void *pBitmap, uint32_t bitmapSize)
{
    if (bitmapSize != 0)
    {
        for (int32_t portIndex = 0; (portIndex < m_PortCount); portIndex++)
        {
            uint32_t maskOffset = (portIndex + 1) / 8;
            if (maskOffset < bitmapSize)
            {
                uint8_t mask = ((uint8_t *)pBitmap)[maskOffset];
                HubPortNumber portNumber = portIndex + 1;
                if (mask & (1 << (portNumber - (maskOffset * 8))))
                {
                    m_Ports[portIndex]->Fsm::QueueFsmEvent(Port::Event_PortIndication);
                }
            }
            else
            {
                NN_USB_LOG_ERROR("HubDevice::HandlePortEvent() bad bitmapSize of %d.\n",
                                 bitmapSize);
                break;
            }
        }
    }
}

void HubDevice::HandleRootHubPortEvent(uint32_t portChangeMask)
{
    m_IsPortIndicationArmed = false;

    /* hub with active removal option is being taken down and should no longer
       honor indications */
    if (GetTerminationOptions() == 0)
    {
        uint32_t bitmap = portChangeMask >> (m_RootBasePortNumber - 1);
        HandlePortEvent(&bitmap, sizeof(bitmap));
    }
}

void HubDevice::HandleExternalHubPortEvent(UsbRequestBlock *pUrb)
{
    Result result;
    m_IsPortIndicationArmed = false;
    auto& xfer = pUrb->m_Xfer[0];

    // hub with active removal option is being taken down and should no longer
    // honor indications
    if (GetTerminationOptions() != 0)
    {
        return;
    }

    // this means that the hub is going away
    if (ResultEndpointClosed::Includes(xfer.result))
    {
        return;
    }

    if (xfer.result.IsSuccess())
    {
        HandlePortEvent(&m_pDmaBuffers->hubEventBuffer, xfer.xferredSize);
        ArmPortIndicationInterrupt();
    }
    else
    {
        NN_USB_LOG_UPON_ERROR(xfer.result);
        Reset();
    }
}

bool HubDevice::IsCreatingDevice()
{
    bool isInProgress = false;
    for (int32_t portIndex = 0; portIndex < m_PortCount; portIndex++)
    {
        Port *pPort = m_Ports[portIndex];
        if (pPort != nullptr)
        {
            Device *pChildDevice = nullptr;
            if ((isInProgress = pPort->IsCreatingDevice()))
            {
                break;
            }
            if ((pChildDevice = pPort->m_pChildDevice) != nullptr)
            {
                if (pChildDevice->GetDeviceClass() == UsbClass_Hub)
                {
                    HubDevice *pHub = static_cast<HubDevice *>(pChildDevice);
                    if ((isInProgress = pHub->IsCreatingDevice()))
                    {
                        break;
                    }
                }
            }
        }
    }
    return isInProgress;
}

HubDevice::Port* HubDevice::GrantExclusivity()
{
    Port *pPortGranted = nullptr;
    for (int32_t portIndex = 0; portIndex < m_PortCount; portIndex++)
    {
        Port *pPort = m_Ports[portIndex];
        if (pPort != nullptr)
        {
            Device *pChildDevice = nullptr;
            if ((pChildDevice = pPort->m_pChildDevice) != nullptr)
            {
                if (pChildDevice->GetDeviceClass() == UsbClass_Hub)
                {
                    HubDevice *pHub = static_cast<HubDevice *>(pChildDevice);
                    if ((pPortGranted = pHub->GrantExclusivity()) != nullptr)
                    {
                        break;
                    }
                }
            }
            else if (pPort->IsWaitingForExclusivity())
            {
                pPort->GrantExclusivity();
                pPortGranted = pPort;
                break;
            }
        }
    }
    return pPortGranted;
}

bool HubDevice::IsRootHubCreatingDevice()
{
    bool isInProgress;
    HubDevice* pRootHub = this;
    HubDevice* pHub = this;
    while((pHub = static_cast<HubDevice*>(pHub->GetParentDevice())) != nullptr)
    {
        pRootHub = pHub;
    }
    isInProgress = pRootHub->IsCreatingDevice();

    return isInProgress;
}

void HubDevice::HandlePortTerminationDone(Port *pPort)
{
    Device::QueueFsmEvent(Device::Event_DeviceSpecificComplete);
}

void HubDevice::ReEvaluateEnumerationExclusivity()
{
    if ( !IsRootHubCreatingDevice())
    {
        GrantExclusivity();
    }
}


Result HubDevice::DeviceSpecificInitStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    enum DeviceSpecificInitSubstate
    {
        DeviceSpecificInitSubstate_Start = 0,
        DeviceSpecificInitSubstate_ReadingHubDescriptor,
        DeviceSpecificInitSubstate_SettingHubDepth,
        DeviceSpecificInitSubstate_UpdatingDeviceContext,
        DeviceSpecificInitSubstate_InitializingEndpoint,
        DeviceSpecificInitSubstate_End,
    };

   /*
    * Substate handling
    */
    switch(Fsm::GetSubstate())
    {
    case DeviceSpecificInitSubstate_Start:
    {
        GetObjectName(m_ObjectName, sizeof(m_ObjectName));
        m_pPlatform             = m_pHc->GetPlatformController();
        m_IsPortIndicationArmed = false;
        m_IsSuperSpeedHub       = (GetSpeed() == UsbDeviceSpeed_Super);
        if(!m_IsRootHub)
        {
            HubDevice* pRootHub = static_cast<HubDevice*>(GetRootDevice());
            m_RootBasePortNumber = pRootHub->m_RootBasePortNumber;
        }
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
            m_pDmaBuffers = reinterpret_cast<DmaBuffers *>(
                detail::UsbMemoryCallocAligned(Util::RoundupDmaBufferSize(sizeof(DmaBuffers)),
                                               HwLimitDmaBufferAlignmentSize,
                                               "DmaBuffers")
            )
        );
        NN_USB_BREAK_UPON_ERROR(m_pPlatform->DoSmmu().Map(nullptr, m_pDmaBuffers,
                                                          Util::RoundupDmaBufferSize(sizeof(DmaBuffers))));
        if(m_IsRootHub)
        {
            Fsm::SetSubstate(DeviceSpecificInitSubstate_End);
        }
        else
        {
            Fsm::AddSubstate(1);
        }
        Fsm::QueueFsmEvent(Device::Event_Try);
        break;
    }
    case DeviceSpecificInitSubstate_ReadingHubDescriptor:
    {
        if(pEvent->eventId == Device::Event_Try)
        {
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetDescriptor,
                                              BmRequestType(Class, Device, DeviceToHost),
                                              (((m_IsSuperSpeedHub) ? UsbSsHubDescriptorType : UsbHubDescriptorType) << 8) | 0, 0,
                                              (m_IsSuperSpeedHub) ? sizeof(UsbSsHubDescriptor) : sizeof(UsbHubDescriptor),
                                              &m_pDmaBuffers->descriptorBuffers, false,
                                              ControlTransferCompletionCallback,
                                              this, HsLimitControlXferTimeout)
            );
        }
        else if(pEvent->eventId == Device::Event_ControlTransferComplete)
        {
            uint32_t transferredSize = pEvent->args[0].size;
            result = pEvent->status;

            // We expect short transfer
            if (ResultShortPacket::Includes(result))
            {
                if (((transferredSize >= NN_USB_SS_HUB_DESCRIPTOR_SIZE(1)) && m_IsSuperSpeedHub) ||
                    ((transferredSize >= NN_USB_HUB_DESCRIPTOR_SIZE(1)) && !m_IsSuperSpeedHub))
                {
                    result = ResultSuccess();
                }
            }
            NN_USB_BREAK_UPON_ERROR(result);

            // Accept and parse descriptor
            memcpy(&m_HubDescriptor, &m_pDmaBuffers->descriptorBuffers, sizeof(m_HubDescriptor));
            m_PortCount = (m_IsSuperSpeedHub) ? m_HubDescriptor.super.bNbrPorts :
                m_HubDescriptor.standard.bNbrPorts;
            Fsm::AddSubstate(1);
            Fsm::QueueFsmEvent(Device::Event_Try);
        }
        break;
    }
    case DeviceSpecificInitSubstate_SettingHubDepth:
    {
        if(pEvent->eventId == Device::Event_Try)
        {
           if(!m_IsSuperSpeedHub)
           {
               Fsm::AddSubstate(1);
               Fsm::QueueFsmEvent(Device::Event_Try);
               break;
           }
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_SetHubDepth,
                                              BmRequestType(Class, Device, HostToDevice),
                                              static_cast<uint16_t>(GetHierarchicalLevel() - 1), 0, 0,
                                              nullptr, false, ControlTransferCompletionCallback,
                                              this, HsLimitControlXferTimeout)
            );
        }
        else if(pEvent->eventId == Device::Event_ControlTransferComplete)
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            Fsm::AddSubstate(1);
            Fsm::QueueFsmEvent(Device::Event_Try);
        }
        break;
    }
    case DeviceSpecificInitSubstate_UpdatingDeviceContext:
    {
        if(pEvent->eventId == Device::Event_Try)
        {
            uint16_t wHubCharacteristics = (m_IsSuperSpeedHub) ? m_HubDescriptor.super.wHubCharacteristics :
                m_HubDescriptor.standard.wHubCharacteristics;

            // Update device context with hub specific parameters
            HostControllerDriverDeviceContext* pDeviceContext = GetHostControllerDriverDeviceContext();
            pDeviceContext->hub.isMultiTtCapable   = false; // we never enable the multi-tt
            pDeviceContext->hub.portCount          = m_PortCount;
            pDeviceContext->hub.maxExitLatencyInUs = 0;
            pDeviceContext->hub.ttThinkTime        = (GetSpeed() == UsbDeviceSpeed_High) ?
                NN_USB_GET_FIELD32(HubCharacteristicsTtThinkTime, wHubCharacteristics) : 0;

            m_pHc->UpdateDeviceContext(pDeviceContext);

            Fsm::AddSubstate(1);
            Fsm::QueueFsmEvent(Device::Event_Try);
        }
        break;
    }
    case DeviceSpecificInitSubstate_InitializingEndpoint:
    {
        if(pEvent->eventId == Device::Event_Try)
        {
            // External hubs need INT endpoint
            EndpointNumber epNum = -1;
            for (EndpointNumber i = 0; i < MaxUserEndpointPairCount; i++)
            {
                if (UsbEndpointIsValid(m_EpIn[i].pDescriptor) && UsbEndpointIsInt(m_EpIn[i].pDescriptor))
                {
                    epNum = i + 1;
                    break;
                }
            }
            NN_USB_BREAK_UNLESS(epNum > 0, ResultInvalidDescriptor());
            int32_t epIndex                 = epNum - 1;
            m_EpIn[epIndex].epDir           = UsbEndpointDirection_ToHost;
            m_EpIn[epIndex].epType          = UsbEndpointType_Int;
            m_EpIn[epIndex].epNumber        = epNum;
            m_EpIn[epIndex].maxUrbCount     = 2;
            NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(m_EpIn[epIndex].pHcEp = new HostControllerDriverEndpoint(m_pHs, m_pHc));
            result = m_EpIn[epIndex].pHcEp->InitializeAsync(&m_EpIn[epIndex], m_pHcDevContext,
                                                            Device::EndpointInitializeDoneCallback, this);
            if (result.IsFailure())
            {
                delete m_EpIn[epIndex].pHcEp;
                m_EpIn[epIndex].pHcEp = nullptr;
                NN_USB_BREAK_UPON_ERROR(result);
            }
            m_pIntHep = &m_EpIn[epIndex];
        }
        else if(pEvent->eventId == Device::Event_EndpointInitializeComplete)
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            Fsm::AddSubstate(1);
            Fsm::QueueFsmEvent(Device::Event_Try);
        }
        break;
    }
    case DeviceSpecificInitSubstate_End:
    {
        if(pEvent->eventId == Device::Event_Try)
        {
            // Check init status
            NN_USB_BREAK_UPON_ERROR(pEvent->status);

            // Start the hub
            NN_USB_BREAK_UPON_ERROR(StartHub());

            // Allow device enum to proceed
            Fsm::SetState(State_Configured);
        }
        break;
    }
    default:
        break;
    }

   /*
    * Superstate event handling
    */
    switch(pEvent->eventId)
    {
    case Device::Event_Error:
    {
        Device::ReportInitializeProgress(InitializeProgress_Error);
        result = ResultSuccess();
        break;
    }
    case Device::Event_TerminateRequest:
    {
        Fsm::SetState(Device::State_DeviceSpecificFinalize);
        result = ResultSuccess();
        break;
    }
    default:
        break;
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Device::Event_Error, result);
    }

    return result;
} // NOLINT(impl/function_size)'

Result HubDevice::DeviceSpecificFinalizeStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            for (int32_t i = 0; i < HsLimitMaxHubPortsCount; i++)
            {
                if (m_Ports[i] != nullptr)
                {
                    m_Ports[i]->Terminate();
                    Fsm::AddSubstate(1);
                }
            }
            if (Fsm::GetSubstate() == 0)
            {
                Fsm::QueueFsmEvent(Device::Event_Continue);
            }
            break;
        }
    case Device::Event_DeviceSpecificComplete:
        {
            if (Fsm::AddSubstate(-1) == 0)
            {
                Fsm::QueueFsmEvent(Device::Event_Continue);
            }
            break;
        }
    case Device::Event_Continue:
        {
            Fsm::SetState(State_FinalizingSelf);
            break;
        }
    case Device::Event_EndpointTerminateComplete:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_EpDestroyPendMask &= ~Util::GetEndpointMask(pHep);

            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

//--------------------------------------------------------------------------
//  HubDevice::Port implementation
//--------------------------------------------------------------------------

const char *HubDevice::Port::StateNames[HubDevice::Port::State_Maximum] =
{
    "Null",
    "Init",
    "PendingSelectiveEnum",
    "PoweringOn",
    "ClearingAll",
    "NotConnected",
    "ConnectionDebounce",
    "WaitingForExclusivity",
    "ResettingPort",
    "ResetRecovery",
    "InitHcContext",
    "ReadDeviceDescriptor",
    "CreatingChildDevice",
    "ChildDeviceActive",
    "DestroyingChildDevice",
    "TerminatingPort",
};

const char *HubDevice::Port::EventNames[HubDevice::Port::Event_Maximum] =
{
    "Try",
    "ReTry",
    "Skip",
    "PrepareExit",
    "Done",
    "EndpointInitializeComplete",
    "EndpointTerminateComplete",
    "ControlTransferComplete",
    "Ready",
    "Exclusive",
    "PortStatus",
    "PortFeatureOpDone",
    "PortIndication",
    "Error",
    "Reset",
    "Freeze",
    "Suspend",
    "Resume",
    "Activate",
    "TerminateRequest",
    "ChildDeviceCreateComplete",
    "ChildDeviceCreateFail",
    "Timeout",
    "ChildDeviceTerminateComplete",
    "HostControllerDeviceContextCreate",
    "HostControllerDeviceContextDestroy",
    "ReadMaxPacketSize0",
    "UpdateEp0",
    "ReadDeviceDescriptor",
};

void* HubDevice::Port::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, Device::ObjectMemAlignmentSize, "HubDevice::Port");
}

void HubDevice::Port::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "HubDevice::Port");
}

Result HubDevice::Port::Initialize()
{
    Result result = ResultSuccess();

    do
    {
        char fsmName[HsLimitMaxDebugNameSize];

        if(m_pHub->m_IsRootHub)
        {
            m_HcPortNumber = (m_pHub->m_RootBasePortNumber - 1) + m_PortNumber;
        }
        GetObjectName(fsmName, HsLimitMaxDebugNameSize);
        NN_USB_BREAK_UPON_ERROR(Fsm::Initialize(fsmName,
                                                Port::EventNames, Port::Event_Maximum,
                                                Port::StateNames, Port::State_Maximum,
                                                Port::State_Init));
        Fsm::InitStaticTimedFsmEvent(&m_TimedReadyEvent, Port::Event_Ready);
        Fsm::InitStaticTimedFsmEvent(&m_TimeoutEvent, Port::Event_Timeout);
        Fsm::InitStaticTimedFsmEvent(&m_EpTerminationRetryTimer, Port::Event_PrepareExit);
        m_IsPortStatusQueryPending = false;
    }while (false);


    return result;
}

void HubDevice::Port::Terminate()
{
    m_TerminationPending = true;
    Fsm::QueueFsmEvent(Event_TerminateRequest);
}

void HubDevice::Port::Finalize()
{
    Fsm::Finalize();
}

void HubDevice::Port::GetObjectName(char *pObjName, uint32_t maxSize)
{
    if (m_pHub->m_IsRootHub)
    {
        nn::util::SNPrintf(pObjName, maxSize,
                           "%sRootHubDevice P%d",
                           Util::GetSpeedDescription(m_pHub->GetRootDevice()->GetSpeed()),
                           m_PortNumber);
    }
    else
    {
        nn::util::SNPrintf(pObjName, maxSize,
                           "%sHubDevice%d-L%d/PP%d/A%02x P%d",
                           Util::GetSpeedDescription(m_pHub->GetRootDevice()->GetSpeed()),
                           m_pHub->GetUid(),
                           m_pHub->m_Level, m_pHub->m_ParentPortNumber,
                           m_pHub->m_Address, m_PortNumber);
    }

}

bool HubDevice::Port::IsTerminated()
{
    return (Fsm::GetState() == Port::State_Null) ? true : false;
}

bool HubDevice::Port::IsCreatingDevice()
{
    bool isInProgress = false;

    switch (Fsm::GetState())
    {
    case Port::State_ResettingPort:
    case Port::State_ResetRecovery:
    case Port::State_InitializeHostControllerDeviceContext:
    case Port::State_ReadDeviceDescriptor:
        isInProgress = true;
        break;
    case Port::State_CreatingChildDevice:
        if ( !m_isChildDeviceAddressingDone)
        {
            isInProgress = true;
        }
        break;
    default:
        break;
    }

    return isInProgress;
}

bool HubDevice::Port::IsWaitingForExclusivity()
{
    return (Fsm::GetState() == Port::State_WaitingForExclusivity) ? true : false;
}

void HubDevice::Port::GrantExclusivity()
{
    Fsm::QueueFsmEvent(Port::Event_Exclusive);
}

Result HubDevice::Port::IssueAdministrativeCommand(AdministrativeCommand command)
{
    Result result = ResultSuccess();
    switch(command)
    {
    case AdministrativeCommand_Freeze:
        Fsm::QueueFsmEvent(Port::Event_Freeze);
        break;
    case AdministrativeCommand_Reset:
        Fsm::QueueFsmEvent(Port::Event_Reset);
        break;
    case AdministrativeCommand_Suspend:
        Fsm::QueueFsmEvent(Port::Event_Suspend);
        break;
    case AdministrativeCommand_Resume:
        Fsm::QueueFsmEvent(Port::Event_Resume);
        break;
    case AdministrativeCommand_Activate:
        Fsm::QueueFsmEvent(Port::Event_Activate);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
    return result;
}

void HubDevice::Port::SetFeatureAsync(uint8_t portFeat)
{
    Result result = ResultSuccess();
    HostControllerDriver::PortAttribute portAttr = HostControllerDriver::PortAttribute_Invalid;
    switch (portFeat)
    {
    case UsbHubPortFeature_Reset:
        portAttr = HostControllerDriver::PortAttribute_Reset;
        break;
    case UsbHubPortFeature_Power:
        portAttr = HostControllerDriver::PortAttribute_Power;
        break;
    case UsbHubPortFeature_Suspend:
        portAttr = HostControllerDriver::PortAttribute_Suspend;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
    if (result.IsSuccess())
    {

        if (m_pHub->m_IsRootHub)
        {
            if(UsbHubPortFeature_Power == portFeat)
            {
                m_pHub->m_pPlatform->SetRootHubPortPower(m_PortNumber, true);
            }
            Fsm::QueueFsmEvent(Event_PortFeatureOpDone,
                               m_pHub->m_pHc->SetRootHubPortAttribute(m_HcPortNumber, portAttr, true));
        }
        else
        {
            result = m_pHub->m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_SetFeature,
                                                           BmRequestType(Class, Other, HostToDevice),
                                                           portFeat, m_PortNumber,
                                                           0, nullptr, false,
                                                           PortFeatureCompletionCallback,
                                                           this, HsLimitControlXferTimeout);
            if(result.IsSuccess())
            {
                m_PendingFeatOptCount++;
            }
        }
    }
    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Port::Event_Error, result);
    }
}

void HubDevice::Port::ClearFeatureAsync(uint8_t portFeat)
{
    Result result = ResultSuccess();
    uint16_t wPortChange = 0;
    bool isRhOnly = false;
    HostControllerDriver::PortAttribute portAttr = HostControllerDriver::PortAttribute_Invalid;
    switch (portFeat)
    {
    case UsbHubPortFeature_CConnetion:
        wPortChange = UsbPortStatusChange_Connection;
        break;
    case UsbHubPortFeature_CEnable:
        wPortChange = UsbPortStatusChange_Enable;
        break;
    case UsbHubPortFeature_CReset:
        wPortChange = UsbPortStatusChange_Reset;
        break;
    case UsbHubPortFeature_Power:
        portAttr = HostControllerDriver::PortAttribute_Power;
        break;
    case UsbHubPortFeature_Suspend:
        portAttr = HostControllerDriver::PortAttribute_Suspend;
        break;
    case UsbHubPortFeature_CSuspend:
        wPortChange = UsbPortStatusChange_Suspend;
        break;
    case UsbHubPortFeature_Reset:
        portAttr = HostControllerDriver::PortAttribute_Reset;
        isRhOnly = true;
        break;
    case UsbHubPortFeature_Enable:
        portAttr = HostControllerDriver::PortAttribute_Enable;
        break;
    case UsbHubPortFeature_CPortLinkState:
        wPortChange = UsbPortStatusChange_LinkState;
        break;
    case UsbHubPortFeature_CBhPortReset:
        wPortChange = UsbPortStatusChange_BhReset;
        break;
    case UsbHubPortFeature_CPortConfigError:
        wPortChange = UsbPortStatusChange_ConfigError;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
    if (result.IsSuccess())
    {
        if (m_pHub->m_IsRootHub)
        {
            if(UsbHubPortFeature_Power == portFeat)
            {
                m_pHub->m_pPlatform->SetRootHubPortPower(m_PortNumber, false);
            }
            if (portAttr != HostControllerDriver::PortAttribute_Invalid)
            {
                Fsm::QueueFsmEvent(Event_PortFeatureOpDone,
                                   m_pHub->m_pHc->SetRootHubPortAttribute(m_HcPortNumber, portAttr, false));
            }
            else
            {
                Fsm::QueueFsmEvent(Event_PortFeatureOpDone,
                                   m_pHub->m_pHc->ClearRootHubPortStatus(m_HcPortNumber, wPortChange));
            }
        }
        else if ( !isRhOnly)
        {
            result = m_pHub->m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_ClearFeature,
                                                           BmRequestType(Class, Other, HostToDevice),
                                                           portFeat, m_PortNumber,
                                                           0, nullptr, false,
                                                           PortFeatureCompletionCallback,
                                                           this, HsLimitControlXferTimeout);
            if(result.IsSuccess())
            {
                m_PendingFeatOptCount++;
            }
        }
        else
        {
            Fsm::QueueFsmEvent(Event_PortFeatureOpDone, result);
        }
    }
    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Port::Event_Error, result);
    }
}

Result HubDevice::Port::QueryPortStatusAsync()
{
    Result result = ResultSuccess();

    if ( !m_IsPortStatusQueryPending)
    {
        if (m_pHub->m_IsRootHub)
        {
            if ((result = m_pHc->GetRootHubPortStatus(m_HcPortNumber, &m_PortStatus)).IsSuccess())
            {
                Fsm::QueueFsmEvent(Event_PortStatus);
            }
        }
        else
        {
            result = m_pHub->m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetStatus,
                                                           BmRequestType(Class, Other, DeviceToHost),
                                                           0, m_PortNumber,
                                                           sizeof(UsbHubPortStatus), m_pDmaPortStatus, false,
                                                           PortStatusCallback,
                                                           this, HsLimitControlXferTimeout);
        }

        if (result.IsSuccess())
        {
            m_IsPortStatusQueryPending = true;
        }
        else
        {
            Fsm::QueueFsmEvent(Port::Event_Error, result);
        }
    }

    return result;
}

UsbDeviceSpeed HubDevice::Port::GetPortSpeed()
{
    UsbDeviceSpeed speed = UsbDeviceSpeed_Invalid;

    do
    {
        // Root hubs let host controller driver decide directly
        if (m_pHub->m_IsRootHub)
        {
            speed = m_pHc->GetPortSpeed(m_HcPortNumber);
            break;
        }

        // Not valid if no connection
        if(!(m_PortStatus.wPortStatus & UsbPortStatus_Connection))
        {
            break;
        }

        switch(m_pHub->GetSpeed())
        {
        case UsbDeviceSpeed_Super:
            // Superspeed port cannot do lower speeds
            speed = UsbDeviceSpeed_Super;
            break;
        case UsbDeviceSpeed_High:
            if (m_PortStatus.wPortStatus & UsbPortStatus_HighSpeed)
            {
                speed = UsbDeviceSpeed_High;
            }
            else if(m_PortStatus.wPortStatus & UsbPortStatus_LowSpeed)
            {
                speed = UsbDeviceSpeed_Low;
            }
            else
            {
                speed = UsbDeviceSpeed_Full;
            }
            break;
        case UsbDeviceSpeed_Full:
            speed = (m_PortStatus.wPortStatus & UsbPortStatus_LowSpeed) ?
                UsbDeviceSpeed_Low : UsbDeviceSpeed_Full;
            break;
        case UsbDeviceSpeed_Low:
            speed = UsbDeviceSpeed_Low;
            break;
        default:
            break;
        }
    }while(false);

    return speed;
}

Result HubDevice::Port::CreateChildDevice(UsbDeviceDescriptor *pDevDesc)
{
    Result result = ResultSuccess();

    if (m_pChildDevice == nullptr)
    {
        switch (pDevDesc->bDeviceClass)
        {
        case UsbClass_Hub:
            if ((m_pChildDevice = new HubDevice(m_pHs, m_pHcDevContext, m_pHub, m_PortNumber, m_PortSpeed,
                                                pDevDesc, m_pHub->GetStartupMode())) != nullptr)
            {
                result = m_pChildDevice->InitializeAsync(ChildDeviceInitializeProgressCallback, this);
            }
            break;
        default:
            if ((m_pChildDevice = new Device(m_pHs, m_pHcDevContext, m_pHub, m_PortNumber, m_PortSpeed,
                                             pDevDesc, m_pHub->GetStartupMode())) != nullptr)
            {
                result = m_pChildDevice->InitializeAsync(ChildDeviceInitializeProgressCallback, this);
            }
            break;
        }
        if (m_pChildDevice == nullptr)
        {
            result = ResultMemAllocFailure();
        }
        else if ( !result.IsSuccess() && (m_pChildDevice != nullptr))
        {
            delete m_pChildDevice;
            m_pChildDevice = nullptr;
        }
    }
    else
    {
        result = ResultInternalStateError();
    }

    return result;
}

void HubDevice::Port::ChildDeviceInitializeProgressCallback(InitializeProgress progress, void *context)
{
    Port *pParentPort = reinterpret_cast<Port *>(context);
    HubDevice *pHub = pParentPort->m_pHub;
    switch (progress)
    {
    case InitializeProgress_AddressAssigned:
        pParentPort->m_isChildDeviceAddressingDone = true;
        pHub->ReEvaluateEnumerationExclusivity();
        break;
    case InitializeProgress_Complete:
        pParentPort->Fsm::QueueFsmEvent(Port::Event_ChildDeviceCreateComplete);
        break;
    default:
    case InitializeProgress_Error:
        pParentPort->Fsm::QueueFsmEvent(Port::Event_ChildDeviceCreateFail);
        break;
    }
}

void HubDevice::Port::ChildDeviceTerminateDoneCallback(Device *pDevice, Result status, void *context)
{
    Port *pParentPort = reinterpret_cast<Port *>(context);
    pParentPort->Fsm::QueueFsmEvent(Port::Event_ChildDeviceTerminateComplete, status);
}

void HubDevice::Port::HostControllerDriverEndpointInitializeDoneCallback(Hs* pHs, Result status,
                                                                         HostControllerDriverEndpoint* pHcEp, void* context)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_EndpointInitializeComplete, status);
}

void HubDevice::Port::HostControllerDriverEndpointTerminateDoneCallback(Hs *pHs, Result status, HostControllerDriverEndpoint *pEp, void *context)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_EndpointTerminateComplete, status);
}

void HubDevice::Port::ReadMaxPacketSize0CompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_UpdateEp0, status, transferredSize);
}

void HubDevice::Port::UpdateEp0CompletionCallback(Hs* pHs, Result result, void* context)
{
    HubDevice::Port* pThis = reinterpret_cast<HubDevice::Port*>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_ReadDeviceDescriptor, result);
}

void HubDevice::Port::ControlTransferCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_ControlTransferComplete, status, transferredSize);
}

void HubDevice::Port::PortFeatureCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->m_PendingFeatOptCount--;
    pThis->Fsm::QueueFsmEvent(Port::Event_PortFeatureOpDone, status);
}

void HubDevice::Port::PortStatusCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    HubDevice::Port *pThis = reinterpret_cast<HubDevice::Port *>(context);
    pThis->m_PortStatus = *pThis->m_pDmaPortStatus;
    pThis->Fsm::QueueFsmEvent(Port::Event_PortStatus, status);
}

void HubDevice::Port::CreateHostControllerDeviceContextCallback(Hs* pHs, Result result,
                                                                HostControllerDriverDeviceContext* pDeviceContext,
                                                                void* context)
{
    HubDevice::Port* pThis = reinterpret_cast<HubDevice::Port*>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_HostControllerDeviceContextCreate, result, pDeviceContext);
}

void HubDevice::Port::DestroyHostControllerDeviceContextCallback(Hs* pHs, Result result, void* context)
{
    HubDevice::Port* pThis = reinterpret_cast<HubDevice::Port*>(context);
    pThis->Fsm::QueueFsmEvent(Port::Event_HostControllerDeviceContextDestroy, result);
}

Result  HubDevice::Port::FsmHandler(int32_t state, LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (state)
    {
    case Port::State_Null:
        break;
    case Port::State_Init:
        result = InitStateHandler(pEvent);
        break;
    case Port::State_PendingSelectiveEnum:
        result = PendingSelectiveEnumStateHandler(pEvent);
        break;
    case Port::State_PoweringOn:
        result = PoweringOnStateHandler(pEvent);
        break;
    case Port::State_ClearingAll:
        result = ClearingAllStateHandler(pEvent);
        break;
    case Port::State_NotConnected:
        result = NotConnectedStateHandler(pEvent);
        break;
    case Port::State_ConnectionDebounce:
        result = ConnectionDebounceStateHandler(pEvent);
        break;
    case Port::State_WaitingForExclusivity:
        result = WaitingForExclusivityStateHandler(pEvent);
        break;
    case Port::State_ResettingPort:
        result = ResettingPortStateHandler(pEvent);
        break;
    case Port::State_ResetRecovery:
        result = ResetRecoveryStateHandler(pEvent);
        break;
    case Port::State_InitializeHostControllerDeviceContext:
        result = InitializeHostControllerDeviceContextStateHandler(pEvent);
        break;
    case State_ReadDeviceDescriptor:
        result = ReadDeviceDescriptorStateHandler(pEvent);
        break;
    case Port::State_CreatingChildDevice:
        result = CreatingChildDeviceStateHandler(pEvent);
        break;
    case Port::State_ChildDeviceActive:
        result = ChildDeviceActiveStateHandler(pEvent);
        break;
    case Port::State_DestroyingChildDevice:
        result = DestroyingChildDeviceStateHandler(pEvent);
        break;
    case Port::State_TerminatingPort:
        result = TerminatingPortStateHandler(pEvent);
        break;
    default:
        NN_USB_ABORT("HubDevice::Port FSM state %d invalid", state);
        break;
    }


    return result;
}

Result HubDevice::Port::InitStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_PendingAdminCommand = AdministrativeCommand_None;
            m_SuccessiveResetAttempts = 0;
            m_SuccessiveDevInitAttempts = 0;
            m_DeviceTerminationCause = 0;
            m_TerminationPending = false;
            break;
        }
    case Port::Event_Try:
        {
            int32_t nextState = State_PoweringOn;
            if (m_pHub->GetStartupMode() == StartupMode_SelectiveEnumeration)
            {
                nextState = State_PendingSelectiveEnum;
            }
            Fsm::SetState(nextState);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            GetObjectName(m_ObjectName, sizeof(m_ObjectName));
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::PendingSelectiveEnumStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {

            break;
        }
    case Port::Event_Try:
        {

            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::PoweringOnStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_Retries = 0;
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
    case Port::Event_ReTry:
        {
            SetFeatureAsync(UsbHubPortFeature_Power);
            break;
        }
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            if (pEvent->status.IsSuccess())
            {
                uint8_t bPwrOn2PwrGood = (m_pHub->m_IsSuperSpeedHub) ? m_pHub->m_HubDescriptor.super.bPwrOn2PwrGood :
                    m_pHub->m_HubDescriptor.standard.bPwrOn2PwrGood;
                Fsm::StartTimedFsmEvent(&m_TimedReadyEvent,
                                        nn::TimeSpan::FromMicroSeconds(bPwrOn2PwrGood * 2000));
            }
            else if (m_Retries++ < StaticPortConfig_MaxRetryNum)
            {
                Fsm::QueueFsmEvent(Port::Event_ReTry);
            }
            else
            {
                NN_USB_LOG_ERROR("%s - Failed to power on after %d retries. Stop.\n",
                                 m_ObjectName, StaticPortConfig_MaxRetryNum);
            }
            break;
        }
    case Port::Event_Ready:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_TimedReadyEvent);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::ClearingAllStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    enum ClearingAllSubstate
    {
        ClearingAllSubstate_Disabling,
        ClearingAllSubstate_StepA,
        ClearingAllSubstate_StepB,
        ClearingAllSubstate_StepC,
        ClearingAllSubstate_StepD,
        ClearingAllSubstate_Done,
    };

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_pHub->ReEvaluateEnumerationExclusivity();
            Fsm::SetSubstate(ClearingAllSubstate_Disabling);
            m_Retries = 0;
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
    case Port::Event_ReTry:
        {
            if (Fsm::GetSubstate() == ClearingAllSubstate_Disabling)
            {
                if(m_pHub->m_IsSuperSpeedHub)
                {
                    ClearFeatureAsync(UsbHubPortFeature_CPortConfigError);
                }
                else
                {
                    ClearFeatureAsync(UsbHubPortFeature_Enable);
                }
            }
            else if (Fsm::GetSubstate() == ClearingAllSubstate_StepA)
            {
                if(m_pHub->m_IsSuperSpeedHub)
                {
                    ClearFeatureAsync(UsbHubPortFeature_CPortLinkState);
                }
                else
                {
                    ClearFeatureAsync(UsbHubPortFeature_CEnable);
                }
            }
            else if (Fsm::GetSubstate() == ClearingAllSubstate_StepB)
            {
                ClearFeatureAsync(UsbHubPortFeature_CReset);
            }
            else if (Fsm::GetSubstate() == ClearingAllSubstate_StepC)
            {
                ClearFeatureAsync(UsbHubPortFeature_CConnetion);
            }
            else if (Fsm::GetSubstate() == ClearingAllSubstate_StepD)
            {
                if(m_pHub->m_IsSuperSpeedHub)
                {
                    ClearFeatureAsync(UsbHubPortFeature_CBhPortReset);
                }
                else
                {
                    ClearFeatureAsync(UsbHubPortFeature_CSuspend);
                }
            }
            else
            {
                Fsm::SetState(Port::State_NotConnected);
            }
            break;
        }
    case Port::Event_Skip:
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            if (pEvent->status.IsSuccess())
            {
                m_Retries = 0;
                Fsm::AddSubstate(1);
                Fsm::QueueFsmEvent(Port::Event_Try);
            }
            else if (m_Retries++ < StaticPortConfig_MaxRetryNum)
            {
                Fsm::QueueFsmEvent(Port::Event_ReTry);
            }
            else
            {
                NN_USB_LOG_ERROR("%s - Failed to clear feature after %d retries. Stop.\n",
                                 m_ObjectName, StaticPortConfig_MaxRetryNum);
            }
            break;
        }
    case Port::Event_PortIndication:
        {
            // ignore for now
            break;
        }
    case Port::Event_PortStatus:
        {
            m_IsPortStatusQueryPending = false;
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
} // NOLINT(impl/function_size)'

Result HubDevice::Port::NotConnectedStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_PendingFeatOptCount = 0;

            // Listen for new connection events
            m_pHub->ArmPortIndicationInterrupt();

            // Query current port status
            QueryPortStatusAsync();
            break;
        }
    case Port::Event_PortIndication:
        {
            if(m_PendingFeatOptCount == 0)
            {
                ClearFeatureAsync(UsbHubPortFeature_CConnetion);
                if(m_pHub->m_IsSuperSpeedHub)
                {
                    ClearFeatureAsync(UsbHubPortFeature_CPortLinkState);
                }
            }

            // initiate another port status query only if one not already pending */
            QueryPortStatusAsync();

            // persistently disabled is cleared by re-insertion
            m_PendingAdminCommand       = AdministrativeCommand_None;
            m_SuccessiveDevInitAttempts = 0;
            break;
        }
    case Port::Event_PortStatus:
        {
            m_IsPortStatusQueryPending = false;
            if ( !pEvent->status.IsSuccess())
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }
            if (m_PendingAdminCommand == AdministrativeCommand_Freeze)
            {
                // Denied, we froze this port
                break;
            }
            if (m_PortStatus.wPortChange & UsbPortStatusChange_ConfigError)
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }
            if (m_PortStatus.wPortStatus & UsbPortStatus_Connection)
            {
                NN_USB_LOG_INFO("%s - Connection detected.\n", m_ObjectName);
                if (m_pHs->m_DeviceManager.IsTotalDeviceLimitExceeded())
                {
                    NN_USB_LOG_WARN("%s - Maximum device limit exceeded, ignoring connection.\n",
                                    m_ObjectName);
                    break;
                }
                if (m_pHub->GetHierarchicalLevel() > HsLimitMaxHubHierarchyCount)
                {
                    NN_USB_LOG_WARN("%s - Maximum hub hierarchy exceeded, ignoring connection.\n",
                                    m_ObjectName);
                    break;
                }
                if (m_SuccessiveDevInitAttempts > StaticPortConfug_MaxDevInitRetryCount)
                {
                    NN_USB_LOG_WARN("%s - Enumeration failed for %d consecutive attempts, disabling port.\n",
                                    m_ObjectName, m_SuccessiveDevInitAttempts);
                    break;
                }
                Fsm::SetState(Port::State_ConnectionDebounce);
            }
            break;
        }
    case Port::Event_PortFeatureOpDone:
        {
            break;
        }
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::ConnectionDebounceStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            // Per section 7.1.7.3 of USB 2.0 specification, at least 100 ms must be allowed for attatch debounce (T3: TATTDB).
            Fsm::StartTimedFsmEvent(&m_TimedReadyEvent,
                                    nn::TimeSpan::FromMicroSeconds(StaticPortConfig_HubPortAttachDebounceInUs));
            break;
        }
    case Port::Event_Ready:
        {
            ClearFeatureAsync(UsbHubPortFeature_CConnetion);
            break;
        }
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            Fsm::SetState(Port::State_WaitingForExclusivity);
            break;
        }
    case Port::Event_PortIndication:
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_TimedReadyEvent);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::WaitingForExclusivityStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
    case Port::Event_Exclusive:
        {
            if (m_pHub->IsRootHubCreatingDevice())
            {
                NN_USB_LOG_INFO("%s - Waiting for exclusivity before continuing.\n",
                                m_ObjectName);
            }
            else
            {
                Fsm::SetState(Port::State_ResettingPort);
            }
            break;
        }
    case Port::Event_PortIndication:
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::ResettingPortStateHandler(LocalEventDataType *pEvent)
{
    enum ResettingPortSubstate
    {
        ResettingPortSubstate_ClearCConnection,
        ResettingPortSubstate_ClearCEnable,
        ResettingPortSubstate_ClearCReset,
        ResettingPortSubstate_BeginReset,
        ResettingPortSubstate_PortStatus,
    };

    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            NN_USB_SATURATED_ADD_U32(m_SuccessiveResetAttempts, 1);
            Fsm::SetSubstate(ResettingPortSubstate_ClearCConnection);
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
        {
            if (Fsm::GetSubstate() == ResettingPortSubstate_ClearCConnection)
            {
                ClearFeatureAsync(UsbHubPortFeature_CConnetion);
            }
            else if (Fsm::GetSubstate() == ResettingPortSubstate_ClearCEnable)
            {
                if(m_pHub->m_IsSuperSpeedHub)
                {
                    Fsm::QueueFsmEvent(Port::Event_Skip, ResultSuccess());
                    break;
                }
                ClearFeatureAsync(UsbHubPortFeature_CEnable);
            }
            else if (Fsm::GetSubstate() == ResettingPortSubstate_ClearCReset)
            {
                ClearFeatureAsync(UsbHubPortFeature_CReset);
            }
            else if (Fsm::GetSubstate() == ResettingPortSubstate_BeginReset)
            {
                SetFeatureAsync(UsbHubPortFeature_Reset);
            }
            else if (Fsm::GetSubstate() == ResettingPortSubstate_PortStatus)
            {
                // Root hub won't get an indication when reset is done
                if (m_pHub->m_IsRootHub)
                {
                    Fsm::ReStartTimedFsmEvent(&m_TimedReadyEvent,
                                              nn::TimeSpan::FromMicroSeconds(StaticPortConfig_ResetRootPortDelayInUs));
                }

                // Set a completion limit
                Fsm::StartTimedFsmEvent(&m_TimeoutEvent,
                                        nn::TimeSpan::FromMicroSeconds(StaticPortConfig_ResetTimeoutInUs));
            }
            break;
        }
    case Port::Event_Skip:
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            if ( !pEvent->status.IsSuccess())
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }
            if (Fsm::GetSubstate() < ResettingPortSubstate_PortStatus)
            {
                Fsm::AddSubstate(1);
                Fsm::QueueFsmEvent(Port::Event_Try);
            }
            break;
        }
    case Port::Event_Ready:
    case Port::Event_PortIndication:
        {
            if (Fsm::GetSubstate() == ResettingPortSubstate_PortStatus)
            {
                QueryPortStatusAsync();
            }
            break;
        }
    case Port::Event_PortStatus:
        {
            m_IsPortStatusQueryPending = false;
            if ( !pEvent->status.IsSuccess())
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }

            // if not connected, this state is stale
            if ((m_PortStatus.wPortStatus & UsbPortStatus_Connection) &&
                !(m_PortStatus.wPortChange & UsbPortStatusChange_Connection))
            {
                if (m_PortStatus.wPortStatus & UsbPortStatus_Enable)
                {
                    Fsm::SetState(Port::State_ResetRecovery);
                    break;
                }
            }

            // root port specific retry
            if (m_pHub->m_IsRootHub && (m_PortStatus.wPortStatus & UsbPortStatus_Connection))
            {
                Fsm::ReStartTimedFsmEvent(&m_TimedReadyEvent,
                                          nn::TimeSpan::FromMicroSeconds(StaticPortConfig_ResetRootPortDelayInUs));
                break;
            }

            // otherwise, give up on this connection attempt
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_Timeout:
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_TimeoutEvent);
            StopTimedFsmEvent(&m_TimedReadyEvent);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
} // NOLINT(impl/function_size)

Result HubDevice::Port::ResetRecoveryStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    enum ResetRecoverySubstate
    {
        ResetRecoverySubstate_ClearCEnable,
        ResetRecoverySubstate_ClearCReset,
        ResetRecoverySubstate_ClearCConnection,
        ResetRecoverySubstate_Wait,
    };

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_SuccessiveResetAttempts = 0;
            if(m_pHub->m_IsSuperSpeedHub)
            {
                Fsm::SetSubstate(ResetRecoverySubstate_ClearCReset);
                ClearFeatureAsync(UsbHubPortFeature_CReset);
            }
            else
            {
                Fsm::SetSubstate(ResetRecoverySubstate_ClearCEnable);
                ClearFeatureAsync(UsbHubPortFeature_CEnable);
            }
            break;
        }
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            if ( !pEvent->status.IsSuccess())
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }
            Fsm::AddSubstate(1);
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
        {
            if (Fsm::GetSubstate() == ResetRecoverySubstate_ClearCEnable)
            {
                ClearFeatureAsync(UsbHubPortFeature_CEnable);
            }
            else if (Fsm::GetSubstate() == ResetRecoverySubstate_ClearCReset)
            {
                ClearFeatureAsync(UsbHubPortFeature_CReset);
            }
            else if (Fsm::GetSubstate() == ResetRecoverySubstate_ClearCConnection)
            {
                ClearFeatureAsync(UsbHubPortFeature_CConnetion);
            }
            else
            {
                // Per section 7.1.7.3 of USB 2.0 specification, at least 10 ms must be allowed for reset recovery (6: TRSTRCY).
                Fsm::StartTimedFsmEvent(&m_TimedReadyEvent,
                                        nn::TimeSpan::FromMicroSeconds(StaticPortConfug_ResetRecoveryInUs));
            }
            break;
        }
    case Port::Event_Ready:
        {
            QueryPortStatusAsync();
            break;
        }
    case Port::Event_PortStatus:
        {
            m_IsPortStatusQueryPending = false;
            if (Fsm::IsEventStale(pEvent)) break;
            if ( !pEvent->status.IsSuccess())
            {
                Fsm::QueueFsmEvent(Port::Event_Error, pEvent->status);
                break;
            }
            if (m_PortStatus.wPortChange & UsbPortStatusChange_Connection)
            {
                Fsm::SetState(Port::State_ClearingAll);
            }
            else
            {
                if ( !(m_PortStatus.wPortStatus & (UsbPortStatus_Connection | UsbPortStatus_Enable)))
                {
                    Fsm::SetState(Port::State_ClearingAll);
                }
                else
                {
                    Fsm::SetState(Port::State_InitializeHostControllerDeviceContext);
                }
            }
            break;
        }
    case Port::Event_PortIndication:
        {
            if(Fsm::GetSubstate() == ResetRecoverySubstate_Wait)
            {
                Fsm::SetState(Port::State_ClearingAll);
            }
            break;
        }
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_ClearingAll);
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_TimedReadyEvent);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
} // NOLINT(impl/function_size)'

Result HubDevice::Port::InitializeHostControllerDeviceContextStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::SetSubstate(Port::State_ReadDeviceDescriptor);

            // Keep track of attempts
            NN_USB_SATURATED_ADD_U32(m_SuccessiveDevInitAttempts, 1);

            // Get speed now that we have valid port status
            m_PortSpeed = GetPortSpeed();

            // Prepare copy of future port hierarchy
            HubPortNumber portHierarchy[HsLimitMaxHubHierarchyCount];
            m_pHub->GetHubPortHierarchy(portHierarchy);
            portHierarchy[m_pHub->GetHierarchicalLevel()] = m_PortNumber;
            portHierarchy[0] += (m_pHub->m_RootBasePortNumber - 1);

            // Resolve split transaction parameters
            uint8_t ttDevice = 0;
            HubPortNumber ttPort = 0;
            if((m_PortSpeed != UsbDeviceSpeed_Super) && (m_PortSpeed != UsbDeviceSpeed_High) && !m_pHub->m_IsRootHub)
            {
                Device* pFirstHighSpeedParent = nullptr;
                ttPort = portHierarchy[m_pHub->GetHierarchicalLevel()];
                if(m_pHub->FindHighSpeedParent(&ttPort, &pFirstHighSpeedParent))
                {
                    HostControllerDriverDeviceContext* pCtx = pFirstHighSpeedParent->GetHostControllerDriverDeviceContext();
                    if(pCtx != nullptr) ttDevice = pCtx->slotId;
                }
            }

            // Ask host controller to create the device context
            result = m_pHc->CreateDeviceContextAsync(portHierarchy, m_PortSpeed, ttDevice, ttPort,
                                                     CreateHostControllerDeviceContextCallback, this);
            break;
        }
    case Port::Event_HostControllerDeviceContextCreate:
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            m_pHcDevContext = reinterpret_cast<HostControllerDriverDeviceContext*>(pEvent->args[0].pData);
            Fsm::SetState(Fsm::GetSubstate());
            break;
        }
    case Port::Event_PortIndication:
    case Port::Event_Reset:
    case Port::Event_Freeze:
    case Port::Event_TerminateRequest:
        {
            // Wait for CreateDeviceContextAsync() completion
            Fsm::SetSubstate(Port::State_DestroyingChildDevice);
            break;
        }
    case Port::Event_Error:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Port::Event_Error, result);
    }
    return result;
} // NOLINT(impl/function_size)

Result HubDevice::Port::ReadDeviceDescriptorStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        Fsm::SetSubstate(Port::State_DestroyingChildDevice);

        // Prepare temporary control transfer mechanisms, just to get device descriptor
        m_CtrlHep.epDir                     = UsbEndpointDirection_Control;
        m_CtrlHep.epType                    = UsbEndpointType_Control;
        m_CtrlHep.epNumber                  = 0;
        m_CtrlHep.maxUrbCount               = 1;
        m_CtrlHep.isShortTransferError      = true;

        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
            m_CtrlHep.pHcEp = new HostControllerDriverEndpoint(m_pHs, m_pHc)
        );

        // Endpoint init is deferred
        result = m_CtrlHep.pHcEp->InitializeAsync(
            &m_CtrlHep, m_pHcDevContext,
            HostControllerDriverEndpointInitializeDoneCallback, this
        );
        if (result.IsFailure())
        {
            delete m_CtrlHep.pHcEp;
            m_CtrlHep.pHcEp = nullptr;

            NN_USB_BREAK_UPON_ERROR(result);
        }

        m_HcCtrlMgr.Initialize(m_CtrlHep.pHcEp, m_ObjectName);

        break;

    case Port::Event_EndpointInitializeComplete:
        NN_USB_BREAK_UPON_ERROR(pEvent->status);

        if (m_PortSpeed == UsbDeviceSpeed_Full)
        {
            Fsm::QueueFsmEvent(Port::Event_ReadMaxPacketSize0, ResultSuccess());
        }
        else
        {
            Fsm::QueueFsmEvent(Port::Event_ReadDeviceDescriptor, ResultSuccess());
        }
        break;

    case Port::Event_ReadMaxPacketSize0:
        NN_USB_BREAK_UPON_ERROR(pEvent->status);

        NN_USB_LOG_INFO("%s - Attempting to read bMaxPacketSize0 of %s speed device...\n",
                        m_ObjectName, Util::GetSpeedDescription(m_PortSpeed));
        NN_USB_BREAK_UPON_ERROR(
            m_HcCtrlMgr.SubmitAsync(
                UsbCtrlXferReq_GetDescriptor,
                BmRequestType(Standard, Device, DeviceToHost),
                (UsbDescriptorType_Device << 8) | 0,
                0,
                8,
                &m_pDmaDescriptorBuffers->device,
                false,
                ReadMaxPacketSize0CompletionCallback,
                this,
                HsLimitControlXferTimeout
            )
        );
        break;

    case Port::Event_UpdateEp0:
    {
        NN_USB_BREAK_UPON_ERROR(pEvent->status);

        uint32_t maxPacketSize0 = m_pDmaDescriptorBuffers->device.bMaxPacketSize0;

        // We only try to get MaxPacketSize0 for FS device, and only following
        // values are valid.
        if (maxPacketSize0 != 8  && maxPacketSize0 != 16 &&
            maxPacketSize0 != 32 && maxPacketSize0 != 64  )
        {
            result = ResultInvalidDescriptor();
            break;
        }

        result = pEvent->status;
        NN_USB_BREAK_UPON_ERROR(result);

        NN_USB_LOG_INFO("%s - Updating Control Endpoint Context of %s speed device...\n",
                        m_ObjectName, Util::GetSpeedDescription(m_PortSpeed));

        NN_USB_BREAK_UPON_ERROR(
            m_pHc->UpdateEp0Async(m_pHcDevContext,
                                  maxPacketSize0,
                                  UpdateEp0CompletionCallback,
                                  this)
        );

        break;
    }

    case Port::Event_ReadDeviceDescriptor:
        NN_USB_BREAK_UPON_ERROR(pEvent->status);

        NN_USB_LOG_INFO("%s - Attempting to read descriptor of %s speed device...\n",
                        m_ObjectName, Util::GetSpeedDescription(m_PortSpeed));
        NN_USB_BREAK_UPON_ERROR(
            m_HcCtrlMgr.SubmitAsync(
                UsbCtrlXferReq_GetDescriptor,
                BmRequestType(Standard, Device, DeviceToHost),
                (UsbDescriptorType_Device << 8) | 0,
                0,
                sizeof(UsbDeviceDescriptor),
                &m_pDmaDescriptorBuffers->device,
                false,
                ControlTransferCompletionCallback,
                this,
                HsLimitControlXferTimeout
            )
        );
        break;

    case Port::Event_ControlTransferComplete:
        if (Fsm::IsEventStale(pEvent)) break;

        // accept completion status
        result = pEvent->status;
        NN_USB_BREAK_UPON_ERROR(result);

        NN_USB_LOG_INFO("%s - Discovered %s speed device of class %s.\n",
                        m_ObjectName, Util::GetSpeedDescription(m_PortSpeed),
                        Util::GetClassCodeDescription(m_pDmaDescriptorBuffers->device.bDeviceClass));
        Fsm::SetSubstate(Port::State_CreatingChildDevice);
        Fsm::QueueFsmEvent(Port::Event_PrepareExit);
        break;

    case Port::Event_PortIndication:
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            if (Fsm::IsEventStale(pEvent)) break;
            Fsm::SetSubstate(Port::State_DestroyingChildDevice);
            Fsm::QueueFsmEvent(Port::Event_PrepareExit);
            break;
        }
    case Port::Event_PrepareExit:
        {
            // nothing to prepare, done now
            if (m_CtrlHep.pHcEp == nullptr)
            {
                Fsm::SetState(Fsm::GetSubstate());
                break;
            }

            // endpoint needs to be terminated
            if (!m_CtrlHep.pHcEp->IsTerminationPending())
            {
                result = m_CtrlHep.pHcEp->TerminateAsync(
                    HostControllerDriverEndpointTerminateDoneCallback, this
                );

                // Perhaps it's still in ResetPending. Retry in 500ms.
                if (ResultResourceBusy::Includes(result))
                {
                    Fsm::ReStartTimedFsmEvent(&m_EpTerminationRetryTimer,
                                              nn::TimeSpan::FromMilliSeconds(500));
                    result = ResultSuccess();
                }
            }
            break;
        }
    case Port::Event_EndpointTerminateComplete:
        {
            Fsm::SetState(Fsm::GetSubstate());
            break;
        }
    case Port::Event_TerminateRequest:
        {
            Fsm::SetSubstate(Port::State_DestroyingChildDevice);
            Fsm::QueueFsmEvent(Port::Event_PrepareExit);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_EpTerminationRetryTimer);

            if (m_CtrlHep.pHcEp != nullptr)
            {
                m_HcCtrlMgr.Finalize();
                NN_USB_ABORT_UNLESS_SUCCESS(m_CtrlHep.pHcEp->Finalize());
                delete m_CtrlHep.pHcEp;
                m_CtrlHep.pHcEp = nullptr;
            }
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Port::Event_Error, result);
    }

    return result;
} // NOLINT(impl/function_size)

Result HubDevice::Port::CreatingChildDeviceStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::SetSubstate(Port::State_ChildDeviceActive);
            m_isChildDeviceAddressingDone = false;
            NN_USB_BREAK_UPON_ERROR(CreateChildDevice(&m_pDmaDescriptorBuffers->device));
            break;
        }
    case Port::Event_ChildDeviceCreateComplete:
        {
            Fsm::SetState(Fsm::GetSubstate());
            break;
        }
    case Port::Event_PortIndication:
       {
           Fsm::SetSubstate(Port::State_DestroyingChildDevice);
           break;
       }
    case Port::Event_ChildDeviceCreateFail:
    case Port::Event_TerminateRequest:
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result HubDevice::Port::ChildDeviceActiveStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_SuccessiveDevInitAttempts = 0;
            break;
        }
    case Port::Event_Try:
        {

            break;
        }
    case Port::Event_PortIndication:
        {
            m_DeviceTerminationCause |= TerminationOptions_Disconnected;
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Port::Event_Error:
    case Port::Event_Reset:
    case Port::Event_Freeze:
    case Port::Event_TerminateRequest:
        {
            Fsm::SetState(Port::State_DestroyingChildDevice);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}


Result HubDevice::Port::DestroyingChildDeviceStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    enum DestroyingChildDeviceSubstate
    {
        DestroyingChildDeviceSubstate_DeviceObject,
        DestroyingChildDeviceSubstate_DeviceContext,
        DestroyingChildDeviceSubstate_Done,
    };

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
        {
            if (m_pChildDevice != nullptr)
            {
                NN_USB_ABORT_UNLESS_SUCCESS(
                    m_pChildDevice->TerminateAsync(
                        m_DeviceTerminationCause | m_pHub->GetTerminationOptions(),
                        ChildDeviceTerminateDoneCallback, this
                    )
                );
                break;
            }
            if(m_pHcDevContext != nullptr)
            {
                result = m_pHc->DestroyDeviceContextAsync(
                    m_pHcDevContext,
                    DestroyHostControllerDeviceContextCallback, this
                );

                if (result.IsSuccess())
                {
                    break;
                }
            }

            if (m_TerminationPending)
            {
                Fsm::SetState(Port::State_TerminatingPort);
            }
            else
            {
                Fsm::SetState(Port::State_ClearingAll);
            }
            break;
        }
    case Port::Event_ChildDeviceTerminateComplete:
        {
            if (m_pChildDevice != nullptr)
            {
                char devObjName[HsLimitMaxDebugNameSize];
                m_pChildDevice->GetObjectName(devObjName, sizeof(devObjName));
                NN_USB_LOG_INFO("%s - Child Finalizing: %s...\n",
                                m_ObjectName, devObjName);
                if (m_pChildDevice->GetDeviceClass() == UsbClass_Hub)
                {
                    HubDevice *pHub = static_cast<HubDevice *>(m_pChildDevice);
                    pHub->Finalize();
                    delete pHub;
                }
                else
                {
                    m_pChildDevice->Finalize();
                    delete m_pChildDevice;
                }
                m_pChildDevice = nullptr;
            }
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_HostControllerDeviceContextDestroy:
        {
            m_pHcDevContext = nullptr;
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Reset:
    case Port::Event_Freeze:
        {
            // not applicable at this point, ignore
            break;
        }
    case Port::Event_PortIndication:
    case Port::Event_TerminateRequest:
        {
            break;
        }

    case Port::Event_UpdateEp0:
    case Port::Event_ReadDeviceDescriptor:
    case Port::Event_ControlTransferComplete:
        // possible leftover from State_ReadDeviceDescriptor, ignore
        break;

    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
} // NOLINT(impl/function_size)

Result HubDevice::Port::TerminatingPortStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    enum TerminatingPortSubstate
    {
        TerminatingPortSubstate_Suspend,
        TerminatingPortSubstate_PowerOff,
        TerminatingPortSubstate_Done,
    };

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::SetSubstate(TerminatingPortSubstate_Done);
            if (m_pHub->GetTerminationOptions() & TerminationOptions_Disconnected)
            {
                Fsm::QueueFsmEvent(Port::Event_Done);
                break;
            }
            if (m_pHub->GetTerminationOptions() & TerminationOptions_Suspend)
            {
                Fsm::SetSubstate(TerminatingPortSubstate_Suspend);
                Fsm::QueueFsmEvent(Port::Event_Try);
                break;
            }
            if (m_pHub->GetTerminationOptions() & TerminationOptions_PowerOff)
            {
                Fsm::SetSubstate(TerminatingPortSubstate_PowerOff);
                Fsm::QueueFsmEvent(Port::Event_Try);
                break;
            }
            Fsm::QueueFsmEvent(Port::Event_Try);
            break;
        }
    case Port::Event_Try:
        {
            if (Fsm::GetSubstate() == TerminatingPortSubstate_Suspend)
            {
                ClearFeatureAsync(UsbHubPortFeature_Enable);
            }
            else if ((Fsm::GetSubstate() == TerminatingPortSubstate_PowerOff) &&
                     (m_pHub->GetTerminationOptions() & TerminationOptions_PowerOff))
            {
                ClearFeatureAsync(UsbHubPortFeature_Power);
            }
            else
            {
                Fsm::QueueFsmEvent(Port::Event_Done);
            }
            break;
        }
    case Port::Event_PortFeatureOpDone:
        {
            if (Fsm::IsEventStale(pEvent))
            {
                break;
            }
            if (pEvent->status.IsSuccess())
            {
                Fsm::AddSubstate(1);
                Fsm::QueueFsmEvent(Port::Event_Try);
                break;
            }
            else
            {
                Fsm::QueueFsmEvent(Port::Event_Error);
            }
            break;
        }
    case Port::Event_Error:
    case Port::Event_Done:
        {
            Fsm::SetState(Port::State_Null);
            break;
        }
    case Port::Event_PortStatus:
    case Port::Event_PortIndication:
    case Port::Event_TerminateRequest:
        {
            // ignore, it's over
            break;
        }
    case Fsm::Event_Exit:
        {
            m_pHub->HandlePortTerminationDone(this);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}



} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
