﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Host Controller base class
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class Hs;
class Device;
class HostControllerDriver;
class HubDevice;
struct HostControllerDriverDeviceContext;

class HostControllerDriverEndpoint
{
public:
    enum State
    {
        State_New,
        State_OpenPending,
        State_Open,
        State_ClosePending,
        State_Closed,
        State_ResetPending,
        State_Error
    };

    typedef void (*InitializeCallback)(Hs* pHs, Result status, HostControllerDriverEndpoint* pEp, void* context);
    typedef void (*TerminateCallback)(Hs* pHs, Result status, HostControllerDriverEndpoint* pEp, void *context);
    typedef void (*ResetCallback)(Hs* pHs,  Result status, HostControllerDriverEndpoint* pEp, void *context);

    HostControllerDriverEndpoint(Hs *pHs, HostControllerDriver *pHc)
        : m_pHs(pHs)
        , m_pHc(pHc)
        , m_State(State_New)
        , m_pDriverEp(nullptr)
        , m_InitializeCallback(nullptr)
        , m_InitializeContext(nullptr)
        , m_TerminateCallback(nullptr)
        , m_TerminateContext(nullptr)
        , m_ResetCallback(nullptr)
        , m_ResetContext(nullptr)
        , m_pHep(nullptr)
        , m_UrbPool(pHs, this)
        , m_pDeviceContext(nullptr)
    {
        NN_UNUSED(m_pHs);
    }
    ~HostControllerDriverEndpoint()
    {
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

    Result InitializeAsync(HostEndpoint *pHep, HostControllerDriverDeviceContext* pDeviceContext,
                           InitializeCallback callback, void* context);
    Result ResetAsync(ResetCallback callback, void *context);
    Result TerminateAsync(TerminateCallback callback, void *context);
    Result Finalize();
    bool IsTerminationPending();
    HostEndpoint* GetHostEndpoint();
    uint8_t GetDeviceAddress();
    UsbDeviceSpeed GetDeviceSpeed();
    PlatformController* GetPlatformController();
    void SetDriverEndpoint(void *pDrvEp);
    void* GetDriverEndpoint();
    HostControllerDriverDeviceContext* GetDeviceContext();
    void ReportOpenComplete(Result status);
    void ReportResetComplete(Result status);
    void ReportCloseComplete(Result status);

    Result CreatePrivateUrb(nn::dd::ProcessHandle     procHandle,
                            uint64_t                  procVa,
                            uint32_t                  size,
                            bool                      isSmmuMapDynamic,
                            UsbBusToken               token,
                            UsbRequestBlock::Callback callback,
                            void                     *context,
                            uint32_t                  timeoutInMs,
                            UsbRequestBlock         **ppCreatedUrb);
    Result CreateCtrlUrb(nn::dd::ProcessHandle     procHandle,
                         uint64_t                  procVa,
                         UsbCtrlRequest&           ctrlRequest,
                         bool                      isSmmuMapDynamic,
                         UsbRequestBlock::Callback callback,
                         void                     *context,
                         uint32_t                  timeoutInMs,
                         UsbRequestBlock         **ppCreatedUrb);
    Result CreateUrb(ClientEpSession::UrbDeferredRequest   *pDeferredRequest,
                      UsbRequestBlock                      **ppCreatedUrb);
    Result SubmitUrb(UsbRequestBlock *pUrb);
    void   DestroyUrb(UsbRequestBlock *pUrb);

    // Methods intended to be called by lower level host controller driver implementations
    void CompleteUrb(UsbRequestBlock *pUrb);

    void TimerCallback(LocalEventDataType *pData);

private:
    static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;

    Hs*                   m_pHs;
    HostControllerDriver* m_pHc;
    State                 m_State;
    void*                 m_pDriverEp;
    InitializeCallback    m_InitializeCallback;
    void*                 m_InitializeContext;
    TerminateCallback     m_TerminateCallback;
    void*                 m_TerminateContext;
    ResetCallback         m_ResetCallback;
    void*                 m_ResetContext;
    HostEndpoint*         m_pHep;
    UrbPool               m_UrbPool;

    PlatformController*   m_pPlatform;
    HostControllerDriverDeviceContext* m_pDeviceContext;

    void DoTerminateCallback(Result status);
};

class HostControllerDriver
{
    friend class HostControllerDriverEndpoint;
public:
    enum PortAttribute
    {
        PortAttribute_Invalid = 0,
        PortAttribute_Connection,
        PortAttribute_Reset,
        PortAttribute_Enable,
        PortAttribute_Suspend,
        PortAttribute_Power,
        PortAttribute_Owner,
    };
    enum ControllerType
    {
        ControllerType_Invalid,
        ControllerType_Ohci,
        ControllerType_Ehci,
        ControllerType_Xhci
    };
    typedef uint32_t UpdateDeviceMask;

    typedef void (*CreateDeviceContextCallback)(Hs*, Result, HostControllerDriverDeviceContext*, void*);
    typedef void (*DestroyDeviceContextCallback)(Hs*, Result, void*);
    typedef void (*AddressDeviceCallback)(Hs*, Result, void*);
    typedef void (*UpdateDeviceCallback)(Hs*, Result, void*);
    typedef void (*UpdateEp0Callback)(Hs*, Result, void*);

    nn::util::IntrusiveListNode  m_ListNode;


    NN_IMPLICIT HostControllerDriver(Hs *pHs, ControllerType controllerType, StartupMode mode, PlatformController* pPlatformController);
    virtual ~HostControllerDriver();
    virtual Result Initialize();
    virtual Result Finalize();
    int GetObjectName(char *pName, uint32_t maxSize);
    HubDevice* GetRootHubDevice();
    PlatformController* GetPlatformController();
    ControllerType GetControllerType();

    // Methods intended to be called by lower level host controller driver implementations
    void ReportPortEvent();
    Result SetPortEventHandler(void *context, void (*cbf)(void *, uint32_t portChangeMask), uint32_t portMask);
    Result ClearPortEventHandler(void *context);

    // Methods implemented by specific host controller driver
    virtual Result CreateDeviceContextAsync(HubPortNumber *portHierarchy,
                                            UsbDeviceSpeed speed, uint8_t ttDevice, HubPortNumber ttPort,
                                            CreateDeviceContextCallback callback, void* context) = 0;
    virtual Result DestroyDeviceContextAsync(HostControllerDriverDeviceContext* pDeviceContext,
                                             DestroyDeviceContextCallback callback, void* context) = 0;
    virtual Result AddressDeviceAsync(HostControllerDriverDeviceContext* pDeviceContext,
                                      AddressDeviceCallback callback, void* context) = 0;
    virtual void   UpdateDeviceContext(HostControllerDriverDeviceContext* pDeviceContext) = 0;
    virtual Result UpdateEp0Async(HostControllerDriverDeviceContext* pDeviceContext, uint8_t maxPacketSize0,
                                  UpdateEp0Callback callback, void *context) = 0;
    virtual Result OpenEpAsync(HostControllerDriverEndpoint *pEp) = 0;
    virtual Result ResetEpAsync(HostControllerDriverEndpoint *pEp) = 0;
    virtual Result CloseEpAsync(HostControllerDriverEndpoint *pEp) = 0;
    virtual Result SubmitUrbAsync(UsbRequestBlock *pUrb) = 0;
    virtual Result GetRootHubPortStatus(HubPortNumber portNumber, UsbHubPortStatus *pStatus) = 0;
    virtual Result ClearRootHubPortStatus(HubPortNumber portNumber, uint16_t wPortStatus) = 0;
    virtual Result SetRootHubPortAttribute(HubPortNumber portNumber, PortAttribute attribute, bool isTrue) = 0;
    virtual Result ResetRootHub() = 0;
    virtual uint32_t GetCurrentFrameId() = 0;
    virtual UsbDeviceSpeed GetPortSpeed(HubPortNumber portNumber) = 0;
    virtual void   Isr() = 0;
    virtual void OnControllerStall() = 0;
    virtual void SetTestMode(uint32_t port, TestMode mode) = 0;

protected:
    Hs*                     m_pHs;
    ControllerType          m_ControllerType;
    StartupMode             m_StartupMode;
    int32_t                 m_PortCount;
    char                    m_HcName[HsLimitMaxDebugNameSize];
    PlatformController*     m_pPlatform;

private:
    struct PortEventHandler
    {
        void*     context;
        void     (*callback)(void *, uint32_t);
        uint32_t  mask;
    };
    PortEventHandler        m_PortEventHandlers[2];
};


struct HostControllerDriverDeviceContext
{
    HostControllerDriver*  pHc;
    HubPortNumber          portHierarchy[HsLimitMaxHubHierarchyCount];
    UsbDeviceSpeed         speed;
    uint8_t                address;
    uint8_t                slotId;
    uint8_t                ttParentDevice;      // address or slot id
    HubPortNumber          ttParentPort;

    // needed if this device is an external hub
    struct
    {
        bool               isMultiTtCapable;
        int32_t            portCount;
        uint16_t           maxExitLatencyInUs;
        uint8_t            ttThinkTime;          // section 11.18.2 of the USB2 spec
    }hub;

    // needed for completion callbacks
    struct
    {
        void*  context;
        union
        {
            HostControllerDriver::CreateDeviceContextCallback  create;
            HostControllerDriver::DestroyDeviceContextCallback destroy;
            HostControllerDriver::AddressDeviceCallback        address;
            HostControllerDriver::UpdateDeviceCallback         update;
        }callback;
    }completion;
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

