﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief Finite state machine driver
 *
 * @details
 */

#define NN_USB_FSM_LOG(format, ...)                             \
    nn::usb::detail::Log(nn::usb::LogLevel_Info,                \
                         m_LogModule,                           \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)

namespace nn { namespace usb { namespace hs {

class Hs;

class Fsm
{
public:
    enum Event
    {
        Event_Invalid = 0,
        Event_Entry,
        Event_Exit,
        Event_Base
    };

    Fsm(Hs *pMain, LogModule module)
    {
        m_pMain           = pMain;
        m_State           = 0;
        m_NextState       = 0;
        m_CurrentEvent    = 0;
        memset(m_FsmName, 0, sizeof(m_FsmName));
        m_EventNames      = nullptr;
        m_StateNames      = nullptr;
        m_NumEvents       = 0;
        m_NumStates       = 0;
        m_LogModule       = module;
    }

    virtual ~Fsm()
    {

    }

    LocalEventPoolType* GetLocalEventPool();
    int32_t GetState();
    int32_t GetCurrentEvent();
    int32_t GetSubstate();
    void SetSubstate(int32_t subState);
    int32_t AddSubstate(int32_t subStateOffset);

protected:
    Result Initialize(char *name,
                      const char **eventNames, int32_t numEvents,
                      const char **stateNames, int32_t numStates,
                      int32_t initialState);
    Result Finalize();
    void SetName(const char *name);
    const char* GetName();
    void QueueFsmEvent(LocalEventDataType *pEvent);
    void QueueFsmEvent(int32_t eventId);
    void QueueFsmEvent(int32_t eventId, Result status);
    void QueueFsmEvent(int32_t eventId, Result status, uint32_t data);
    void QueueFsmEvent(int32_t eventId, void *pData);
    void QueueFsmEvent(int32_t eventId, Result status, void *pData);
    Result DispatchFsmEvent(int32_t eventId, void *pData);
    Result DispatchFsmEvent(int32_t eventId);
    LocalEventType* CreateTimedFsmEvent(int32_t eventId);
    void DestroyTimedFsmEvent(LocalEventType *pEvent);
    void InitStaticTimedFsmEvent(LocalEventType *pEvent, int32_t eventId);
    void StartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime);
    void ReStartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime);
    void StopTimedFsmEvent(LocalEventType *pEvent);
    virtual Result FsmHandler(int32_t state, LocalEventDataType *pEvent) = 0;
    void SetState(int32_t nextState);
    bool IsEventStale(LocalEventDataType *pEventData);
    void Schedule();

private:
    Hs*                 m_pMain;
    int32_t             m_State;
    int32_t             m_NextState;
    int32_t             m_Substate;
    int32_t             m_CurrentEvent;
    char                m_FsmName[HsLimitMaxDebugNameSize];
    const char**        m_EventNames;
    const char**        m_StateNames;
    int32_t             m_NumEvents;
    int32_t             m_NumStates;
    LogModule           m_LogModule;
    LocalEventPoolType  m_LocalEventPool;
    LocalEventType      m_LocalEventStorage[32];

    Result HandleEvent(LocalEventDataType *pEvent);
    static void HandleEventStatic(void *context, LocalEventType *pEvent)
    {
        Fsm *pFsm = static_cast<Fsm *>(context);
        pFsm->HandleEvent(&pEvent->data);
    }
};

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn


