﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class Device;
struct DeviceInterface;

class DeviceManager
{
public:
    class ManagedDevice
    {
    public:
        friend DeviceManager;

        nn::util::IntrusiveListNode  m_ListNode;

        ManagedDevice(Hs *pHs, Device *pDevice)
            : m_pHs(pHs)
            , m_pDm(nullptr)
            , m_pDevice(pDevice)
            , m_IsHub(false)
            , m_IsRootHub(false)
            , m_pIfArray(nullptr)
            , m_Uid(0)
        {
        }
        ~ManagedDevice()
        {
        }
        Result Initialize(DeviceManager *pDm, bool isHub, bool isRootHub);
        Result Finalize();
        Result Offer(DeviceInterface *ifArray);
        Result Revoke();
        void   HostEndpointCreateConfirm(HostEndpoint *pHep, Result status);
        void   HostEndpointDestroyConfirm(HostEndpoint *pHep);
        DeviceUid GetUid();
        bool IsHub();
        bool IsRootHub();
    private:
        Hs*                 m_pHs;
        DeviceManager*      m_pDm;
        Device*             m_pDevice;
        bool                m_IsHub;
        bool                m_IsRootHub;
        DeviceInterface*    m_pIfArray;
        DeviceUid           m_Uid;
    };

    class ManagedInterface : public Fsm
    {
    public:
        friend DeviceManager;
        friend ManagedDevice;
        static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;

        nn::util::IntrusiveListNode  m_ListNode;

        ManagedInterface(Hs *pHs, DeviceManager *pDm,
                         ManagedDevice *pManagedDevice, DeviceInterface *pIf)
            : Fsm(pHs, LogModule_HsFsmDeviceManager)
            , m_pHs(pHs)
            , m_pDm(pDm)
            , m_pManagedDevice(pManagedDevice)
            , m_pIf(pIf)
            , m_pClient(nullptr)
            , m_OpenEndpointMask(0)
            , m_pPendingReleaseRequest(nullptr)
            , m_InterfaceHandle(-1)
        {
        }
        ~ManagedInterface()
        {
        }
        void * operator new(size_t size) NN_NOEXCEPT;
        void operator delete(void *p, size_t size) NN_NOEXCEPT;
        Result Initialize();
        Result Finalize();
        ManagedDevice* GetManagedDevice();
        Device* GetDevice();
        DeviceInterface* GetDeviceInterface();
        InterfaceHandle GetInterfaceHandle();
        void SubmitIfAdminRequest(DeferredRequestBase *pRequest);
        void SetAltIfConfirm(Result status);
    private:
        typedef nn::util::IntrusiveList
        < DeferredRequestBase,nn::util::IntrusiveListMemberNodeTraits<DeferredRequestBase, &DeferredRequestBase::m_ListNode>> AdminRequestListType;

        enum State
        {
            State_Null,
            State_Offered,
            State_Acquired,
            State_Releasing,
            State_Revoked,
            State_Maximum
        };
        enum Event
        {
            Event_Try = Fsm::Event_Base,
            Event_Continue,
            Event_AcquireRequest,
            Event_ReleaseRequest,
            Event_Revoke,
            Event_AdminRequest,
            Event_EndpointCreateConfirm,
            Event_EndpointDestroyConfirm,
            Event_SetAltIfConfirm,
            Event_Error,
            Event_Maximum,
        };
        static const char *EventNames[Event_Maximum];
        static const char *StateNames[State_Maximum];

        Hs*                                             m_pHs;
        DeviceManager*                                  m_pDm;
        ManagedDevice*                                  m_pManagedDevice;
        DeviceInterface*                                m_pIf;
        ClientRootSession*                              m_pClient;
        UsbEndpointMask                                 m_OpenEndpointMask;
        ClientIfSession::ReleaseUsbIfDeferredRequest*   m_pPendingReleaseRequest;
        InterfaceHandle                                 m_InterfaceHandle;
        AdminRequestListType                            m_AdminRequestList;

        bool   InitiateManagedEndpointDestruction();
        void DispatchNextAdminRequest();
        void HandleAcquiredEndpointCreateConfirm(HostEndpoint *pHep, Result status);
        void HandleAcquiredEndpointDestroyConfirm(UsbEndpointMask singleDestroyedEndpoint);
        void CompleteCurrentAdminRequest(Result result);

        Result FsmHandler(int32_t state, LocalEventDataType *pEvent);
        Result OfferedStateHandler(LocalEventDataType *pEvent);
        Result AcquiredStateHandler(LocalEventDataType *pEvent);
        Result ReleasingStateHandler(LocalEventDataType *pEvent);
        Result RevokedStateHandler(LocalEventDataType *pEvent);
    };


    explicit DeviceManager(Hs *pHs)
        : m_pHs(pHs)
        , m_TotalDeviceCount(0)
        , m_RollingDeviceUid(0)
    {
    }
    ~DeviceManager()
    {
    }
    Result Initialize();
    Result Finalize();

    bool IsTotalDeviceLimitExceeded();

    Result AddDevice(DeviceManager::ManagedDevice *pManagedDevice,
                     DeviceInterface *ifArray);
    Result RemoveDevice(DeviceManager::ManagedDevice *pManagedDevice);
    Result GetProfiles(InterfaceHandle   ifHandle,
                       DeviceProfile    *pDeviceProfile,
                       InterfaceProfile *pInterfaceProfile);
    Result GetCurrentFrameId(InterfaceHandle ifHandle, FrameNumber *frame);

    Result SubmitUrbAsync(InterfaceHandle ifHandle,
                          EndpointNumber epNumber,
                          UsbEndpointDirection epDirection,
                          nn::dd::ProcessHandle processHandle,
                          uint64_t bufferAddress,
                          uint32_t bufferSize);

    Result QueryAllInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest);
    Result QueryAvailableInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest);
    Result QueryAcquiredInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest);
    Result NewUsbIfAvailableEvent(ClientRootSession::NewUsbIfAvailableEventDeferredRequest *pRequest);
    Result AcquireUsbIf(ClientRootSession::AcquireUsbIfDeferredRequest *pRequest);
    Result ReleaseUsbIf(ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest);
    Result SubmitControlTransfer(ClientIfSession::CtrlXferDeferredRequest *pRequest);
    Result SubmitDeferredIfAdminRequest(InterfaceHandle ifHandle, DeferredRequestBase *pRequest);
    Result SubmitUrb(ClientEpSession::UrbDeferredRequest *pRequest);
    Result ResetDevice(ClientIfSession::ResetDeviceDeferredRequest* pRequest);

private:
    typedef detail::NnList<ManagedDevice>     DeviceListType;
    typedef detail::NnList<ManagedInterface>  InterfaceListType;
    SafeHandleManager<ManagedInterface, InterfaceHandle, ResultInterfaceInvalid, HsLimitMaxInterfacesCount, 0x5000> m_InterfaceHandleManager;
    Hs*                   m_pHs;
    int32_t               m_TotalDeviceCount;
    DeviceUid             m_RollingDeviceUid;

    // Device lists
    DeviceListType        m_RootHubList;
    DeviceListType        m_ExternalDeviceList;

    // Interface Lists
    InterfaceListType     m_OfferedIfList;
    InterfaceListType     m_AcquiredIfList;
};



} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

