﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

void* Device::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, Device::ObjectMemAlignmentSize,
                                        "Device");
}
void Device::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "Device");
}

const char *Device::EventNames[Event_Maximum] =
{
    "Try",
    "Skip",
    "AddressComplete",
    "Continue",
    "DeviceSpecificComplete",
    "ControlTransferComplete",
    "SetAltIfComplete",
    "SetIfRequest",
    "SuspendRequest",
    "ResumeRequest",
    "EndpointRequest",
    "EndpointInitializeComplete",
    "EndpointTerminateComplete",
    "Error",
    "TerminateRequest",
    "RecoveryOk",
    "PowerManagementRequest",
};

const char *Device::StateNames[State_Maximum] =
{
    "Null",
    "Initialize",
    "AssignAddress",
    "IdentifyConfiguration",
    "RetrieveStrings",
    "SetConfiguration",
    "WarmRecovery",
    "DeviceSpecificInit",
    "Configured",
    "Suspended",
    "DeviceSpecificFinalize",
    "FinalizingSelf",
};

Result Device::InitializeAsync(InitializeCallback callback, void *context)
{
    Result result = ResultSuccess();

    m_InitializeCallback  = callback;
    m_InitializeContext   = context;

    if (m_pParent != nullptr)
    {
        m_pParent->GetHubPortHierarchy(m_PortHierarchy);
        m_PortHierarchy[m_pParent->GetHierarchicalLevel()] = m_ParentPortNumber;
        m_Level = m_pParent->GetHierarchicalLevel() + 1;
        m_pHc = m_pParent->GetHostControllerDriver();
    }
    else
    {
        m_DeviceDescriptor.bDeviceClass = UsbClass_Hub;
        m_Level = 0;
    }

    // Resolve root
    Device* pTemp = this;
    m_pRootDevice = this;
    while((pTemp = pTemp->GetParentDevice()) != nullptr)
    {
        m_pRootDevice = pTemp;
    }

    m_pPlatform = m_pHc->GetPlatformController();
    GetObjectName(m_ObjectName, sizeof(m_ObjectName));

    NN_USB_ABORT_UPON_ERROR(
        m_ManagedDevice.Initialize(
            &m_pHs->m_DeviceManager,
            (GetDeviceClass() == UsbClass_Hub),
            (GetDeviceClass() == UsbClass_Hub) && (GetHierarchicalLevel() == 0)
        )
    );

    NN_USB_LOG_INFO("%s - Initializing, speed=%s, vid=0x%x, pid=0x%x, class=%s.\n",
                    m_ObjectName, Util::GetSpeedDescription(m_Speed),
                    m_DeviceDescriptor.idVendor, m_DeviceDescriptor.idProduct,
                    Util::GetClassCodeDescription(m_DeviceDescriptor.bDeviceClass));
    NN_USB_ABORT_UPON_ERROR(Fsm::Initialize(m_ObjectName,
                                            EventNames, Event_Maximum,
                                            StateNames, State_Maximum,
                                            State_Initialize));
    Fsm::InitStaticTimedFsmEvent(&m_EpTerminationRetryTimer, Event_TerminateRequest);
    Fsm::InitStaticTimedFsmEvent(&m_SetAddrDelayEvent,       Event_Continue);

    return result;
}

Result Device::Finalize()
{
    Result result = ResultSuccess();

    NN_USB_LOG_INFO("%s - Finalizing.\n", m_ObjectName);

    Fsm::Finalize();

    NN_USB_ABORT_UPON_ERROR(m_ManagedDevice.Finalize());

    return result;
}

Result Device::TerminateAsync(TerminationOptionMask options, TerminateCompleteCallback completeCbf, void *context)
{
    Result result = ResultSuccess();

    NN_USB_LOG_INFO("%s - Terminating with options", m_ObjectName);
    if (options & TerminationOptions_Disconnected)
    {
        NN_SDK_LOG(" Disconnected");
    }
    if (options & TerminationOptions_Suspend)
    {
        NN_SDK_LOG(" Suspend");
    }
    if (options & TerminationOptions_PowerOff)
    {
        NN_SDK_LOG(" PowerOff");
    }
    NN_SDK_LOG("\n");

    m_TerminationOptionMask     = options | TerminationOptions_Valid;
    m_TerminateCompleteCallback = completeCbf;
    m_TerminateCompleteContext  = context;
    Fsm::QueueFsmEvent(Event_TerminateRequest);

    return result;
}

void Device::GetDeviceDescriptor(UsbDeviceDescriptor *pReturnedDescriptor)
{
    *pReturnedDescriptor = m_DeviceDescriptor;
}

void Device::GetConfigDescriptor(UsbConfigDescriptor *pReturnedDescriptor)
{
    if (m_pActCfg != nullptr)
    {
        *pReturnedDescriptor = *m_pActCfg;
    }
    else
    {
        memset(pReturnedDescriptor, 0, sizeof(UsbConfigDescriptor));
    }
}

void Device::GetDeviceProfile(DeviceProfile *pReturnedProfile)
{
    GetObjectName(pReturnedProfile->deviceDebugName, sizeof(pReturnedProfile->deviceDebugName));

    pReturnedProfile->deviceSpeed = GetSpeed();
    pReturnedProfile->deviceUid   = GetUid();

    GetDeviceDescriptor(&pReturnedProfile->deviceDesc);
    GetConfigDescriptor(&pReturnedProfile->cfgDesc);

    pReturnedProfile->enumerationTimestamp = GetEnumerationTimeStamp();
}

Result Device::GetInterfaceProfile(int32_t ifIndex, InterfaceProfile *pReturnedProfile)
{
    Result result = ResultInterfaceInvalid();
    memset(pReturnedProfile->epInDesc, 0, sizeof(pReturnedProfile->epInDesc));
    memset(pReturnedProfile->epOutDesc, 0, sizeof(pReturnedProfile->epInDesc));
    memset(pReturnedProfile->epInCompDesc, 0, sizeof(pReturnedProfile->epInCompDesc));
    memset(pReturnedProfile->epOutCompDesc, 0, sizeof(pReturnedProfile->epOutCompDesc));

    DeviceInterface *pDif = GetDeviceInterface(ifIndex);

    if (pDif)
    {
        if (pDif->pActAltSetting != nullptr)
        {
            result = ResultSuccess();

            // pass back top level information
            pReturnedProfile->deviceUid       = GetUid();
            pReturnedProfile->altSettingCount = pDif->altSettingCount;
            pReturnedProfile->ifDesc          = *pDif->pActAltSetting;

            // Pass back information about endpoints bound to this interface
            for (EndpointNumber epNumber = 1; epNumber <= MaxUserEndpointPairCount; epNumber++)
            {
                HostEndpoint *pHep;
                if (GetHostEndpoint(epNumber, UsbEndpointDirection_ToHost, &pHep).IsSuccess())
                {
                    if (pHep->ifMask & (1 << ifIndex))
                    {
                        pReturnedProfile->epInDesc[epNumber] = *pHep->pDescriptor;

                        if (pHep->pCompanionDescriptor != nullptr)
                        {
                            pReturnedProfile->epInCompDesc[epNumber] = *pHep->pCompanionDescriptor;
                        }
                    }
                }
                if (GetHostEndpoint(epNumber, UsbEndpointDirection_ToDevice, &pHep).IsSuccess())
                {
                    if (pHep->ifMask & (1 << ifIndex))
                    {
                        pReturnedProfile->epOutDesc[epNumber]     = *pHep->pDescriptor;

                        if (pHep->pCompanionDescriptor != nullptr)
                        {
                            pReturnedProfile->epOutCompDesc[epNumber] = *pHep->pCompanionDescriptor;
                        }
                    }
                }
            }
        }
    }
    return result;
}

Result Device::GetAlternateInterfaceProfile(int32_t ifIndex, uint8_t bAlternateSetting, InterfaceProfile *pReturnedProfile)
{
    Result result = ResultSuccess();

    memset(pReturnedProfile->epInDesc, 0, sizeof(pReturnedProfile->epInDesc));
    memset(pReturnedProfile->epOutDesc, 0, sizeof(pReturnedProfile->epInDesc));
    memset(pReturnedProfile->epInCompDesc, 0, sizeof(pReturnedProfile->epInCompDesc));
    memset(pReturnedProfile->epOutCompDesc, 0, sizeof(pReturnedProfile->epOutCompDesc));

    if (ifIndex < 0)
    {
        return ResultInvalidParameter();
    }

    if (bAlternateSetting >= HsLimitMaxAltInterfaceSettingsCount)
    {
        return ResultImplementationLimit();
    }

    DeviceInterface *pDif = GetDeviceInterface(ifIndex);

    if (pDif == nullptr || pDif->altSettingArray[bAlternateSetting] == nullptr)
    {
        return ResultInvalidParameter();
    }

    UsbEndpointDescriptors endpointInDescriptors[MaxUserEndpointPairCount];
    UsbEndpointDescriptors endpointOutDescriptors[MaxUserEndpointPairCount];
    memset(endpointInDescriptors, 0, sizeof(endpointInDescriptors));
    memset(endpointOutDescriptors, 0, sizeof(endpointOutDescriptors));
    UsbInterfaceDescriptor *pIntDesc = pDif->altSettingArray[bAlternateSetting];
    if ((result = ParseEndpoints(m_pActCfg, pIntDesc, endpointInDescriptors, endpointOutDescriptors)).IsSuccess())
    {
        // pass back top level information
        pReturnedProfile->deviceUid       = GetUid();
        pReturnedProfile->altSettingCount = pDif->altSettingCount;
        pReturnedProfile->ifDesc          = *pDif->altSettingArray[bAlternateSetting];

        // Pass back information about endpoints bound to this interface
        for (EndpointNumber epNumber = 1; epNumber <= MaxUserEndpointPairCount; epNumber++)
        {
            if (endpointInDescriptors[epNumber - 1].pStandard != nullptr)
            {
                pReturnedProfile->epInDesc[epNumber]      = *endpointInDescriptors[epNumber - 1].pStandard;
            }
            if (endpointInDescriptors[epNumber - 1].pCompanion != nullptr)
            {
                pReturnedProfile->epInCompDesc[epNumber]  = *endpointInDescriptors[epNumber - 1].pCompanion;
            }
            if (endpointOutDescriptors[epNumber - 1].pStandard != nullptr)
            {
                pReturnedProfile->epOutDesc[epNumber]     = *endpointOutDescriptors[epNumber - 1].pStandard;
            }
            if (endpointOutDescriptors[epNumber - 1].pCompanion != nullptr)
            {
                pReturnedProfile->epOutCompDesc[epNumber] = *endpointOutDescriptors[epNumber - 1].pCompanion;
            }
        }
    }

    return result;
}

Result Device::SetInterface(int32_t ifIndex, uint8_t bAlternateSetting)
{
    Result result = ResultInterfaceInvalid();

    do
    {
        DeviceInterface *pDif = GetDeviceInterface(ifIndex);

        if (pDif)
        {
            result = ResultSuccess();

            if (bAlternateSetting >= HsLimitMaxAltInterfaceSettingsCount)
            {
                NN_USB_BREAK_UPON_ERROR(ResultImplementationLimit());
            }
            if (pDif->altSettingArray[bAlternateSetting] == nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(ResultInvalidParameter());
            }

            // There cannot be any open endpoints which belong to this interface
            for (int32_t i = 0; i < MaxUserEndpointPairCount; i++)
            {
                if ((m_EpIn[i].ifMask & (1 << ifIndex)) && (m_EpIn[i].pHcEp != nullptr))
                {
                    return ResultResourceBusy();
                }

                if ((m_EpOut[i].ifMask & (1 << ifIndex)) && (m_EpOut[i].pHcEp != nullptr))
                {
                    return ResultResourceBusy();
                }
            }

            // Check if the new setting requires the endpoint currently owned by other interface
            NN_USB_BREAK_UPON_ERROR(ParseAndTryEndpoints(ifIndex, bAlternateSetting));

            // Unbind all endpoints which were associated with this interface
            for (int32_t i = 0; i < MaxUserEndpointPairCount; i++)
            {
                m_EpIn[i].ifMask &= ~(1 << ifIndex);
                m_EpOut[i].ifMask &= ~(1 << ifIndex);

                // Performing set interface always sets device data toggle
                // back to DATA0, per section 9.4.5 of the USB 2.0 standard
                m_EpIn[i].toggle = false;
                m_EpOut[i].toggle = false;
            }

            // Now finally tell the device we are cutting over to alternate setting
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_SetInterface,
                                              BmRequestType(Standard, Interface, HostToDevice),
                                              bAlternateSetting, ifIndex,
                                              0, nullptr, false,
                                              SetAltIfCompletionCallback,
                                              GetDeviceInterface(ifIndex),
                                              HsLimitControlXferTimeout)
            );
            pDif->pActAltSetting = pDif->altSettingArray[bAlternateSetting];
        }
    }while (false);

    return result;
}

void Device::GetHubPortHierarchy(HubPortNumber *pHierarchy)
{
    memcpy(pHierarchy, m_PortHierarchy, sizeof(m_PortHierarchy));
}

int32_t Device::GetHierarchicalLevel()
{
    return m_Level;
}

HubPortNumber Device::GetParentPort()
{
    return m_ParentPortNumber;
}

Device* Device::GetParentDevice()
{
    return m_pParent;
}

Device* Device::GetRootDevice()
{
    return m_pRootDevice;
}

int Device::GetObjectName(char *pObjName, uint32_t maxSize)
{
    char hcName[HsLimitMaxDebugNameSize] = { 0 };
    m_pHc->GetObjectName(hcName, sizeof(hcName));
    return nn::util::SNPrintf(pObjName, maxSize,
                              "%sDevice%d-%s/L%d/P%d/A%02x",
                              Util::GetSpeedDescription(GetSpeed()),
                              GetUid(), hcName, m_Level, m_ParentPortNumber, m_Address);
}

HostControllerDriver* Device::GetHostControllerDriver()
{
    return m_pHc;
}

HostControllerDriverDeviceContext* Device::GetHostControllerDriverDeviceContext()
{
    return m_pHcDevContext;
}

bool Device::IsAddressAssigned()
{
    return (m_Address != 0) ? true : false;
}

uint8_t Device::GetDeviceClass()
{
    return m_DeviceDescriptor.bDeviceClass;
}

StartupMode Device::GetStartupMode()
{
    return m_StartupMode;
}

DeviceUid Device::GetUid()
{
    return m_ManagedDevice.GetUid();
}

UsbDeviceSpeed Device::GetSpeed()
{
    return m_Speed;
}

uint8_t Device::GetAddress()
{
    return m_Address;
}

Device::TerminationOptionMask Device::GetTerminationOptions()
{
    return m_TerminationOptionMask;
}

bool Device::FindHighSpeedParent(HubPortNumber *pRetPort, Device** ppRetDevice)
{
    bool isFound = false;
    Device *pP1 = nullptr;
    Device *pP0 = this;
    while (pP0 != nullptr)
    {
        if (pP0->GetSpeed() == UsbDeviceSpeed_High)
        {
            if (ppRetDevice != nullptr)
            {
                *ppRetDevice = pP0;
            }
            if ((pP1 != nullptr) && (pRetPort != nullptr))
            {
                *pRetPort = pP1->GetParentPort();
            }
            isFound = true;
            break;
        }
        pP1 = pP0;
        pP0 = pP0->GetParentDevice();
    }
    return isFound;
}

Result Device::CreateManagedHostEndpoint(int32_t ifIndex, UsbEndpointType epType, EndpointNumber epNumber, UsbEndpointDirection epDir,
                                         uint16_t maxUrbCount, uint32_t maxTransferSize, bool isToggleReset, void *adminContext)
{
    HostEndpoint *pHep = nullptr;
    bool isAllocated = false;
    Result result = ResultSuccess();

    // Deprecated parameter. Remove later.
    NN_UNUSED(maxTransferSize);

    do
    {
        NN_USB_BREAK_UPON_ERROR(GetHostEndpoint(epNumber, epDir, &pHep));
        if (Fsm::GetState() != State_Configured)
        {
            NN_USB_BREAK_UPON_ERROR(ResultInvalidDeviceState());
        }
        if ( !((1 << ifIndex) & pHep->ifMask))
        {
            NN_USB_BREAK_UPON_ERROR(ResultInvalidParameter());
        }
        if (epType != UsbGetEndpointType(pHep->pDescriptor))
        {
            NN_USB_BREAK_UPON_ERROR(ResultInvalidParameter());
        }
        if (maxUrbCount > HsLimitMaxUrbPerEndpointCount)
        {
            NN_USB_BREAK_UPON_ERROR(ResultImplementationLimit());
        }
        if (maxUrbCount == 0)
        {
            NN_USB_BREAK_UPON_ERROR(ResultInvalidParameter());
        }
        if (pHep->pHcEp != nullptr)
        {
            NN_USB_BREAK_UPON_ERROR(ResultResourceBusy());
        }
        pHep->epType               = epType;
        pHep->epNumber             = epNumber;
        pHep->epDir                = epDir;
        pHep->maxUrbCount          = maxUrbCount;
        pHep->splitTxnPortNumber   = 0;
        pHep->splitTxnAddress      = 0;
        pHep->toggle               = (isToggleReset) ? 0 : pHep->toggle;
        pHep->isShortTransferError = false;
        pHep->adminContext         = adminContext;
        NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(pHep->pHcEp = new HostControllerDriverEndpoint(m_pHs, m_pHc));
        isAllocated = true;
        NN_USB_BREAK_UPON_ERROR(pHep->pHcEp->InitializeAsync(pHep, m_pHcDevContext,
                                                             EndpointInitializeDoneCallback, this));
    }while (false);

    if ( !result.IsSuccess() && pHep && isAllocated)
    {
        delete pHep->pHcEp;
        pHep->pHcEp = nullptr;
    }

    return result;
}

Result Device::DestroyManagedHostEndpoint(HostEndpoint *pHep, void *adminContext)
{
    Result result = ResultSuccess();

    do
    {
        if (pHep->pHcEp == nullptr)
        {
            // This is common, such as device disconnect while class driver running
            // so don't do any warning
            result = ResultEndpointStateInvalid();
            break;
        }
        pHep->adminContext = adminContext;
        NN_USB_BREAK_UPON_ERROR(pHep->pHcEp->TerminateAsync(EndpointTerminateDoneCallback, this));
        m_EpDestroyPendMask |= Util::GetEndpointMask(pHep);
    }while (false);

    return result;
}

Result Device::GetHostEndpoint(EndpointNumber epNumber, UsbEndpointDirection epDir, HostEndpoint **ppHeP)
{
    Result result = ResultSuccess();
    if (epNumber == 0)
    {
        *ppHeP = &m_Ep0;
    }
    else if ((epNumber <= MaxUserEndpointPairCount) && (epNumber > 0))
    {
        int32_t epIndex = epNumber - 1;
        if (epDir == UsbEndpointDirection_ToDevice)
        {
            *ppHeP = &m_EpOut[epIndex];
        }
        else if (epDir == UsbEndpointDirection_ToHost)
        {
            *ppHeP = &m_EpIn[epIndex];
        }
        else
        {
            result = ResultInvalidParameter();
        }
    }
    else
    {
        result = ResultInvalidParameter();
    }
    return result;
}

Result Device::GetManagedHostEndpoint(EndpointNumber epNumber, UsbEndpointDirection epDirection, HostEndpoint **ppHep)
{
    Result result;
    HostEndpoint *pHep = nullptr;
    if ((result = GetHostEndpoint(epNumber, epDirection, &pHep)).IsSuccess())
    {
        if (Fsm::GetState() == State_Configured)
        {
            if(pHep->pHcEp != nullptr)
            {
                *ppHep = pHep;
            }
            else
            {
                result = ResultEndpointStateInvalid();
            }
        }
        else
        {
            result = ResultInvalidDeviceState();
        }
    }
    return result;
}

Result Device::GetControlTransferManager(ControlTransferManager **ppCtrlMgr)
{
    Result result = ResultSuccess();

    // external client cannot do control transfers unless device is in proper state
    if (Fsm::GetState() == State_Configured)
    {
        *ppCtrlMgr = &m_CtrlTransferMgr;
    }
    else
    {
        result = ResultInvalidDeviceState();
    }

    return result;
}

nn::TimeSpan Device::GetEnumerationTimeStamp()
{
    return m_EnumerationTimestamp;
}

Result Device::Reset()
{
    HubDevice::Port* pPort = HubDevice::GetParentPort(this);

    // This should happen only if this device is the RootHub, but we shall never
    // call the Reset() on RootHub.
    NN_USB_ABORT_IF_NULL(pPort);

    return pPort->IssueAdministrativeCommand(HubDevice::Port::AdministrativeCommand_Reset);
}

void Device::ReportInitializeProgress(InitializeProgress progress)
{
    if (m_InitializeCallback != nullptr)
    {
        (*m_InitializeCallback)(progress, m_InitializeContext);
    }
}

void Device::MarkHostEndpointAsInvalid(HostEndpoint *pHep)
{
    pHep->epDir     = UsbEndpointDirection_Invalid;
    pHep->epType    = UsbEndpointType_Invalid;
}

void Device::ControlTransferCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    Device *pThis = reinterpret_cast<Device *>(context);
    pThis->QueueFsmEvent(Event_ControlTransferComplete, status, transferredSize);
}

void Device::SetAltIfCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    DeviceInterface *pDif = reinterpret_cast<DeviceInterface *>(context);
    pDif->pDevice->QueueFsmEvent(Event_SetAltIfComplete, status, pDif);
}

void Device::EndpointInitializeDoneCallback(Hs* pHs, Result status, HostControllerDriverEndpoint* pEp, void* context)
{
    Device *pThis = reinterpret_cast<Device *>(context);
    HostEndpoint *pHep = pEp->GetHostEndpoint();

    if (status.IsFailure())
    {
        NN_USB_ABORT_UNLESS_SUCCESS(pHep->pHcEp->Finalize());
        delete pHep->pHcEp;
        pHep->pHcEp = nullptr;
    }
    pThis->Fsm::QueueFsmEvent(Event_EndpointInitializeComplete, status, pHep);
}

void Device::EndpointTerminateDoneCallback(Hs *pHs, Result status, HostControllerDriverEndpoint *pEp, void *context)
{
    Device *pThis = reinterpret_cast<Device *>(context);
    HostEndpoint *pHep = pEp->GetHostEndpoint();
    pThis->Fsm::QueueFsmEvent(Event_EndpointTerminateComplete, status, pHep);
}

void  Device::AddressDeviceDoneCallback(Hs* pHs, Result status, void *context)
{
    Device *pThis = reinterpret_cast<Device *>(context);
    pThis->Fsm::QueueFsmEvent(Event_AddressComplete, status);
}

// Sanity check the type and size of some descriptors
Result Device::CheckDescriptors(char *buffer, size_t length)
{
    size_t offset = 0;
    size_t remain = length;

    UsbConfigDescriptor *pCfg = reinterpret_cast<UsbConfigDescriptor*>(buffer);

    // At least contains a Configuration Descriptor
    if (length < UsbDescriptorSize_Config)
    {
        return ResultInvalidDescriptor();
    }

    // Sanity check the Configuration Descriptor
    if (pCfg->bLength         != UsbDescriptorSize_Config ||
        pCfg->bDescriptorType != UsbDescriptorType_Config ||
        pCfg->wTotalLength    != length                    )
    {
        return ResultInvalidDescriptor();
    }

    offset += UsbDescriptorSize_Config;
    remain -= UsbDescriptorSize_Config;

    // Iterate through all sub-descriptors
    while (remain)
    {
        uint8_t bLength;
        uint8_t bDescriptorType;

        // Any descriptor should be at least 2 bytes (bLength + bDescriptorType)
        if (remain < 2)
        {
            return ResultInvalidDescriptor();
        }

        bLength         = buffer[offset];
        bDescriptorType = buffer[offset + 1];

        if (bLength > remain)
        {
            return ResultInvalidDescriptor();
        }

        if (bDescriptorType == UsbDescriptorType_Interface &&
            bLength         != UsbDescriptorSize_Interface  )
        {
            return ResultInvalidDescriptor();
        }

        // UAC endpoint bLength could be larger
        if (bDescriptorType == UsbDescriptorType_Endpoint &&
            bLength         <  UsbDescriptorSize_Endpoint  )
        {
            return ResultInvalidDescriptor();
        }

        if (bDescriptorType == UsbDescriptorType_EndpointCompanion &&
            bLength         != UsbDescriptorSize_EndpointCompanion  )
        {
            return ResultInvalidDescriptor();
        }

        offset += bLength;
        remain -= bLength;
    }

    return ResultSuccess();
}

Result Device::ParseInterfaces(uint32_t cfgIndex)
{
    NN_USB_ABORT_UNLESS(cfgIndex < NN_USB_ARRAY_SIZE(m_CfgDescArray));
    NN_USB_ABORT_IF_NULL(m_CfgDescArray[cfgIndex]);

    // set active configuration
    m_pActCfg = m_CfgDescArray[cfgIndex];

    // accommodate unusual devices which offer no interfaces
    if(m_pActCfg->bNumInterfaces == 0)
    {
        NN_USB_LOG_INFO("%s - This device has no interfaces.\n", m_ObjectName);
        return ResultSuccess();
    }

    // parse and apply configuration descriptor
    uint32_t offset = UsbDescriptorSize_Config;
    while (offset < m_pActCfg->wTotalLength)
    {
        uint8_t *pByte = reinterpret_cast<uint8_t *>(m_pActCfg) + offset;
        uint8_t descSize = pByte[0];
        uint8_t descType = pByte[1];

        if (descType == UsbDescriptorType_Interface)
        {
            UsbInterfaceDescriptor *pIntDesc = reinterpret_cast<UsbInterfaceDescriptor*>(pByte);

            if (pIntDesc->bInterfaceNumber  <  HsLimitMaxInterfacesPerConfigurationCount &&
                pIntDesc->bAlternateSetting <  HsLimitMaxAltInterfaceSettingsCount       &&
                pIntDesc->bNumEndpoints     <= UsbLimitMaxEndpointPairCount               )
            {
                DeviceInterface *pDi = &m_IfArray[pIntDesc->bInterfaceNumber];

                if (pDi->altSettingArray[pIntDesc->bAlternateSetting] != nullptr)
                {
                    return ResultInvalidDescriptor();
                }

                pDi->altSettingArray[pIntDesc->bAlternateSetting] = pIntDesc;
                pDi->altSettingCount++;
            }
            else
            {
                // ignore the interface
            }

        }

        offset += descSize;
    }

    return ResultSuccess();
}

Result Device::ParseEndpoints(UsbConfigDescriptor    *pCfgDesc,
                              UsbInterfaceDescriptor *pIntDesc,
                              UsbEndpointDescriptors *epInDescriptors,
                              UsbEndpointDescriptors *epOutDescriptors)
{
    UsbEndpointDescriptor* pLastStandardEndpointDescriptor = nullptr;
    uint32_t offset = static_cast<uint32_t>(
        (uintptr_t)pIntDesc - (uintptr_t)pCfgDesc
    ) + UsbDescriptorSize_Interface;

    uint32_t endpointCount  = 0;
    uint32_t companionCount = 0;
    UsbEndpointDescriptors *pEpDescriptors = nullptr;

    while (offset < pCfgDesc->wTotalLength)
    {
        uint8_t *pByte = (uint8_t *)pCfgDesc + offset;
        uint8_t descSize = pByte[0];
        uint8_t descType = pByte[1];

        if (descType == UsbDescriptorType_Interface)
        {
            break;
        }
        else if (descType == UsbDescriptorType_Endpoint)
        {
            UsbEndpointDescriptor *pEpDesc  = (UsbEndpointDescriptor *)pByte;
            EndpointNumber         epNumber = UsbGetEndpointNumber(pEpDesc);
            int32_t                epIndex  = epNumber - 1;

            if (!Util::IsValidEndpointDescriptor(pEpDesc, m_Speed))
            {
                return ResultInvalidDescriptor();
            }

            if (UsbEndpointIsHostToDevice(pEpDesc))
            {
                pEpDescriptors = &epOutDescriptors[epIndex];
            }
            else
            {
                pEpDescriptors = &epInDescriptors[epIndex];
            }

            if (pEpDescriptors->pStandard != nullptr)
            {
                return ResultInvalidDescriptor();
            }

            pEpDescriptors->pStandard = pEpDesc;

            endpointCount++;
            pLastStandardEndpointDescriptor = pEpDesc;
        }
        else if(descType == UsbDescriptorType_EndpointCompanion)
        {
            UsbEndpointCompanionDescriptor *pEpCompDesc = (UsbEndpointCompanionDescriptor *)pByte;

            if (endpointCount != companionCount + 1)
            {
                return ResultInvalidDescriptor();
            }

            // Companion desdcriptors immediately follow corresponding standard descriptor
            int32_t epIndex = UsbGetEndpointNumber(pLastStandardEndpointDescriptor) - 1;
            if(UsbEndpointIsHostToDevice(pLastStandardEndpointDescriptor))
            {
                pEpDescriptors = &epOutDescriptors[epIndex];
            }
            else if(UsbEndpointIsDeviceToHost(pLastStandardEndpointDescriptor))
            {
                pEpDescriptors = &epInDescriptors[epIndex];
            }

            if (pEpDescriptors->pCompanion != nullptr)
            {
                return ResultInvalidDescriptor();
            }

            pEpDescriptors->pCompanion = pEpCompDesc;

            companionCount++;
            pLastStandardEndpointDescriptor = nullptr;
        }
        else
        {
            // ignore
        }

        offset += descSize;
    }

    if (endpointCount != pIntDesc->bNumEndpoints)
    {
        return ResultInvalidDescriptor();
    }

    if (m_Speed == UsbDeviceSpeed_Super   ||
        m_Speed == UsbDeviceSpeed_SuperPlus)
    {
        if (endpointCount != companionCount)
        {
            return ResultInvalidDescriptor();
        }
    }
    else
    {
        if (companionCount != 0)
        {
            return ResultInvalidDescriptor();
        }
    }

    return ResultSuccess();
}

Result Device::ParseAndTryEndpoints(int32_t ifIndex, uint8_t bAlternateSetting)
{
    Result result = ResultSuccess();
    UsbEndpointDescriptors  epInDescriptors[MaxUserEndpointPairCount];
    UsbEndpointDescriptors  epOutDescriptors[MaxUserEndpointPairCount];
    DeviceInterface        *pDif     = &m_IfArray[ifIndex];
    UsbInterfaceDescriptor *pIntDesc = pDif->altSettingArray[bAlternateSetting];

    memset(epInDescriptors,  0, sizeof(epInDescriptors));
    memset(epOutDescriptors, 0, sizeof(epOutDescriptors));

    NN_USB_RETURN_UPON_ERROR(
        ParseEndpoints(m_pActCfg, pIntDesc, epInDescriptors, epOutDescriptors)
    );

    for (int32_t index = 0; index < MaxUserEndpointPairCount; index++)
    {
        HostEndpoint        *pHeP;
        UsbEndpointDirection epDir;
        EndpointNumber       epNumber;

        // to host
        if (epInDescriptors[index].pStandard != nullptr)
        {
            epDir    = UsbGetEndpointDirection(epInDescriptors[index].pStandard);
            epNumber = UsbGetEndpointNumber   (epInDescriptors[index].pStandard);

            NN_USB_RETURN_UPON_ERROR(GetHostEndpoint(epNumber, epDir, &pHeP));

            if (pHeP->ifMask != (1U << ifIndex) && pHeP->ifMask != 0)
            {
                return ResultResourceBusy();
            }
        }

        // to device
        if (epOutDescriptors[index].pStandard != nullptr)
        {
            epDir    = UsbGetEndpointDirection(epOutDescriptors[index].pStandard);
            epNumber = UsbGetEndpointNumber   (epOutDescriptors[index].pStandard);

            NN_USB_RETURN_UPON_ERROR(GetHostEndpoint(epNumber, epDir, &pHeP));

            if (pHeP->ifMask != (1U << ifIndex) && pHeP->ifMask != 0)
            {
                return ResultResourceBusy();
            }
        }
    }

    return result;
}

Result Device::ParseAndApplyEndpoints(int32_t ifIndex, uint8_t bAlternateSetting)
{
    Result result = ResultSuccess();
    UsbEndpointDescriptors  epInDescriptors[MaxUserEndpointPairCount];
    UsbEndpointDescriptors  epOutDescriptors[MaxUserEndpointPairCount];
    DeviceInterface        *pDif     = &m_IfArray[ifIndex];
    UsbInterfaceDescriptor *pIntDesc = pDif->altSettingArray[bAlternateSetting];

    memset(epInDescriptors,  0, sizeof(epInDescriptors));
    memset(epOutDescriptors, 0, sizeof(epOutDescriptors));

    NN_USB_RETURN_UPON_ERROR(
        ParseEndpoints(m_pActCfg, pIntDesc, epInDescriptors, epOutDescriptors)
    );

    for (int32_t index = 0; index < MaxUserEndpointPairCount; index++)
    {
        HostEndpoint        *pHeP;
        UsbEndpointDirection epDir;
        EndpointNumber       epNumber;

        // to host
        if (epInDescriptors[index].pStandard != nullptr)
        {
            epDir    = UsbGetEndpointDirection(epInDescriptors[index].pStandard);
            epNumber = UsbGetEndpointNumber   (epInDescriptors[index].pStandard);

            NN_USB_RETURN_UPON_ERROR(GetHostEndpoint(epNumber, epDir, &pHeP));

            pHeP->ifMask              |= (1 << ifIndex);
            pHeP->pDescriptor          = epInDescriptors[index].pStandard;
            pHeP->pCompanionDescriptor = epInDescriptors[index].pCompanion;
        }

        // to device
        if (epOutDescriptors[index].pStandard != nullptr)
        {
            epDir    = UsbGetEndpointDirection(epOutDescriptors[index].pStandard);
            epNumber = UsbGetEndpointNumber   (epOutDescriptors[index].pStandard);

            NN_USB_RETURN_UPON_ERROR(GetHostEndpoint(epNumber, epDir, &pHeP));

            pHeP->ifMask              |= (1 << ifIndex);
            pHeP->pDescriptor          = epOutDescriptors[index].pStandard;
            pHeP->pCompanionDescriptor = epOutDescriptors[index].pCompanion;
        }
    }

    if (result.IsSuccess())
    {
        pDif->pActAltSetting = pIntDesc;
    }

    return result;
}

Result Device::FsmHandler(int32_t state, LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (state)
    {
    case State_Null:
        break;
    case State_Initialize:
        result = InitializeStateHandler(pEvent);
        break;
    case State_AssignAddress:
        result = AssignAddressStateHandler(pEvent);
        break;
    case State_IdentifyConfiguration:
        result = IdentifyConfigurationStateHandler(pEvent);
        break;
    case State_RetrieveStrings:
        result = RetrieveStringsStateHandler(pEvent);
        break;
    case State_SetConfiguration:
        result = SetConfigurationStateHandler(pEvent);
        break;
    case State_WarmRecovery:
        result = WarmRecoveryStateHandler(pEvent);
        break;
    case State_DeviceSpecificInit:
        result = DeviceSpecificInitStateHandler(pEvent);
        break;
    case State_Configured:
        result = ConfiguredStateHandler(pEvent);
        break;
    case State_Suspended:
        result = SuspendedStateHandler(pEvent);
        break;
    case State_DeviceSpecificFinalize:
        result = DeviceSpecificFinalizeStateHandler(pEvent);
        break;
    case State_FinalizingSelf:
        result = FinalizingSelfStateHandler(pEvent);
        break;
    default:
        NN_USB_ABORT("Device FSM state %d invalid", state);
        break;
    }
    return result;
}

Result Device::InitializeStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            if (m_ManagedDevice.IsRootHub())
            {
                // Nothing to do for root hubs
                Fsm::QueueFsmEvent(Event_Skip, result);
            }
            else
            {
                Fsm::QueueFsmEvent(Event_Try, result);
            }
            break;
        }
    case Event_Skip:
        {
            Fsm::SetState(State_DeviceSpecificInit);
            break;
        }
    case Event_Try:
        {
            m_Ep0.epDir                     = UsbEndpointDirection_Control;
            m_Ep0.epType                    = UsbEndpointType_Control;
            m_Ep0.epNumber                  = 0;
            m_Ep0.maxUrbCount               = 1;
            m_Ep0.isShortTransferError      = true;

            NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
                m_Ep0.pHcEp = new HostControllerDriverEndpoint(m_pHs, m_pHc)
            );

            result = m_Ep0.pHcEp->InitializeAsync(
                &m_Ep0, m_pHcDevContext, EndpointInitializeDoneCallback, this
            );
            if (result.IsFailure())
            {
                delete m_Ep0.pHcEp;
                m_Ep0.pHcEp = nullptr;
            }

            break;
        }
    case Event_EndpointInitializeComplete:
        {
            if (pEvent->status.IsSuccess())
            {
                m_CtrlTransferMgr.Initialize(m_Ep0.pHcEp, m_ObjectName);
                Fsm::SetState(State_AssignAddress);
            }
            else
            {
                NN_USB_ABORT_UNLESS_SUCCESS(m_Ep0.pHcEp->Finalize());
                delete m_Ep0.pHcEp;
                m_Ep0.pHcEp = nullptr;
            }
            result = pEvent->status;
            break;
        }
    case Event_Error:
        {
            ReportInitializeProgress(InitializeProgress_Error);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if (result.IsFailure())
    {
        Fsm::QueueFsmEvent(Event_Error, result);
    }

    return result;
}

Result Device::AssignAddressStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            NN_USB_BREAK_UPON_ERROR(
                m_pHc->AddressDeviceAsync(m_pHcDevContext, AddressDeviceDoneCallback, this)
            );
            break;
        }
    case Event_AddressComplete:
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            m_Address = m_pHcDevContext->address;

            Fsm::StartTimedFsmEvent(&m_SetAddrDelayEvent, nn::TimeSpan::FromMicroSeconds(10000));
            break;
        }
    case Event_Error:
        {
            ReportInitializeProgress(InitializeProgress_Error);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_SetAddrDelayEvent);
            break;
        }
    case Event_Continue:
        {
            // Update fsm name for correct debug tracing
            GetObjectName(m_ObjectName, HsLimitMaxDebugNameSize);
            Fsm::SetName(m_ObjectName);

            // Report progress at major milestone
            ReportInitializeProgress(InitializeProgress_AddressAssigned);

            // Continue
            Fsm::SetState(State_IdentifyConfiguration);
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Event_Error, result);
    }

    return result;
}

Result Device::IdentifyConfigurationStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_Substate = 1;
            /*
             * TODO: 64K page aligned memory allocation gives big pressure to the
             *       allocator. Perhaps change this to per-use basis allocation?
             */
            NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
                m_pCtrlTransferBuffer = m_pPlatform->DoSmmu().AllocMemory(
                    nullptr, ControlTransferBufferSize, "m_pCtrlTransferBuffer"
                )
            );
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetDescriptor,
                                              BmRequestType(Standard, Device, DeviceToHost),
                                              ((UsbDescriptorType_Config << 8) | 0), m_Substate >> 1,
                                              sizeof(UsbConfigDescriptor), m_pCtrlTransferBuffer, false,
                                              ControlTransferCompletionCallback,
                                              this, HsLimitControlXferTimeout)
            );
            break;
        }

    case Event_ControlTransferComplete:
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            uint32_t lastTransferSize = pEvent->args[0].size;
            UsbConfigDescriptor *pCfgDesc = reinterpret_cast<UsbConfigDescriptor *>(m_pCtrlTransferBuffer);
            uint32_t nextTransferSize = pCfgDesc->wTotalLength;

            // sanity check
            if (pCfgDesc->bLength         != UsbDescriptorSize_Config ||
                pCfgDesc->bDescriptorType != UsbDescriptorType_Config ||
                m_DeviceDescriptor.bNumConfigurations == 0)
            {
                result = ResultInvalidDescriptor();
                break;
            }

            if ((m_Substate % 2) == 0)
            {
                // save off the full descriptor
                int32_t cfgDescIndex = (m_Substate >> 1) - 1;

                NN_USB_BREAK_UPON_ERROR(
                    CheckDescriptors((char*)m_pCtrlTransferBuffer, lastTransferSize)
                );

                NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
                    (m_CfgDescArray[cfgDescIndex] = reinterpret_cast<UsbConfigDescriptor *>(detail::UsbMemoryAlloc(lastTransferSize,
                                                                                                           "UsbConfigDescriptor"))));
                memcpy(m_CfgDescArray[cfgDescIndex], m_pCtrlTransferBuffer, lastTransferSize);
                nextTransferSize = sizeof(UsbConfigDescriptor);
            }
            if (((m_Substate >> 1) < m_DeviceDescriptor.bNumConfigurations) &&
                ((m_Substate >> 1) < HsLimitMaxConfigurationsPerDeviceCount))
            {

                NN_USB_BREAK_UPON_ERROR(
                    m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetDescriptor,
                                                  BmRequestType(Standard, Device, DeviceToHost),
                                                  ((UsbDescriptorType_Config << 8) | 0), m_Substate >> 1,
                                                  nextTransferSize, m_pCtrlTransferBuffer, false,
                                                  ControlTransferCompletionCallback,
                                                  this, HsLimitControlXferTimeout)
                );
                m_Substate++;
            }
            else
            {
                NN_USB_BREAK_UPON_ERROR(ParseInterfaces(DefaultConfigurationIndex));
                for (unsigned ifIndex = 0; ifIndex < NN_USB_ARRAY_SIZE(m_IfArray); ifIndex++)
                {
                    uint8_t bAlternateSetting = DefaultAltIfIndex;
                    if (m_IfArray[ifIndex].altSettingArray[bAlternateSetting] != nullptr)
                    {
                        NN_USB_BREAK_UPON_ERROR(ParseAndApplyEndpoints(ifIndex, bAlternateSetting));
                    }
                }

                if (result.IsSuccess())
                {
                    Fsm::SetState(State_RetrieveStrings);
                }
            }
            break;
        }

    case Event_Try:
        {
            break;
        }
    case Event_Error:
        {
            ReportInitializeProgress(InitializeProgress_Error);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Event_Error, result);
    }

    return result;
} // NOLINT(impl/function_size)

Result Device::RetrieveStringsStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_Substate = 0;
            if (m_pDevStringLangIds == nullptr)
            {
                NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
                    m_pDevStringLangIds = reinterpret_cast<UsbStringDescriptor *>(
                        detail::UsbMemoryCalloc(sizeof(UsbStringDescriptor), "m_pDevStringLangIds")
                    )
                );
            }
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetDescriptor,
                                              BmRequestType(Standard, Device, DeviceToHost),
                                              ((UsbDescriptorType_String << 8) | 0), 0,
                                              sizeof(UsbStringDescriptor), m_pCtrlTransferBuffer, false,
                                              ControlTransferCompletionCallback,
                                              this, HsLimitControlXferTimeout));
            break;
        }
    case Event_ControlTransferComplete:
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            uint32_t lastTransferSize = pEvent->args[0].size;
            memcpy(m_pDevStringLangIds, m_pCtrlTransferBuffer, lastTransferSize);
            if (m_pDevStringLangIds->bDescriptorType == UsbDescriptorType_String)
            {
                if (m_Substate == 0)
                {
                    int32_t maxLanguages =
                        (m_pDevStringLangIds->bLength -
                         (sizeof(m_pDevStringLangIds->bDescriptorType) + sizeof(m_pDevStringLangIds->bLength))) / sizeof(uint16_t);
                    maxLanguages = (maxLanguages < MaxStringLanguageCount) ? maxLanguages : MaxStringLanguageCount;
                    size_t descriptorSize = 2 + 2 * maxLanguages;

                    detail::UsbMemoryFree(m_pDevStringLangIds, "m_pDevStringLangIds");
                    NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(
                        m_pDevStringLangIds = reinterpret_cast<UsbStringDescriptor *>(
                            detail::UsbMemoryCalloc(descriptorSize, "m_pDevStringLangIds")
                        )
                    );
                    NN_USB_BREAK_UPON_ERROR(
                        m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_GetDescriptor,
                                                      BmRequestType(Standard, Device, DeviceToHost),
                                                      ((UsbDescriptorType_String << 8) | 0), 0,
                                                      descriptorSize,
                                                      m_pCtrlTransferBuffer, false,
                                                      ControlTransferCompletionCallback,
                                                      this, HsLimitControlXferTimeout)
                    );
                }
                else
                {
                    // successful completion
                    Fsm::QueueFsmEvent(Event_Continue);
                }
            }
            else
            {
                // bad response
                Fsm::QueueFsmEvent(Event_Error);
            }
            m_Substate++;
            break;
        }
    case Event_Error:
        {
            if (m_pDevStringLangIds != nullptr)
            {
                detail::UsbMemoryFree(m_pDevStringLangIds, "m_pDevStringLangIds");
                m_pDevStringLangIds = nullptr;
            }
            Fsm::QueueFsmEvent(Event_Continue);
            break;
        }
    case Event_Continue:
        {
            Fsm::SetState((GetStartupMode() == StartupMode_WarmRecovery) ? State_WarmRecovery : State_SetConfiguration);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            // Teardown the buffer since we don't need it anymore
            if (m_pCtrlTransferBuffer != nullptr)
            {
                m_pPlatform->DoSmmu().FreeMemory(m_pCtrlTransferBuffer, "m_pCtrlTransferBuffer");
                m_pCtrlTransferBuffer = nullptr;
            }
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Event_Error, result);
    }

    return result;
} // NOLINT(impl/function_size)

Result Device::SetConfigurationStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            NN_USB_BREAK_UPON_ERROR(
                m_CtrlTransferMgr.SubmitAsync(UsbCtrlXferReq_SetConfiguration,
                                              BmRequestType(Standard, Device, HostToDevice),
                                              m_pActCfg->bConfigurationValue, 0,
                                              0, nullptr, false,
                                              ControlTransferCompletionCallback,
                                              this, HsLimitControlXferTimeout)
            );
            break;
        }
    case Event_ControlTransferComplete:
        {
            NN_USB_BREAK_UPON_ERROR(pEvent->status);
            Fsm::SetState(State_DeviceSpecificInit);
            if(detail::IsLogVerbose())
            {
                Util::PrintDescriptors(m_pActCfg, &m_DeviceDescriptor, m_Speed,
                                       m_IfArray, m_EpIn, m_EpOut);
            }
            break;
        }
    case Event_Error:
        {
            ReportInitializeProgress(InitializeProgress_Error);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    if ( !result.IsSuccess())
    {
        Fsm::QueueFsmEvent(Event_Error, result);
    }

    return result;
}

Result Device::WarmRecoveryStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {

            break;
        }
    case Event_Try:
        {

            break;
        }
    case Event_Error:
        {
            ReportInitializeProgress(InitializeProgress_Error);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Device::DeviceSpecificInitStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::QueueFsmEvent(Event_Skip);
            break;
        }
    case Event_Skip:
        {
            Fsm::SetState(State_Configured);
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Device::ConfiguredStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            NN_USB_LOG_INFO("%s - Configured, speed=%s, vid=0x%x, pid=0x%x, class=%s.\n",
                           m_ObjectName, Util::GetSpeedDescription(m_Speed),
                           m_DeviceDescriptor.idVendor, m_DeviceDescriptor.idProduct,
                           Util::GetClassCodeDescription(m_DeviceDescriptor.bDeviceClass));
            m_EnumerationTimestamp = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            ReportInitializeProgress(InitializeProgress_Complete);
            m_ManagedDevice.Offer(m_IfArray);
            break;
        }
    case Event_EndpointInitializeComplete:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_ManagedDevice.HostEndpointCreateConfirm(pHep, pEvent->status);
            break;
        }
    case Event_EndpointTerminateComplete:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_EpDestroyPendMask &= ~Util::GetEndpointMask(pHep);
            if (pHep->pHcEp == nullptr) break;
            NN_USB_BREAK_UPON_ERROR(pHep->pHcEp->Finalize());
            delete pHep->pHcEp;
            pHep->pHcEp = nullptr;
            m_ManagedDevice.HostEndpointDestroyConfirm(pHep);
            break;
        }
    case Event_SetAltIfComplete:
        {
            DeviceInterface *pDif = reinterpret_cast<DeviceInterface *>(pEvent->args[0].pData);
            pDif->pManagedInterface->SetAltIfConfirm(ParseAndApplyEndpoints(pDif->ifIndex,
                                                                            pDif->pActAltSetting->bAlternateSetting));
            break;
        }
    case Event_Error:
        {
            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            m_ManagedDevice.Revoke();
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Device::SuspendedStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {

            break;
        }
    case Event_Try:
        {

            break;
        }
    case Event_TerminateRequest:
        {
            Fsm::SetState(State_DeviceSpecificFinalize);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Device::DeviceSpecificFinalizeStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            Fsm::QueueFsmEvent(Event_Skip);
            break;
        }
    case Event_Skip:
        {
            Fsm::SetState(State_FinalizingSelf);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Device::FinalizingSelfStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        if (m_EpDestroyPendMask == 0)
        {
            Fsm::QueueFsmEvent(Event_TerminateRequest);
        }
        else
        {
            // The pending request will send Event_EndpointTerminateComplete on completion
        }

        break;

    case Event_TerminateRequest:
        // Find an opened endpoint and close it
        if ((m_Ep0.pHcEp != nullptr) && !(m_EpDestroyPendMask & Util::GetEndpointMask(&m_Ep0)))
        {
            result = m_Ep0.pHcEp->TerminateAsync(EndpointTerminateDoneCallback, this);
            if(result.IsSuccess())
            {
                m_EpDestroyPendMask |= Util::GetEndpointMask(&m_Ep0);
                break;
            }
        }
        for (int32_t i = 0; i < MaxUserEndpointPairCount; i++)
        {
            if ((m_EpIn[i].pHcEp != nullptr) && !(m_EpDestroyPendMask & Util::GetEndpointMask(&m_EpIn[i])))
            {
                result = m_EpIn[i].pHcEp->TerminateAsync(EndpointTerminateDoneCallback, this);
                if(result.IsSuccess())
                {
                    m_EpDestroyPendMask |= Util::GetEndpointMask(&m_EpIn[i]);
                    break;
                }
            }
            if ((m_EpOut[i].pHcEp != nullptr) && !(m_EpDestroyPendMask & Util::GetEndpointMask(&m_EpOut[i])))
            {
                result = m_EpOut[i].pHcEp->TerminateAsync(EndpointTerminateDoneCallback, this);
                if(result.IsSuccess())
                {
                    m_EpDestroyPendMask |= Util::GetEndpointMask(&m_EpOut[i]);
                    break;
                }
            }
        }

        // All endpoints closed
        if ((m_EpDestroyPendMask == 0) && result.IsSuccess())
        {
            StopTimedFsmEvent(&m_EpTerminationRetryTimer);
            Fsm::QueueFsmEvent(Event_Continue);
        }
        else if (ResultResourceBusy::Includes(result)) // open/close/reset pending
        {
            // Retry in 500ms
            Fsm::ReStartTimedFsmEvent(&m_EpTerminationRetryTimer,
                                      nn::TimeSpan::FromMilliSeconds(500));
        }
        break;

    case Event_EndpointTerminateComplete:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_EpDestroyPendMask &= ~Util::GetEndpointMask(pHep);

            if (pHep->epNumber == 0)
            {
                // Finalize after EP0 being terminated, so no more events will
                // be created / received after this point for CtrlXferMgr FSM.
                m_CtrlTransferMgr.Finalize();
            }

            NN_USB_ABORT_UNLESS_SUCCESS(pHep->pHcEp->Finalize());
            delete pHep->pHcEp;
            pHep->pHcEp = nullptr;

            Fsm::QueueFsmEvent(Event_TerminateRequest);
            break;
        }
    case Event_Continue:
        {
            if (m_pCtrlTransferBuffer != nullptr)
            {
                m_pPlatform->DoSmmu().FreeMemory(m_pCtrlTransferBuffer, "m_pCtrlTransferBuffer");
                m_pCtrlTransferBuffer = nullptr;
            }
            for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_CfgDescArray); i++)
            {
                if (m_CfgDescArray[i] != nullptr)
                {
                    detail::UsbMemoryFree(m_CfgDescArray[i], "UsbConfigDescriptor");
                    m_CfgDescArray[i] = nullptr;
                }
            }
            if (m_pDevStringLangIds != nullptr)
            {
                detail::UsbMemoryFree(m_pDevStringLangIds, "m_pDevStringLangIds");
                m_pDevStringLangIds = nullptr;
            }
            Fsm::SetState(State_Null);
            break;
        }
    case Fsm::Event_Exit:
        {
            if (m_TerminateCompleteCallback != nullptr)
            {
                (*m_TerminateCompleteCallback)(this, result, m_TerminateCompleteContext);
            }
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
} // NOLINT(impl/function_size)


DeviceInterface* Device::GetDeviceInterface(int32_t ifIndex)
{
    for (int i = 0; i < HsLimitMaxInterfacesPerConfigurationCount; i++)
    {
        DeviceInterface *pDif = m_IfArray + i;

        if ((pDif->pActAltSetting != nullptr) && (pDif->ifIndex == ifIndex))
        {
            return pDif;
        }
    }

    return nullptr;
}


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
