﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief External request deferral
 *
 * @details This class handles deferment of external API
 *          requests
 */

#pragma once

#include <nn/os.h>

namespace nn { namespace usb { namespace hs {

class Hs;

class DeferredRequestBase
{
public:
    typedef std::function<Result()> ActionHandler;
    typedef std::function<void()>   CompletionHandler;

    nn::util::IntrusiveListNode m_ListNode;

    DeferredRequestBase(nn::os::MessageQueue&  requestQueue,
                        char                   category,
                        int32_t                command)
        : m_RequestQueue(requestQueue)
        , m_AvailabilitySemaphore(1, 1)
        , m_Category(category)
        , m_Command(command)
        , m_Result(ResultSuccess())
        , m_TransferredSize(0)
    {
    }


    virtual ~DeferredRequestBase()
    {
    }

    void AsynchronousRequest(ActionHandler action, CompletionHandler complete);
    void SynchronousRequest (ActionHandler action, CompletionHandler complete);

    void Dispatch();

    void CompleteRequest();
    void CompleteRequest(Result status);
    void CompleteRequest(Result status, uint32_t transferredSize);

    void Flush();

    int32_t            GetCommand();
    char               GetCategory();

    Result             GetActionResult();
    uint32_t           GetXferredSize();

private:
    nn::os::MessageQueue&   m_RequestQueue;

    nn::os::Semaphore       m_AvailabilitySemaphore;

    ActionHandler           m_Action;
    CompletionHandler       m_Complete;

    char                    m_Category;
    int32_t                 m_Command;

    Result                  m_Result;
    uint32_t                m_TransferredSize;
};

template<char Category, class RequestDataType>
class DeferredRequest : public DeferredRequestBase
{
public:
    DeferredRequest(nn::os::MessageQueue&  requestQueue,
                    int32_t                command)
        : DeferredRequestBase(requestQueue, Category, command)
    {
    }

    ~DeferredRequest()
    {
    }

public:
    RequestDataType m_Data;
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

