﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class Hs;
class DeviceManager;

class ClientRootSession : public nn::sf::ISharedObject
{
public:
    enum Command
    {
        Command_Invalid = 0,
        Command_QueryAllInterfaces,
        Command_QueryAvailableInterfaces,
        Command_QueryAcquiredInterfaces,
        Command_AcquireUsbIf,
        Command_NewUsbIfAvailableEvent,
    };
    struct QueryUsbIfArgDataType
    {
        ClientRootSession     *pClient;
        DeviceFilter*          pDevFilter;
        InterfaceQueryOutput*  pQueryOut;
        int32_t                maxOutputCount;
        int32_t*               pQueryOutCount;
    };
    struct AcquireUsbIfArgDataType
    {
        ClientRootSession     *pClient;
        InterfaceHandle        ifHandle;
        DeviceProfile         *pDeviceProfile;
        InterfaceProfile      *pInterfaceProfile;
    };
    struct GetDescriptorStringArgDataType
    {
        InterfaceHandle              ifHandle;
        uint8_t                      index;
        uint32_t                     maxOutputSize;
        ClientRootSession           *pClient;

        nn::dd::ProcessHandle        procHandle;
        uint64_t                     procVa;
        void*                        pBuffer;
    };
    struct NewUsbIfAvailableEventDataType
    {
        DeviceFilter deviceFilter;
        bool         isMatched;
    };

    typedef DeferredRequest<'R', QueryUsbIfArgDataType>          QueryUsbIfDeferredRequest;
    typedef DeferredRequest<'R', NewUsbIfAvailableEventDataType> NewUsbIfAvailableEventDeferredRequest;
    typedef DeferredRequest<'R', AcquireUsbIfArgDataType>        AcquireUsbIfDeferredRequest;
    typedef DeferredRequest<'R', GetDescriptorStringArgDataType> GetDescriptorStringDeferredRequest;

    NN_IMPLICIT ClientRootSession(Hs *pHs) NN_NOEXCEPT;
    NN_IMPLICIT ~ClientRootSession()NN_NOEXCEPT;

    // Client process memory space binding
    Result BindClientProcess(nn::sf::NativeHandle processHandle) NN_NOEXCEPT;

    // Interface Query
    Result QueryAllInterfaces(nn::sf::Out<int32_t> outCount,
                              const nn::sf::OutBuffer &output,
                              DeviceFilter devId)NN_NOEXCEPT;
    Result QueryAvailableInterfaces(nn::sf::Out<int32_t> outCount,
                                    const nn::sf::OutBuffer &output,
                                    DeviceFilter devId)NN_NOEXCEPT;
    Result QueryAcquiredInterfaces(nn::sf::Out<int32_t> outCount,
                                   const nn::sf::OutBuffer &output)NN_NOEXCEPT;

    // Interface Events
    Result CreateInterfaceAvailableEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle,
                                         uint8_t filterIndex,
                                         DeviceFilter deviceFilter)NN_NOEXCEPT;
    Result DestroyInterfaceAvailableEvent(uint8_t filterIndex)NN_NOEXCEPT;
    Result GetInterfaceStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle)NN_NOEXCEPT;

    Result AcquireUsbIf(InterfaceHandle ifHandle,
                        nn::sf::Out<nn::sf::SharedPointer<nn::usb::hs::IClientIfSession>> outSession,
                        const nn::sf::OutBuffer &outputDeviceProfile,
                        const nn::sf::OutBuffer &outputInterfaceProfile) NN_NOEXCEPT;

    Result SetTestMode(uint32_t port, TestMode mode, int32_t offset,
                       nn::sf::Out<uint32_t> outStrength) NN_NOEXCEPT;

    // Methods called by subordinate sessions
    void AddClientIfSession(ClientIfSession *pClientIf);
    void RemoveClientIfSession(ClientIfSession *pClientIf);

    ClientId GetId();

    void AdvertiseDeviceInterface(UsbDeviceDescriptor* pDeviceDescriptor,
                                  UsbInterfaceDescriptor* pInterfaceDescriptor);
    void SignalIfStateChange(InterfaceHandle ifHandle);

private:
    class IfAvailableNotifier
    {
    public:
        IfAvailableNotifier() : m_IsValid(false), m_Event(nn::os::EventClearMode_AutoClear, true)
        {
            memset(&m_Filter, 0, sizeof(m_Filter));
        }
        bool                 m_IsValid;
        DeviceFilter         m_Filter;
        nn::os::SystemEvent  m_Event;
    };
    typedef detail::NnList<ClientIfSession> ClientIfSessionListType;

    Hs                             *m_pHs;
    nn::os::Mutex                   m_AccessMutex;
    ClientIfSessionListType         m_IfSessionList;
    bool                            m_IsInitialized;
    IfAvailableNotifier             m_IfAvailableNotifiers[HsLimitMaxClientInterfaceAvailabilityEventCount];
    ClientId                        m_ClientId;
    nn::os::SystemEvent             m_IfStateChangeEvent;
    nn::sf::NativeHandle            m_ProcessHandle;

private:
    ClientIfSession* GetClientIfSession(InterfaceHandle ifHandle);
    int GetClientIfSessionCount();
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

