﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

ClientRootSession::ClientRootSession(Hs *pHs) NN_NOEXCEPT
    : m_pHs(pHs)
    , m_AccessMutex(true)
    , m_IsInitialized(true)
    , m_IfStateChangeEvent(nn::os::EventClearMode_AutoClear, true)
{
    m_ClientId = m_pHs->AllocateClientId(this);
    NN_USB_ABORT_UNLESS(m_ClientId > 0);
    NN_USB_LOG_INFO("ClientRootSession() clientId=%d, freeMem=0x%x\n",
                    m_ClientId, detail::UsbMemoryGetTotalFreeSize());
}

ClientRootSession::~ClientRootSession() NN_NOEXCEPT
{
    NN_USB_ABORT_UNLESS(GetClientIfSessionCount() == 0);

    m_pHs->FreeClientId(m_ClientId);

    m_IsInitialized = false;
    NN_USB_LOG_INFO("~ClientRootSession() clientId=%d, freeMem=0x%x\n",
                    m_ClientId, detail::UsbMemoryGetTotalFreeSize());
}

Result ClientRootSession::BindClientProcess(nn::sf::NativeHandle processHandle) NN_NOEXCEPT
{
    m_ProcessHandle = std::move(processHandle);

    return ResultSuccess();
}

Result ClientRootSession::QueryAllInterfaces(nn::sf::Out<int32_t> outCount,
                                             const nn::sf::OutBuffer& output,
                                             DeviceFilter devId) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    int32_t queryOutCount = 0;

    QueryUsbIfDeferredRequest request(m_pHs->GetRequestQueue(),
                                      Command_QueryAllInterfaces);

    request.m_Data.pClient        = this;
    request.m_Data.pDevFilter     = &devId;
    request.m_Data.pQueryOut      = reinterpret_cast<InterfaceQueryOutput *>(output.GetPointerUnsafe());
    request.m_Data.maxOutputCount = output.GetSize() / sizeof(InterfaceQueryOutput);
    request.m_Data.pQueryOutCount = &queryOutCount;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.QueryAllInterfaces(&request);
        },
        [&] {
            result = request.GetActionResult();
            outCount.Set(queryOutCount);
        }
    );

    return result;
}

Result ClientRootSession::QueryAvailableInterfaces(nn::sf::Out<int32_t> outCount,
                                                   const nn::sf::OutBuffer& output,
                                                   DeviceFilter devId) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    int32_t queryOutCount = 0;

    QueryUsbIfDeferredRequest request(m_pHs->GetRequestQueue(),
                                      Command_QueryAvailableInterfaces);

    request.m_Data.pClient        = this;
    request.m_Data.pDevFilter     = &devId;
    request.m_Data.pQueryOut      = reinterpret_cast<InterfaceQueryOutput *>(output.GetPointerUnsafe());
    request.m_Data.maxOutputCount = output.GetSize() / sizeof(InterfaceQueryOutput);
    request.m_Data.pQueryOutCount = &queryOutCount;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.QueryAvailableInterfaces(&request);
        },
        [&] {
            result = request.GetActionResult();
            outCount.Set(queryOutCount);
        }
    );

    return result;
}

Result ClientRootSession::QueryAcquiredInterfaces(nn::sf::Out<int32_t> outCount,
                                                  const nn::sf::OutBuffer& output) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    int32_t queryOutCount = 0;

    QueryUsbIfDeferredRequest request(m_pHs->GetRequestQueue(),
                                      Command_QueryAcquiredInterfaces);

    request.m_Data.pClient        = this;
    request.m_Data.pDevFilter     = nullptr;
    request.m_Data.pQueryOut      = reinterpret_cast<InterfaceQueryOutput *>(output.GetPointerUnsafe());
    request.m_Data.maxOutputCount = output.GetSize() / sizeof(InterfaceQueryOutput);
    request.m_Data.pQueryOutCount = &queryOutCount;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.QueryAcquiredInterfaces(&request);
        },
        [&] {
            result = request.GetActionResult();
            outCount.Set(queryOutCount);
        }
    );

    return result;
}

Result ClientRootSession::CreateInterfaceAvailableEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle,
                                                        uint8_t filterIndex,
                                                        DeviceFilter deviceFilter) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (filterIndex >= HsLimitMaxClientInterfaceAvailabilityEventCount)
    {
        return ResultImplementationLimit();
    }

    NewUsbIfAvailableEventDeferredRequest request(m_pHs->GetRequestQueue(),
                                                  Command_NewUsbIfAvailableEvent);

    request.m_Data.isMatched    = false;
    request.m_Data.deviceFilter = deviceFilter;

    m_IfAvailableNotifiers[filterIndex].m_Filter  = deviceFilter;
    m_IfAvailableNotifiers[filterIndex].m_IsValid = true;
    eventHandle.Set(
        nn::sf::NativeHandle(m_IfAvailableNotifiers[filterIndex].m_Event.GetReadableHandle(), false)
    );

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.NewUsbIfAvailableEvent(&request);
        },
        [&] {
            result = request.GetActionResult();
            if (result.IsSuccess() && request.m_Data.isMatched)
            {
                m_IfAvailableNotifiers[filterIndex].m_Event.Signal();
            }
        }
    );

    return result;
}

Result ClientRootSession::DestroyInterfaceAvailableEvent(uint8_t filterIndex)NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (filterIndex >= HsLimitMaxClientInterfaceAvailabilityEventCount)
    {
        return ResultImplementationLimit();
    }

    m_IfAvailableNotifiers[filterIndex].m_IsValid = false;
    memset(&m_IfAvailableNotifiers[filterIndex].m_Filter, 0, sizeof(m_IfAvailableNotifiers[filterIndex].m_Filter));

    return result;
}

Result ClientRootSession::GetInterfaceStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle)NN_NOEXCEPT
{
    Result result = ResultSuccess();
    eventHandle.Set(nn::sf::NativeHandle(m_IfStateChangeEvent.GetReadableHandle(), false));
    return result;
}

Result ClientRootSession::AcquireUsbIf(InterfaceHandle ifHandle,
                                       nn::sf::Out<nn::sf::SharedPointer<nn::usb::hs::IClientIfSession>> outSession,
                                       const nn::sf::OutBuffer& outputDeviceProfile,
                                       const nn::sf::OutBuffer& outputInterfaceProfile) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    DeviceProfile *pOutputDeviceProfile = reinterpret_cast<DeviceProfile *>(
        outputDeviceProfile.GetPointerUnsafe()
    );
    InterfaceProfile *pOutputInterfaceProfile = reinterpret_cast<InterfaceProfile *>(
        outputInterfaceProfile.GetPointerUnsafe()
    );

    if (outputDeviceProfile.GetSize() != sizeof(DeviceProfile))
    {
        return ResultBufferSizeError();
    }

    if (outputInterfaceProfile.GetSize() != sizeof(InterfaceProfile))
    {
        return ResultBufferSizeError();
    }

    if (GetClientIfSessionCount() >= HsLimitMaxInterfacesPerClientCount)
    {
        return ResultImplementationLimit();
    }

    AcquireUsbIfDeferredRequest request(m_pHs->GetRequestQueue(),
                                        Command_AcquireUsbIf);

    request.m_Data.pClient           = this;
    request.m_Data.ifHandle          = ifHandle;
    request.m_Data.pDeviceProfile    = pOutputDeviceProfile;
    request.m_Data.pInterfaceProfile = pOutputInterfaceProfile;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.AcquireUsbIf(&request);
        },
        [&] {
            result = request.GetActionResult();
            if (result.IsFailure())
            {
                return;
            }

            nn::sf::SharedPointer<IClientIfSession> p =
                detail::Factory::CreateSharedEmplaced<IClientIfSession, ClientIfSession>(
                    detail::UsbMemoryGetAllocator(),
                    m_pHs,
                    this,
                    pOutputDeviceProfile,
                    pOutputInterfaceProfile,
                    reinterpret_cast<nn::dd::ProcessHandle>(m_ProcessHandle.GetOsHandle())
                );

            if (p != nullptr)
            {
                *outSession = std::move(p);
            }
            else
            {
                result = ResultMemAllocFailure();
            }
        }
    );

    return result;
}

Result ClientRootSession::SetTestMode(uint32_t port, TestMode mode, int32_t offset,
                                      nn::sf::Out<uint32_t> outStrength) NN_NOEXCEPT
{
    Result result = ResultNotImplemented();

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    result = m_pHs->m_pController->SetTestMode(port, mode, offset, outStrength.GetPointer());
#endif

    return result;
}

void ClientRootSession::AddClientIfSession(ClientIfSession *pClientIf)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    m_IfSessionList.push_back(*pClientIf);
}

void ClientRootSession::RemoveClientIfSession(ClientIfSession *pClient)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    m_IfSessionList.erase(m_IfSessionList.iterator_to(*pClient));
}

ClientId ClientRootSession::GetId()
{
    return m_ClientId;
}

void ClientRootSession::AdvertiseDeviceInterface(UsbDeviceDescriptor* pDeviceDescriptor,
                                                 UsbInterfaceDescriptor* pInterfaceDescriptor)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

    for(uint8_t index=0; index < NN_USB_ARRAY_SIZE(m_IfAvailableNotifiers); index++)
    {
        IfAvailableNotifier* pIfNotify = &m_IfAvailableNotifiers[index];
        if(pIfNotify->m_IsValid)
        {
            if (Util::EvaluateDeviceId(&pIfNotify->m_Filter, pInterfaceDescriptor, pDeviceDescriptor))
            {
                pIfNotify->m_Event.Signal();
            }
        }
    }
}

void ClientRootSession::SignalIfStateChange(InterfaceHandle ifHandle)
{
    // Signal the common state change event
    m_IfStateChangeEvent.Signal();

    // Also signal the specific interface
    ClientIfSession* pClientIfSession = GetClientIfSession(ifHandle);
    if(pClientIfSession) pClientIfSession->SignalStateChange();
}

ClientIfSession* ClientRootSession::GetClientIfSession(InterfaceHandle ifHandle)
{
    ClientIfSession* pIfSession = nullptr;
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    for (ClientIfSessionListType::iterator it = m_IfSessionList.begin();
         it != m_IfSessionList.end(); it++)
    {
        if(it->GetHandle() == ifHandle)
        {
            pIfSession = &(*it);
            break;
        }
    }
    return pIfSession;
}

int ClientRootSession::GetClientIfSessionCount()
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return m_IfSessionList.size();
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
