﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

ClientIfSession::ClientIfSession(Hs                    *pHs,
                                 ClientRootSession     *pClient,
                                 DeviceProfile         *pDeviceProfile,
                                 InterfaceProfile      *pInterfaceProfile,
                                 nn::dd::ProcessHandle  processHandle) NN_NOEXCEPT
    : m_pHs(pHs)
    , m_pClient(pClient, true)
    , m_DeviceProfile(*pDeviceProfile)
    , m_InterfaceProfile(*pInterfaceProfile)
    , m_AccessMutex(true)
    , m_IsInitialized(true)
    , m_StateChangeEvent(nn::os::EventClearMode_AutoClear, true)
    , m_ProcessHandle(processHandle)
    , m_CtrlXferRequest(m_pHs->GetRequestQueue(), Command_ControlTransfer)
    , m_CtrlXferCompletionEvent(nn::os::EventClearMode_ManualClear, true)
{
    NN_USB_LOG_INFO("ClientIfSession clientId=%d, deviceUid=0x%x, ifHandle=0x%x\n",
                    m_pClient->GetId(), m_InterfaceProfile.deviceUid, m_InterfaceProfile.handle);
    m_pClient->AddClientIfSession(this);

    m_CtrlXferRequest.m_Data.ifHandle      = GetHandle();
    m_CtrlXferRequest.m_Data.timeoutInMs   = 20000; // 20s
    m_CtrlXferRequest.m_Data.procHandle    = m_ProcessHandle;
}

ClientIfSession::~ClientIfSession() NN_NOEXCEPT
{
    m_IsInitialized = false;

    // Make sure any ongoing ctrl xfer is finished / aborted
    m_CtrlXferRequest.Flush();

    // Release the interface in the host stack
    ReleaseUsbIfDeferredRequest request(m_pHs->GetRequestQueue(),
                                        Command_ReleaseUsbIf);
    request.m_Data.ifHandle = GetHandle();

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.ReleaseUsbIf(&request);
        },
        [&] {
            NN_USB_ABORT_UNLESS(GetClientEpSessionCount() == 0);
            m_pClient->RemoveClientIfSession(this);
        }
    );

    NN_USB_LOG_INFO("~ClientIfSession clientId=%d, deviceUid=0x%x, ifHandle=0x%x\n",
                    m_pClient->GetId(), m_InterfaceProfile.deviceUid, m_InterfaceProfile.handle);
}

Result ClientIfSession::GetStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT
{
    eventHandle.Set(nn::sf::NativeHandle(m_StateChangeEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

Result ClientIfSession::GetCtrlXferCompletionEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT
{
    eventHandle.Set(nn::sf::NativeHandle(m_CtrlXferCompletionEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

Result ClientIfSession::SetInterface(uint8_t bAlternateSetting,
                                     const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
{
    Result result;
    if (outBuffer.GetSize() != sizeof(InterfaceProfile))
    {
        return ResultBufferSizeError();
    }

    SetInterfaceDeferredRequest request(m_pHs->GetRequestQueue(),
                                        Command_SetInterface);

    request.m_Data.pData = reinterpret_cast<InterfaceProfile *>(outBuffer.GetPointerUnsafe());
    request.m_Data.bAlternateSetting = bAlternateSetting;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientIfSession::GetInterface(const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
{
    Result result;
    if (outBuffer.GetSize() != sizeof(InterfaceProfile))
    {
        return ResultBufferSizeError();
    }

    GetInterfaceDeferredRequest request(m_pHs->GetRequestQueue(),
                                        Command_GetInterface);
    request.m_Data.pData = reinterpret_cast<InterfaceProfile *>(outBuffer.GetPointerUnsafe());;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientIfSession::GetAlternateInterface(uint8_t bAlternateSetting,
                                              const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
{
    Result result;
    if (outBuffer.GetSize() != sizeof(InterfaceProfile))
    {
        return ResultBufferSizeError();
    }

    GetInterfaceDeferredRequest request(m_pHs->GetRequestQueue(),
                                        Command_GetAltUsbIf);
    request.m_Data.bAlternateSetting = bAlternateSetting;
    request.m_Data.pData             =
        reinterpret_cast<InterfaceProfile *>(outBuffer.GetPointerUnsafe());

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientIfSession::GetCurrentFrame(nn::sf::Out<FrameNumber> outFrame) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    FrameNumber frame;

    result = m_pHs->m_DeviceManager.GetCurrentFrameId(GetHandle(), &frame);

    if (result.IsSuccess())
    {
        outFrame.Set(frame);
    }

    return result;
}

Result ClientIfSession::CtrlXferAsync(uint8_t         bmRequestType,
                                      uint8_t         bRequest,
                                      uint16_t        wValue,
                                      uint16_t        wIndex,
                                      uint16_t        wLength,
                                      uint64_t        procVa) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_CtrlXferRequest.m_Data.bmRequestType = bmRequestType;
    m_CtrlXferRequest.m_Data.bRequest      = bRequest;
    m_CtrlXferRequest.m_Data.wValue        = wValue;
    m_CtrlXferRequest.m_Data.wIndex        = wIndex;
    m_CtrlXferRequest.m_Data.wLength       = wLength;

    m_CtrlXferRequest.m_Data.procVa        = procVa;

    m_CtrlXferRequest.m_Data.xferredSize   = 0;

    m_CtrlXferRequest.AsynchronousRequest(
        [=] {
            return m_pHs->m_DeviceManager.SubmitControlTransfer(&m_CtrlXferRequest);
        },
        [=] {
            m_CtrlXferRequest.m_Data.result      = m_CtrlXferRequest.GetActionResult();
            m_CtrlXferRequest.m_Data.xferredSize = m_CtrlXferRequest.GetXferredSize();

            m_CtrlXferCompletionEvent.Signal();
        }
    );

    return result;
}

Result ClientIfSession::GetCtrlXferReport(nn::sf::OutBuffer buffer) NN_NOEXCEPT
{
    if (buffer.GetSize() != sizeof(XferReport))
    {
        return ResultBufferSizeError();
    }

    XferReport *pReport = reinterpret_cast<XferReport*>(buffer.GetPointerUnsafe());

    pReport->id              = 0; // ignore
    pReport->result          = m_CtrlXferRequest.m_Data.result;
    pReport->requestedSize   = m_CtrlXferRequest.m_Data.wLength;
    pReport->transferredSize = m_CtrlXferRequest.m_Data.xferredSize;

    return ResultSuccess();
}

Result ClientIfSession::ResetDevice() NN_NOEXCEPT
{
    Result result;
    ResetDeviceDeferredRequest request(m_pHs->GetRequestQueue(),
                                       Command_ResetDevice);
    request.m_Data.ifHandle = GetHandle();
    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.ResetDevice(&request);
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientIfSession::OpenUsbEp(UsbEndpointType epType,
                                  EndpointNumber epNumber,
                                  UsbEndpointDirection epDirection,
                                  uint16_t maxUrbCount,
                                  uint32_t maxXferSize,
                                  nn::sf::Out<UsbEndpointDescriptor> outEpDescriptor,
                                  nn::sf::Out<nn::sf::SharedPointer<IClientEpSession>> outSession)
{
    Result result = ResultSuccess();

    if(GetClientEpSessionCount() >= UsbLimitMaxEndpointPairCount)
    {
        return ResultImplementationLimit();
    }

    OpenUsbEpDeferredRequest request(m_pHs->GetRequestQueue(),
                                     Command_OpenUsbEp);

    request.m_Data.epType      = epType;
    request.m_Data.epNumber    = epNumber;
    request.m_Data.epDirection = epDirection;
    request.m_Data.maxUrbCount = maxUrbCount;
    request.m_Data.maxXferSize = maxXferSize;
    request.m_Data.pReturnedEpDescriptor = outEpDescriptor.GetPointer();

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
            if (result.IsFailure())
            {
                return;
            }

            nn::sf::SharedPointer<IClientEpSession> p =
                detail::Factory::CreateSharedEmplaced<IClientEpSession, ClientEpSession>(
                    detail::UsbMemoryGetAllocator(),
                    m_pHs,
                    m_pClient->GetId(),
                    this,
                    epType,
                    epNumber,
                    epDirection,
                    maxUrbCount,
                    maxXferSize,
                    m_ProcessHandle
                );

            if (p != nullptr)
            {
                *outSession = std::move(p);
            }
            else
            {
                result = ResultMemAllocFailure();
            }
        }
    );

    return result;
}

void ClientIfSession::AddClientEpSession(ClientEpSession *pClientEp)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    m_EpSessionList.push_back(*pClientEp);
}

void ClientIfSession::RemoveClientEpSession(ClientEpSession *pClientEp)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    m_EpSessionList.erase(m_EpSessionList.iterator_to(*pClientEp));
}

InterfaceProfile& ClientIfSession::GetInterfaceProfile()
{
    return m_InterfaceProfile;
}

InterfaceHandle ClientIfSession::GetHandle()
{
    return m_InterfaceProfile.handle;
}

void ClientIfSession::SignalStateChange()
{
    m_StateChangeEvent.Signal();
}

int ClientIfSession::GetClientEpSessionCount()
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return m_EpSessionList.size();
}


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
