﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

#include <nn/usb/detail/usb_Ring.h>
#include <nn/os/os_TransferMemory.h>

namespace nn { namespace usb { namespace hs {

class Hs;
class ClientRootSession;
class ClientIfSession;
class DeviceManager;

class ClientEpSession
{
public:
    enum Command
    {
        Command_Invalid = 0,
        Command_Close,
        Command_Urb,
    };

    struct CloseArgDataType;
    struct UrbArgDataType;

    typedef DeferredRequest<'E', CloseArgDataType> CloseDeferredRequest;
    typedef DeferredRequest<'E', UrbArgDataType>   UrbDeferredRequest;

    struct CloseArgDataType
    {
        EndpointNumber        epNumber;
        UsbEndpointDirection  epDirection;
    };

    struct UrbArgDataType
    {
        // Uniq ID for the URB
        uint32_t              id;

        // where does the data go or come from?
        InterfaceHandle       ifHandle;
        EndpointNumber        epNumber;
        UsbEndpointDirection  epDirection;

        // where is the data?
        nn::dd::ProcessHandle procHandle;
        detail::SmmuSpace    *pSmmuSpace;
        uint64_t              procVa;

        // Xfer definition
        XferStatus            status;
        uint32_t              xferCount;
        struct {
            uint32_t          size;
            uint32_t          xferredSize;
            Result            result;
        } xfers[HsLimitMaxXferPerUrbCount];

        // isoch specific
        SchedulePolicy        policy;
        uint32_t              frameId;

        // for convenience
        uint64_t              context;

        // deprecated
        uint32_t              timeoutInMs;
    };

    nn::util::IntrusiveListNode  m_ListNode;

    ClientEpSession(Hs                      *pHs,
                    ClientId                 clientId,
                    ClientIfSession         *pIf,
                    UsbEndpointType          epType,
                    EndpointNumber           epNumber,
                    UsbEndpointDirection     epDirection,
                    uint16_t                 maxUrbCount,
                    uint32_t                 maxXferSize,
                    nn::dd::ProcessHandle    processHandle) NN_NOEXCEPT;
    ~ClientEpSession()NN_NOEXCEPT;

    Result PopulateRing() NN_NOEXCEPT;

    Result CreateSmmuSpace(uint64_t              procVa,
                           uint32_t              size) NN_NOEXCEPT;
    Result DestroySmmuSpace() NN_NOEXCEPT;

    Result ShareReportRing(nn::sf::NativeHandle ringHandle, uint32_t size) NN_NOEXCEPT;

    Result GetCompletionEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT;

    Result BatchBufferAsync(nn::sf::Out<uint32_t>      pOutXferId,
                            uint64_t                   procVa,
                            nn::sf::InBuffer           sizeBuffer,
                            uint32_t                   xferCount,
                            uint64_t                   context,
                            SchedulePolicy             policy,
                            uint32_t                   frameId) NN_NOEXCEPT;

    Result BatchBufferAsync(uint32_t                  *pOutXferId,
                            uint64_t                   procVa,
                            const uint32_t            *sizeArray,
                            uint32_t                   xferCount,
                            uint64_t                   context,
                            SchedulePolicy             policy,
                            uint32_t                   frameId) NN_NOEXCEPT;

    Result PostBufferAsync(nn::sf::Out<uint32_t>      pOutXferId,
                           uint64_t                   address,
                           uint32_t                   bytes,
                           uint64_t                   context) NN_NOEXCEPT;

    Result GetXferReport(nn::sf::Out<uint32_t> pOutCount,
                         nn::sf::OutBuffer     buffer,
                         uint32_t              count) NN_NOEXCEPT;

    Result ReOpen();
    Result Close();

private:
    class RequestRing : public detail::RingBase<UrbDeferredRequest>
    {
    public:
        typedef std::function<void(uint32_t, UrbDeferredRequest*)> Initializer;

        virtual ~RequestRing() NN_NOEXCEPT { /* do nothing */ };
        Result Initialize(uint32_t itemCount, Initializer initializer) NN_NOEXCEPT;
        void   Finalize() NN_NOEXCEPT;
    };

    class ReportRing : public detail::RingBase<XferReport>
    {
    public:
        ReportRing() NN_NOEXCEPT;
        virtual ~ReportRing() NN_NOEXCEPT { /* do nothing */ };

        Result Initialize(uint32_t maxUrbCount) NN_NOEXCEPT;
        Result Initialize(uint32_t maxUrbCount, nn::os::NativeHandle ringHandle, uint32_t size) NN_NOEXCEPT;
        void   Finalize() NN_NOEXCEPT;

    private:
        Result Attach(uint32_t itemCount) NN_NOEXCEPT;
        Result Attach(uint32_t itemCount, nn::os::NativeHandle ringHandle, uint32_t size) NN_NOEXCEPT;
        void   Detach() NN_NOEXCEPT;
        void   Populate() NN_NOEXCEPT;

    private:
        nn::os::TransferMemoryType m_XferMem;
        bool                       m_IsShared;
    };

    Hs*                                      m_pHs;
    ClientId                                 m_ClientId;
    nn::sf::SharedPointer<ClientIfSession>   m_pIfSession;

    UsbEndpointType                          m_EpType;
    EndpointNumber                           m_EndpointNumber;
    UsbEndpointDirection                     m_EpDirection;

    uint16_t                                 m_MaxUrbCount;
    uint32_t                                 m_MaxXferSize;

    nn::os::SystemEvent                      m_CompletionEvent;
    nn::dd::ProcessHandle                    m_ProcessHandle;

    RequestRing                              m_RequestRing;
    ReportRing                               m_ReportRing;
    nn::os::Mutex                            m_RequestMutex;
    nn::os::Mutex                            m_ReportMutex;

    detail::SmmuSpace                        m_SmmuSpace;
};

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

