﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn { namespace usb { namespace hs {

Result Hs::Initialize(detail::UsbPlatform* pPlatform)
{
    Result result = ResultSuccess();

    do
    {
        m_pPlatform = pPlatform;
        nn::os::InitializeMultiWait(&m_MultiWait);
        NN_USB_BREAK_UPON_ERROR(m_ServiceManager.Initialize());
        NN_USB_BREAK_UPON_ERROR(m_DeviceManager.Initialize());

        // For all ports on which main thread may block
        for (int portIndex = 0; portIndex < MainThreadPort_MaxPorts; portIndex++)
        {
            switch (portIndex)
            {
            case MainThreadPort_ReEvaluate:
                nn::os::InitializeEvent(&m_ReEvaluateEvent, false, nn::os::EventClearMode_ManualClear);
                nn::os::InitializeMultiWaitHolder(&m_Holder[portIndex], &m_ReEvaluateEvent);
                break;
            case MainThreadPort_LocalEventTimer:
                m_LocalEventManager.Initialize(&m_Holder[portIndex]);
                break;
            case MainThreadPort_Message:
                nn::os::InitializeMultiWaitHolder(&m_Holder[portIndex], m_MessageQueue.GetBase(),
                                                  nn::os::MessageQueueWaitType_WaitForNotEmpty);
                break;
            default:
                break;
            }
            nn::os::SetMultiWaitHolderUserData(&m_Holder[portIndex], portIndex);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[portIndex]);
        }
        nn::os::CreateThread(&m_MainThread, MainThreadEntry,
                             this, m_MainThreadStack,
                             sizeof(m_MainThreadStack),
                             NN_SYSTEM_THREAD_PRIORITY(usb,HsThread));
        nn::os::SetThreadNamePointer(&m_MainThread, NN_SYSTEM_THREAD_NAME(usb, HsThread));
        m_MainThreadRun = true;
        nn::os::StartThread(&m_MainThread);

    }while (false);

    return result;
}

Result Hs::Finalize()
{
    Result result = ResultSuccess();

    do
    {
        NN_USB_BREAK_UPON_ERROR(m_ServiceManager.Finalize());
        NN_USB_BREAK_UPON_ERROR(m_DeviceManager.Finalize());
        if(!m_MainThreadRun) break;
        m_MainThreadRun = false;
        nn::os::SignalEvent(&m_ReEvaluateEvent);
        nn::os::WaitThread(&m_MainThread);
        nn::os::DestroyThread(&m_MainThread);
    }while (false);

    return result;
}

Result Hs::BindComplex(ComplexId complexId)
{
    detail::UsbComplex *pComplex = m_pPlatform->GetComplex(complexId);

    NN_USB_ABORT_IF_NULL(pComplex);

    m_pController = static_cast<PlatformController*>(
        pComplex->GetController(UsbCapability_Host | UsbCapability_SuperSpeed)
    );

    NN_USB_ABORT_IF_NULL(m_pController);

    return m_pController->Bind(this);
}

Result Hs::UnbindComplex()
{
    NN_USB_ABORT_IF_NULL(m_pController);

    Result result = m_pController->Unbind();

    m_pController = nullptr;

    return result;
}

void Hs::MainThread()
{
    while (m_MainThreadRun)
    {
        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&m_MultiWait);
        {
            std::lock_guard<nn::os::Mutex> lock(m_StackMutex);
            int32_t portIndex = GetMultiWaitHolderUserData(holder);
            switch (portIndex)
            {
            case MainThreadPort_ReEvaluate:
                nn::os::ClearEvent(&m_ReEvaluateEvent);
                break;
            case MainThreadPort_LocalEventTimer:
                m_LocalEventManager.DispatchTimedEvents();
                break;
            case MainThreadPort_Message:
                uintptr_t msg;
                while (m_MessageQueue.TryReceive(&msg))
                {
                    DeferredRequestBase *pDeferredRequest = reinterpret_cast<DeferredRequestBase *>(msg);
                    pDeferredRequest->Dispatch();
                }
                break;
            default:
                break;
            }
            m_LocalEventManager.DispatchEvents();
        }
    }
}

nn::os::MessageQueue& Hs::GetRequestQueue()
{
    return m_MessageQueue;
}

void Hs::SendDeferredRequest(uintptr_t message)
{
    m_MessageQueue.Send(message);
}

ClientId Hs::AllocateClientId(ClientRootSession* pClient)
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientsMutex);

    ClientId id = -1;
    for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_pClients); i++)
    {
        if (m_pClients[i] == nullptr)
        {
            m_pClients[i] = pClient;
            id = static_cast<ClientId>(i + 1);
            break;
        }
    }
    return id;
}

void Hs::FreeClientId(ClientId id)
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientsMutex);

    int32_t i = static_cast<int32_t>(id - 1);
    if((i >= 0) && (i < NN_USB_ARRAY_COUNT32(m_pClients)))
    {
        m_pClients[i] = nullptr;
    }
}

void Hs::AdvertiseDeviceInterfaceToClients(UsbDeviceDescriptor* pDeviceDescriptor,
                                           UsbInterfaceDescriptor* pInterfaceDescriptor)
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientsMutex);

    for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_pClients); i++)
    {
        if (m_pClients[i] != nullptr)
        {
            m_pClients[i]->AdvertiseDeviceInterface(pDeviceDescriptor, pInterfaceDescriptor);
        }
    }
}

void Hs::LockStackMutex()
{
    m_StackMutex.Lock();
}

void Hs::UnlockStackMutex()
{
    m_LocalEventManager.DispatchEvents();
    m_StackMutex.Unlock();
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
