﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_DsProtocol.h
 * @brief   USB Device Stack Protocol Layer
 */

#include "usb_DsCommon.h"
#include "usb_DsDescriptor.h"
#include "usb_DsUrbRing.h"
#include "usb_DsChargingDevice.h"

namespace nn {
namespace usb {
namespace ds {

class DsController;
class DsDummyController;

class DsProtocolEndPoint
{
public:
    DsProtocolEndPoint() NN_NOEXCEPT;

    ~DsProtocolEndPoint() NN_NOEXCEPT;

    void Enable() NN_NOEXCEPT;
    void Disable() NN_NOEXCEPT;
    bool IsEnabled() NN_NOEXCEPT;

    DsUrbRing*GetUrbRingPtr() NN_NOEXCEPT;
    DsUrbRing*AllocUrbRing(DsController *pController, uint8_t address) NN_NOEXCEPT;
    void FreeUrbRing() NN_NOEXCEPT;

private:
    DsUrbRing* m_UrbRingPtr;
    bool m_IsEnabled;
};


class DsProtocol
{
//    friend class DsDefaultInterface;

public:

    explicit DsProtocol() NN_NOEXCEPT;
    ~DsProtocol() NN_NOEXCEPT;

    Result Initialize() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result ClearDeviceData() NN_NOEXCEPT;
    Result AddUsbStringDescriptor(UsbStringDescriptor *pUsbStringDescriptor, uint8_t *pIndex) NN_NOEXCEPT;
    Result DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT;
    Result SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;
    Result SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT;
    Result AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pData, int bytes) NN_NOEXCEPT;
    Result SetReportData(uint8_t bInterfaceNumber, uint8_t *pData, int bytes) NN_NOEXCEPT;

    // Attach controller
    Result AttachDsController(DsController *pController) NN_NOEXCEPT;
    Result DetachDsController() NN_NOEXCEPT;

    // Device
    Result EnableDevice() NN_NOEXCEPT;
    Result DisableDevice() NN_NOEXCEPT;

    // Interface
    Result RegisterInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT;
    Result UnRegisterInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT;

    Result EnableInterface(uint8_t ifNum) NN_NOEXCEPT;
    Result DisableInterface(uint8_t ifNum) NN_NOEXCEPT;
    bool   IsInterfaceEnabled(uint8_t ifNum) NN_NOEXCEPT;

    // Endpoint
    Result RegisterEndpoint(uint8_t endpointAddress, uint8_t ifNum) NN_NOEXCEPT;
    Result UnRegisterEndpoint(uint8_t endpointAddress) NN_NOEXCEPT;

    // USB Status
    Result   SetState(UsbState state) NN_NOEXCEPT;
    UsbState GetState(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    void     BindStateChangeEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    void     UnbindStateChangeEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;

    // SETUP Packet Handling
    Result SetSetupPacket(uint8_t epAddr, UsbCtrlRequest *pSetup) NN_NOEXCEPT;
    Result GetSetupPacket(uint8_t epAddr, char *buffer, size_t size,
                          nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result BindSetupEvent(uint8_t epAddr, uint8_t ifNum, nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result UnbindSetupEvent(uint8_t epAddr, uint8_t ifNum) NN_NOEXCEPT;
    Result DoCtrlRequest() NN_NOEXCEPT;

    // I/O
    Result PostBufferAsync(uint8_t                epAddr,
                           nn::dd::ProcessHandle  processHandle,
                           uint64_t               procVa,
                           uint32_t               bytes,
                           uint32_t              *pUrbId) NN_NOEXCEPT;

    Result PostBuffer(uint8_t                 epAddr,
                      nn::dd::ProcessHandle   processHandle,
                      uint64_t                procVa,
                      uint32_t                bytes,
                      uint32_t               *pBytesTransferred) NN_NOEXCEPT;

    Result PostBuffer(uint8_t   epAddr,
                      void     *buffer,
                      uint32_t  bytes,
                      uint32_t *pBytesTransferred) NN_NOEXCEPT;

    nn::os::SystemEventType* GetCompletionEvent(uint8_t epAddr) NN_NOEXCEPT;

    Result GetUrbReport(uint8_t epAddr, UrbReport *pReport) NN_NOEXCEPT;

    Result Stall(uint8_t epAddr) NN_NOEXCEPT;
    Result Cancel(uint8_t epAddr) NN_NOEXCEPT;
    Result SetZlt(uint8_t epAddr, bool zlt) NN_NOEXCEPT;
    void CompleteBuffer(uint8_t epAddr, Result result, uint32_t bytesRemaining, bool lockMutex) NN_NOEXCEPT;
    DsDescriptor *GetDescriptorPointer() {return  &m_Descriptor;}

    // Function to allow controller and protocol to share proper locks
    void LockEndPointMutex() NN_NOEXCEPT;
    void UnlockEndPointMutex() NN_NOEXCEPT;

private:
    enum CtrlFlow
    {
        CtrlFlow_In,
        CtrlFlow_Out,
        CtrlFlow_None,

        CtrlFlow_Invalid
    };

    typedef void (DsProtocol::*CtrlHandler)(UsbCtrlRequest *pCtrl);
    typedef struct {
        uint8_t     tmask;
        uint8_t     type;
        uint8_t     rmask;
        uint8_t     request;
        CtrlHandler handler;
    } CtrlHandlerRecord;

private:

    // DsDevice client reference, used to control DsChargingDevice
    int                         m_DsDeviceRef;
    bool                        m_DeviceIsEnabled;

    // dummy controller used when the real controller is disabled
    static DsDummyController    s_DummyController;

    DsController               *m_pController;
    DsController               *m_pPhysicalController;
    DsDescriptor                m_Descriptor;
    DsChargingDevice            m_DsChargingDevice;

    // active configure number
    uint8_t                     m_Configuration;

    bool                        m_IsInterfaceEnabled[DsLimitMaxInterfacesPerConfigurationCount];

    // some protocol operations must be serialized
    nn::os::Mutex               m_ProtocolMutex;

    // USB State
    UsbState                                 m_State;
    nn::os::Mutex                            m_StateChangeMutex;
    detail::StdList<nn::os::SystemEvent*>    m_StateChangeEventList;

    // Endpoint State
    DsProtocolEndPoint          m_DsEndPoint[UsbLimitMaxEndpointsCount];
    nn::os::Mutex               m_EndPointMutex;


    // default control pipe
    UsbCtrlRequest              m_SetupPacket;
    nn::os::Mutex               m_SetupMutex;
    nn::os::SystemEvent        *m_pSetupEvent[DsLimitMaxInterfacesPerConfigurationCount];

    nn::os::ThreadType                 m_CtrlPipeThread;
    static const int32_t               m_ThreadStackSize = 1024 * 16;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t m_ThreadStack[m_ThreadStackSize];

    nn::os::EventType           m_CtrlRequestEvent;
    nn::os::EventType           m_CtrlIOEvent;
    nn::os::EventType           m_CtrlFinalizeEvent;

    nn::os::MultiWaitType       m_MultiWait;
    nn::os::MultiWaitHolderType m_CtrlRequestHolder;
    nn::os::MultiWaitHolderType m_CtrlIOHolder;
    nn::os::MultiWaitHolderType m_CtrlFinalizeHolder;

    NN_USB_DMA_ALIGN uint8_t    m_TxBuffer[4096];
    NN_USB_DMA_ALIGN uint8_t    m_RxBuffer[4096];

    CtrlFlow  m_CtrlFlow;

    static const CtrlHandlerRecord CtrlHandlerTable[];

private:



    static void CtrlPipeThread(void *arg) NN_NOEXCEPT
    {
        DsProtocol *pThis = reinterpret_cast<DsProtocol *>(arg);

        pThis->CtrlPipeLoop();
    }

    void   CtrlPipeLoop() NN_NOEXCEPT;
    void   HandleCtrlRequest() NN_NOEXCEPT;

    // control request handlers
    void   GetStatus(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   ClearFeature(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   SetFeature(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   SetAddress(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   GetDescriptor(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   GetConfiguration(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   SetConfiguration(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;

    // Required for USB 3.0
    void   SetSel(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   SetIsocDelay(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;

    // For Charging Device
    void   GetReportDescriptor(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;
    void   SetIdle(UsbCtrlRequest *pCtrl) NN_NOEXCEPT;

    // Runtime
    void   EnableDisableEndpoints(bool enable, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;
    void   DisableAllEndpoints() NN_NOEXCEPT;

    void AttachDevice() NN_NOEXCEPT;
    void DetachDevice() NN_NOEXCEPT;

    DsProtocolEndPoint &GetEndPointFromAddress(uint8_t epAddr) NN_NOEXCEPT;
    Result Enqueue(uint32_t *urbId, uint8_t epAddr, nn::dd::ProcessHandle processHandle, uint64_t procVa, uint32_t bytes) NN_NOEXCEPT;
    Result EnqueueAndBlock(uint32_t *urbId, uint8_t epAddr, nn::dd::ProcessHandle processHandle, uint64_t procVa, uint32_t bytes) NN_NOEXCEPT;
    void SetControllerAllEndPoints() NN_NOEXCEPT;

    void   GetStatusFromHost() NN_NOEXCEPT;
    void   SetStatusToHost() NN_NOEXCEPT;
};


} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
