﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_DsDescriptor.h
 *
 * @brief   USB Device Stack Descriptor Container
 */

#include "usb_DsCommon.h"
//#include "usb_DsDefaultInterface.h"
#include "usb_DsStringTable.h"
#include "usb_DsInterfaceData.h"

namespace nn {
namespace usb {
namespace ds {

const int BinaryObjectStoreSize = 32;

class DsProtocol;

class DsDescriptor
{
public:
    DsDescriptor() NN_NOEXCEPT : m_Mutex(false) { }
    ~DsDescriptor() NN_NOEXCEPT { }

    Result Initialize(DsProtocol *pProtocol) NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result ClearDeviceData() NN_NOEXCEPT;
    Result AddUsbStringDescriptor(UsbStringDescriptor *pUsbStringDescriptor, uint8_t *pIndex) NN_NOEXCEPT;
    Result DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT;
    Result SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;
    Result SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT;
    Result AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pData, int bytes) NN_NOEXCEPT;
    Result GetConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pDest, int *pSize) NN_NOEXCEPT;
    Result SetReportData(uint8_t bInterfaceNumber, uint8_t *pData, int bytes) NN_NOEXCEPT;

    Result AddInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT;
    Result DelInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT;

    Result AddEndpoint(uint8_t epAddress, uint8_t ifNum) NN_NOEXCEPT;
    Result DelEndpoint(uint8_t epAddress) NN_NOEXCEPT;

    int GetStringDescriptor(void *pDestination, uint8_t index, int length) NN_NOEXCEPT;
    int GetDeviceDescriptor(void *pDestination, int length, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;
    int GetConfigurationDescriptor(void *pDestination, uint8_t index, int length) NN_NOEXCEPT;
    int GetBinaryObjectStore(void *pDestination, int length) NN_NOEXCEPT;
    int GetReportData(uint8_t bInterfaceNumber, uint8_t *pDest, int length) NN_NOEXCEPT;

    void UpdateConfigurationDescriptor(UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;

    int  GetEndpointByIndex(void *pDestination, int index) NN_NOEXCEPT;

    uint8_t GetParentInterface(uint8_t epAddress) NN_NOEXCEPT;
    uint8_t GetParentInterfaceByIndex(int epIndex) NN_NOEXCEPT;

private:

    // total number of entries in string table
    static const int StringTableEntries = 16;

    nn::os::Mutex           m_Mutex;

    DsProtocol             *m_pProtocol;

    // storage for the descriptors
    UsbDeviceDescriptor     m_DeviceDescriptorLowFullSpeed;
    UsbDeviceDescriptor     m_DeviceDescriptorHighSpeed;
    UsbDeviceDescriptor     m_DeviceDescriptorSuperSpeed;
    UsbConfigDescriptor     m_ConfigurationDescriptor;
    bool                    m_InterfaceIsOccupied[DsLimitMaxInterfacesPerConfigurationCount];
    UsbEndpointDescriptor   m_EndpointDescriptor[UsbLimitMaxEndpointsCount];
    uint8_t                 m_BinaryObjectStore[BinaryObjectStoreSize];
    int                     m_BinaryObjectStoreSize;

    DsStringTable           m_StringTable;
    DsInterfaceData         m_InterfaceData;

//    UsbHidDescriptor        m_HidDescriptor;

    // endpoint-interface relationship
    uint32_t                m_EpBitmap[DsLimitMaxInterfacesPerConfigurationCount];
    uint32_t                m_EpParent[UsbLimitMaxEndpointsCount];

    // storage used to hold configuration descriptor and all associated
    // interface and endpoint descriptors
    //
    // TODO: confirm the below:
    // This buffer should be able to hold
    //   - 1 Configuration descriptor (9 bytes)
    //   - 8 Interface descriptors (9 * 8  bytes)
    //   - 32 Endpoint descriptors (9 * 32 bytes)
    //   - 32 SuperSpeed Endpoint Companion descriptors (6 * 32 bytes)
    //   - 32 SuperSpeedPlus Isochronous Endpoint Companion descriptors (8 * 32 bytes)
    // which is 817 bytes in total
    uint8_t                 m_Buffer[1024];
    uint8_t                 m_Length;
};

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
