﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*
 * @file    usb_DsController30-soc.tegra.h
 * @brief   TX1 XUSB Device Controller
 */

#include "usb_DsCommon.h"
#include "usb_DsController.h"

#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MemoryFence.h>

namespace nn {
namespace usb {

namespace detail
{
    class UsbComplexTegra21x;
}

namespace ds
{

/**
 * @brief Maximum size of the event ring
 */
const int32_t  UsbXusbDevMaxRingSize = 4096;

/**
 * @brief Maximum Number of Endpoint Context
 */
const int32_t  UsbXusbDevNumEndpointContext     = 32;

/**
 * @brief Maximum amount of transfer rings
 */
const int32_t  UsbXusbDevControlTransferRingSize= 8;
const int32_t  UsbXusbDevBulkTransferRingSize   = 64;
const int32_t  UsbXusbDevIntTransferRingSize    = 8;

/**
 * @brief Maximum size of the Endpoint Ring Segment Type
 */
const int32_t  UsbXusbDevErstSize               = 2;

/**
 * @brief Macros for FE register
 */
const uint32_t UsbXusbDevFeHsFsPiSelect         = 2;
const uint32_t UsbXusbDevFeSsPiSelect           = 1;
const uint32_t UsbXusbDevFeRestorePortReg       = 0;

const uint64_t UsbXusbDevBase                   = 0x00000000700d0000;

const uint8_t WaitForSetup                      = 0;
const uint8_t SetupPktProcessInProgress         = 1;
const uint8_t DataStageTransferToHost           = 2; // IN
const uint8_t DataStageReceivedFromHost         = 3; // OUT
const uint8_t StatusStageTransferToHost         = 4;
const uint8_t StatusStageReceivedFromHost       = 5;

const uint32_t EpStoppedRegTimeoutInuSecs        = 100;
const uint32_t StateChgRegTimeoutInuSecs         = 100;
const uint32_t ReloadRegTimeoutInuSecs           = 100;

const uint32_t ControlPacketQueueDepth           = 5;

const uint32_t PortScPsMask                      = 0xf;
const uint32_t PortScPsShift                     = 10;

const uint32_t Pls_U0                           = (0x0 << 5); // 0
const uint32_t Pls_U3                           = (0x3 << 5); // 0x60
const uint32_t Pls_Disabled                     = (0x4 << 5); // 0x80
const uint32_t Pls_RxDetect                     = (0x5 << 5); // 0xa0
const uint32_t Pls_Inactive                     = (0x6 << 5); // 0xc0
const uint32_t Pls_Polling                      = (0x7 << 5); // 0xe0
const uint32_t Pls_Resume                       = (0xf << 5); // 0x1e0

// @IMPLEMENT Add a MACRO for this
const uint32_t PortPmRwe                        = (1 << 3);
const uint32_t PortPmU1E                        = (1 << 28);
const uint32_t PortPmU2E                        = (1 << 29);
const uint32_t PortPmFrwe                       = (1 << 30);

const uint32_t TrbMaxBufferSize                 = 64 * 1024;//65536;

const uint32_t PrimeNotRcvdWar                  = 0;
const unsigned int  MaxStreamRejectedRetryCount = 20;

const uint8_t  UsbRecipMask                     = 0x1f;
const uint16_t UsbDeviceOtgStatusSelector       = 0xF000;

const uint8_t UsbDevStatU1Enabled               = 2; // ch9 // transition into U1 state
const uint8_t UsbDevStatU2Enabled               = 3; // ch9 // transition into U2 state

// Interface status, Figure 9-5 USB 3.0 spec
const uint8_t UsbIntrfStatFuncRwCap             = 1;
const uint8_t UsbIntrfStatFuncRw                = 2;

const uint8_t UsbDirAddrCtrl                    = 0x80;

// XHCI spec 4.5.1 Device Context Index
// The DCI of the Slot Context is 0.
// For Device Context Indices 1-31, the following rules apply:
// 1) For Isoch, Interrupt, or Bulk type endpoints the DCI is calculated from the Endpoint Number and
//    Direction with the following formula;
//    DCI = (Endpoint Number * 2) + Direction, where Direction = ‘0’ for OUT endpoints and ‘1’ for IN endpoints.
// 2) For Control type endpoints:
//    DCI = (Endpoint Number * 2) + 1.

const uint32_t UsbEndpointControlIndex           = 0x00; // Control EP index is always 0
const uint32_t UsbEndpointBulkOutIndex           = 0x02; // even numbers
const uint32_t UsbEndpointBulkInIndex            = 0x03; // odd numbers

//----------------------------------------------------------------------------
//  Helper Macros
//----------------------------------------------------------------------------
#define NN_USB_DEFINE_SET_FIELD_VALUE(field, _struct, value)                      \
                              (_struct = (((_struct) & ~(field ## _Mask)) |       \
                              (((value) << field ## _Shift) & (field ## _Mask))))

#define NN_USB_DEFINE_GET_FIELD_VALUE(field, value) \
                              (((value) & (field ## _Mask)) >> (field ## _Shift))

#define NN_USB_DEFINE_SET_BIT(BitName) (1 << BitName)

#define NN_USB_DEFINE_DB_TARGET(x) (((x) & 0xff) << 8)

#define NN_USB_DEFINE_DB_STREAMID(x) (((x) & 0xffff) << 16)

#define NN_USB_DEFINE_DEV_ADDR(x) (((x) & 0x7f) << 24)

#define NN_USB_DEFINE_RT_IMOD_IMODI(x) ((x) & 0xffff)
#define NN_USB_DEFINE_RT_IMOD_IMODC(x) (((x) & 0xffff) << 16)

#define NN_USB_PORTSC_PLS(x)    (((x) & 0xf) << 5)


// endpoint context
// 7.6.9 Data Structures
struct EpContext
{
    volatile uint32_t    epDw0;
    volatile uint32_t    epDw1;
    volatile uint32_t    epDw2;
    volatile uint32_t    epDw3;
    volatile uint32_t    epDw4;
    volatile uint32_t    epDw5;
    volatile uint32_t    epDw6;
    volatile uint32_t    epDw7;
    volatile uint32_t    epDw8;
    volatile uint32_t    epDw9;
    volatile uint32_t    epDw10;
    volatile uint32_t    epDw11;
    volatile uint32_t    epDw12;
    volatile uint32_t    epDw13;
    volatile uint32_t    epDw14;
    volatile uint32_t    epDw15;
} __attribute__((__packed__));

#define BUILD_EP_CONTEXT_RW(name, member, shift, mask)                              \
static inline uint32_t EpContextRead##name(struct EpContext *pEpConetxt)            \
{                                                                                   \
    return (pEpConetxt->member >> shift) & mask;                                    \
};                                                                                  \
static inline void EpContextWrite##name(struct EpContext *pEpConetxt, uint32_t val) \
{                                                                                   \
    uint32_t tmp = pEpConetxt->member & ~(mask << shift);                           \
    tmp |= (val & mask) << shift;                                                   \
    pEpConetxt->member = tmp;                                                       \
};                                                                                  \

BUILD_EP_CONTEXT_RW(State, epDw0, 0, 0x7)
BUILD_EP_CONTEXT_RW(Mult, epDw0, 8, 0x3)
BUILD_EP_CONTEXT_RW(MaxPstreams, epDw0, 10, 0x1f)
BUILD_EP_CONTEXT_RW(Lsa, epDw0, 15, 0x1)
BUILD_EP_CONTEXT_RW(Interval, epDw0, 16, 0xff)
BUILD_EP_CONTEXT_RW(Cerr, epDw1, 1, 0x3)
BUILD_EP_CONTEXT_RW(Type, epDw1, 3, 0x7)
BUILD_EP_CONTEXT_RW(Hid, epDw1, 7, 0x1)
BUILD_EP_CONTEXT_RW(MaxBurstSize, epDw1, 8, 0xff)
BUILD_EP_CONTEXT_RW(MaxPacketSize, epDw1, 16, 0xffff)
BUILD_EP_CONTEXT_RW(Dcs, epDw2, 0, 0x1)
BUILD_EP_CONTEXT_RW(DeqLo, epDw2, 0, 0xfffffff0)
BUILD_EP_CONTEXT_RW(DeqHi, epDw3, 0, 0xffffffff)
BUILD_EP_CONTEXT_RW(AvgTrbLen, epDw4, 0, 0xffff)
BUILD_EP_CONTEXT_RW(MaxEsitPayload, epDw4, 16, 0xffff)
BUILD_EP_CONTEXT_RW(Edtla, epDw5, 0, 0xffffff)
BUILD_EP_CONTEXT_RW(SeqNum, epDw5, 24, 0xff)
BUILD_EP_CONTEXT_RW(PartialTd, epDw5, 25, 0x1)
BUILD_EP_CONTEXT_RW(CerrCnt, epDw6, 18, 0x3)
BUILD_EP_CONTEXT_RW(DataOffset, epDw7, 0, 0x1ffff)
BUILD_EP_CONTEXT_RW(NumTrbs, epDw7, 22, 0x1f)
BUILD_EP_CONTEXT_RW(DevAddr, epDw11, 0, 0x7f)

static inline void EpContextWriteDeqPtr(struct EpContext *pEpConetxt, uint64_t addr)
{
    EpContextWriteDeqLo(pEpConetxt, static_cast<uint32_t>((addr & 0xffffffff)));
    EpContextWriteDeqHi(pEpConetxt, static_cast<uint32_t>(addr >> 32));
}


//////////////////////////////////////////////////////////////////////////////
struct Trb
{
    volatile uint32_t    dataBufPtrLo;
    volatile uint32_t    dataBufPtrHi;
    volatile uint32_t    status;
    volatile uint32_t    control;
} __attribute__((__packed__));


#define BUILD_TRB_RW(name, member, shift, mask)                     \
static inline uint32_t TrbRead##name(struct Trb *pTrb)              \
{                                                                   \
    return (pTrb->member >> shift) & mask;                          \
}                                                                   \
static inline void TrbWrite##name(struct Trb *pTrb, uint32_t val)   \
{                                                                   \
    uint32_t tmp;                                                   \
                                                                    \
    tmp = pTrb->member & ~(mask << shift);                          \
    tmp |= (val & mask) << shift;                                   \
    pTrb->member = tmp;                                             \
}


BUILD_TRB_RW(DataLo, dataBufPtrLo, 0, 0xffffffff)
BUILD_TRB_RW(DataHi, dataBufPtrHi, 0, 0xffffffff)
BUILD_TRB_RW(SeqNum, status, 0, 0xffff)
BUILD_TRB_RW(TransferLen, status, 0, 0xffffff)
BUILD_TRB_RW(TdSize, status, 17, 0x1f)
BUILD_TRB_RW(CmplCode, status, 24, 0xff)
BUILD_TRB_RW(Cycle, control, 0, 0x1)
BUILD_TRB_RW(ToggleCycle, control, 1, 0x1)
BUILD_TRB_RW(Isp, control, 2, 0x1)
BUILD_TRB_RW(Chain, control, 4, 0x1)
BUILD_TRB_RW(Ioc, control, 5, 0x1)
BUILD_TRB_RW(Type, control, 10, 0x3f)
BUILD_TRB_RW(StreamId, control, 16, 0xffff)
BUILD_TRB_RW(EndpointId, control, 16, 0x1f)
BUILD_TRB_RW(Tlbpc, control, 16, 0xf)
BUILD_TRB_RW(DataStageDir, control, 16, 0x1)
BUILD_TRB_RW(FrameId, control, 20, 0x7ff)
BUILD_TRB_RW(Sia, control, 31, 0x1)

static inline uint64_t TrbReadDataPtr(struct Trb *pTrb)
{
    return ((uint64_t)TrbReadDataHi(pTrb) << 32) | TrbReadDataLo(pTrb);
}

static inline void TrbWriteDataPtr(struct Trb *pTrb, uint64_t addr)
{
    TrbWriteDataLo(pTrb, static_cast<uint32_t>(addr & 0xffffffff));
    TrbWriteDataHi(pTrb, static_cast<uint32_t>(addr >> 32));
}


enum EventTrbDw
{
    // EventTrbDw2
    EventTrbDw2_TranLen_Mask              = 0x00FFFFFF,
    EventTrbDw2_TranLen_Shift             = 0,
    EventTrbDw2_SeqNum_Mask               = 0x0000FFFF,
    EventTrbDw2_SeqNum_Shift              = 0,
    EventTrbDw2_ComplCode_Mask            = 0xFF000000,
    EventTrbDw2_ComplCode_Shift           = 24,
    // EventTrbDw3
    EventTrbDw3_CycleBit_Mask             = 0x00000001,
    EventTrbDw3_CycleBit_Shift            = 0,
    EventTrbDw3_Type_Mask                 = 0x0000FC00,
    EventTrbDw3_Type_Shift                = 10,
    EventTrbDw3_EndpointId_Mask           = 0x001F0000,
    EventTrbDw3_EndpointId_Shift          = 16
};

enum EndpointState
{
    EndpointState_Disabled = 0,
    EndpointState_Running,
    EndpointState_Halted,
    EndpointState_Stopped,
    EndpointState_Error
};

enum EndpointType
{
    EndpointType_Invalid = 0,
    EndpointType_IsochOut,
    EndpointType_BulkOut,
    EndpointType_IntrOut,
    EndpointType_Cntrl,
    EndpointType_IsochIn,
    EndpointType_BulkIn,
    EndpointType_IntrIn
};

// From Table 127 of the XHCI spec
enum TrbCmplCode
{
    TrbCmplCode_Invalid = 0,
    TrbCmplCode_Success,
    TrbCmplCode_DataBufferErr,
    TrbCmplCode_BabbleDetectedErr,
    TrbCmplCode_UsbTransErr,
    TrbCmplCode_TrbErr,  /*5*/
    TrbCmplCode_TrbStall,
    TrbCmplCode_InvalidStreamTypeErr = 10,
    TrbCmplCode_Short_Pkt = 13,
    TrbCmplCode_RingUnderrun,
    TrbCmplCode_RingOverrun, /*15*/
    TrbCmplCode_EventRingFullErr = 21,
    TrbCmplCode_Stopped = 26,
    TrbCmplCode_IsochBufferOverrun = 31,
    /*192-224 vendor defined error*/
    TrbCmplCode_StreamNumpError = 219,
    TrbCmplCode_PrimePipeReceived,
    TrbCmplCode_HostRejected,
    TrbCmplCode_CtrlDirErr,
    TrbCmplCode_CtrlSeqnumErr,
};

// From Table 128 of the XHCI spec
enum TrbType
{
    TrbType_Rsvd = 0,
    TrbType_Transfer_Normal,
    TrbType_Transfer_Setup_Stage,
    TrbType_Transfer_Data_Stage,
    TrbType_Transfer_Status_Stage,
    TrbType_Transfer_Data_Isoch,   // 5
    TrbType_Link,
    TrbType_Evt_Transfer         = 32,
    TrbType_Evt_PortStatusChange = 34,
    TrbType_Transfer_Stream      = 48,
    TrbType_Evt_SetupPkt         = 63
};

class DsControllerTegra30 : public DsController
{
public:
    DsControllerTegra30() NN_NOEXCEPT
        : DsController(UsbCapability_HighSpeed)
        , m_DataTransferComplete(false)
        , m_pComplex(NULL)
        , m_StreamRejected(0)
        , m_DevAddr(0)
        , m_Ccs(0)
        , m_SetupStatus(WaitForSetup)
        , m_DeviceState(UsbState_Default)
        , m_ResumeState(0)
        , m_CtrlSeqNum(0)
        , m_ControllerRefCount(0)
        , m_WaitForSecPrc(false)
        , m_WaitForSecPrcRefCount(0)
        , m_WaitForCsc(false)
        , m_WaitForCscRefCount(0)
    {
        m_XusbDevIrqHandler = [&] {
            nn::os::ClearInterruptEvent(&m_XusbDevIrqEvent);
            protocol.LockEndPointMutex();
            XusbDeviceIrqHandler();
            protocol.UnlockEndPointMutex();
        };

        m_PortResetWarHandler = [&] {
            nn::os::ClearTimerEvent(&m_PortResetWarEvent);
            PortResetWarWork();
        };

        m_PlcResetWarHandler = [&] {
            nn::os::ClearTimerEvent(&m_PlcResetWarEvent);
            PlcResetWarWork();
        };
    }

    virtual ~DsControllerTegra30() NN_NOEXCEPT
    {
        // do nothing
    }

    virtual Result Initialize(detail::UsbComplex *pComplex,
                              detail::UsbController::Config* pConfig) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Finalize() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Enable() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Disable() NN_NOEXCEPT NN_OVERRIDE;

    // device functions
    virtual Result Attach() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Detach() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Ctrl(UsbCtrlRequest *pCtrl) NN_NOEXCEPT NN_OVERRIDE;
    virtual UsbDeviceSpeed GetConnectSpeed() NN_NOEXCEPT NN_OVERRIDE;

    // endpoint functions
    virtual Result EnableEndpoint(UsbEndpointDescriptor *pUsbEndpointDescriptor, UsbEndpointCompanionDescriptor *pUsbEndpointCompanionDescriptor) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DisableEndpoint(uint8_t bEndpointAddress) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result FlushEndpoint(uint8_t bEndpointAddress) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result StallEndpoint(uint8_t bEndpointAddress) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetZlt(uint8_t bEndpointAddress, bool zlt) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result PostBufferAsync(uint8_t   endpointAddress,
                                   nn::dd::ProcessHandle processHandle,
                                   uint64_t  procVa,
                                   uint32_t  bytes) NN_NOEXCEPT NN_OVERRIDE;

private:
    //////////////////////////////////////////////////////////////////////////
    // XUSB Device Controller Standard Constants
    //////////////////////////////////////////////////////////////////////////
    /**
    * @brief XUSB Device controller Registers
    *
    */
    static const size_t DeviceControllerBaseOffset = 0x0;

    // Xusb Device Controller Id Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlIdBase,0x0);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlIdBase,ErstMax,16,5);// Event Ring Segment Table Max i.e. number of ERST entries

    // Xusb Device Controller Doorbell Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDoorbellBase, 0x04);

    // Xusb Device Controller Event Ring Segment Table Size Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtSgmtRingSz,0x08);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtSgmtRingSz,Erst0sz,      0, 16);// Event Ring Segment Table 0
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtSgmtRingSz,Erst1sz,     16, 16);// Event Ring Segment Table 1

    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtSgmtRingSzRsvd,0x0c);

    // Xusb Device Controller Event Ring Segment 0 Base Address Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingSgmt0AddrLo,0x10);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt0AddrLo,Erst0BalLOAddrRsvd,  0,  4);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt0AddrLo,Erst0BalLOAddrLO,    4, 28);// Event Ring Segment Table 0 Low Address
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingSgmt0AddrHi,0x14);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt0AddrHi,Erst0BalHIAddrHI,    0, 32);// Event Ring Segment Table 0 High Address


    // Xusb Device Controller Event Ring Segment 1 Base Address Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingSgmt1AddrLo,0x18);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt1AddrLo,Erst1BalLOAddrRsvd,  0,  4);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt1AddrLo,Erst1BalLOAddrLO,    4, 28);// Event Ring Segment Table 1 Low Address
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingSgmt1AddrHi,0x1c);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingSgmt1AddrHi,Erst1BalHIAddrHI,    0, 32);// Event Ring Segment Table 1 High Address

    // Xusb Device Controller Event Ring Dequeue Pointer Register
    // EvtRingDpLo Reflects the current value of ERDP (Event Ring Dequeue Pointer),
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingDpLo,0x20);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingDpLo,Ehb,          3, 1 );// RW1C: Event Handler Busy Flag
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingDpLo,ErdpLOAddrLO, 4, 28);// Event Ring Dequeue Pointer Low Address
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingDpHi,0x24);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingDpHi,ErdpHIAddrHI, 0, 32);// Event Ring Dequeue Pointer High Address

    // Xusb Device Controller Event Ring Enqueue Pointer Register
    // EvtRingEqLo reflects the current value of EREP (Event Ring Enqueue Pointer);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingEqLo,0x28);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingEqLo,EreploEcs,    0, 1); // Event Ring Enqueue Pointer empty capability set (ECS)
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingEqLo,ErepLOAddrLO, 4, 28);// Event Ring Enqueue Pointer Low Address
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEvtRingEqHi,0x2c);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEvtRingEqHi,ErepHIAddrHI, 0, 32);// Event Ring Enqueue Pointer High Address

    // Xusb Device Controller Control Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevControl,  0x30);
    NN_USB_DEFINE_FIELD32(XusbDevControl, Run,    0,  1); // Start/stop DC execution - do not write unless DC is halted?
    NN_USB_DEFINE_FIELD32(XusbDevControl, Lse,    1,  1); // Generate Event on PORTSC change
    NN_USB_DEFINE_FIELD32(XusbDevControl, Ie,     4,  1); // Interrupt enable: Enable legacy/smi interrupt for pending events
    NN_USB_DEFINE_FIELD32(XusbDevControl, SmiEvt, 5,  1); // System management interrupts Event
    NN_USB_DEFINE_FIELD32(XusbDevControl, SmiDse, 6,  1); // Enable SMI interrupt for device system errors
    NN_USB_DEFINE_FIELD32(XusbDevControl, Ewe,    7,  1); // Enable Wrap Event - '1' means xDC generates an event when MFINDEX wraps
    NN_USB_DEFINE_FIELD32(XusbDevControl, DevAddr,24, 6); // Address assigned to the device DUT (Device Under Test)
    NN_USB_DEFINE_FIELD32(XusbDevControl, Enable, 31, 1); // Device control Enable used to enable device mode operation

    // Xusb Device Controller Status Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlStatus,0x34);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlStatus, Rc,  0,  1); // RW1C Run Change - set when run bit transitions from 1 to 0
    NN_USB_DEFINE_FIELD32(XusbDevCtrlStatus, Ip,  4,  1); // RW1C Interrupt Pending-set by HW when an event is added to the event ring
    NN_USB_DEFINE_FIELD32(XusbDevCtrlStatus, Dse, 5,  1); // RW1C Device System Error bit set when error is received on FPCI wrapper


    // Xusb Device Controller Runtime IMOD (Interrupt Moderation)
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlRtImod,0x38);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlRtImod,IMODC,        0, 16); // Interrupt Moderation Interval
    NN_USB_DEFINE_FIELD32(XusbDevCtrlRtImod,IMODI,       16, 16); // Interrupt Moderation Counter

    // Xusb Device Controller Port Status Change
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlPortSc,0x3C);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Ccs,  0, 1); // Current Connect Status (CCS) – RW
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Ped,  1, 1); // Port Enabled/Disabled (PED) – RW
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Pr,   4, 1); // Port Reset (PR) – R
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Pls,  5, 4); // Port Link State (PLS)
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Ps,  10, 4); // Port Speed (PS) R
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Lws, 16, 1); // Port Link State Write Strobe (LWS)
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Csc, 17, 1); // Connect Status Change (CSC) – RW1CS
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Wrc, 19, 1); // Warm Port Reset Change (WRC) – RW1CS
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Prc, 21, 1); // Port Reset Change (PRC) – RW
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Plc, 22, 1); // Port Link State Change (PLC) – RW
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Cec, 23, 1); // Port Config Error Change (CEC) – RW1CS
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortSc, Wpr, 30, 1); // Warm Port Reset (WPR) – R

    // Xusb Device Controller xtended Capabilities Port
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEcpLo,0x40); // Extended Capabilities Port Low Address
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEcpLo,EcpLOAddrLO, 6, 26);

    // Xusb Device Controller xtended Capabilities Port
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEcpHi,0x44);// Extended Capabilities Port High Address
    NN_USB_DEFINE_FIELD32(XusbDevCtrlEcpHi,EcpHIAddrHI, 0, 32);

    // Xusb Device Controller MicroFrame Index
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlMfIndex,     0x48);

    // Xusb Device Controller Port PM Status and Control Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlPortPm,      0x4c);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,L1state,    0, 2);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Rwe,        3, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Hird,       4, 4);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,U2timeout,  8, 8);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,U1timeout, 16, 8);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Fla,       24, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Vba,       25, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Woc,       26, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Wod,       27, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,U1e,       28, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,U2e,       29, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,Frwe,      30, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortPm,PngCya,    31, 1);

    // Xusb Device Controller EP HALT register used by SW to program an EP to respond to host request with STALL transaction
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEpHalt,      0x50);

    // Xusb Device Controller EP Pause register used by SW to program the DUT to stop processing the EP
    // and respond host requests with a NRDY/NAK
    // 31:0 R/W
    // 0h: DCI_NO (default)
    // 1h: DCI_YES
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEpPause,     0x54);

    // Xusb Device Controller EP Reload bits are used by SW to direct the Device BI to reload the EP context for that
    // EP and update its internal EP related variables like EP state
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEpReload,    0x58);

    // Xusb Device Controller EP STCHG is set by HW whenever halt/pause/run bit is cleared by SW, to indicate completion
    // of processing of the SW request.
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlEpStchg,     0x5c);

    // Xusb Device Controller Flowcontrol threshold values for HSFS NAK - SW needs to enable this if more than 4 IN or 4
    // OUT EPs are configured
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlFlowCntrl,   0x60);

    // Xusb Device Controller Device notification registers, for SW to direct HW to send a device notification TP data
    // should be written and updated before writing 1 to the trigger.
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevNotifLo,  0x64);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevNotifHi,  0x68);

    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlPortHalt,0x6c);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortHalt,HltLtssm,          0, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortHalt,StChgReq,         20, 1);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlPortHalt,StChgIntrEnabled, 24, 1);

    // Xusb Device Controller Testmode control register, for enabling DUT to enter testmode and send specific patterns
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevTestMode, 0x70);

    // Xusb Device Controller Per EP bit to indicate whether EP is loaded/active in BI
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevEpThreadActive,0x74);

    // Xusb Device Controller EP_STOPPED register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevEpStopped,0x78);

    // Xusb Device Controller Soft Reset Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlDevSoftRst,  0x84);

    // Xusb Device Controller HS FSPI Registers
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt0,  0x100);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt1,  0x104);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt2,  0x108);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt3,  0x10c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt4,  0x110);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt5,  0x114);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt6,  0x118);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt7,  0x11c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt8,  0x120);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt9,  0x124);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt10, 0x128);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt11, 0x12c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt12, 0x130);
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCnt13, 0x134);

    // Xusb Device Controller HS FSPI Control Registers
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlHsFspiCtrl,  0x138);

    /**
    * @brief SSPX undefined in TRM
    *
    */
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlSspxCorePadctl4,        0x750);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlSspxCorePadctl4,           RxDatVldTimeoutU3,0,20); // Mask 0xfffff
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlSspxCoreCnt0,           0x610);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlSspxCoreCnt0,PingTbrst,    0,7);         // Mask 0xff
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlSspxCoreCnt30,          0x688);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlSspxCoreCnt30,LmpitpTimer, 0,20);        // Mask 0xfffff
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlSspxCoreCnt32,          0x690);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlSspxCoreCnt32,PollTbrstMax,0,7);         // Mask 0xff
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlSspiTransfer,           0x858);

    /**
    * @brief Config Dev Fe undefined in TRM
    *
    */
    NN_USB_DEFINE_REG_OFFSET(XusbDevCtrlCfgDevFe,          0x85c);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlCfgDevFe,RegSelect,    0,  2);
    NN_USB_DEFINE_FIELD32(XusbDevCtrlCfgDevFe,SsRetry,     29,  1);

    /**
    * @brief XUSB Device PCI Config Registers (fpci)
    *
    */

    // Xusb Device Controller PCI config registers
    static const size_t FpciConfigRegistersOffset = 0x8000;
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg0,    0x00);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg0,       VendorId,                0, 16);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg0,       DeviceId,               16, 16);
    static const uint16_t XusbDevFpciCfg0Tegra21xVenId = 0x10de;
    static const uint16_t XusbDevFpciCfg0Tegra21xDevId = 0x0fad;

    // Xusb Device Controller FPCI Configuration Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg1,     0x04);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg1,        IoSpaceEnabled,        0,  1);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg1,        MemorySpaceEnabled,    1,  1);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg1,        BusMasterEnabled,      2,  1);

    // Xusb Device Controller PCI Revision ID and Class Code Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg2,     0x08);

    // Xusb Device Controller PCI Configuration Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg3,     0x0c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg4,     0x10);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg4,        BaseAddress,          15,  16);
    //NN_USB_DEFINE_FIELD32(XusbDevFpciCfg4,      BaseAddress,          16,  16);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg5,     0x14);
    NN_USB_DEFINE_FIELD32(XusbDevFpciCfg5,        BaseAddress,           0,  32);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg6,     0x18);
    //NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg11,  0x28);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg12,    0x30);

    // Xusb Device Controller PCI Capability Pointer Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg13,    0x34);

    // Xusb Device Controller PCI Configuration Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg14,    0x38);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg15,    0x3c);

    // Xusb Device Controller Backdoor register write for updating subsystem ID/vendor ID.
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg16,    0x40);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg17,    0x44);
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciCfg18,    0x48);

    // XUSB Device PCI Config Registers MSI Message Control and Capability

    // MSI_CTRL is the main control register for MSI support.
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiCtrl,  0xc0);

    // MSI Message Address Register 84h
    // MSI_ADDR1 specifies the lower 32 bits to which an MSI memory write transaction should occur
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiAddr1, 0xc4);

    // MSI Message Address Register 88h
    // MSI_ADDR1 specifies the higher 32 bits to which an MSI memory write transaction should occur
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiAddr2, 0xc8);

    // MSI Message Data Register 8Ch
    // The MSI_DATA register contains the system-specified message.
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiData, 0xcc);

    // MSI Message Data Register 60h
    // The MSI_MASK register enables software to mask or defer message generation on a per-vector basis.
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiMask, 0xd0);

    // MSI Pending Bits Register 64h
    // The MSI_PEND register shows which MSI vectors have pending interrupts
    NN_USB_DEFINE_REG_OFFSET(XusbDevFpciMsiPend, 0xd4);


    /**
    * @brief XUSB Device Registers (ipfs)
    *
    * IFPS is the bus interface block to the rest of the system.
    * The XUSB Device controller is connected to the APB bus and it is wrapped with protocol conversion logic
    * because it was also used in other 'raptor' chips.
    * IPFS converts transactions between both the slave APB interface and the master MC(Mem Controller) protocol
    * mapping them to a bus protocol referred to as ‘FPCI’ which is the native interface of the XUSB Device Controller.
    * To be able to access the core configuration registers (FPCI), some wrapper registers(IPFS) should be
    * programmed accordingly.
    * IPFS is a flexible block designed to support multiple IPs which need various BAR addresses and sizes
    */
    static const size_t IpfsDeviceRegistersOffset = 0x9000;

    // AXI is the AMBA Advanced eXtensible Interface, a more advanced bus than AHB defined as part of AMBA 3.
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar0Sz,    0x00);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsAxiBar0Sz,       Bar0Size,            0,   20);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar1Sz,    0x04);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar2Sz,    0x08);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar3Sz,    0x0c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar0Start, 0x40);
    // The start of the FPCI address space mapped into the BARi range of PCI memory space.
    NN_USB_DEFINE_FIELD32(XusbDevIpfsAxiBar0Start,    AxiBar0Start,       12,   19);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar1Start, 0x44);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar2Start, 0x48);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBar3Start, 0x4c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciBar0,     0x80);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsFpciBar0,        FpciBar0AccessType, 0,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsFpciBar0,        FpciBar0Start,      4,   27);

    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciBar1,     0x84);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciBar2,     0x88);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciBar3,     0x8c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiMsiBarSz,  0xc0);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsAxiBarSt0,    0xc4);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiFpciBarSt0,0xc8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec00,     0x100);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec10,     0x104);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec20,     0x108);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec30,     0x10c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec40,     0x110);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec50,     0x114);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec60,     0x118);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiVec70,     0x11c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec00,   0x140);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec10,   0x144);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec20,   0x148);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec30,   0x14c);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec40,   0x150);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec50,   0x154);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec60,   0x158);
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMsiEnVec70,   0x15c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsConfig,       0x180);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          EnFpci,              0,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          DfpciPassPw,         1,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          DfpciRsPassPw,       2,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          DfpciPwPassNpw,      3,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          UfpciPwPassNpw,      4,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          UfpciPassPw,         5,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          UfpciPwPassPw,       6,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          UfpciMsiaw,          7,   2);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          MsiVecEmpty,         9,   1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          TargetWriteIdle,     10,  1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          TargetReadIdle,      11,  3);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          WrIntrlv3731Cya,     14,  1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          WdataLeadCya,        15,  2);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          InitiatorWriteIdle,  17,  1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          InitiatorReadIdle,   18,  1);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          PwNoDevselErrCya,    19,  12);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsConfig,          ClkenOverride,       31,  1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciErrorMasks,0x184);
    // MPCORE is a Multi-processor CPU core, a generic term for a CPU capable of operating as part of an SMP group.
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIntrMasks,    0x188);
    NN_USB_DEFINE_FIELD32(XusbDevIpfsIntrMasks,       Int,               0,  8);// Interrupt to MPCORE gated by mask.
    NN_USB_DEFINE_FIELD32(XusbDevIpfsIntrMasks,       Msi,               8,  8);// MSI to MPCORE gated by mask
    NN_USB_DEFINE_FIELD32(XusbDevIpfsIntrMasks,       IpInt,            16,  1);// IP(SATA/AZA) interrupt to MPCORE gated by mask

    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIntrCode,          0x18c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIntrSignature,     0x190);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsUpperFpciAddr,     0x194);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIfpsIntrEnable,    0x198);

    // Upstream FPCI Configuration
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIntrUfpciConfig,   0x19c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsCfgRevId,          0x1a0);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsFpciTimeout,       0x1a4);

    // Top of Memory Limit
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsTom,               0x1a8);

    // Initiator ISO PW Response Pending
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIsoPwRespPending,  0x1ac);

    // Initiator Non-ISO PW Response Pending
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsNisoPwRespPending, 0x1b0);

    // IPFS Interrupt Status
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsIntrStatus,        0x1b4);

    // Downstream FPCI Byte Enables
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsDfpciBen,          0x1b8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsClkGateHysteresis, 0x1bc);

    // Memory Client Interface Fifo Control (where applicable) and Clock Gating Control Register
    NN_USB_DEFINE_REG_OFFSET(XusbDevIpfsMccifFifoCtrl,     0x1dc);

    //////////////////////////////////////////////////////////////////////////
    //  XUSB Device Controller Standard Types
    //////////////////////////////////////////////////////////////////////////
    /*
    The event TRBs are generated by the hw for these transactions:
    - IN DMA engine
    - Out DMA engine
    - Control (Setup), Flow control threshold
    - Port status change

    See 6.4.2.1 Transfer Event TRB
    */
    struct EventTrb
    {
        volatile uint32_t    trbPointerLo;
        volatile uint32_t    trbPointerHi;
        volatile uint32_t    eventTrbDw2;
        volatile uint32_t    eventTrbDw3;
    } __attribute__((__packed__));



    struct CtrlPacket
    {
        UsbCtrlRequest UsbCtrlReq;
        uint16_t SeqNum;
    };

    // FIXME - EndPoint becomes class and actually zeros its data or defaults it to usable values
    struct Endpoint
    {
        UsbEndpointDescriptor           usbEndpointDescriptor;
        UsbEndpointCompanionDescriptor  usbEndpointCompanionDescriptor;

        Trb                             *pFirstTransferTrb;
        Trb                             *pEnqPtrTransferTrb;
        Trb                             *pDeqPtrTransferTrb;
        Trb                             *pLinkTrb;

        uint32_t                        numTrbsAva;
        uint32_t                        numTrbsStillNeeded;
        uint32_t                        trbCountForShortTransfer;
        uint32_t                        bytesRemainingForShortTransfer;
        uint64_t                        bytesSent;
        uint64_t                        addressOfTransferDataSent;

        uint8_t                         *pTransferRingBuffer;
        nn::dd::DeviceVirtualAddress    transferRingIoVa;

        // For transfers larger than a ring size
        bool                            bulkTransferComplete;
        bool                            transferRingFull;

        // The Cycle bit field in a TRB identifies the location of the Enqueue Pointer in a Transfer Ring,
        // eliminating the need to define a physical Enqueue Pointer register
        uint8_t                         pcs; // Producer Cycle State (PCS)
        uint8_t                         streamRejectedRetryCount;
        // The Device Context Index (DCI) is used to reference the respective element of the Device Context data structure
        uint8_t                         dci; // Device Context Index (DCI)
    };

    //////////////////////////////////////////////////////////////////////////
    // Private Data
    //////////////////////////////////////////////////////////////////////////
    // Storage for transfer rings. Must have proper alignment and size for sMMU map operation.
    // Consecutive DMA blocks
    NN_USB_DMA_ALIGN uint8_t m_EvtRingSeg0         [NN_USB_ROUNDUP_SIZE(sizeof(EventTrb) * UsbXusbDevMaxRingSize * 2, nn::dd::DeviceAddressSpaceMemoryRegionAlignment)];
    NN_USB_DMA_ALIGN uint8_t m_EpContext           [NN_USB_ROUNDUP_SIZE(sizeof(EpContext) * UsbXusbDevNumEndpointContext, nn::dd::DeviceAddressSpaceMemoryRegionAlignment)];

    detail::RegisterBlock                XusbDeviceBaseRegisters;
    detail::RegisterBlock                XusbDevFpciConfigRegisters;
    detail::RegisterBlock                XusbDeviceIpfsRegisters;

    EventTrb                             *m_pEvtRingSeg0;
    EpContext                            *m_pEpContext;

    EventTrb                             *m_pEvtDqPtr;       // used for parsing the event ring segments
    EventTrb                             *m_pEvtSeg1LastTrb; // used for parsing the event ring segments
    nn::dd::DeviceVirtualAddress          m_pEvtDqIoVaPtr;
    nn::dd::DeviceVirtualAddress          m_pEvtDqIoVaBasePtr;
                                          // 0 slot context-1 control ep - 2 Bulk OUT - 3 Bulk IN, etc
    Endpoint                              m_EndpointArray[UsbXusbDevNumEndpointContext];
    bool                                  m_DataTransferComplete;
    size_t                                m_SizeOfControlTransferRing;
    size_t                                m_SizeOfBulkTransferRing;

    detail::UsbComplexTegra21x           *m_pComplex;
    detail::UsbPlatform                  *m_pPlatform;
    DsControllerTegra30                  *m_pXusbDeviceController;

    // Device Interrupt
    detail::IrqHandlerType                m_XusbDevIrqHandler;
    nn::os::MultiWaitHolderType           m_XusbDevIrqHolder;
    nn::os::InterruptEventType            m_XusbDevIrqEvent;

    static const nn::os::InterruptName Usb3DevInterruptName    = 32 + 44; // 76

    UsbDeviceSpeed                        m_Speed;
    uint32_t                              m_StreamRejected;
    uint16_t                              m_DevAddr;
    uint8_t                               m_Ccs; // Keep Current CS to make sure we don't parse EvtRing0 unnecessarily
    uint8_t                               m_SetupStatus;
    uint8_t                               m_DeviceState;
    uint8_t                               m_ResumeState;
    uint16_t                              m_CtrlSeqNum;

    int                                   m_ControllerRefCount;
    bool                                  m_ConnectedToHost;

    // Workarounds
    detail::IrqHandlerType                m_PortResetWarHandler;
    nn::os::TimerEventType                m_PortResetWarEvent;
    nn::os::MultiWaitHolderType           m_PortResetWarEventHolder;
    bool                                  m_WaitForSecPrc; // wait for Port Reset Change
    volatile int                          m_WaitForSecPrcRefCount;

    detail::IrqHandlerType                m_PlcResetWarHandler;
    nn::os::TimerEventType                m_PlcResetWarEvent;
    nn::os::MultiWaitHolderType           m_PlcResetWarEventHolder;
    bool                                  m_WaitForCsc;// Wait for Port Status Change
    volatile int                          m_WaitForCscRefCount;

    //////////////////////////////////////////////////////////////////////////
    // Internal methods
    //////////////////////////////////////////////////////////////////////////
    void XusbDeviceIrqHandler();
    Result InitializeXusbDevice() NN_NOEXCEPT;
    Result FinalizeXusbDevice() NN_NOEXCEPT;
    Result InitializeDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT;
    Result FinalizeDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT;
    Result MapDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT;
    void UnMapDriverMemory() NN_NOEXCEPT;
    void EnableRxDetect() NN_NOEXCEPT; // DisableRxDetect not required
    void SetInterruptModeration(uint32_t us);

    // For configuring the control EP
    Endpoint *UsbEpStructSetup(UsbEndpointDescriptor *pUsbEndpointDescriptor, UsbEndpointCompanionDescriptor *pUsbEndpointCompanionDescriptor) NN_NOEXCEPT;

    // For processing Event Ring Events
    void HandleTrbEvent(EventTrb *pEvent) NN_NOEXCEPT;

    // For handling Port status change
    void HandleConnectionStatusChange(uint32_t portStatusChange) NN_NOEXCEPT;
    void HandlePortLinkStateChange(uint32_t portStatusChange) NN_NOEXCEPT;
    void HandlePortStatusChange() NN_NOEXCEPT;
    void UpdateEp0MaxPacketSize() NN_NOEXCEPT;
    void EnableDoorBell() NN_NOEXCEPT;
    void SetDeviceSpeed(uint32_t portStatusChange) NN_NOEXCEPT;

    // For polling register status
    void PollEpStopped(const char *_fmt, size_t _val) NN_NOEXCEPT;
    void PollStchg(const char *_fmt, size_t _val) NN_NOEXCEPT;

    // For Halts/resets
    void Reset() NN_NOEXCEPT;

    // For Work Arounds
    void PlcResetWarWork() NN_NOEXCEPT;
    void PortResetWarWork() NN_NOEXCEPT;
    void RetryStreamRejectedWork(Endpoint *pEndpoint) NN_NOEXCEPT;
    void PortpmConfigWar() NN_NOEXCEPT;

    // For Resume
    void ResumeState(bool deviceInit) NN_NOEXCEPT;
    void DoorbellForUnpause(uint32_t pausedBits) NN_NOEXCEPT;

    // For handling control packets
    void CheckSetupPkt(EventTrb *pEvent) NN_NOEXCEPT;
    void HandleSetupPacket(UsbCtrlRequest *pUsbCtrlReq, uint16_t seqNum) NN_NOEXCEPT;

    Result HandleDataPacket(uint8_t epAddr, uint64_t transferData, uint32_t transferLength) NN_NOEXCEPT;
    Result BuildEp0DataIn(Endpoint *pEp0, uint32_t numTrbsNeeded, uint64_t transferData,uint64_t transferLength) NN_NOEXCEPT;

    // TRB completion
    Trb* TranTrbDmaToVirt(Endpoint *pEndpoint, uint64_t address) NN_NOEXCEPT;
    Result HandleDataPacketCmpl(EventTrb *pEvent) NN_NOEXCEPT;
    void UpdateEpDequeuePtr(EventTrb *pEvent, Endpoint *pEndpoint) NN_NOEXCEPT;
    void HandleCmplCodeSuccess(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT;
    bool HandleCmplCodeShortPkt(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT;
    void HandleCmplCodeHostRejected(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT;
    void HandleCmplCodePrimePipeReceived(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT;
    void HandleCmplCodeCtrlSeqnumErr(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT;

    // called directly from 'set'
    // For sending and completing a control request status
    void HandleEp0StatusComplete(Endpoint *pEndpoint) NN_NOEXCEPT;
    void Ep0Status() NN_NOEXCEPT;

    // For Bulk EPs
    Result InitializeEndpoint(Endpoint *pEndpoint) NN_NOEXCEPT;
    void InitializeEpContext(Endpoint *pEndpoint) NN_NOEXCEPT;
    void FinalizeEpContext(Endpoint *pEndpoint) NN_NOEXCEPT;
    Result Ringdoorbell(Endpoint *pEndpoint) NN_NOEXCEPT;
    Result QueueTrbs(Endpoint *pEndpoint, uint32_t transferRingSize,uint32_t numTrbsNeeded, uint64_t transferData, uint64_t BufferLength) NN_NOEXCEPT;
    uint32_t RoomOnRing(uint32_t transferRingSize, Trb *pRing, Trb *pEnqPtr, Trb *dQptr) NN_NOEXCEPT;
    inline int UsbSsMaxStreams(const UsbEndpointCompanionDescriptor *pCompEpDescriptor) NN_NOEXCEPT;
    Trb* CheckEndofSegment(Endpoint *pEndpoint, Trb *pTrb) NN_NOEXCEPT;

    int UsbAddressToEndPointIndex(uint8_t address) NN_NOEXCEPT;

    // Utility
    void ReadPoll32(size_t offset, uint32_t mask, uint32_t value) NN_NOEXCEPT;

    // Endpoint utility
    uint8_t     EndpointAttributes(Endpoint *pEndpoint);
    uint8_t     EndpointType(Endpoint *pEndpoint);
    uint8_t     EndpointDirection(Endpoint *pEndpoint);
    uint16_t    EndpointMaxPacketSize(Endpoint *pEndpoint);
    uint8_t     EndpointInterval(Endpoint *pEndpoint);
    uint8_t     EndpointCompMaxBurst(Endpoint *pEndpoint);
    uint8_t     EndpointCompAttributes(Endpoint *pEndpoint);
    uint16_t    EndpointCompBytesPerInterval(Endpoint *pEndpoint);
    void        EndpointInitializeTransferRing(Endpoint *pEndpoint);

    // Endpoint primitives
    void EpReload(int epIndex) NN_NOEXCEPT;
    void EpPause(int epIndex) NN_NOEXCEPT;
    void EpUnpause(int epIndex) NN_NOEXCEPT;
    void EpHalt(int epIndex) NN_NOEXCEPT;
    void EpUnhalt(int epIndex) NN_NOEXCEPT;
    void EpClearStop(int epIndex) NN_NOEXCEPT;
    void EpWaitForStopped(int epIndex) NN_NOEXCEPT;
    void EpWaitForInactive(int epIndex) NN_NOEXCEPT;

};

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
